/*
 * postscript.c
 *
 * Report everything on the style data base
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97, 98 Akim Demaille, Miguel Santana
 * $Id: report.c,v 1.2 1998/01/15 15:00:47 demaille Exp $
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "a2ps.h"
#include "styles.h"
#include "report.h"
#include "routines.h"
#include "utilsty.h"
#include "pathwalk.h"

#define DEFAULT_ALPHA1 ((ustring) \
	"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_")
#define DEFAULT_ALPHA2 ((ustring) \
	"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_0123456789")
#define DEFAULT_SENSITIVENESS CASE_INSENSITIVE

extern char *program_name;

static void report_escapes PARAMS ((STYLE lang,
				    const char *prefix, const char *end));


static const_ustring
char_to_string (uchar c)
{
  ustring res;
  res = ALLOC (uchar, 3);

  *res = NUL;

  switch (c)
    {
    case '\n':
      return (const_ustring) "$";
    case '\\':
    case '\"':
    case '\'':
      ustrcat (res, "\\");
    default:
      USTRCCAT (res, c);
    }

  return res;
}

static int
is_reserved (const_ustring string)
{
  static const char *reserved[] =
  {
    "by", "written", "version",
    "---", "\\carriagereturn", "comment", "Current",
    "\\copyright", "\\trademark", "\\register",
    "\\varcopyright", "\\vartrademark", "\\varregister",
    "\\forall", "\\exists", "\\suchthat", "\\cong", "\\Alpha",
    "\\Beta", "\\Chi", "\\Delta", "\\Epsilon", "\\Phi", "\\Gamma",
    "\\Eta", "\\Iota", "\\vartheta", "\\Kappa", "\\Lambda", "\\Mu",
    "\\Nu", "\\Omicron", "\\Pi", "\\Theta", "\\Rho", "\\Sigma", "\\Tau",
    "\\Upsilon", "\\varsigma", "\\Omega", "\\Xi", "\\Psi", "\\Zeta",
    "\\therefore", "\\perp", "\\radicalex", "\\alpha", "\\beta", "\\chi",
    "\\delta", "\\epsilon", "\\phi", "\\gamma", "\\eta", "\\iota",
    "\\varphi", "\\kappa", "\\lambda", "\\mu", "\\nu", "\\omicron",
    "\\pi", "\\theta", "\\rho", "\\sigma", "\\tau", "\\upsilon",
    "\\varpi", "\\omega", "\\xi", "\\psi", "\\zeta", "\\sim",
    "\\varUpsilon", "\\prime", "\\leq", "\\infty", "\\florin",
    "\\clubsuit", "\\diamondsuit", "\\heartsuit", "\\spadesuit",
    "\\leftrightarrow", "\\leftarrow", "\\uparrow", "\\rightarrow",
    "\\downarrow", "\\circ", "\\pm", "\\geq", "\\times", "\\propto",
    "\\partial", "\\bullet", "\\div", "\\neq", "\\equiv", "\\approx",
    "\\ldots", "\\aleph", "\\Im", "\\Re", "\\wp", "\\otimes", "\\oplus",
    "\\emptyset", "\\cap", "\\cup", "\\supset", "\\supseteq",
    "\\not\\subset", "\\subset", "\\subseteq", "\\in", "\\not\\in",
    "\\angle", "\\nabla", "\\register", "\\trademark", "\\prod", "\\surd",
    "\\cdot", "\\not", "\\wedge", "\\vee", "\\Leftrightarrow",
    "\\Leftarrow", "\\Uparrow", "\\Rightarrow", "\\Downarrow",
    "\\vardiamondsuit", "\\langle", "\\varregister", "\\vartrademark",
    "\\sum", "\\lceil", "\\lfloor", "\\rangle", "\\int", "\\rceil",
    "\\rfloor", "style", "is", "in", "end", "first", "second", "alphabet",
    "case", "sensitive", "insensitive", "keywords", "optional",
    "sequences", "operators", "are", "Plain", "Keyword", "Keyword_strong",
    "Comment", "Comment_strong", "Label", "Label_strong", "String",
    "Symbol", "Tag1", "Tag2", "Tag3", "Tag4", "Dec1", "Dec2", "Encoding",
    "exceptions"
  };
  int i;

  for (i = 0; i < (sizeof (reserved) / sizeof (reserved[0])); i++)
    if (ustrequ (string, reserved[i]))
      return TRUE;
  return FALSE;
}

static const_ustring
string_to_string (const_ustring string)
{
  ustring res;

  if (!strpbrk ((char *) string, "#, \t\r\f\n\"\\\'()+/")
      && !is_reserved (string))
    return string;

  res = ALLOC (uchar, 4000);
  ustrcpy (res, "\"");

  for (; *string; string++)
    ustrcat (res, char_to_string (*string));

  ustrcat (res, "\"");
  return res;
}

/* Take priviledged access to job */
extern print_job *job;

/************************************************************************/
/*                              report styles                           */
/************************************************************************/
/*
 *      This is a correspondance between LaTeX chars and font symbol
 *              (extracted by ./getwx)
 */
uchar *Symbol_to_LaTeX[256] =
{
/* .notdef (/0) */ UNULL,
/* .notdef (/1) */ UNULL,
/* .notdef (/2) */ UNULL,
/* .notdef (/3) */ UNULL,
/* .notdef (/4) */ UNULL,
/* .notdef (/5) */ UNULL,
/* .notdef (/6) */ UNULL,
/* .notdef (/7) */ UNULL,
/* .notdef (/8) */ UNULL,
/* .notdef (/9) */ UNULL,
/* .notdef (/10) */ UNULL,
/* .notdef (/11) */ UNULL,
/* .notdef (/12) */ UNULL,
/* .notdef (/13) */ UNULL,
/* .notdef (/14) */ UNULL,
/* .notdef (/15) */ UNULL,
/* .notdef (/16) */ UNULL,
/* .notdef (/17) */ UNULL,
/* .notdef (/18) */ UNULL,
/* .notdef (/19) */ UNULL,
/* .notdef (/20) */ UNULL,
/* .notdef (/21) */ UNULL,
/* .notdef (/22) */ UNULL,
/* .notdef (/23) */ UNULL,
/* .notdef (/24) */ UNULL,
/* .notdef (/25) */ UNULL,
/* .notdef (/26) */ UNULL,
/* .notdef (/27) */ UNULL,
/* .notdef (/28) */ UNULL,
/* .notdef (/29) */ UNULL,
/* .notdef (/30) */ UNULL,
/* .notdef (/31) */ UNULL,
/* space (/32) */ (ustring) " ",
/* exclam (/33) */ (ustring) "!",
/* universal (/34) */ (ustring) "\\forall",
/* numbersign (/35) */ (ustring) "\\#",
/* existential (/36) */ (ustring) "\\exists",
/* percent (/37) */ (ustring) "\\%",
/* ampersand (/38) */ (ustring) "\\&",
/* suchthat (/39) */ (ustring) "\\suchthat",
/* parenleft (/40) */ (ustring) "(",
/* parenright (/41) */ (ustring) ")",
/* asteriskmath (/42) */ (ustring) "*",
/* plus (/43) */ (ustring) "+",
/* comma (/44) */ (ustring) ",",
/* minus (/45) */ (ustring) "-",
/* period (/46) */ (ustring) ".",
/* slash (/47) */ (ustring) "/",
/* zero (/48) */ (ustring) "0",
/* one (/49) */ (ustring) "1",
/* two (/50) */ (ustring) "2",
/* three (/51) */ (ustring) "3",
/* four (/52) */ (ustring) "4",
/* five (/53) */ (ustring) "5",
/* six (/54) */ (ustring) "6",
/* seven (/55) */ (ustring) "7",
/* eight (/56) */ (ustring) "8",
/* nine (/57) */ (ustring) "9",
/* colon (/58) */ (ustring) ":",
/* semicolon (/59) */ (ustring) ";",
/* less (/60) */ (ustring) "<",
/* equal (/61) */ (ustring) "=",
/* greater (/62) */ (ustring) ">",
/* question (/63) */ (ustring) "?",
/* congruent (/64) */ (ustring) "\\cong",
/* Alpha (/65) */ (ustring) "\\Alpha",
/* Beta (/66) */ (ustring) "\\Beta",
/* Chi (/67) */ (ustring) "\\Chi",
/* Delta (/68) */ (ustring) "\\Delta",
/* Epsilon (/69) */ (ustring) "\\Epsilon",
/* Phi (/70) */ (ustring) "\\Phi",
/* Gamma (/71) */ (ustring) "\\Gamma",
/* Eta (/72) */ (ustring) "\\Eta",
/* Iota (/73) */ (ustring) "\\Iota",
/* theta1 (/74) */ (ustring) "\\vartheta",
/* Kappa (/75) */ (ustring) "\\Kappa",
/* Lambda (/76) */ (ustring) "\\Lambda",
/* Mu (/77) */ (ustring) "\\Mu",
/* Nu (/78) */ (ustring) "\\Nu",
/* Omicron (/79) */ (ustring) "\\Omicron",
/* Pi (/80) */ (ustring) "\\Pi",
/* Theta (/81) */ (ustring) "\\Theta",
/* Rho (/82) */ (ustring) "\\Rho",
/* Sigma (/83) */ (ustring) "\\Sigma",
/* Tau (/84) */ (ustring) "\\Tau",
/* Upsilon (/85) */ (ustring) "\\Upsilon",
/* sigma1 (/86) */ (ustring) "\\varsigma",
/* Omega (/87) */ (ustring) "\\Omega",
/* Xi (/88) */ (ustring) "\\Xi",
/* Psi (/89) */ (ustring) "\\Psi",
/* Zeta (/90) */ (ustring) "\\Zeta",
/* bracketleft (/91) */ (ustring) "[",
/* therefore (/92) */ (ustring) "\\therefore",
/* bracketright (/93) */ (ustring) "]",
/* perpendicular (/94) */ (ustring) "\\perp",
/* underscore (/95) */ (ustring) "\\_",
/* radicalex (/96) */ (ustring) "\\radicalex",
/* alpha (/97) */ (ustring) "\\alpha",
/* beta (/98) */ (ustring) "\\beta",
/* chi (/99) */ (ustring) "\\chi",
/* delta (/100) */ (ustring) "\\delta",
/* epsilon (/101) */ (ustring) "\\epsilon",
/* phi (/102) */ (ustring) "\\phi",
/* gamma (/103) */ (ustring) "\\gamma",
/* eta (/104) */ (ustring) "\\eta",
/* iota (/105) */ (ustring) "\\iota",
/* phi1 (/106) */ (ustring) "\\phi",
/* kappa (/107) */ (ustring) "\\kappa",
/* lambda (/108) */ (ustring) "\\lambda",
/* mu (/109) */ (ustring) "\\mu",
/* nu (/110) */ (ustring) "\\nu",
/* omicron (/111) */ (ustring) "\\omicron",
/* pi (/112) */ (ustring) "\\pi",
/* theta (/113) */ (ustring) "\\theta",
/* rho (/114) */ (ustring) "\\rho",
/* sigma (/115) */ (ustring) "\\sigma",
/* tau (/116) */ (ustring) "\\tau",
/* upsilon (/117) */ (ustring) "\\upsilon",
/* omega1 (/118) */ (ustring) "\\varpi",
/* omega (/119) */ (ustring) "\\omega",
/* xi (/120) */ (ustring) "\\xi",
/* psi (/121) */ (ustring) "\\psi",
/* zeta (/122) */ (ustring) "\\zeta",
/* braceleft (/123) */ (ustring) "{",
/* bar (/124) */ (ustring) "|",
/* braceright (/125) */ (ustring) "}",
/* similar (/126) */ (ustring) "\\sim",
/* .notdef (/127) */ UNULL,
/* .notdef (/128) */ UNULL,
/* .notdef (/129) */ UNULL,
/* .notdef (/130) */ UNULL,
/* .notdef (/131) */ UNULL,
/* .notdef (/132) */ UNULL,
/* .notdef (/133) */ UNULL,
/* .notdef (/134) */ UNULL,
/* .notdef (/135) */ UNULL,
/* .notdef (/136) */ UNULL,
/* .notdef (/137) */ UNULL,
/* .notdef (/138) */ UNULL,
/* .notdef (/139) */ UNULL,
/* .notdef (/140) */ UNULL,
/* .notdef (/141) */ UNULL,
/* .notdef (/142) */ UNULL,
/* .notdef (/143) */ UNULL,
/* .notdef (/144) */ UNULL,
/* .notdef (/145) */ UNULL,
/* .notdef (/146) */ UNULL,
/* .notdef (/147) */ UNULL,
/* .notdef (/148) */ UNULL,
/* .notdef (/149) */ UNULL,
/* .notdef (/150) */ UNULL,
/* .notdef (/151) */ UNULL,
/* .notdef (/152) */ UNULL,
/* .notdef (/153) */ UNULL,
/* .notdef (/154) */ UNULL,
/* .notdef (/155) */ UNULL,
/* .notdef (/156) */ UNULL,
/* .notdef (/157) */ UNULL,
/* .notdef (/158) */ UNULL,
/* .notdef (/159) */ UNULL,
/* .notdef (/160) */ UNULL,
/* Upsilon1 (/161) */ (ustring) "\\varUpsilon",
/***/
/* minute (/162) */ (ustring) "\\prime",
/* lessequal (/163) */ (ustring) "\\leq",
/* fraction (/164) */ (ustring) "/",
/* infinity (/165) */ (ustring) "\\infty",
/* florin (/166) */ (ustring) "\\florin",
/****/
/* club (/167) */ (ustring) "\\clubsuit",
/* diamond (/168) */ (ustring) "\\diamondsuit",
/* heart (/169) */ (ustring) "\\heartsuit",
/* spade (/170) */ (ustring) "\\spadesuit",
/* arrowboth (/171) */ (ustring) "\\leftrightarrow",
/* arrowleft (/172) */ (ustring) "\\leftarrow",
/* arrowup (/173) */ (ustring) "\\uparrow",
/* arrowright (/174) */ (ustring) "\\rightarrow",
/* arrowdown (/175) */ (ustring) "\\downarrow",
/* degree (/176) */ (ustring) "\\circ",
/***/
/* plusminus (/177) */ (ustring) "\\pm",
/* second (/178) */ (ustring) "''",
/* greaterequal (/179) */ (ustring) "\\geq",
/* multiply (/180) */ (ustring) "\\times",
/* proportional (/181) */ (ustring) "\\propto",
/* partialdiff (/182) */ (ustring) "\\partial",
/* bullet (/183) */ (ustring) "\\bullet",
/* divide (/184) */ (ustring) "\\div",
/* notequal (/185) */ (ustring) "\\neq",
/* equivalence (/186) */ (ustring) "\\equiv",
/* approxequal (/187) */ (ustring) "\\approx",
/* ellipsis (/188) */ (ustring) "\\ldots",
/* arrowvertex (/189) */ UNULL,
/* arrowhorizex (/190) */ (ustring) "---",
/* carriagereturn (/191) */ (ustring) "\\carriagereturn",
/* aleph (/192) */ (ustring) "\\aleph",
/* Ifraktur (/193) */ (ustring) "\\Im",
/* Rfraktur (/194) */ (ustring) "\\Re",
/* weierstrass (/195) */ (ustring) "\\wp",
/* circlemultiply (/196) */ (ustring) "\\otimes",
/* circleplus (/197) */ (ustring) "\\oplus",
/* emptyset (/198) */ (ustring) "\\emptyset",
/* intersection (/199) */ (ustring) "\\cap",
/* union (/200) */ (ustring) "\\cup",
/* propersuperset (/201) */ (ustring) "\\supset",
/* reflexsuperset (/202) */ (ustring) "\\supseteq",
/* notsubset (/203) */ (ustring) "\\not\\subset",
/* propersubset (/204) */ (ustring) "\\subset",
/* reflexsubset (/205) */ (ustring) "\\subseteq",
/* element (/206) */ (ustring) "\\in",
/* notelement (/207) */ (ustring) "\\not\\in",
/* angle (/208) */ (ustring) "\\angle",
/* gradient (/209) */ (ustring) "\\nabla",
/* registerserif (/210) */ (ustring) "\\register",
/****/
/* copyrightserif (/211) */ (ustring) "\\copyright",
/****/
/* trademarkserif (/212) */ (ustring) "\\trademark",
/****/
/* product (/213) */ (ustring) "\\prod",
/* radical (/214) */ (ustring) "\\surd",
/* dotmath (/215) */ (ustring) "\\cdot",
/* logicalnot (/216) */ (ustring) "\\not",
/* logicaland (/217) */ (ustring) "\\wedge",
/* logicalor (/218) */ (ustring) "\\vee",
/* arrowdblboth (/219) */ (ustring) "\\Leftrightarrow",
/* arrowdblleft (/220) */ (ustring) "\\Leftarrow",
/* arrowdblup (/221) */ (ustring) "\\Uparrow",
/* arrowdblright (/222) */ (ustring) "\\Rightarrow",
/* arrowdbldown (/223) */ (ustring) "\\Downarrow",
/* lozenge (/224) */ (ustring) "\\diamondsuit",
/****/
/* angleleft (/225) */ (ustring) "\\langle",
/* registersans (/226) */ (ustring) "\\register",
/***/
/* copyrightsans (/227) */ (ustring) "\\copyright",
/***/
/* trademarksans (/228) */ (ustring) "\\trademark",
/****/
/* summation (/229) */ (ustring) "\\sum",
/* parenlefttp (/230) */ UNULL,
/* parenleftex (/231) */ UNULL,
/* parenleftbt (/232) */ UNULL,
/* bracketlefttp (/233) */ (ustring) "\\lceil",
/* bracketleftex (/234) */ UNULL,
/* bracketleftbt (/235) */ (ustring) "\\lfloor",
/* bracelefttp (/236) */ UNULL,
/* braceleftmid (/237) */ UNULL,
/* braceleftbt (/238) */ UNULL,
/* braceex (/239) */ UNULL,
/* .notdef (/240) */ (ustring) "\\apple",
/* angleright (/241) */ (ustring) "\\rangle",
/* integral (/242) */ (ustring) "\\int",
/* integraltp (/243) */ UNULL,
/* integralex (/244) */ UNULL,
/* integralbt (/245) */ UNULL,
/* parenrighttp (/246) */ UNULL,
/* parenrightex (/247) */ UNULL,
/* parenrightbt (/248) */ UNULL,
/* bracketrighttp (/249) */ (ustring) "\\rceil",
/* bracketrightex (/250) */ UNULL,
/* bracketrightbt (/251) */ (ustring) "\\rfloor",
/* bracerighttp (/252) */ UNULL,
/* bracerightmid (/253) */ UNULL,
/* bracerightbt (/254) */ UNULL,
  UNULL
};

/*
 * Give the content of an alphabet
 */
static void
report_alphabet (const uchar * alphabet)
{
  uchar letter;

  printf ("   \"");
  while (*alphabet)
    {
      for (letter = *alphabet; letter <= *(alphabet + 1); letter++)
	printf ("%s", char_to_string (letter));
      alphabet += 2;
    }
  printf ("\"\n");
}

/*
 * Give the alphabets of lang
 */
static void
report_alphabets (STYLE lang)
{
  if (!ustrequ (languages[lang].first_characters, DEFAULT_ALPHA1))
    {
      printf ("first alphabet is\n");
      report_alphabet (languages[lang].first_characters);
    }

  if (!ustrequ (languages[lang].following_characters, DEFAULT_ALPHA2))
    {
      printf ("second alphabet is\n");
      report_alphabet (languages[lang].following_characters);
    }
}

/*
 * Give the description note of lang
 */
static void
report_description (STYLE lang)
{
  const uchar *desc = languages[lang].description;

  printf ("version 1.0\n");

  if (*desc != NUL)
    {
      printf ("comment is\n%s\nend comment\n", string_to_string (desc));
    }
}

/*
 * Give the abbreviations admited of lang
 */
static void
report_abbreviations (STYLE lang)
{
  const char **abbr = languages[lang].abbreviations;

  printf ("# %s files\n", languages[lang].name);
  for (abbr = languages[lang].abbreviations; **abbr; abbr++)
    {
      printf ("*.%s\t\t\t%s\n", *abbr, languages[lang].name);
    }
  printf ("\n");
}

/*
 * Give the case sensitivity of lang
 */
static void
report_sensitivity (STYLE lang)
{
  if (languages[lang].sensitiveness != DEFAULT_SENSITIVENESS)
    printf ("case %s\n",
	    ((languages[lang].sensitiveness == CASE_SENSITIVE)
	     ? "sensitive" : "insensitive"));
}

/*
 * Give the keywords of lang
 */
static void
report_keywords (STYLE lang)
{
  const keyword *array = languages[lang].keywords;
  face_t face;

  while (!IS_EMPTY (array->theKeyword))
    {
      face = array->theFont;
      printf ("keywords in %s are\n   ", xface_to_string (face));
      while (!IS_EMPTY ((array + 1)->theKeyword)
	     && (array + 1)->theFont == face)
	printf ("%s, ", string_to_string ((array++)->theKeyword));
      printf ("%s\n", string_to_string ((array)->theKeyword));
      printf ("end keywords\n");
      array++;
    }
}

/*
 * Give the symbols of lang
 */
static void
report_sym (STYLE lang, const symbol * array, const char *what)
{
  int count = 0;

  if (array->theKeyword)
    {
      if (lang != prescript_style)
	printf ("optional ");
      printf ("%s are\n   ", what);
      while (!IS_EMPTY (array->theKeyword))
	{
	  /* the original word */
	  printf ("%s", string_to_string (array->theKeyword));
	  if (array->theFont == SYMBOL)
	    {
	      ustring string = array->theSymbol;
	      for ( /*skip */ ; *string; string++)
		if (Symbol_to_LaTeX[*string])
		  if (ustrequ (Symbol_to_LaTeX[*string], ")"))
		    printf (" \")\"");
		  else if (ustrequ (Symbol_to_LaTeX[*string], "("))
		    printf (" \"(\"");
		  else
		    printf (" %s", Symbol_to_LaTeX[*string]);
		else
		  printf (" %c", *string);
	    }
	  else if (ustrequ (array->theKeyword, array->theSymbol)
		   && (array->theFont == PLAIN))
	    {
	      /* Nothing */ ;
	    }
	  else if (array->theFont == PLAIN)
	    printf (" %s", string_to_string (array->theSymbol));
	  else
	    printf (" %s %s", string_to_string (array->theSymbol),
		    xface_to_string (array->theFont));

	  printf ("%s", IS_EMPTY ((array + 1)->theKeyword) ? "\n" : ",\n   ");
	  array++;
	  count++;
	}
      printf ("end %s\n", what);
    }
}

/*
 * Give the symbols of lang
 */
static void
report_symbols (STYLE lang)
{
  report_sym (lang, languages[lang].regulars, "keywords");
}

/*
 * Give the specials of lang
 */
static void
report_specials (STYLE lang)
{
  report_sym (lang, languages[lang].specials, "operators");
}

/*
 * Give the sequences of lang
 */
static void
report_sequence (STYLE lang, const sequence * seq)
{
  /* Is it a char sequence ? */
  if (ustrequ (seq->theOpening, "\'")
      && ustrequ (seq->theClosing, "\'")
      && languages[lang].verbatims == languages[c_style].verbatims)
    printf ("    C-char");
  /* Is it a C string seq ? */
  else if (ustrequ (seq->theOpening, "\"")
	   && ustrequ (seq->theClosing, "\"")
	   && languages[lang].verbatims == languages[c_style].verbatims)
    printf ("    C-string");
  /* Is it an up to the end of line? */
  else if (ustrequ (string_to_string (seq->theClosing), "$")
	   && (seq->theOpeningFont == seq->theInsideFont))
    printf ("    %s %s", string_to_string (seq->theOpening),
	    xface_to_string (seq->theOpeningFont));
  /* Is it a char seq? */
  else if ((ustrequ (seq->theOpening, "\'")
	    && ustrequ (seq->theClosing, "\'"))
	   || (ustrequ (seq->theOpening, "\"")
	       && ustrequ (seq->theClosing, "\""))
	   || (ustrequ (seq->theOpening, "(")
	       && ustrequ (seq->theClosing, ")")
	       && lang == postscript_style))
    {
      printf ("    %s %s %s %s %s\n",
	      string_to_string (seq->theOpening),
	      xface_to_string (seq->theOpeningFont),
	      xface_to_string (seq->theInsideFont),
	      string_to_string (seq->theClosing),
	      xface_to_string (seq->theClosingFont));
      report_escapes (lang, "       ", "");
    }
  else
    printf ("    %s %s %s %s %s",
	    string_to_string (seq->theOpening),
	    xface_to_string (seq->theOpeningFont),
	    xface_to_string (seq->theInsideFont),
	    string_to_string (seq->theClosing),
	    xface_to_string (seq->theClosingFont));
}

static void
report_sequences (STYLE lang)
{
  const sequence *array = languages[lang].sequences;

  if (!IS_EMPTY (array->theOpening))
    {
      printf ("sequences are\n");
      while (!IS_EMPTY (array->theOpening))
	{
	  report_sequence (lang, array);
	  printf ("%s", IS_EMPTY ((array + 1)->theOpening) ? "\n" : ",\n");
	  array++;
	}
      printf ("end sequences\n");
    }
}

/*
 * Give the verbatims of lang
 */
static void
report_verbatims (STYLE lang)
{
  const uchar **array = languages[lang].verbatims;

  if (array && !IS_EMPTY (*array) && array != languages[c_style].verbatims)
    {
      fprintf (stderr, "Verbatims %s: %p\n", languages[lang].name,
	       languages[lang].escapes);
      printf ("exceptions are\n   ");
      while (!IS_EMPTY (*array))
	{
	  printf ("%s", string_to_string (*array));
	  printf ("%s",
		  *(array + 1) ? ", " : "\n");
	  array++;
	}
      printf ("end exceptions\n");
    }
}

/*
 * Give the escapes of lang
 */
static void
report_escapes (STYLE lang, const char *prefix, const char *end)
{
  const uchar **array = languages[lang].escapes;

  if (array && !IS_EMPTY (*array))
    {
      fprintf (stderr, "Escapes %s: %p\n", languages[lang].name,
	       languages[lang].escapes);
      printf ("%sexceptions are\n   %s", prefix, prefix);
      while (!IS_EMPTY (*array))
	{
	  printf ("%s", string_to_string (*array));
	  printf ("%s",
		  *(array + 1) ? ", " : "\n");
	  array++;
	}
      printf ("%send exceptions%s", prefix, end);
    }
}

/*
 * Report about a language 
 */
static void
report_language (STYLE lang)
{
  printf ("style %s is\n", languages[lang].name);
  report_description (lang);
  report_alphabets (lang);
  report_sensitivity (lang);
  report_keywords (lang);
  report_symbols (lang);
  report_specials (lang);
  report_sequences (lang);
  report_verbatims (lang);
/*  report_escapes(lang, "", "\n"); */
  printf ("end style\n");
}

/************************************************************************/
/*                              global functions                        */
/************************************************************************/
/*
 * The whole report about styles
 */
void
report (void)
{
  STYLE lang;
  static STYLE don_t_refresh[] =
  {
    fortran_style, perl_style
  };

  /* Build the sheets.map.proto */
  freopen ("sheets.map.proto", "w", stdout);
  for (lang = 0; lang < nbr_languages; lang++)
    {
      report_abbreviations (lang);
    }

  for (lang = 0; lang < nbr_languages; lang++)
    {
      char buf[256];
      int dont;
      int noooo = 0;
      fprintf (stderr, "%s\n", languages[lang].name);
      for (dont = 0; dont < (sizeof (don_t_refresh) / sizeof (STYLE)); dont++)
	if (don_t_refresh[dont] == lang)
	  noooo = 1;

      if (noooo)
	sprintf (buf, "%s.ssh.new", languages[lang].name);
      else
	sprintf (buf, "%s.ssh", languages[lang].name);
      freopen (buf, "w", stdout);
      report_language (lang);
    }

  exit (EXIT_SUCCESS);
}
