#include <netdb.h> /* for herr_str() only */
#include <stdio.h>
#include <string.h>
#ifdef __STDC__
# include <stdlib.h>
#else
# include <malloc.h>
#endif
#include "wutils.h"
#include "wdefs.h"

/*  
 *
 *
 *                             Internal routines
 *
 *
 */  


/*  
 *  If `r' is NULL return a count of characters needed to hold the resulting
 *  nul terminated string.  If `s' is not a valid string return 0.  In either
 *  case set `*end' to point to the character that caused scanning to stop.
 *  
 *  If `r' is non-NULL it is assumed to point to sufficient storage to
 *  contain a "dequoted", nul terminated version of `s'.
 */  
static int strfsm(s, r, end)
  const char *s;
  char *r;
  const char **end;
{
  typedef enum { END, ERROR, IN, SLASH, START } States;
  States state = START;
  int count = 0;

  while (1)
  {
    switch (state)
    {
    case END: *end = s ; return r ? *r = '\0', 1 : count+1;
    case ERROR: *end = s; return 0;
    case IN:
      switch (*s)
      {
      case '"':  state = END;   s++; break;
      case '\\': state = SLASH; s++; break;
      case '\0': state = ERROR; break;
      default:   state = IN; r ? *r++ = *s++ : (s++, count++); break;
      }
      break;
    case SLASH:
      switch (*s)
      {
      case '\0': state = ERROR; break;
      default:   state = IN; r ? *r++ = *s++ : (s++, count++); break;
      }
      break;
    case START:
      switch (*s++)
      {
      case '"': state = IN; break;
      default:  state = ERROR; break;
      }
      break;
    }
  }
}


char *strndup(s, len)
  char *s;
  int len;
{
  char *p = malloc(len + 1);

  strncpy(p, s, len);
  *(p + len) = '\0';
  return p;
}


/*  
 *
 *
 *                             External routines
 *
 *
 */  

const char *herr_str()
{
#ifndef HOST_NOT_FOUND
  return "unknown host?";
#else
  switch (h_errno)
  {
  case HOST_NOT_FOUND: return "authoritive answer: host not found";
  case TRY_AGAIN: return "non-authoritive answer: host not found (or SERVERFAIL)";
  case NO_RECOVERY: return "non recoverable error";
  case NO_DATA: return "valid name, but no data record";
  default:
    return "unknown host?";
  }
#endif
}


static int wdebug_ = 1;


int wGetDebug()
{
  return wdebug_;
}


int wSetDebug(lev)
  int lev;
{
  int olev = wdebug_;

  wdebug_ = lev;
  return olev;
}


/*  
 *  Return non-zero if `c' occurs in the string `s'; 0 otherwie (including if
 *  c is '\0').
 */  
int char_in(c, s)
  int c;
  const char *s;
{
  while (*s)
  {
    if (c == *s++) return c ;
  }
  return 0;
}


/*  
 *  Assume `s' is a nul terminated string.
 *  
 *  Return a pointer to the nul character.
 */  
char *strend(s)
  char *s;
{
  while (*s++);
  return s-1;
}


const char *wskip(s)
  const char *s;
{
  return s + strspn(s, WHITE_SPACE);
}


/*  
 *  A name is a sequence of characters matching [A-Za-z][A-Za-z0-9-]*
 */  
char *get_name(s, end)
  const char *s;
  const char **end;
{
#define is_digit(c) \
   ((c) >= '0' && (c) <= '9')
#define is_name_char(c) \
  (((c) >= 'A' && (c) <= 'Z') || \
   ((c) >= 'a' && (c) <= 'z') || \
   is_digit(c) || \
   (c) == '-')

  char c;
  const char *e;
  const char *p;

  p = e = wskip(s);
  if ( ! (is_name_char(*p) || is_digit(*p) || (*p == '-')))
  {
    d2fprintf(stderr, "get_name: failed with [%s].\n", s);
    *end = e;
    return (char *)0;
  }

  while (c = *++e, is_name_char(c)) /* empty */;
  *end = e;
  return strndup(p, e - p);

#undef is_digit
#undef is_name_char
}


char *get_number(s, end)
  const char *s;
  const char **end;
{
#define is_digit(c) \
   ((c) >= '0' && (c) <= '9')

   const char *p;

   p = s = wskip(s);
   while (is_digit(*p))
   {
     p++;
   }
   *end = p;
   return p == s ? (char *)0 : strndup(s, p - s);
}

/*  
 *  Expect a string of the form `"<stuff>"', where <stuff> is a sequence of
 *  characters, with the restrictions:
 *  
 *  - double quotes are preceded by an odd number of backslashes
 *  
 *  - if a backslash is the last character it must be preceded by an odd
 *    number of backslashes.
 */    
char *dequote(s, end)
  const char *s;
  const char **end;
{
  int space;

  if ( ! (space = strfsm(s, (char *)0, end)))
  {
    d2fprintf(stderr, "dequote: strfsm() (counting pass) failed on [%s].\n", s);
    return (char *)0;
  }
  else
  {
    char *t = malloc(space);

    if (t)
    {
      if ( ! strfsm(s, t, end))
      {
        d2fprintf(stderr, "dequote: strfsm() (copying pass) failed on [%s].\n", s);
        return (char *)0;
      }
    }
    return t;
  }
}


/*  
 *  A string must have one of two forms: that of a name (i.e. recognized by
 *  get_name(), or a quoted string.
 */    
char *get_value(s, end)
  const char *s;
  const char **end;
{
  const char *p;

  p = wskip(s);
  if (*p != '"') return get_name(p, end);
  else return dequote(p, end);
}


const char *getNameStr(s, lhs, rhs, end)
  const char *s;
  char **lhs;
  char **rhs;
  const char **end;
{
  const char *e;
  char *l;  /* temp for lhs */
  const char *p;
  char *r;  /* temp for rhs */

  p = wskip(s);
  if ( ! (l = get_name(p, &e)))
  {
    *end = e;
    return (char *)0;
  }
  p = wskip(e);
  if ( ! (r = get_value(p, &e)))
  {
    free(l);
    *end = e;
    return (char *)0;
  }
  *lhs = l; *rhs = r; *end = e;
  return s;
}


/*  
 *  Name = String
 *  
 *  If the text in `s' consists of "<name>=<string>", allocate space for both
 *  <name> and <string>, returning them in `*lhs' and `*rhs' respectively.
 *  
 *  If the correct sequence of characters is not found, return NULL.  In
 *  either case `*end' is set to point to the character that caused the
 *  searching to stop.
 */  
const char *getNEqS(s, lhs, rhs, end)
  const char *s;
  char **lhs;
  char **rhs;
  const char **end;
{
  const char *e;
  char *l;  /* temp for lhs */
  const char *p;
  char *r;  /* temp for rhs */

  p = wskip(s);
  if ( ! (l = get_name(p, &e)))
  {
    *end = e;
    d2fprintf(stderr, "getNEqS: failed to get name in [%s].\n", p);
    return (char *)0;
  }
  p = wskip(e);
  if (*p != WEQUAL)
  {
    free(l);
    *end = p;
    d2fprintf(stderr, "getNEqS: failed to find `=' first in [%s].\n", p);
    return (const char *)0;
  }
  p = wskip(p+1);
  if ( ! (r = get_value(p, &e)))
  {
    free(l);
    *end = e;
    d2fprintf(stderr, "getNEqS: failed to find value in [%s].\n", p);
    return (char *)0;
  }
  *lhs = l; *rhs = r; *end = e;
  return s;
}


const char *getStr(lst, v)
  const StrVal *lst;
  Val v;
{
  while (lst->str)
  {
    if (v == lst->val) break;
    lst++;
  }
  return lst->str;
}


Val getVal(lst, s)
  const StrVal *lst;
  const char *s;
{
  while (lst->str)
  {
    if (strcasecmp(s, lst->str) == 0) break;
    lst++;
  }
  return lst->val;
}


void wFreeList(w)
  WItem *w;
{
  if (w)
  {
    if (w->cons) wFreeList(w->cons);
    if (w->next) wFreeList(w->next);
    if (w->lhs) free(w->lhs);
    if (w->rhs) free(w->rhs);
    free(w);
  }
}


WItem *wNewWItem()
{
  WItem *w = (WItem *)malloc(sizeof *w);

  if (w)
  {
    w->type = w->format = NO_VAL;
    w->lhs = w->rhs = (const char *)0;
    w->cons = w->next = (WItem *)0;
  }
  return w;
}

