#ifdef __STDC__
# include <memory.h>
# include <stdlib.h>
#else
# include <malloc.h>
#endif
#include <errno.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "wquery.h"
#include "wdefs.h"
#include "wutils.h"


extern int sys_nerr;
extern char *sys_errlist[];


static WItem *constraints_ proto_((/*WItem *h,*/ const char *s, const char **end));
static WItem *general_search_ proto_((WItem *h, const char *s, const char **end));
static WItem *parse_as proto_((WItem *(func)(), Val type, const char *s, const char **end));
static WItem *parse_as proto_((WItem *(* func)(), Val type, const char *s, const char **end));
static WItem *search_list_ proto_((WItem *h, const char *s, const char **end));
static WItem *search_pair proto_((const char *s, const char **end));
static WItem *short_search proto_((const char *s, const char **end));
static WItem *system_command_ proto_((WItem *h, const char *s, const char **end));

command_set_t command_set[] = {
   { C_HELP, "HELP", "Get help on a topic"},
   { C_DESCRIBE, "DESCRIBE", "Describe something??"},
   { C_COMMANDS, "COMMANDS", "Get this list"},
   { C_CONSTRAINTS, "CONSTRAINTS", "Get list of supported constraints"},
   { C_LIST, "LIST", "List supported templates describe"},
   { C_VERSION, "VERSION", "Version of this server"},
   { C_LAST, ""}
};


/*  
 *
 *
 *                              Internal routines
 *
 *
 */  

/*  
 *  Look for a list of constraints.
 *  
 *  Return 1 or 0 depending on success.
 *  
 *  In either case set `*end' to point to the character that caused scanning
 *  to stop.
 */  
static WItem *constraints_(/*h,*/ s, end)
/*  WItem *h;*/
  const char *s;
  const char **end;
{
  char *lhs;
  char *rhs;

  if ( ! getNEqS(s, &lhs, &rhs, end))
  {
    d2fprintf(stderr, "constraints: getNEqS() failed on [%s].\n", s);
    return (WItem *)0;
  }
  else
  {
    WItem *w = wNewWItem();
    const char *p;

    if ( ! w)
    {
      free(lhs); free(rhs);
      return (WItem *)0;
    }
    else
    {
      w->type = T_CONS;
      w->lhs = lhs;
      w->rhs = rhs;

      p = wskip(*end);
      if (*p != CONS_SEP)
      {
        *end = p;
        return w;
      }
      else
      {
        w->next = constraints_(/*h,*/ p+1, end);
        if (w->next)
        {
          return w;
        }
        else
        {
          wFreeList(w);
          return (WItem *)0;
        }
      }
    }
  }
}


static WItem *general_search_(h, s, end)
  WItem *h;
  const char *s;
  const char **end;
{
  WItem *w = wNewWItem();

  if (w)
  {
    char *p = strdup(s);

    if ( ! p)
    {
      wFreeList(w);
      return (WItem *)0;
    }
    else
    {
      w->type = T_SEARCH;
      w->format = F_ALL;
      w->lhs = p;
    }
  }
  return w;
}


static WItem *system_command_(h, s, end)
  WItem *h;
  const char *s;
  const char **end;
{
#define SYS_TERM ":"

  WItem *t;
  const char *e;
  char *name;
  const char *p;

  p = wskip(s);

#if 0
  if(p[0] != CMD_START)
    return (WItem *)0;
  else
    p++;
#endif

  if( ! (name = get_name(p, &e))) return (WItem *)0;

  /*  
   *  Check that a valid character terminates the command name.
   *  e.g. disallow '}help,zot'
   */  
  if ( ! char_in(*e, CMD_END) && *e != '\0')
  {
    d2fprintf(stderr, "system_command: bad command-argument separator [%c].\n", *e);
    return (WItem *)0;
  }

  t = wNewWItem();
  if (t)
  {
    int i;

    for(i = 0; i < C_LAST; i++){
      if(strcasecmp(command_set[i].c_name, name) == 0){
        t->type = T_SYSTEM;
        t->lhs = name;
	break;
      }
    }
    if(i == C_LAST){
      wFreeList(t);
      return (WItem *)0;
    }
  }
  else
  {
    free(name);
    return (WItem *)0;
  }

  p = wskip(e);
  if (*p == '\0')
  {
    return t;
  }
  else if (*p != GCONS_START) /* it was white space followd by an argument */
  {
    const char *a = get_value(p, &e);

    if (a)
    {
      t->rhs = a;
    }
    else
    {
      wFreeList(t);
      return (WItem *)0;
    }
  }

  /* what's next ought to be \0 or a global constraint separator */
  p = wskip(e);
  if (*p == '\0')
  {
    return t;
  }
  else if (*p == GCONS_START)
  {
    WItem *c = constraints_(/*h,*/ p+1, &e);

    if (c)
    {
      h->cons = c;
      return t;
    }
    else
    {
      wFreeList(t);
      return (WItem *)0;
    }
  }
  else
  {
    d2fprintf(stderr, "system_command: improper command ending in [%s].\n", s);
    wFreeList(t);
    return (WItem *)0;
  }
}


static WItem *parse_as(func, type, s, end)
  WItem *(* func)();
  Val type;
  const char *s;
  const char **end;
{
  WItem *w = wNewWItem();

  if (w)
  {
    WItem *l = func(w, s, end);

    if ( ! l)
    {
      wFreeList(w);
      w = (WItem *)0;
    }
    else
    {
      w->type = type;
      w->format = F_HEAD;
      w->next = l;
    }
  }
  return w;
}

static WItem *single_constraints(w, p, end)
  WItem *w;			       
  const char *p;
  const char **end;
{
  if (*p == LCONS_START)
  {
    w->cons = constraints_(/*h,*/ p+1, end);
    if ( ! w->cons)
    {
      wFreeList(w);
      return (WItem *)0;
    }
    return w;
  }
  else
    return (WItem *)0;
}


static WItem *search_single(s, end)
  const char *s;
  const char **end;
{
  char *p;
  char *x;
  WItem *w = wNewWItem();
  WItem *t = wNewWItem();

  p = wskip(s);

  if((*end[0] == '\0') || (x = strchr(GEN_SRCH_EXCL, *end[0])))
  {

    w->type = t->type = T_SEARCH;
    w->format = F_VALUE;
    w->lhs = strndup(p, *end - p);
    w->rhs = strdup("");

    t->format = F_ATTR;
    t->lhs = strndup(p, *end - p);
    t->rhs = strdup("");

    w -> next = t;   

    p = wskip(*end);
    *end = p;

    single_constraints(w, p, end);

    return w;
  }
  else if(!x){

    /* we stopped at a non exclusion character */

    if(!(x = strpbrk(p, GEN_SRCH_EXCL))){

      /* Last string ? */

      x = p + strlen(p);
    }

    if(*end[0] == QUOTE){
      char *y;

      /* quoted string ? */

      if(y = dequote(p, &x)){

	/* quoted string */

	w->type = T_SEARCH;
	w->format = F_VALUE;
	w->lhs = y;
	w->rhs = strdup("");

	t->type = T_SEARCH;
	t->format = F_ATTR;
	t->lhs = y;
	t->rhs = strdup("");

	w -> next = t;

	*end = x;
	p = wskip(*end);
	*end = p;
	single_constraints(w, p, end);

	t-> cons = w -> cons;
	return w;
      }
      else{
	/* Bad quoting */

        d2fprintf(stderr, "search_single: improper quoting in [%s].\n", p);
	free(y);
	wFreeList(w);
	wFreeList(t);
	return (WItem *)0;      
      }
    }
    else{

      *end = x;
      w->type = T_SEARCH;
      w->format = F_VALUE;
      w->lhs = strndup(p, *end - p);
      w->rhs = strdup("");

      t->type = T_SEARCH;
      t->format = F_ATTR;
      t->lhs = strndup(p, *end - p);
      t->rhs = strdup("");

      w->next = t;

      p = wskip(*end);
      *end = p;

      single_constraints(w, p, end);

      t-> cons = w -> cons;

      return w;

    }
  }
  else{
    p = wskip(*end);
    *end = p;
    single_constraints(w, p, end);
    t-> cons = w -> cons;
    return w;
  }
}

/*  
 *  We must deal with three cases:
 *  
 *  1) foo=zot -- where foo is the name of an attribute (e.g. NAME) and zot
 *                is a value of name for which to look.
 *  
 *  2) foo=zot -- where foo is the name of a specifier (e.g. TEMPLATE or
 *                ATTRIBUTE) and zot is the name of an attribute (e.g. NAME).
 *
 *  3) foo=zot -- where foo is neither an attribute, nor a specifier; in
 *                which case we [ask Peter!] /bug/
 */  
static WItem *search_pair(s, end)
  const char *s;
  const char **end;
{
  char *lhs;
  char *rhs;
  const char *p;

  p = wskip(s);
  if ( ! getNEqS(p, &lhs, &rhs, end))
  {
    WItem *w = wNewWItem();

    d2fprintf(stderr, "search_pair: getNEqS() failed on [%s].\n", p);

    w = search_single(p, end);

    if( !w){
       wFreeList(w);

       d2fprintf(stderr, "search_pair: search_single() failed on [%s].\n", p);
       return (WItem *)0;
    }
    else
      return w;
  }
  else
  {
    static StrVal specs[] =
    {
      { "attribute", F_ATTR },
      { "handle",    F_HAND },
      { "template",  F_TEMPL },
      { "value",     F_VALUE },

      { (const char *)0, NO_VAL }
    };
    Val sval;
    WItem *w = wNewWItem();

    if ( ! w)
    {
      free(lhs); free(rhs);
      return (WItem *)0;
    }
    else
    {
      w->type = T_SEARCH;
      sval = getVal(specs, lhs);
      if (sval == NO_VAL)
      {
        w->format = F_ATTR_VAL;
        w->lhs = lhs;
        w->rhs = rhs;
      }
      else
      {
        /*  
         *  If we receive something that could have been specified using the
         *  short form, then store it in the short form.  I.e.  the value is
         *  put in `w->lhs' and `w->rhs' remains NULL.
         */  
        w->format = sval;
        w->lhs = rhs;
        free(lhs);
      }

      p = wskip(*end);
      *end = p;
      if (*p == LCONS_START)
      {
        w->cons = constraints_(/*h,*/ p+1, end);
        if ( ! w->cons)
        {
          wFreeList(w);
          return (WItem *)0;
        }
      }

      return w;
    }
  }
}


static WItem *short_search(s, end)
  const char *s;
  const char **end;
{
  char *name;
  const char *p;
  Val format;

  switch (*s)
  {
  case SRCH_ALL:    format = F_ALL;   break;
  case SRCH_ATTR:   format = F_ATTR;  break;
  case SRCH_HANDLE: format = F_HAND;  break;
  case SRCH_TEMPL:  format = F_TEMPL; break;
  case SRCH_VALUE:  format = F_VALUE; break;

  default: return (WItem *)0;
  }
  
  p = wskip(s+1);
  if ( ! (name = get_name(p, end)))
  {
    d2fprintf(stderr, "short_search: get_name() failed on [%s].\n", p);
    return (WItem *)0;
  }
  else
  {
    WItem *w = wNewWItem();

    if (w)
    {
      w->type = T_SEARCH;
      w->format = format;
      w->lhs = name;

      d3fprintf(stderr, "short_search: name is [%s].\n", name);

      p = wskip(*end);
      *end = p;
      if (*p == LCONS_START)
      {
        w->cons = constraints_(/*h,*/ p+1, end);
        if ( ! w->cons)
        {
          wFreeList(w);
          w = (WItem *)0;
        }
      }
    }
    return w;
  }
}


/*  
 *  Given a probable search list, parse it into separate entries
 *  corresponding to each search term.
 *  
 *  `s' is a pointer to the search string to be parsed.  `w' is a pointer to
 *  the head of a linked list of structures, each of which corresponds to one
 *  search term.
 *  
 *  If the string is successfully parsed we return a non-NULL pointer, and
 *  the list of structures attached to `w'.  If unsuccessful in parsing we
 *  return a NULL pointer.
 *  
 *  The calling routine is responsible for disposing of `w'.
 */  
static WItem *search_list_(h, s, end)
  WItem *h;
  const char *s;
  const char **end;
{
  WItem *t;
  const char *e;
  const char *p;

  p = wskip(s);
  t = char_in(*p, SHORT_CHARS) ? short_search(p, &e) : search_pair(p, &e);

  if ( ! t)
  {
    return (WItem *)0;
  }
  else
  {
    p = wskip(e);
    switch (*p)
    {
    case SRCH_SEP:
      p = wskip(p+1);
      t->next = search_list_(h, p, end);
      if (t->next)
      {
        return t;
      }
      else
      {
        wFreeList(t);
        return (WItem *)0;
      }
      break;

    case GCONS_START:
      h->cons = constraints_(/*h,*/ p+1, &e);
      if ( ! h->cons)
      {
        return (WItem *)0;
      }
      else
      {
        p = wskip(e);
        if (*p == '\0')
        {
          return t;
        }
        else
        {
          wFreeList(t);
          return (WItem *)0;
        }
      }
      break;

    case '\0':
      return t;

    default:
      return (WItem *)0;
    }
  }
}


int wQueryCons_(w, r)
  WItem *w;
  char *r;
{
  int count = 1;                /* default return value if r is non-NULL */

  if ( ! r)
  {
    count = strlen(w->lhs) + 1 + 1 + strlen(w->rhs) + 1; /* add quotes around rhs */
  }
  else
  {
    strcat(r, w->lhs);
    r = strend(r);
    *r++ = WEQUAL;
    *r++ = QUOTE;
    strcat(r, w->rhs);
    r = strend(r);
    *r++ = QUOTE;
  }

  return count;
}

  
int wQuerySrch_(w, r)
  WItem *w;
  char *r;
{
  int count = 1;                /* default return value if r is non-NULL */

  if (w->format == F_ATTR_VAL)
  {
    if ( ! r)
    {
      count = strlen(w->lhs) + 1 + 1 + strlen(w->rhs) + 1; /* add quotes around rhs */
    }
    else
    {
      strcat(r, w->lhs);
      r = strend(r);
      *r++ = WEQUAL;
      *r++ = QUOTE;
      strcat(r, w->rhs);
      r = strend(r);
      *r++ = QUOTE;
    }
  }
  else
  {
    static StrVal spectxt[] =
    {
      { "attribute", F_ATTR },
      { "handle", F_HAND },
      { "template", F_TEMPL },
      { "value", F_VALUE },

      { (const char *)0, NO_VAL }
    };
    const char *txt;

    txt = getStr(spectxt, w->format);
    if ( ! txt)
    {
      /*  
       *  We assume it's a general query.  First check the RHS, then the LHS
       *  for the query string.
       */      
      if ( ! r)
      {
        count = 1 + strlen(w->rhs ? w->rhs : w->lhs);
      }
      else
      {
        *r++ = SRCH_ALL;
        strcat(r, w->rhs ? w->rhs : w->lhs);
      }
    }
    else
    {
      int txtlen = strlen(txt);

      /*  
       *  We accept items either in the long form (e.g. LHS=template,
       *  RHS=foo) or in the short form (e.g. LHS=foo, RHS is NULL).  We also
       *  put quotes around the RHS.
       */    
      if ( ! r)
      {
        count = txtlen + 1 + strlen(w->rhs ? w->rhs : w->lhs);
      }
      else
      {
        strcat(r, txt);
        r = strend(r);
        *r++ = WEQUAL;
        *r++ = QUOTE;
        strcat(r, w->rhs ? w->rhs : w->lhs);
        r = strend(r);
        *r++ = QUOTE;
      }
    }
  }

  return count;
}

  
int wQuerySys_(w, r)
  WItem *w;
  char *r;
{
  int count = 1;                /* default return value if r is non-NULL */

  if ( ! r)
  {
    count = strlen(w->lhs);
    if (w->rhs) count += strlen(w->rhs);
    return count;
  }
  else
  {
    strcat(r, w->lhs);
    if (w->rhs)
    {
      strcat(r, " ");
      strcat(r, w->rhs);
    }
  }

  return count;
}

  
int wQueryStr_(w, r)
  WItem *w;
  char *r;
{
  WItem *t = w;
  int count = 0;

  if (t)
  {
    if (t->format == F_HEAD)
    {
      t = t->next;
      if (t)
      {
        do
        {
          switch (t->type)
          {
          case T_SEARCH:
            if ( ! r)
            {
              count += wQuerySrch_(t, r) + 1; /* ... + 1; room for search term separator */
            }
            else
            {
              wQuerySrch_(t, r);
              r = strend(r);
            }

            /*  
             *  check local constraints
             */  
            if (t->cons && t->cons->next)
            {
              WItem *c = t->cons->next;

              do
              {
                if ( ! r)
                {
                  count += 1 + wQueryCons_(c, r);
                }
                else
                {
                  *r++ = CONS_SEP; /* would be bug if CONS_SEP != LCONS_START */
                  wQueryCons_(c, r);
                  r = strend(r);
                }
                c = c->next;
              }
              while (c);
            }
            if (r && t->next) *r++ = SRCH_SEP; /* cheating? */
            break;

          case T_SYSTEM:
            if ( ! r)
            {
              count += wQuerySys_(t, r);
            }
            else
            {
              wQuerySys_(t, r);
              r = strend(r);
            }
            break;

          default:
            goto error;
            break;
          }

          t = t->next;
        }
        while (t);

        /*  
         *  check global constraints
         */  
        if (w->cons && w->cons->next)
        {
          WItem *c = w->cons->next;

          if (r)                /* count increment taken care of in loop */
          {
            *r++ = GCONS_START;
          }

          do
          {
            if ( ! r)
            {
              count += 1 + wQueryCons_(c, r);
            }
            else
            {
              wQueryCons_(c, r);
              r = strend(r);
              if (c->next) *r++ = CONS_SEP;
            }
            c = c->next;
          }
          while (c);
        }
      }

      return r ? 1 : count;
    }
  }

 error:
  if (r) free(r);
  return 0;
}


/*  
 *
 *
 *                              External routines
 *
 *
 */  


void wPrintList(fp, w)
  FILE *fp;
  const WItem *w;
{
  static StrVal types[] =
  {
    /* WHOIS++ Item Types */

    { "constraint", T_CONS   },
    { "search",     T_SEARCH },
    { "system",     T_SYSTEM },
    { "reply",      T_REPLY  },

    { "no_value",   NO_VAL   }
  };

  static StrVal formats[] =
  {
    /* WItem Query Formats */

    { "all",             F_ALL      },
    { "attr",            F_ATTR     },
    { "value",           F_VALUE    },
    { "template",        F_TEMPL    },
    { "handle",          F_HAND     },
    { "head",            F_HEAD     },
    { "attribute-value", F_ATTR_VAL },

    /* WItem Response Formats */

    { "abridged",    R_ABRIDGED },
    { "full",        R_FULL     },
    { "full_format", R_FULL_FMT },
    { "handle",      R_HANDLE   },
    { "mime",        R_MIME     },
    { "point",       R_POINT    },
    { "summary",     R_SUM      },

    { "no_value",    NO_VAL     }
  };

  if (w)
  {
    const char *t = getStr(types, w->type);
    const char *f = getStr(formats, w->format);

    fprintf(fp, "Type: %s, Format: %s\n", t ? t : "<?>", f ? f : "<?>");
    fprintf(fp, "LHS = `%s'\nRHS = `%s'\n", w->lhs ? w->lhs : "", w->rhs ? w->rhs : "");
    if (w->cons)
    {
      fprintf(fp, "Constraints:\n");
      wPrintList(fp, w->cons);
    }
    wPrintList(fp, w->next);
  }
}


int wPutLine(fp, s)
  FILE *fp;
  const char *s;
{
  int r = fputs(s, fp) != EOF && fputs(CRLF, fp) != EOF;
  fflush(fp);
  return r;
}


#define INADDR_NONE ((unsigned long)-1)

/*  
 *  Connect to whois service on `host'.  If `port' is greater than zero use
 *  that port instead.
 *
 *  `host' may be a fully qualified domain name, or a dotted decimal address.
 */  
int wConnect(host, port, ifp, ofp)
  const char *host;
  int port;
  FILE **ifp;
  FILE **ofp;
{
  int fd;
  struct hostent *hp;
  struct servent *sp;
  struct sockaddr_in srvaddr;
  unsigned long inaddr;
  
  ptr_check(host, "host", "wConnect", 0);
  ptr_check(ifp,  "ifp",  "wConnect", 0);
  ptr_check(ofp,  "ofp",  "wConnect", 0);

  memset((char *)&srvaddr, 0, sizeof srvaddr);
  srvaddr.sin_family = AF_INET;
  if (port <= 0)
  {
    if ((sp = getservbyname("whois", "tcp")))
    {
      port = sp->s_port;
    }
    else
    {
      d1fprintf(stderr, "wConnect: getservbyname(\"whois\", \"tcp\") failed: unknown service?\n");
      return 0;
    }
  }

  srvaddr.sin_port = htons((u_short)port);

  if ((inaddr = inet_addr(host)) != INADDR_NONE)
  {
    memcpy((char *)&srvaddr.sin_addr, (char *)&inaddr, sizeof inaddr);
  }
  else
  {
    if ( ! (hp = gethostbyname(host)))
    {
      d1fprintf(stderr, "wConnect: gethostbyname(\"%s\") failed: %s.\n",
                host, herr_str());
      return 0;
    }
    memcpy((char *)&srvaddr.sin_addr, (char *)hp->h_addr, hp->h_length);
  }

  if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
  {
    d1fprintf(stderr, "wConnect: socket(AF_INET, SOCK_STREAM, 0) failed: %s.\n",
              errno < sys_nerr ? sys_errlist[errno] : "unknown error");
    return 0;
  }

  if (connect(fd, (struct sockaddr *)&srvaddr, sizeof srvaddr) < 0)
  {
    d1fprintf(stderr, "wConnect: connect() to `%s' failed: %s.\n",
              host, errno < sys_nerr ? sys_errlist[errno] : "unknown error");
    close(fd);
    return 0;
  }

  if ( ! (*ifp = fdopen(fd, "r")))
  {
    d1fprintf(stderr, "wConnect: fdopen(%d, \"r\") failed.\n", fd);
    close(fd);
    return 0;
  }

  if ( ! (*ofp = fdopen(fd, "w")))
  {
    d1fprintf(stderr, "wConnect: fdopen(%d, \"w\") failed.\n", fd);
    fclose(ifp);
    close(fd);
  }

  return 1;
}


char *wQueryStr(w)
  const WItem *w;
{
  char *s = (char *)0;
  int space;

  space = wQueryStr_(w, (char *)0);
  if (space != 0)
  {
    s = malloc(space);

    if (s)
    {
      *s = '\0';
      wQueryStr_(w, s);
    }
  }

  return s;
}


WItem *wQueryParse(s)
  const char *s;
{
  WItem *w = (WItem *)0;
  const char *e;
  const char *p;

  p = wskip(s);
  w = parse_as(system_command_, T_SYSTEM, p, &e);
  if ( ! w)
  {
    w = parse_as(search_list_, T_SEARCH, p, &e);
    if ( ! w)
    {
      w = parse_as(general_search_, T_SEARCH, p, &e);
    }
  }

  return w;
}
