/*
 * in6_debug.c  --  Insipired by Craig Metz's Net/2 in6_debug.c, but
 *                  not quite as heavyweight (initially, anyway).
 *
 * The idea is to have globals here, and dump netinet6/ data structures.
 *
 * Copyright 1995 by Dan McDonald, Bao Phan, and Randall Atkinson,
 *	All Rights Reserved.  
 *      All Rights under this copyright have been assigned to NRL.
 */

/*----------------------------------------------------------------------
#       @(#)COPYRIGHT   1.1a (NRL) 17 August 1995

COPYRIGHT NOTICE

All of the documentation and software included in this software
distribution from the US Naval Research Laboratory (NRL) are
copyrighted by their respective developers.

This software and documentation were developed at NRL by various
people.  Those developers have each copyrighted the portions that they
developed at NRL and have assigned All Rights for those portions to
NRL.  Outside the USA, NRL also has copyright on the software
developed at NRL. The affected files all contain specific copyright
notices and those notices must be retained in any derived work.

NRL LICENSE

NRL grants permission for redistribution and use in source and binary
forms, with or without modification, of the software and documentation
created at NRL provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:

        This product includes software developed at the Information
        Technology Division, US Naval Research Laboratory.

4. Neither the name of the NRL nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THE SOFTWARE PROVIDED BY NRL IS PROVIDED BY NRL AND CONTRIBUTORS ``AS
IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL NRL OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation
are those of the authors and should not be interpreted as representing
official policies, either expressed or implied, of the US Naval
Research Laboratory (NRL).

----------------------------------------------------------------------*/

#define GENERIC_DEBUG_C 1

#ifdef linux
#include <netkey/osdep_linux.h>
#include <linux/skbuff.h>
#ifndef KERNEL
#include <netinet/in.h>
#endif /* KERNEL */
#include <linux/module.h>

#include <net/security.h>
#else /* linux */
#include <sys/param.h>
#include <sys/socket.h>
#ifdef KERNEL
#include <sys/mbuf.h>
#include <sys/systm.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/route.h>
#endif /* KERNEL */
#include <netinet/in.h>
#ifdef KERNEL
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/in_pcb.h>
#endif /* KERNEL */

#include <netkey/osdep_44bsd.h>
#endif /* linux */

#include <netkey/generic_debug.h>

#ifdef linux
int in6_debug_level = IDL_FINISHED + 1;
#endif /* linux */

#ifdef INET6
#include <netinet6/in6_debug.h>
#endif /* INET6 */

/*----------------------------------------------------------------------
 * Dump a data buffer 
 ----------------------------------------------------------------------*/
void dump_buf DEFARGS((buf, len),
     caddr_t buf AND
     int len)
{
#ifdef _GENERIC_DEBUG
  int i;

  printf("buf at %08lx len=%x(%d):", (unsigned long)buf, len, len);
  if (!buf)
    return;

  for (i = 0; i < len; i++) {
    if (!(i % 32))
      printf("\n");
    if (!(i % 2))
      printf(" ");
    printf("%02x", *(u_int8_t *)(buf++));
  }
  printf("\n");
#endif /* _GENERIC_DEBUG */
}

void dump_flags DEFARGS((f, i),
struct dump_flags *f AND
int i)
{
#ifdef _GENERIC_DEBUG
  int j;

  j = 0;

  while(i && f->name) {
    if (i & f->val) {
      if (j)
	printf(" ");
      else
        j = 1;
      printf(f->name);
      i &= ~(f->val);
    }
    f++;
  }
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump an IPv4 address in x.x.x.x form.
 ----------------------------------------------------------------------*/
void dump_in_addr DEFARGS((in_addr),
     struct in_addr *in_addr)
{
#ifdef _GENERIC_DEBUG
  u_char *chars = (u_char *)in_addr;
  int i = 0;

  if (!in_addr) {
    printf("Dereference a NULL in_addr? I don't think so.\n");
    return;
  }

  while (i < 3)
    printf("%d.",chars[i++]);
  printf("%d [%08x]\n",chars[3], (unsigned int)(*(u_int32_t *)in_addr));
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump an IPv4 socket address.
 ----------------------------------------------------------------------*/
void dump_sockaddr_in DEFARGS((sin),
     struct sockaddr_in *sin)
{
#ifdef _GENERIC_DEBUG
  printf("sockaddr_in at %08lx: ", (unsigned long)sin);
  if (!sin)
    goto ret;

#ifdef SIN_LEN
  printf("len=%x, ", sin->sin_len);
#endif /* SIN_LEN */
  printf("family=%x, port=%x, addr=", sin->sin_family, sin->sin_port);
  dump_in_addr(&sin->sin_addr);
  printf(" zero: ");
  dump_buf(sin->sin_zero, sizeof(sin->sin_zero));

ret:
  printf("\n");
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump a generic socket address.  Use if no family-specific routine is
 * available.
 ----------------------------------------------------------------------*/
void dump_sockaddr DEFARGS((sa),
SOCKADDR *sa)
{
#ifdef _GENERIC_DEBUG
  printf("sockaddr at %08lx: ", (unsigned long)sa);
  if (!sa)
    goto ret;

#ifdef SA_LEN
  printf("len=%x, ", sa->sa_len);
#endif /* SA_LEN */
  printf("family=%x", sa->sa_family);
#ifdef SA_LEN
  printf("data: ");
  dump_buf(sa->sa_data, sa->sa_len - 2);
#endif /* SA_LEN */
ret:
  printf("\n");
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump a socket address, calling a family-specific routine if available.
 ----------------------------------------------------------------------*/
void dump_smart_sockaddr DEFARGS((sa),
SOCKADDR *sa)
{
#ifdef _GENERIC_DEBUG
  if (!sa) {
    printf("sockaddr at NULL\n");
    return;
  }

  switch(sa->sa_family) {
#ifdef INET6
    case AF_INET6:
      dump_sockaddr_in6((struct sockaddr_in6 *)sa);
      break;
#endif /* INET6 */
    case AF_INET:
      dump_sockaddr_in((struct sockaddr_in *)sa);
      break;
#if defined(KERNEL) && !defined(linux)
    case AF_LINK:
      dump_sockaddr_dl((struct sockaddr_dl *)sa);
      break;
#endif /* defined(KERNEL) && !defined(linux) */
    default:
      dump_sockaddr(sa);
      break;
    }
#endif /* _GENERIC_DEBUG */
}

#ifdef KERNEL
#ifdef linux
void dump_skb(struct sk_buff *skb)
{
#ifdef _GENERIC_DEBUG
  printf("skb at %08x:\n", (unsigned)skb);
  printf("  data=%08x head=%08x tail=%08x len=%08x\n", (unsigned)skb->data, (unsigned)skb->head, (unsigned)skb->tail, (unsigned)skb->len);
  dump_buf(skb->data, skb->len);
#endif /* _GENERIC_DEBUG */
}

#ifdef _GENERIC_DEBUG
static struct dump_flags sos_request_flags[] = {
  { SOS_SPEED, "SPEED" },
  { SOS_SECURITY, "SECURITY" },
  { SOS_UNIQUE, "UNIQUE" },
  { SOS_ERRIFLESS, "ERRIFLESS" },
  { 0, NULL }
};
#endif /* _GENERIC_DEBUG */

void dump_sos_request(struct sos_request *s)
{
#ifdef _GENERIC_DEBUG
  printf("sos_request at %08x: command=%x, len=%x, identtype=%x\n", (unsigned)s, s->sos_command, s->sos_len, s->sos_identtype);
  printf("  inflags=%04x(", s->sos_inflags);
  dump_flags(sos_request_flags, s->sos_inflags);
  printf("), outflags=%04x(", s->sos_outflags);
  dump_flags(sos_request_flags, s->sos_outflags);
  printf(")\n");
#endif /* _GENERIC_DEBUG */
}

void dump_sos_requests(struct sos_request *s, int len)
{
#ifdef _GENERIC_DEBUG
  while(len >= sizeof(struct sos_request)) {
    dump_sos_request(s);

    if (s->sos_len > sizeof(struct sos_request)) 
      dump_buf((void *)s + sizeof(struct sos_request), s->sos_len - sizeof(struct sos_request));

    len -= s->sos_len;
    s = (void *)s + s->sos_len;
  }
  if (len) {
    printf("unknown extra data, len=%x: ", (unsigned)len);
    dump_buf((void *)s, len);
  }
#endif /* _GENERIC_DEBUG */
}

#ifdef _GENERIC_DEBUG
static struct dump_flags sos_action_flags[] = {
  { SOS_SKIPPABLE, "SKIPPABLE" },
  { 0, NULL }
};
#endif /* _GENERIC_DEBUG */

void dump_sos_action(struct sos_action *s)
{
#ifdef _GENERIC_DEBUG
  if (!s)
    return;

  printf("sos_action at %08x: next=%08x, f=%08x, arg1=%08x,\n", (unsigned)s, (unsigned)s->next, (unsigned)s->f, (unsigned)s->arg1);
  printf("  arg2=%08x, flags=%08x(", (unsigned)s->arg2, (unsigned)s->flags);
  dump_flags(sos_action_flags, s->flags);
  printf(")\n");
#endif /* _GENERIC_DEBUG */
}

void dump_sos_actions(struct sos_action *s)
{
#ifdef _GENERIC_DEBUG
  while(s) {
    dump_sos_action(s);
    s = s->next;
  }
#endif /* _GENERIC_DEBUG */
}
#else /* linux */
/*----------------------------------------------------------------------
 * Dump a link-layer socket address.  (Not that there are user-level link
 * layer sockets, but there are plenty of link-layer addresses in the kernel.)
 ----------------------------------------------------------------------*/
void dump_sockaddr_dl DEFARGS((sdl),
     struct sockaddr_dl *sdl)
{
#ifdef _GENERIC_DEBUG
  char buf[256];

  if (!sdl) {
	printf("Dereference a NULL sockaddr_dl? I don't think so.\n");
        return;
  }

  printf("sdl_len = %d, sdl_family = %d, sdl_index = %d, sdl_type = %d,\n",
	 sdl->sdl_len, sdl->sdl_family, sdl->sdl_index, sdl->sdl_type);
  buf[sdl->sdl_nlen] = 0;
  if (sdl->sdl_nlen)
    bcopy(sdl->sdl_data,buf,sdl->sdl_nlen);
  printf("sdl_nlen = %d, name = '%s'\n",sdl->sdl_nlen,buf);
  printf("sdl_alen = %d",sdl->sdl_alen);
  if (sdl->sdl_alen) {
    printf(", addr = ");
    dump_buf((caddr_t)sdl->sdl_data + sdl->sdl_nlen, sdl->sdl_alen);
    printf(")");
  }
  printf("\n");
  printf("sdl_slen = %d",sdl->sdl_slen);
  if (sdl->sdl_slen) {
    printf(", addr = ");
    dump_buf((caddr_t)sdl->sdl_data + sdl->sdl_nlen + sdl->sdl_alen,
	     sdl->sdl_alen);
    printf(")");
  }
  printf("\n");
#endif /* _GENERIC_DEBUG */
}

#ifdef _GENERIC_DEBUG
static struct dump_flags mbuf_flags[] = {
  { M_EXT, "EXT" },
  { M_PKTHDR, "PKTHDR" },
  { M_EOR, "EOR" },
  { M_BCAST, "BCAST" },
  { M_MCAST, "MCAST" },
  { M_AUTHENTIC, "AUTHENTIC" },
  { M_DECRYPTED, "DECRYPTED" },
  { M_DAD, "DAD" },
  { 0, NULL }
};
#endif /* _GENERIC_DEBUG */

void dump_mbuf_flags DEFARGS((m), struct mbuf *m)
{
#ifdef _GENERIC_DEBUG
  if (m)
    dump_flags(mbuf_flags, m->m_flags);
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump only the header fields of a single mbuf.
 ----------------------------------------------------------------------*/
void dump_mbuf_hdr DEFARGS((m),
     struct mbuf *m)
{
#ifdef _GENERIC_DEBUG
  printf("mbuf at %08lx\n", (unsigned long)m);
  if (m) {
    printf("\tm_len = %d, m_data = 0x%lx, m_type = %d\n",m->m_len, (unsigned long)m->m_data, m->m_type);
    printf("\tm_flags = 0x%x: ", (unsigned int)m->m_flags);
    dump_mbuf_flags(m);
    if (m->m_flags & M_PKTHDR)
      printf("\n\tm_pkthdr.len = %d, m_pkthdr.rcvif = 0x%lx",m->m_pkthdr.len,
	     (unsigned long)m->m_pkthdr.rcvif);
    printf("\n\tm_next = 0x%lx  m_nextpkt = 0x%lx\n", (unsigned long)m->m_next, (unsigned long)m->m_nextpkt);
  }
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump the entire contents of a single mbuf.
 ----------------------------------------------------------------------*/
void dump_mbuf DEFARGS((m),
     struct mbuf *m)
{
#ifdef _GENERIC_DEBUG
  dump_mbuf_hdr(m);
  printf("m_data:\n");
  dump_buf(m->m_data, m->m_len);
  printf("\n");
#endif /* _GENERIC_DEBUG */
}

void dump_mchain_hdr(struct mbuf *m)
{
#ifdef _GENERIC_DEBUG
  struct mbuf *walker;
  int i;

  for (walker = m, i = 0; walker != NULL && (i < 10); 
       walker = walker->m_next, i++)
    dump_mbuf_hdr(walker);
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump the contents of an mbuf chain.  (WARNING:  Lots of text may
 * result.
 ----------------------------------------------------------------------*/
void dump_mchain DEFARGS((m),
     struct mbuf *m)
{
#ifdef _GENERIC_DEBUG
  struct mbuf *walker;
  int i;

  for (walker = m, i = 0; walker != NULL && (i < 10); 
       walker = walker->m_next, i++)
    dump_mbuf(walker);
#endif /* _GENERIC_DEBUG */
}

void dump_tcpdump DEFARGS((m),
     struct mbuf *m)
{
#ifdef _GENERIC_DEBUG
  int i, j, count;

  for (i = count = 0; m && (i < 10); m = m->m_next, i++) {
    for (j = 0; j < m->m_len; j++, count++) {
      if (!(count % (2 * 8)))
	printf("\n\t\t\t");
      if (!(count % 2))
	printf(" ");
      printf("%02x", (u_int8_t)(m->m_data[j]));
    }
  }
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump an interface address.
 ----------------------------------------------------------------------*/
void dump_ifa DEFARGS((ifa),
     struct ifaddr *ifa)
{
#ifdef _GENERIC_DEBUG
  if (ifa == NULL)
    {
      printf("ifa of NULL.\n");
      return;
    }

  printf("ifa_addr: ");
  dump_smart_sockaddr(ifa->ifa_addr);
  printf("ifa_netmask: ");
  dump_smart_sockaddr(ifa->ifa_netmask);
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump an interface structure.
 ----------------------------------------------------------------------*/
void dump_ifp DEFARGS((ifp),
     struct ifnet *ifp)
{
#ifdef _GENERIC_DEBUG
  if (!ifp) {
    printf("Dereference a NULL ifnet/ifp? I don't think so.\n");
    return;
  }

#ifdef NetBSD
  printf("Interface name: %s.\n", ifp->if_xname);
#else /* NetBSD */
  printf("Interface name: %s.\n",ifp->if_name);
#endif /* NetBSD */
  printf("Interface type: %d.  ",ifp->if_type);
  printf("MTU: %d.\n", (unsigned int)ifp->if_mtu);
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump a route structure (sockaddr/rtentry pair).
 ----------------------------------------------------------------------*/
void dump_route DEFARGS((ro),
     struct route *ro)
{
#ifdef _GENERIC_DEBUG
  if (!ro) {
    printf("Dereference a NULL route? I don't think so.\n");
    return;
  }

  printf("ro_rt = 0x%lx, ro_dst is:\n", (unsigned long)ro->ro_rt);
  dump_smart_sockaddr(&ro->ro_dst);
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump a routing entry.
 ----------------------------------------------------------------------*/
void dump_rtentry DEFARGS((rt),
     struct rtentry *rt)
{
#ifdef _GENERIC_DEBUG
  if (!rt) {
    printf("Dereference a NULL rtentry? I don't think so.\n");
    return;
  }

  printf("rt_key is:\n");
  dump_smart_sockaddr(rt_key(rt));
  printf("rt_mask is:\n");
  dump_smart_sockaddr(rt_mask(rt));
  printf("rt_llinfo = 0x%lx ", (unsigned long)rt->rt_llinfo);
  printf("rt_rmx.rmx_mtu = %d ", (unsigned int)rt->rt_rmx.rmx_mtu);
  printf("rt_refcnt = %d ",rt->rt_refcnt);
  printf("rt_flags = 0x%x\n", (unsigned int)rt->rt_flags);
  printf("rt_ifp is:\n");
  dump_ifp(rt->rt_ifp);
  printf("rt_ifa is:\n");
  dump_ifa(rt->rt_ifa);
#endif /* _GENERIC_DEBUG */
}

/*----------------------------------------------------------------------
 * Dump an Internet (v4/v6) protocol control block.
 ----------------------------------------------------------------------*/
void dump_inpcb DEFARGS((inp),
     struct inpcb *inp)
{
#ifdef _GENERIC_DEBUG
  if (!inp) {
    printf("Dereference a NULL inpcb? I don't think so.\n");
    return;
  }

  printf("inp_next = 0x%lx, inp_prev = 0x%lx, inp_head = 0x%lx.\n", (unsigned long)inp->inp_next,
	 (unsigned long)inp->inp_prev, (unsigned long)inp->inp_head);
  printf("inp_socket = 0x%8lx, inp_ppcb = 0x%08lx\n", (unsigned long)inp->inp_socket, (unsigned long)inp->inp_ppcb);
#ifdef INET6
  printf("faddr, faddr6:\n");
  dump_in_addr(&inp->inp_faddr); dump_in6_addr(&inp->inp_faddr6);
  printf("laddr, laddr6:\n");
  dump_in_addr(&inp->inp_laddr); dump_in6_addr(&inp->inp_laddr6);
#else /* INET6 */
  printf("faddr:\n");
  dump_in_addr(&inp->inp_faddr);
  printf("laddr:\n");
  dump_in_addr(&inp->inp_laddr);
#endif /* INET6 */
  printf("inp_route: ");
  dump_route(&inp->inp_route);
#ifdef INET6
  printf("inp_ipv6:");
  dump_ipv6(&inp->inp_ipv6);
#endif /* INET6 */
#if 0
  printf("inp_ip:");
  printf("<Coming soon.>\n");
#endif /* 0 */
  printf("inp_options = 0x%lx, inp_moptions{6,} = 0x%lx,\n", (unsigned long)inp->inp_options,
	 (unsigned long)inp->inp_moptions);
  printf("inp_flags = 0x%x, inp_fport = %d, inp_lport = %d.\n",
	 (unsigned int)inp->inp_flags, inp->inp_fport, inp->inp_lport);
#endif /* _GENERIC_DEBUG */
}
#endif /* linux */
#endif /* KERNEL */

#ifdef MODULE
int init_module(void)
{
  return 0;
}

int cleanup_module(void)
{
  return 0;
}
#endif /* MODULE */
