/******************************************/
/* planets.h                              */
/*                                        */
/* Header file for VGA Planets            */
/* Structures of DAT- and RST-Files       */
/*                                        */
/* Special thanks to A.M. van den Bos,    */
/* (bos@cs.utwente.nl), who published the */
/* header of PLANMAP                      */
/*                                        */
/* (For more information look in          */
/* readplan.c or in planmap.h)            */
/*                                        */
/* (c) 1995 C. Quix                       */
/* Email: Christoph_Quix@p17.ks.fido.de   */
/******************************************/

#ifndef _stdio_h_
#include <stdio.h>
#endif

/***************************/
/* Definition of datatypes */
/***************************/
#define WORD unsigned short int
#define LONG unsigned int
#define INT signed short int

/***************************/
/* Global constants        */
/***************************/
#define MAXPLANET  500
#define MAXSHIP    500
#define MAXRACE     11
#define MAXHULL    105
#define MAXHULLRACE 20
#define MAXBEAM     10
#define MAXTORP     10
#define MAXENG       9

/*************************/
/* Ship                  */
/*************************/
typedef struct ship  {
	WORD id;
	WORD owner;
	char friendlyCode[3];
	WORD warp;
	INT deltax;
	INT deltay;
	WORD x;
	WORD y;
	WORD engineIndex;
	WORD hullType;
	WORD beamIndex;
	WORD numberOfBeams;
	WORD numberOfBays;
	WORD torpIndex;
	WORD numberOfFightTorp;
	WORD numberOfLaunchers;
	WORD mission;
	WORD enemy;
	WORD towed;
	WORD damage;
	WORD crew;
	WORD clans;
	char name[20];
	WORD fuel;
	WORD trit;
	WORD dura;
	WORD moly;
	WORD supplies;
	WORD clansToPlanet;
	WORD fuelToPlanet;
	WORD tritToPlanet;
	WORD duraToPlanet;
	WORD molyToPlanet;
	WORD suppliesToPlanet;
	WORD planetID;
	WORD clansToShip;
	WORD fuelToShip;
	WORD tritToShip;
	WORD duraToShip;
	WORD molyToShip;
	WORD suppliesToShip;
	WORD shipID;
	WORD intercept;
	WORD megaCredits;
} Ship;   /* 107 bytes */

/*************************/
/* Alien Ship            */
/*************************/
typedef struct alienship  {
	WORD id;
	WORD owner;
	WORD warp;
	WORD x;
	WORD y;
	WORD hull;
	WORD angle;
	char name[20];
} AlienShip; /* 34 bytes */

/*************************/
/* Planet                */
/*************************/
typedef struct planet  {
	WORD owner;
    WORD id;
	char friendlyCode[3];
	WORD mines;
	WORD facs;
	WORD defposts;
	LONG fuel;
	LONG trit;
	LONG dura;
	LONG moly;
	LONG clans;
	LONG supplies;
	LONG megaCredits;
	LONG fuelGround;
	LONG tritGround;
	LONG duraGround;
	LONG molyGround;
	WORD fuelMiningRate;
	WORD tritMiningRate;
	WORD duraMiningRate;
	WORD molyMiningRate;
	WORD colonistTax;
	WORD nativeTax;
	INT  colonistStatus;
	INT  nativeStatus;
	WORD government;
	LONG natives;
	WORD nativeRace;
	WORD temperature;   /* 100 - temperature */
	WORD buildStarbase;
} Planet;  /* 85 bytes */

/*************************/
/* Starbase              */
/*************************/
typedef struct starbase  {
	WORD id;
	WORD owner;
	WORD defposts;
	WORD damage;
	WORD engineTechLevel;
	WORD hullTechLevel;
	WORD beamTechLevel;
	WORD torpTechLevel;
	WORD engines[9];
	WORD hulls[20];      /* #Hulls, index = index in truehull           0x24 */
	WORD beams[10];      /* #beam weapons on starbase index 0=Laser     0x4C */
	WORD launchers[10];  /* #torp lauchers on base, 0=Mark 1            0x60 */
	WORD torps[10];      /* #tops on starbase index 0=Mark 1            0x74 */
	WORD fighters;       /* #fighters on base 0x88 */
	WORD fixID;          /* ID of ship being fixed or recycled */
	WORD fixShip;        /* 0=do nothing, 1=fix ship, 2=recycle ship */
	WORD mission;        /* 1 = Refuel, 2 = Max defence, ... */
	WORD ship;           /* Ship being build (index in Hullspel?) */
	WORD shipEngineIndex;
	WORD shipBeamIndex;
	WORD shipBeamsOnShip;   
	WORD shipTorpIndex; 
	WORD shipTorpsOnShip;   
	WORD shipFightersOnShip;
} Starbase;

/*************************/
/* Message               */
/*************************/
typedef struct message {	
	LONG offset;    /* Position of message in file (*.rst or mdata*.*) */
	WORD length;    /* This dates of ALL messages comes first, after that */
	                /* there is NULL-Byte and then the messagetext */
	char *text;
} Message;

/*************************/
/* ShipXY                */
/*************************/
typedef struct shipxy  {
	WORD x;         /* only non-zero when known  */
	WORD y;
	WORD owner;
	WORD mass;
} ShipXY;

/*************************/
/* Score                 */
/*************************/
typedef struct score {
	WORD planets;
	WORD ships;
	WORD freighters;
	WORD bases;
} Score;

/*************************/
/* VCR                   */
/*************************/
typedef struct vcr  {
	WORD unknown[4];
	WORD mass1;
	WORD mass2;
	char name1[20];         /* ship 1 */
	WORD damage1;
	WORD crew1;
	WORD id1;
	WORD race1;
	WORD bitmap1;            /* Index in RESOURCE.PLN */
	WORD beamIndex1;
	WORD numberOfBeams1;
	WORD numberOfBays1;
	WORD torpIndex1;
	WORD numberOfTorpFighters1;
	WORD numberOfLaunchers1;
	char name2[20];          /* ship2 */
	WORD damage2;
	WORD crew2;
	WORD id2;
	WORD race2;
	WORD bitmap2;            /* Index in RESOURCE.PLN */
	WORD beamIndex2;
	WORD numberOfBeams2;
	WORD numberOfBays2;
	WORD torpIndex2;
	WORD numberOfTorpFighters2;
	WORD numberOfLaunchers2;

	WORD shield1;
	WORD shield2;   /* Initial shields of both ships. */
} Vcr;

/*************************/
/* RST-File              */
/*************************/
typedef struct rstfile {
	/* char miscInfo[33]; */
	WORD numberOfShips;
	struct ship *shipInfo;
	WORD numberOfAlienShips;
	struct alienship *alienShipInfo;
	WORD numberOfPlanets;
	struct planet *planetInfo;
	WORD numberOfStarbases;
	struct starbase *starbaseInfo;
	WORD numberOfMessages;
	struct message *messageInfo;
	struct shipxy shipXYInfo[MAXSHIP];
	char timestamp[18];
	struct score scoreInfo[MAXRACE];
	WORD playernumber;
	char password[20];
	char checkInfo[12];
	WORD turnNumber;
	WORD unknown;
	WORD numberOfVCRs;
	struct vcr *vcrInfo;
} RstFile;

/***************************************************/
/* Structures of global data                       */
/***************************************************/

/******************************/
/* HULLSPEC.DAT               */
/******************************/
typedef struct hullspec { 
	char name[30];
	WORD picture;
	WORD unknown;    /* Always 1, perhaps part of Picture ID; I don't know */
	WORD trit;
	WORD dura;
	WORD moly;
	WORD fuelTank;
	WORD crew;
	WORD numberOfEngines;
	WORD mass;
	WORD techLevel;
	WORD cargo;
	WORD fighterBays;
	WORD torpBays;
	WORD beams;
	WORD megaCredits;
} HullSpec;


/******************************/
/* TRUEHULL.DAT               */
/******************************/
typedef struct truehull  {
	WORD hullIndex[MAXHULLRACE];
} TrueHull;


/******************************/
/* BEAMSPEC.DAT               */
/******************************/
typedef struct beamspec {
	char name[20];
	WORD megaCredits;
	WORD trit;
	WORD dura;
	WORD moly;
	WORD mass;
	WORD techLevel;
	WORD kill;
	WORD explosive;
} BeamSpec;


/******************************/
/* TORPSPEC.DAT               */
/******************************/
typedef struct torpspec { 
	char name[20];
	WORD megaCreditsTorp;           /* Costs of one torpedo of this type */
	WORD megaCreditsLauncher;       /* Money to build torp. launcher */
	WORD trit;
	WORD dura;
	WORD moly;
	WORD mass;
	WORD techLevel;
	WORD kill;
	WORD explosive;
} TorpSpec;


/******************************/
/* ENGSPEC.DAT                */
/******************************/
typedef struct engspec { 
	char name[20];
	WORD megaCredits;
	WORD trit;
	WORD dura;
	WORD moly;
	WORD techLevel;
	LONG fuelUse[9];
} EngSpec;


/******************************/
/* XYPLAN.DAT                 */
/******************************/
typedef struct xyplan  {
	WORD x;
	WORD y;
	WORD unknown;
} XYPlan;


/******************************/
/* PLANET.NM                  */
/******************************/
typedef struct planetName  {
	char name[20];
} PlanetName;


/******************************/
/* RACE.NM                    */
/* 11*30 for long name        */
/* 11*20 for plural name      */
/* 11*12 for short name       */
/******************************/
typedef struct raceName  {
	char longName[30]; 
	char pluralName[20];
	char shortName[12];
} RaceName;

	
/*******************************************************/
/* Functions to read data                              */
/*******************************************************/
void skip(FILE *fp, WORD i);
WORD readWord(FILE *fp);
LONG readLong(FILE *fp);
void readString(FILE *fp, char *str, WORD length);

RstFile*     readRst(FILE *fp);
RstFile*     readDatFiles(WORD raceNumber, char *path);
Ship*        readShips(FILE *fp, WORD i);
AlienShip*   readAlienShips(FILE *fp, WORD i);
Planet*      readPlanets(FILE *fp, WORD i);
Starbase*    readStarbases(FILE *fp, WORD i);
Message*     readMessages(FILE *fp, WORD i, WORD isDat);
Vcr*         readVCRs(FILE *fp, WORD i);

void readShipXY(FILE *fp, ShipXY *xy);  
void readScores(FILE *fp, Score *scores);

void readHullSpec();
void readTrueHulls();
void readBeamSpec();
void readTorpSpec();
void readEngSpec();
void readXYPlan();
void readPlanetNames();
void readRaceNames();


/********************************************************/
/* Functions to print the data                          */
/********************************************************/
void printString(FILE *fp, char *str, WORD length);

void printBeam(FILE *fp, WORD index);
void printTorp(FILE *fp, WORD index);

void printShips(FILE *fp, Ship* ships, ShipXY *xy, WORD numberOfShips);
void printAlienShips(FILE *fp, AlienShip *s, ShipXY *xy, WORD numberOfAlienShips);
void printPlanets(FILE *fp, Planet *p, ShipXY *xy, WORD numberOfPlanets, WORD what);
void printStarbases(FILE *fp,Starbase *s, ShipXY *xy, WORD numberOfStarbases);
void printMessages(FILE *fp, Message *m, WORD numberOfMessages);
void printScores(FILE *fp, Score *s);
void printVCRs(FILE *fp, Vcr *v, WORD numberOfVCRs);

/* Prints an error message, usage text, and exits */
void help();
