/*
 *   Header file for the Portable Host Development Kit (PDK)
 *   Version 2.5
 *
 *   Copyright 1995 by Andrew Sterian, Thomas Voigt, and Steffen Pietsch
 */

#ifndef _PHOSTPDK_H_
#define _PHOSTPDK_H_

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#ifdef __cplusplus
extern "C" {
#endif

#define   RACE_NR		11	  /* Number of races */
#define   SHIP_NR	   500	  /* Max number of ships */
#define   PLANET_NR    500	  /* Number of planets */
#define   BASE_NR	   500	  /* Max number of bases */
#define   MINE_NR	   500	  /* Max number of mines */
#define   HULL_NR	   105	  /* Number of hulls in HULLSPEC.DAT */
#define   ENGINE_NR 	 9	  /* Number of engines in ENGSPEC.DAT */
#define   BEAM_NR		10	  /* Number of beams in BEAMSPEC.DAT */
#define   TORP_NR		10	  /* Number of torps in TORPSPEC.DAT */

#define   LONGNAME_SIZE 	   30	 /* Length of full race name */
#define   SHORTNAME_SIZE	   20	 /* Length of short race name */
#define   ADJECTIVE_SIZE	   12	 /* Length of race adjective */
#define   RACENAME_RACE_SIZE   (LONGNAME_SIZE+SHORTNAME_SIZE+ADJECTIVE_SIZE)
#define   RACENAME_SIZE        (RACENAME_RACE_SIZE*11)
										/* Size of RACE.NM file */
#define   PLANETNAME_SIZE	   20  /* size of a record in PLANET.NM */

#define RACEHULLS			 20  /* Max number of hulls per race */
#define MAX_SPEED			  9  /* Max warp speed on a ship */
#define MAX_BEAMS			 20  /* Max number of beams on a ship */
#define MAX_TUBES			 20  /* Max number of tubes on a ship */
#define MAX_BAYS			 20  /* Max number of bays on a ship */
#define MAX_PLANET_BAYS 	 50  /* max number of bays on a planet/SB */
#define MAX_BASE_DEFENSE	200  /* Max defences for a base */
#define MAX_BASE_FIGHTERS	 60  /* Max fighters for a base */
#define MAX_TECH			 10  /* Max tech level */
#define MAX_DENSITY 		100  /* Max mineral density in a planet */
#define MAX_HAPPY			100  /* Max happiness for colonists and natives */
#define MIN_HAPPY		 (-300)  /* Min happiness for colonists and natives */
#define MAX_COORDINATE	  10000  /* coordinate limits given by planets.exe */
#define MIN_COORDINATE		  0
#define MAX_SHIP_CREDITS  10000   /* max credits allowed on a ship */
#define MAX_WORMHOLES		100   /* maximum supported wormholes */
#define MAX_MINES			700   /* sqrt(249800) + 200 */
#define MAX_FACTORIES		600   /* sqrt(249900) + 100 */
#define MAX_DEFENSE 		550   /* sqrt(249950) +  50 */
#define MAX_POPULATION 25000000UL /* absolute maximum population on a planet */
#define MAX_TAX 			100   /* max tax rate on a planet */
#define MAX_TEMP			100   /* max planet temperature */

typedef   short 		 Int16;
typedef   unsigned short Uns16;

#ifdef __alpha	  /* Alpha AXP defines long as 64 bits */
 typedef   int			  Int32;
 typedef   unsigned int   Uns32;
#else
 typedef   long 		  Int32;
 typedef   unsigned long  Uns32;
#endif

typedef   enum {False, True} Boolean;
typedef   Uns16 RandType_Def;

typedef enum {
	NEUTRONIUM=0,
	TRITANIUM,
	DURANIUM,
	MOLYBDENUM,
	COLONISTS,
	SUPPLIES,
	CREDITS,
	NumCargoTypes
} CargoType_Def;

typedef enum {
	NoOwner=0, NoRace=0,
	Federation=1,
	Gorn=2, Lizards=2,
	Romulans=3, Birdmen=3,
	Klingons=4,
	Orions=5, Privateers=5,
	Borg=6, Cyborg=6,
	Tholians=7, Crystals=7,
	Empire=8,
	Cylons=9, Robots=9,
	Rebels=10,
	Colonies=11
} RaceType_Def;

typedef enum {
	NoNatives=0,
	Humanoid=1,
	Bovinoid=2,
	Reptilian=3,
	Avian=4,
	Amorphous=5,
	Insectoid=6,
	Amphibian=7,
	Ghipsoldal=8,
	Siliconoid=9,
	NumNativeTypes=9
} NativeType_Def;

typedef enum {
	NoGovm=0,
	Anarchy=1,
	PreTribal=2,
	EarlyTribal=3,
	Tribal=4,
	Feudal=5,
	Monarchy=6,
	Representative=7,
	Participatory=8,
	Unity=9,
	NumGovmTypes=9
} NativeGovm_Def;

typedef enum {
	NoMission=0,
	Exploration=1,
	MineSweep=2,
	LayMines=3,
	Kill=4,
	SensorSweep=5,
	Colonize=6,
	Tow=7,
	Intercept=8,
	SpecialMission=9,
	Cloak=10,
	GatherNeutronium=11,
	GatherDuranium=12,
	GatherTritanium=13,
	GatherMolybdenum=14,
	GatherSupplies=15,
	NumShipMissions=15
} ShipMission_Def;

typedef enum {
	NoBaseMission=0,
	Refuel=1,
	MaxDefense=2,
	LoadTorps=3,
	UnloadFreighters=4,
	RepairBase=5,
	ForceSurrender=6
} BaseMission_Def;

typedef enum {
	FIX_MISSION=0,
	RECYCLE_MISSION=1
} BaseFixMission_Def;

typedef enum {
	ENGINE_TECH=0,
	HULL_TECH=1,
	BEAM_TECH=2,
	TORP_TECH=3
} BaseTech_Def;

typedef struct {
	Uns16 mHull;
	Uns16 mEngineType;
	Uns16 mBeamType;
	Uns16 mNumBeams;
	Uns16 mTubeType;
	Uns16 mNumTubes;
} BuildOrder_Struct;

typedef enum {
	ALLY_SHIPS=0,
	ALLY_PLANETS=1,
	ALLY_MINES=2,
	ALLY_COMBAT=3,
	ALLY_VISION=4
} AllianceLevel_Def;

typedef enum {
    ALLY_STATE_OFF,
    ALLY_STATE_ON,
    ALLY_STATE_CONDITIONAL
} AllianceState_Def;

typedef enum {
	GAME_DIR_ONLY	=0x01,		/* search or write only in game subdirectory */
	ROOT_DIR_ONLY	=0x02,		/* search or write only in root subdirectory */
	GAME_OR_ROOT_DIR=0x04,		/* search first in game dir, then in root dir */
								/* This flag only applies to input files */
	REWRITE_MODE	=0x10,		/* rewrite mode for update files */
	APPEND_MODE 	=0x20,		/* append mode rather than write for output files */
	TEXT_MODE		=0x40,		/* open the file in text mode instead of binary */
	NO_MISSING_ERROR=0x80		/* don't complain if input file is missing */
} FileLocation_Def;

typedef enum {
	LANG_English,
	LANG_German,
    LANG_French,
	NumLanguages
} Language_Def;

typedef struct {
	Uns16	 RecycleRate,			/* % of ore recovered from colonizing ships */
			 MeteorRate;			/* % chance of one large meteor */
	Boolean  Minefields,			/* allow minefields */
			 Alchemy,				/* allow alchemy */
			 DeleteOldMessages, 	/* delete old messages -- NOT USED */
			 DisablePasswd; 		/* disable passwords */
	Uns16	 GroundKill[12],		/* tim uses only 1..11, 0 ignored */
			 GroundDefense[12], 	/* ...ground attack, defense ratios */
			 FreeFighters[12],		/* free fighters per starbase */
			 MiningRate[12],		/* mineral mining rate */
			 TaxRate[12];			/* taxation rate */
	Boolean  RebelsBuildFighters,	/* Rebels allowed to build fighters in space */
			 ColoniesBuildFighters, /* Colonies allowed to build fighters in space */
			 RobotsBuildFighters;	/* Robots allowed to build fighters in space */
	Uns16	 CloakFailure;			/* % chance of cloaking failure */
	Boolean  RobCloakedShips;		/* Privateers can rob cloaked ships */
	Uns16	 ScanRange, 			/* Enemy ship scan range */
			 DarksenseRange;		/* Dark sense range */
	Boolean  Hiss,					/* Allow hiss mission */
			 RebelGroundAttack, 	/* Allow RGA */
			 SuperRefit,			/* Allow super refit */
			 WebMines;				/* Allow web mines */
	Uns16	 CloakFuelBurn, 		/* kt of fuel burn required to stay cloaked */
			 SensorRange;			/* range of the sensor sweep mission */
	Boolean  NewNatives,			/* Allow new natives to appear */
			 PlanetsAttack; 		/* Allow planets to attack ships */
	Uns16	 AssimilationRate,		/* assimilation percentage */
			 WebDecay,				/* web decay rate percentage */
			 MineDecay, 			/* mines decay rate percentage */
			 MaxMineRadius, 		/* maximum minefield radius */
			 TransuraniumRate,		/* new mineral formation rate */
			 StructureDecay;		/* structure decay rate */
	Boolean  EatSupplies,			/* overpopulated planets eat supplies */
			 NoFuelMove;			/* ships without fuel can move */
	Uns16	 MineOdds,				/* percentage odds of hitting a mine */
			 WebOdds,				/* percentage odds of hitting a web mine */
			 MineScanRange; 		/* range in which mines are visible */
	Boolean  MinesDestroyMines, 	/* allow mines to destroy mines */
			 EngShieldBonus;		/* allow engine bonus */
	Uns16	 ShieldBonusRate,		/* engine tech bonus rate */
			 ColFighterSweepRate;	/* Colonies fighter sweep rate */
	Boolean  ColSweepWebs;			/* Allow Colonies to sweep web mines */
	Uns16	 MineSweepRate, 		/* rate of mine sweeping with beam weapons */
			 WebSweepRate,			/* rate of web sweeping with beam weapons */
			 HissEffect,			/* hiss effect per ship */
			 RobFailRate;			/* percentage change that robbing fails */
	Boolean  PlanetRebelAttack, 	/* planets can attack Rebels */
			 PlanetKlingonAttack;	/* planets can attack Klingons */
	Uns16	 MineSweepRange,		/* mine sweep range */
			 WebSweepRange; 		/* web sweep range */
	Boolean  ScienceShips;			/* allow science ship mission */
	Uns16	 CloakMineOdds, 		/* 10X percentage of cloaked hitting mine (13 = 1.3%) */
			 NoCloakDamage; 		/* amount of damage that prevents cloaking */
	Boolean  FedCrewBonus;			/* allow Fed combat bonus */

	Uns16	 SmallMeteorOdds;		/* percentage chance per planet of small showers */
	Uns32	 SmallOre[8];			/* Nmin, Dmin, Tmin, Mmin, Nmax, Dmax, Tmax, Mmax */
	Uns16	 LargeMeteorNumber; 	/* number of large meteors to impact */
	Uns32	 LargeOre[8];			/* Nmin, Dmin, Tmin, Mmin, Nmax, Dmax, Tmax, Mmax */
	Boolean  MeteorMessages;		/* send meteor messages when they happen */

	Boolean  OneTow,				/* allow one engine ships to tow */
			 HyperDrives;			/* allow hyperdrive */
	Uns16	 ClimateDeathRate;		/* climate death rate */
	Boolean  GravityWells,			/* allow gravity wells */
			 CrystalsLikeDesert;	/* Crystals grow better on hot planets */
	Boolean  DestroyWebs,			/* allow normal mines to destroy web mines */
			 ClimateLimitsPopulation; /* allow climate to limit population */
    Uns32    MaxPlanetaryIncome;    /* max number of credits to earn per planet */
    Uns16    IonStormActivity;      /* chance of storms as a percentage */
    Boolean  AllowChunneling;       /* allow Firecloud chunneling */
    Boolean  AllowDeluxeSuperSpy;   /* allow Deluxe Super Spy mission */
    Boolean  IonStormsHideMines;    /* Ion storms hide minefields */
    Boolean  AllowGloryDevice;      /* allow Klingon glory device */
    Boolean  AllowAntiCloakShips;   /* allow Loki to uncloak ships */
    Boolean  AllowGamblingShips;    /* allow Lady Royale to generate MC */
    Boolean  AllowCloakedShipAttack;/* allow cloaked ships to attack based on PE */
    Boolean  AllowShipCloning;      /* allow ships to be cloned */
    Boolean  AllowBoardingParties;  /* allow Priv/Crystals to tow capture */
    Boolean  AllowImperialAssault;  /* allow SSD to perform imperial assault */
    Uns16    RamScoopFuelPerLY;     /* amount of fuel to gather per LY travelled */
    Boolean  AllowAdvancedRefinery; /* allow Aries to do refinery */
    Boolean  AllowBioscanners;      /* allow bioscanning */
    Uns16    HullTechNotSlowedByMines; /* min hull tech for ships not to be slowed */

					/* NEW OPTIONS */
	Boolean  UseAccurateFuelModel;	/* mass of ship decreases as fuel burns */
	Uns16	 DefenseForUndetectable; /* min defenses to make planet undetectable */
	Uns16	 FactoriesForDetectable; /* min factories to make planet detectable */
	Uns16	 MinesForDetectable;	 /* min mines to make planet detectable */
	Uns16	 ColFighterSweepRange;	/* range at which colonial fighters can sweep */
	Uns16	 MineHitDamage; 		/* damage per 100 KT of mass */
	Uns16	 WebHitDamage;			/* damage per 100 KT of mass */
	Boolean  AllowRegisteredFunctions; /* allow registered-only functions to be performed */
	Uns16	 GravityWellRange;		/* distance from planets at which ships fall in */
	Language_Def Language[12];		/* preferred language. Element 0 is for host */
	Boolean  AllowPlayerMessages;	/* allow player-to-player messages */
	Uns16	 ScoringMethod; 		/* the method of reporting scores */
	Boolean  TowedShipsBreakFree;	/* allow towed ships to break free */
	Uns16	 NativeClimateDeathRate;/* climate death rate for natives */
	Boolean  AllowMoreThan50Targets[12]; /* allow target.dat part of .RST file to have >50 targets */
	Boolean  CrystalSinTempBehavior; /* growth and max pop for Crystals is sinusoidal */
	Boolean  RGANeedsBeams; 		/* Need beam weapons to do RGA */
	Boolean  AllowRGAOnUnowned; 	/* allow RGA on unowned planets */
	Boolean  CPEnableLanguage;		/* Enable 'language' command processor command */
	Boolean  CPEnableBigTargets;	/* Enable 'bigtargets' command */
	Boolean  CPEnableRaceName;		/* Enable 'racename' command */
	Boolean  CPEnableAllies;		/* Enable 'allies' command */
	Boolean  CPEnableMessage;		/* Enable 'message' command */
	Boolean  CPEnableRumor; 		/* Enable 'rumor' command */
	Boolean  DelayAllianceCommands; /* Alliance management only after combat */
	Uns16	 TerraformRate; 		/* number of degrees of terraforming per turn */
	Uns16	 MaxColTempSlope;		/* max colonists slope on arctic/desert planets */
	Uns16	 WebDrainFuelLoss;		/* amount of fuel to drain from ships in web mines */
	Boolean  AllowWormholes;		/* allow wormholes */
	Uns16	 WrmDisplacement;		/* magnitude of predictable endpoint travel */
	Uns16	 WrmRandDisplacement;	/* magnitude of random endpoint travel */
	Int16	 WrmStabilityAddX10;	/* amount of stability increase per turn X10 */
	Uns16	 WrmRandStability;		/* magnitude of random stability to add */
	Int16	 WrmMassAdd;			/* amount of mass to add per turn */
	Uns16	 WrmRandMass;			/* magnitude of random mass to add */
	Boolean  WrmVoluntaryTravel;	/* require WRT friendly code for travel */
	Uns16	 WrmTravelDistDivisor;	/* distance scaling for wormhole travel */
	Uns16	 WrmTravelWarpSpeed;	/* speed at which ships must travel thru wormholes */
	Boolean  WrmTravelCloaked;		/* allow ships to remain cloaked thru wormholes */
	Uns16	 WrmEntryPowerX100; 	/* entry radius power of wormhole mass */
	Boolean  CPEnableGive;			/* allow the 'give' CP command */
	Boolean  AllowTowCloakedShips;	/* allow cloaked ships to be towed */
	Uns16	 RobCloakedChance;		/* percent chance of robbing cloaked ships */
	Uns16	 PlanetaryTorpsPerTube; /* number of free torps to give per tube */
	Uns16	 UnitsPerTorpRate[12];	/* percent of normal units-per-torp rate */
    Boolean  ESBonusAgainstPlanets; /* E-S bonus applies when fighting planets */
    Uns16    ShipCloneCostRate[12]; /* percentage cost of original ship */
    Boolean  HyperjumpGravityWells; /* allow hyperwarp ships to get sucked in */
    Uns16    NativeCombatSurvRate;  /* percent natives dying when planet loses combat */
    Boolean  AllowPrivTowCapture;    /* Allow Privs to do tow capture */
    Boolean  AllowCrystalTowCapture; /* Allow Crystals to do tow capture */
    char     GameName[32];          /* The name of the game */
    Boolean  RoundWarpWells;        /* Warp wells are round, not square */
    Boolean  CPEnableSend;          /* Enable the 'send' command */
    Boolean  CPEnableJettison;      /* Enable the 'jettison' command */
    Boolean  CumulativePillaging;   /* Allow multiple ships to pillage */
    Boolean  AllowInterceptAttack;  /* Allow XA attacks */
    Uns16    RaceGrowthRate[12];    /* Colonist growth factor */

	Uns16	 BayRechargeRate;		/* recharge rate X10 of fighter bays per bay */
    Int16    BayRechargeBonus;      /* bay number bonus in recharging */
	Uns16	 BeamRechargeRate;		/* recharge rate X10 of beams */
    Int16    BeamRechargeBonus;     /* beam tech bonus in recharging */
	Uns16	 TubeRechargeRate;		/* recharge rate X10 of tubes */
	Uns16	 BeamHitFighterCharge;	/* minimum charge X10 for beams to fire at fighters */
	Uns16	 BeamHitShipCharge; 	/* minimum charge X10 for beams to fire at ships */
	Uns32	 TorpFiringRange;		/* maximum distance at which to fire torps */
	Uns32	 BeamFiringRange;		/* maximum distance at which to fire beams */
	Uns16	 TorpHitOdds;			/* percentage chance of torps hitting */
	Uns16	 BeamHitOdds;			/* percentage chance of beams hitting */
    Int16    BeamHitBonus;          /* bonus factor for beam hit odds */
	Uns16	 StrikesPerFighter; 	/* number of strikes per fighter */
	Uns16	 FighterKillOdds;		/* percent chance that a fighter will destroy another fighter */
	Uns16	 FighterBeamExplosive;	/* damage done by each fighter */
	Uns16	 FighterBeamKill;		/* kill power of each fighter */
	Uns16	 ShipMovementSpeed; 	/* ship closing speed in meters/tick */
	Uns16	 FighterMovementSpeed;	/* fighter closing speed in meters/tick */
	Uns16	 BayLaunchInterval; 	/* minimum number of ticks between launches */
	Uns16	 MaxFightersLaunched;	/* max number of fighters out of ship */
	Boolean  AlternativeCombat; 	/* allow fractional damage, etc. */
	Uns32	 StandoffDistance;		/* minimum inter-ship distance */
	Boolean  PlanetsHaveTubes;		/* allow planets to fire torps */
	Boolean  FireOnAttackFighters;	/* fire first on attack fighters, then retreaters */
    Int16    TorpHitBonus;          /* bonus factor for torp hit odds */
    Int16    TubeRechargeBonus;     /* tube tech bonus in recharging */
	Uns16	 ShieldDamageScaling;	/* relative damage to shields */
	Uns16	 HullDamageScaling; 	/* relative damage to hull */
	Uns16	 CrewKillScaling;		/* relative damage to crew */
} Pconfig_Struct;

/*
 *	Enumeration functions
 */

extern Uns16 *EnumerateShipsWithinRadius(Int16 pX, Int16 pY, double pRadius);
extern Uns16 *EnumerateShipsAt(Uns16 pX, Uns16 pY);
extern Uns16 *EnumerateMinesWithinRadius(Int16 pX, Int16 pY, Uns16 pRadius);
extern Uns16 FindPlanetAtShip(Uns16 lShip);
extern Uns16 FindGravityPlanet(Int16 pX, Int16 pY);
extern Uns16 *EnumeratePlanetsWithin(Int16 pX, Int16 pY, Uns16 pRadius);
extern Uns16 *EnumerateShipsAtPlanet(Uns16 pPlanet);
extern Uns16 *EnumerateMinesCovering(Int16 pX, Int16 pY);
extern Uns16 *EnumerateOverlappingMines(Uns16 pMinefield);

/*
 * Interface to host messaging
 */

extern Boolean WriteExternalMessage(RaceType_Def pRace, const char *pMessage);
extern void ExternalMessageCleanup(void);
extern Boolean WriteAUXHOSTMessage(RaceType_Def pRace, const char *pMessage);
extern void AUXHOSTMessageCleanup(void);
extern Boolean WriteMessageToRST(RaceType_Def pRace, int pArgc,
                                            const char *pArgv[]);

/*
 *	 Reading and writing of PHOST data files
 */

/* Multiple-read routines (single-read routines all in a row) */
extern Boolean ReadGlobalData(void);
extern Boolean ReadHostData(void);

/* Multiple-write routine (single-write routines all in a row) */
extern Boolean WriteHostData(void);

/* Single-read routines */
extern Boolean Read_Hullspec_File(void);
extern Boolean Read_Engspec_File(void);
extern Boolean Read_Beamspec_File(void);
extern Boolean Read_Torpspec_File(void);
extern Boolean Read_Truehull_File(void);
extern Boolean Read_Planetname_File(void);
extern Boolean Read_Planets_File(Int16* pControl);
extern Boolean Read_Ships_File(Int16* pControl);
extern Boolean Read_Bases_File(Int16* pControl1, Int16* pControl2);
extern Boolean Read_Mines_File(void);
extern Boolean Read_HostGen_File(void);
extern Boolean Read_Racenames_File(void);
extern Boolean Read_Xyplan_File(void);
extern Boolean Read_HConfig_File(void);
extern Boolean Read_AuxData_File(void);
extern Boolean Read_Turntime_File(void);
extern Boolean ReadWormholeFile(void);

/* Single-write routines */
extern Boolean Write_Planets_File(Int16 pControl);
extern Boolean Write_Ships_File(Int16 pControl);
extern Boolean Write_Bases_File(Int16 pControl1, Int16 pControl2);
extern Boolean Write_Mines_File(void);
extern Boolean Write_HostGen_File(void);
extern Boolean Write_AuxData_File(void);
extern Boolean Write_Racenames_File(void);
extern Boolean Write_HConfig_File(Boolean PHOSTFormat);
extern Boolean WriteWormholeFile(void);

/*
 *   Initializing dynamic files (optional routines in place of reading from disk)
 */

extern void InitializePlanets(void);
extern void InitializeShips(void);
extern void InitializeBases(void);
extern void InitializeMines(void);
extern void InitializeHostGen(void);

/*
 *	 Player status functions
 */

extern Boolean PlayerIsActive(RaceType_Def pPlayer);
extern void PutPlayerIsActive(RaceType_Def pPlayer, Boolean pActive);
extern const char *PlayerPassword(RaceType_Def pPlayer);
extern void PutPlayerPassword(RaceType_Def pPlayer, const char *pPasswd);

/*
 *	 Build Queue Interface
 */

extern Uns16 BuildQueuePeek(BuildOrder_Struct *pOrder);
extern void BuildQueuePush(Uns16 pBase, BuildOrder_Struct *pOrderPtr);
extern void BuildQueueInvalidate(Uns16 pBase);
extern Uns16 ShipsInBuildQueue(void);

/*
 *	  Alliance interface
 */

extern void AllyAddRequest(RaceType_Def pPlayer, RaceType_Def pRace);
extern void AllyDropRequest(RaceType_Def pPlayer, RaceType_Def pRace);
extern Boolean PlayerAllowsAlly(RaceType_Def pPlayer, RaceType_Def pRace,
											AllianceLevel_Def pLevel);
extern void PutPlayerAllowsAlly(RaceType_Def pPlayer, RaceType_Def pRace,
                   AllianceLevel_Def pLevel, AllianceState_Def pState);
extern Boolean PlayerOffersAlliance(RaceType_Def pPlayer, RaceType_Def pRace);
extern Boolean PlayersAreAllies(RaceType_Def pPlayer, RaceType_Def pRace);

/*
 *	  Utility functions
 */

/* File handling */
extern FILE *OpenInputFile(const char *pName, int pLocation);
extern FILE *OpenOutputFile(const char *pName, int pLocation);
extern FILE *OpenUpdateFile(const char *pName, int pLocation);
extern Boolean RemoveGameFile(const char *pName);
extern Boolean CopyFileToFile(FILE *pSrc, FILE *pDst, Uns32 pSize);
extern Boolean CopyGameFile(const char *pSrcName, const char *pDstName);
extern Boolean RenameGameFile(const char *pSrcName, const char *pDstName);
extern Uns32 FileLength(FILE *lFile);

/* Memory allocation */
extern void *MemAlloc(size_t pAmount);
extern void *MemCalloc(size_t pNumber, size_t pSize);
extern void  MemFree(void *pPtr);

/* Reporting */
extern void Error(const char *pStr, ...);
extern void VError(const char *pStr, va_list pAP);
extern void ErrorExit(const char *pStr, ...);
extern void VErrorExit(const char *pStr, va_list pAP);
extern void Warning(const char *pStr, ...);
extern void VWarning(const char *pStr, va_list pAP);
extern void Info(const char *pStr, ...);
extern void VInfo(const char *pStr, va_list pAP);
extern void AssertFail(const char *pExpr, const char *pFile, int pLine);
#ifdef NDEBUG
#define passert(p) ((void)0)
#else
#define passert(p) ((p) ? ((void)0) : AssertFail(0, __FILE__, __LINE__))
#endif

/* Random numbers */
extern void SetRandomSeed(Uns16 seed);
extern RandType_Def RandomRange(RandType_Def pRange);
extern double RandomReal(void);

/* Cross-platform portability */
extern void WordSwapLong(void *pData, Uns16 pNumWords);
extern void WordSwapShort(void* pData, Uns16 pNumWords);
extern Boolean DOSRead16(Uns16 *pData, Uns16 pNum, FILE *pInFile);
extern Boolean DOSRead32(Uns32 *pData, Uns16 pNum, FILE *pInFile);
extern Boolean DOSWrite16(Uns16 *pData, Uns16 pNum, FILE *pOutFile);
extern Boolean DOSWrite32(Uns32 *pData, Uns16 pNum, FILE *pOutFile);

/* Command-line option processing */
extern int pgetopt(int argc, char *argv[], char *optstring);

/*
 *	 Global variables
 */

extern const char *gRootDirectory;
extern const char *gGameDirectory;
extern FILE *gLogFile;
extern Boolean gNewlyMastered;
extern Pconfig_Struct *gPconfigInfo;
extern char *poptarg;
extern int   poptind;

/*
 *	 Initialization and cleanup
 */

extern void InitPHOSTLib(void);
extern void FreePHOSTLib(void);

/*
 *	 Ship functions
 */

extern Boolean IsShipExist(Uns16 pID);
extern void DeleteShip(Uns16 pID);
extern RaceType_Def ShipOwner(Uns16 pID);
extern Uns16 ShipSpeed(Uns16 pID);
extern Uns16 ShipLocationX(Uns16 pID);
extern Uns16 ShipLocationY(Uns16 pID);
extern Uns16 ShipWaypointX(Uns16 pID);
extern Uns16 ShipWaypointY(Uns16 pID);
extern Int16 ShipRelWaypointX(Uns16 pID);
extern Int16 ShipRelWaypointY(Uns16 pID);
extern Uns16 ShipHull(Uns16 pID);
extern Uns16 ShipEngine(Uns16 pID);
extern Uns16 ShipBeamType(Uns16 pID);
extern Uns16 ShipBeamNumber(Uns16 pID);
extern Uns16 ShipTorpType(Uns16 pID);
extern Uns16 ShipTubeNumber(Uns16 pID);
extern Uns16 ShipBays(Uns16 pID);
extern Uns16 ShipAmmunition(Uns16 pID);
extern ShipMission_Def ShipMission(Uns16 pID);
extern RaceType_Def ShipEnemy(Uns16 pID);
extern Uns16 ShipTowTarget(Uns16 pID);
extern Uns16 ShipInterceptTarget(Uns16 pID);
extern Uns16 ShipDamage(Uns16 pID);
extern Uns16 ShipCrew(Uns16 pID);
extern Uns16 ShipDump(Uns16 pID, CargoType_Def pType);
extern Uns16 ShipDumpPlanet(Uns16 pID);
extern Uns16 ShipTransfer(Uns16 pID, CargoType_Def pType);
extern Uns16 ShipTransferShip(Uns16 pID);
extern char* ShipName(Uns16 pID, char* pName);
extern const char* ShipFC(Uns16 pID, char* pFCode);
extern Uns16 ShipCargo(Uns16 pID, CargoType_Def pType);
extern Uns16 CreateShip(RaceType_Def pOwner);
extern void PutShipCargo(Uns16 pID, CargoType_Def pType, Uns16 pCargo);
extern void PutShipName(Uns16 pID, const char* pName);
extern void PutShipOwner(Uns16 pID, RaceType_Def pOwner);
extern void PutShipFC(Uns16 pID, char* pFCode);
extern void PutShipSpeed(Uns16 pID, Uns16 pSpeed);
extern void PutShipWaypointX(Uns16 pID, Uns16 pWaypoint);
extern void PutShipWaypointY(Uns16 pID, Uns16 pWaypoint);
extern void PutShipRelWaypointX(Uns16 pID, Int16 pRel);
extern void PutShipRelWaypointY(Uns16 pID, Int16 pRel);
extern void PutShipLocationX(Uns16 pID, Uns16 pLocation);
extern void PutShipLocationY(Uns16 pID, Uns16 pLocation);
extern void PutShipHull(Uns16 pID, Uns16 pHull);
extern void PutShipEngine(Uns16 pID, Uns16 pEngine);
extern void PutShipBeamType(Uns16 pID, Uns16 pBeamType);
extern void PutShipBeamNumber(Uns16 pID, Uns16 pBeamNumber);
extern void PutShipTorpType(Uns16 pID, Uns16 pTorpType);
extern void PutShipTubeNumber(Uns16 pID, Uns16 pTubeNumber);
extern void PutShipBays(Uns16 pID, Uns16 pBays);
extern void PutShipAmmunition(Uns16 pID, Uns16 pAmmunition);
extern void PutShipMission(Uns16 pID, ShipMission_Def pMission);
extern void PutShipEnemy(Uns16 pID, RaceType_Def pEnemy);
extern void PutShipTowTarget(Uns16 pID, Uns16 pTarget);
extern void PutShipInterceptTarget(Uns16 pID, Uns16 pTarget);
extern void PutShipDamage(Uns16 pID, Uns16 pDamage);
extern void PutShipCrew(Uns16 pID, Uns16 pCrew);
extern void PutShipDump(Uns16 pID, CargoType_Def pType, Uns16 pCargo);
extern void PutShipDumpPlanet(Uns16 pID, Uns16 pPlanet);
extern void PutShipTransfer(Uns16 pID, CargoType_Def pType, Uns16 pCargo);
extern void PutShipTransferShip(Uns16 pID, Uns16 pShip);

/*
 *	  Planet functions
 */

extern Boolean IsPlanetExist(Uns16 pID);
extern Uns16 PlanetLocationX(Uns16 pID);
extern Uns16 PlanetLocationY(Uns16 pID);
extern RaceType_Def PlanetOwner(Uns16 pID);
extern Uns16 PlanetMines(Uns16 pID);
extern Uns16 PlanetFactories(Uns16 pID);
extern Uns16 PlanetDefense(Uns16 pID);
extern Uns32 PlanetCargo(Uns16 pID, CargoType_Def pType);
extern Uns32 PlanetCore(Uns16 pID, CargoType_Def pType);
extern Uns16 PlanetDensity(Uns16 pID, CargoType_Def pType);
extern Uns16 PlanetColTax(Uns16 pID);
extern Uns16 PlanetNatTax(Uns16 pID);
extern Int16 PlanetColHappy(Uns16 pID);
extern Int16 PlanetNatHappy(Uns16 pID);
extern NativeType_Def PlanetNatType(Uns16 pID);
extern NativeGovm_Def PlanetNatGovm(Uns16 pID);
extern Uns32 PlanetNativePopulation(Uns16 pID);
extern Uns16 PlanetTemp(Uns16 pID);
extern Boolean PlanetBuildBase(Uns16 pID);
extern Boolean PlanetHasNatives(Uns16 pPlanet);
extern const char *PlanetName(Uns16 pID, char *pBuffer);
extern const char* PlanetFCode(Uns16 pID, char* pFCode);
extern const char *PlanetTempString(Uns16 pPlanet);
extern void PutPlanetOwner(Uns16 pID, RaceType_Def pOwner);
extern void PutPlanetFC(Uns16 pID, char* pFCode);
extern void PutPlanetMines(Uns16 pID, Uns16 pMines);
extern void PutPlanetFactories(Uns16 pID, Uns16 pFactories);
extern void PutPlanetDefense(Uns16 pID, Uns16 pDefense);
extern void PutPlanetCargo(Uns16 pID, CargoType_Def pType, Uns32 pAmount);
extern void PutPlanetCore(Uns16 pID, CargoType_Def pType, Uns32 pAmount);
extern void PutPlanetDensity(Uns16 pID, CargoType_Def pType, Uns16 pDensity);
extern void PutPlanetColTax(Uns16 pID, Uns16 pTax);
extern void PutPlanetNatTax(Uns16 pID, Uns16 pTax);
extern void PutPlanetColHappy(Uns16 pID, Int16 pHappy);
extern void PutPlanetNatHappy(Uns16 pID, Int16 pHappy);
extern void PutPlanetNatGovm(Uns16 pID, NativeGovm_Def pGovm);
extern void PutPlanetNativePopulation(Uns16 pID, Uns32 pPopulation);
extern void PutPlanetNatType(Uns16 pID, NativeType_Def pType);
#define PutPlanetNativeType PutPlanetNatType
extern void PutPlanetTemp(Uns16 pID, Uns16 pTemp);
extern void PutPlanetBuildBase(Uns16 pID, Boolean pBuild);

/*
 *	  Base functions
 */

extern void DeleteBase(Uns16 pID);
extern Boolean IsBaseExist(Uns16 pID);
extern Boolean IsBasePresent(Uns16 pID);
extern RaceType_Def BaseOwner(Uns16 pID);
extern Uns16 BaseDefense(Uns16 pID);
extern Uns16 BaseDamage(Uns16 pID);
extern Uns16 BaseFighters(Uns16 pID);
extern BaseMission_Def BaseOrder(Uns16 pID);
extern BaseFixMission_Def BaseFixOrder(Uns16 pID);
extern Uns16 BaseFixingShip(Uns16 pID);
extern Uns16 BaseTech(Uns16 pID, BaseTech_Def pType);
extern Uns16 BaseHulls(Uns16 pID, Uns16 pHullType);
extern Uns16 BaseEngines(Uns16 pID, Uns16 pEngineType);
extern Uns16 BaseBeams(Uns16 pID, Uns16 pBeamType);
extern Uns16 BaseTubes(Uns16 pID, Uns16 pTorpType);
extern Uns16 BaseTorps(Uns16 pID, Uns16 pTorpType);
extern Boolean BaseBuildOrder(Uns16 pID, BuildOrder_Struct *pOrder);
extern void CreateBase(Uns16 pPlanet);
extern void PutBaseOwner(Uns16 pID, RaceType_Def pOwner);
extern void PutBaseDefense(Uns16 pID, Uns16 pDefense);
extern void PutBaseDamage(Uns16 pID, Uns16 pDamage);
extern void PutBaseFighters(Uns16 pID, Uns16 pFighters);
extern void PutBaseOrder(Uns16 pID, BaseMission_Def pOrder);
extern void PutBaseFixOrder(Uns16 pID, BaseFixMission_Def pOrder);
extern void PutBaseFixingShip(Uns16 pID, Uns16 pShip);
extern void PutBaseTech(Uns16 pID, BaseTech_Def pType, Uns16 pTech);
extern void PutBaseHulls(Uns16 pID, Uns16 pHullType, Uns16 pNumber);
extern void PutBaseEngines(Uns16 pID, Uns16 pEngineType, Uns16 pNumber);
extern void PutBaseBeams(Uns16 pID, Uns16 pBeamType, Uns16 pNumber);
extern void PutBaseTubes(Uns16 pID, Uns16 pTorpType, Uns16 pNumber);
extern void PutBaseTorps(Uns16 pID, Uns16 pTorpType, Uns16 pNumber);
extern void PutBaseBuildOrder(Uns16 pID, BuildOrder_Struct *pOrderPtr);
extern void ClearBaseBuildOrder(Uns16 pID);
extern void ClearBaseHulls(Uns16 pID);

/*
 *	  Minefield functions
 */

extern Boolean IsMinefieldExist(Uns16 pID);
extern double MinefieldRadius(Uns16 pID);
extern Uns16 MinefieldPositionX(Uns16 pID);
extern Uns16 MinefieldPositionY(Uns16 pID);
extern RaceType_Def MinefieldOwner(Uns16 pID);
extern Uns32 MinefieldUnits(Uns16 pID);
extern Boolean IsMinefieldWeb(Uns16 pID);
extern void PutMinefieldUnits(Uns16 pID, Uns32 pUnits);
extern Uns16 CreateMinefield(Uns16 pXloc, Uns16 pYloc, RaceType_Def pRace,
						Uns32 pUnits, Boolean pIsWeb);

/*
 *	  Hull functions
 */

extern Uns16 HullPicnumber(Uns16 pHullNr);
extern Uns16 HullTritCost(Uns16 pHullNr);
extern Uns16 HullDurCost(Uns16 pHullNr);
extern Uns16 HullMolyCost(Uns16 pHullNr);
extern Uns16 HullFuelCapacity(Uns16 pHullNr);
extern Uns16 HullCrewComplement(Uns16 pHullNr);
extern Uns16 HullEngineNumber(Uns16 pHullNr);
extern Uns16 HullMass(Uns16 pHullNr);
extern Uns16 HullTechLevel(Uns16 pHullNr);
extern Uns16 HullCargoCapacity(Uns16 pHullNr);
extern Uns16 HullBays(Uns16 pHullNr);
extern Uns16 HullTubeNumber(Uns16 pHullNr);
extern Uns16 HullBeamNumber(Uns16 pHullNr);
extern Uns16 HullMoneyCost(Uns16 pHullNr);
extern Boolean HullDoesAlchemy(Uns16 pHullNr);
extern Boolean HullDoesRefinery(Uns16 pHullNr);
extern Boolean HullHeatsTo50(Uns16 pHullNr);
extern Boolean HullCoolsTo50(Uns16 pHullNr);
extern Boolean HullHeatsTo100(Uns16 pHullNr);
extern Boolean HullCanHyperwarp(Uns16 pHullNr);
extern Boolean HullIsGravitonic(Uns16 pHullNr);
extern Boolean HullScansAllWormholes(Uns16 pHullNr);
extern Boolean HullCanCloak(Uns16 pHullNr);
extern Boolean HullDoesAdvancedRefinery(Uns16 pHullNr);
extern Boolean HullIsGamblingShip(Uns16 pHullNr);
extern Boolean HullIsAntiCloaking(Uns16 pHullNr);
extern Boolean HullDoesImperialAssault(Uns16 pHullNr);
extern Boolean HullDoesChunneling(Uns16 pHullNr);
extern Boolean HullHasRamScoop(Uns16 pHullNr);
extern Boolean HullDoesBioscan(Uns16 pHullNr);
extern Boolean HullHasGloryDevice(Uns16 pHullNr, Boolean *pHighDamage);
extern Uns16 TrueHull(RaceType_Def pRace, Uns16 pHullIndex);
extern const char* HullName(Uns16 pHullNr, char* pBuffer);

/*
 *	  Engine functions
 */

extern char* EngineName(Uns16 pEngNr, char* pBuffer);
extern Uns16 EngMoneyCost(Uns16 pEngNr);
extern Uns16 EngTritCost(Uns16 pEngNr);
extern Uns16 EngDurCost(Uns16 pEngNr);
extern Uns16 EngMolyCost(Uns16 pEngNr);
extern Uns16 EngTechLevel(Uns16 pEngNr);
extern Uns32 EngFuelConsumption(Uns16 pEngNr, Uns16 pSpeed);

/*
 *	 Torpedo/tube functions
 */

extern char* TorpName(Uns16 pTorpNr, char* pBuffer);
extern Uns16 TorpTorpCost(Uns16 pTorpNr);
extern Uns16 TorpTubeCost(Uns16 pTorpNr);
extern Uns16 TorpTritCost(Uns16 pTorpNr);
extern Uns16 TorpDurCost(Uns16 pTorpNr);
extern Uns16 TorpMolyCost(Uns16 pTorpNr);
extern Uns16 TorpTubeMass(Uns16 pTorpNr);
extern Uns16 TorpTechLevel(Uns16 pTorpNr);
extern Uns16 TorpKillPower(Uns16 pTorpNr);
extern Uns16 TorpDestructivePower(Uns16 pTorpNr);

/*
 *	 Beam weapon functions
 */

extern char* BeamName(Uns16 pBeamNr, char* pBuffer);
extern Uns16 BeamMoneyCost(Uns16 pBeamNr);
extern Uns16 BeamTritCost(Uns16 pBeamNr);
extern Uns16 BeamDurCost(Uns16 pBeamNr);
extern Uns16 BeamMolyCost(Uns16 pBeamNr);
extern Uns16 BeamMass(Uns16 pBeamNr);
extern Uns16 BeamTechLevel(Uns16 pBeamNr);
extern Uns16 BeamKillPower(Uns16 pBeamNr);
extern Uns16 BeamDestructivePower(Uns16 pBeamNr);

/*
 *	  Race name functions
 */

extern const char* RaceLongName(RaceType_Def pRace, char* pBuffer);
extern const char* RaceShortName(RaceType_Def pRace, char *pBuffer);
extern const char* RaceNameAdjective(RaceType_Def pRace, char *pBuffer);
extern void PutRaceLongName(RaceType_Def pRace, const char* pBuffer);
extern void PutRaceShortName(RaceType_Def pRace, const char *pBuffer);
extern void PutRaceNameAdjective(RaceType_Def pRace, const char *pBuffer);

/*
 *	  Host run time functions
 */

extern char *HostTimeStamp(char *pTimeStamp);
extern Uns16 TurnNumber(void);

/*
 *	  Wormhole functions
 */
extern Uns16 NumWormholes(void);
extern Uns16 CreateWormhole(void);
extern void DeleteWormhole(Uns16 pID);
extern Uns16 WormholeStabilityCode(Uns16 pID);
extern Uns16 WormholeStartX(Uns16 pID);
extern Uns16 WormholeStartY(Uns16 pID);
extern Uns16 WormholeEndX(Uns16 pID);
extern Uns16 WormholeEndY(Uns16 pID);
extern Int16 WormholeMass(Uns16 pID);
extern float WormholeInstability(Uns16 pID);
extern Uns16 WormholeWaypointStartX(Uns16 pID);
extern Uns16 WormholeWaypointStartY(Uns16 pID);
extern Uns16 WormholeWaypointEndX(Uns16 pID);
extern Uns16 WormholeWaypointEndY(Uns16 pID);
extern void PutWormholeStartX(Uns16 pID, Uns16 pX);
extern void PutWormholeStartY(Uns16 pID, Uns16 pY);
extern void PutWormholeEndX(Uns16 pID, Uns16 pX);
extern void PutWormholeEndY(Uns16 pID, Uns16 pY);
extern void PutWormholeMass(Uns16 pID, Int16 pMass);
extern void PutWormholeInstability(Uns16 pID, float pInstability);
extern void PutWormholeWaypointStartX(Uns16 pID, Uns16 pX);
extern void PutWormholeWaypointStartY(Uns16 pID, Uns16 pY);
extern void PutWormholeWaypointEndX(Uns16 pID, Uns16 pX);
extern void PutWormholeWaypointEndY(Uns16 pID, Uns16 pY);

#ifdef __cplusplus
}
#endif

#endif /* _PHOSTPDK_H_ */

