#include <assert.h>
#include <stdio.h>
#include <ctype.h>
#include "global.h"

#define SAVE_FILE "save.groups"
#define MAX_GROUPS 600
#define NUM_RACES 40

#define PL_ALL -1
#define SET_TECH(T) (T)
#define GET_TECH(T) (T)

struct groups
{
  int groupid;
  int race;
  int num_ships;
  int planet_id;
  char *ship_name;
  char *planet_name;
  char *what;
  float dist_from_planet, quantity;
  float drive, weapon, shield, cargo;
  unsigned char set;
};

static struct group_data
{
  struct groups group[MAX_GROUPS];
  int num_groups;
  char *racenames[NUM_RACES];
  int num_races;
  int no_save;

  struct race_data {
    int num_techs;
    struct tech_level_def {
      float drive, weapon, shield, cargo;	/* tech levels */
    } *tech_level;
  } data[NUM_RACES];	/* data specific to each race */
} g;

typedef struct tech_level_def *techPtr;

extern int line_num;
extern int do_more;
int g_print_group_tech;

init_groups()
{
   /* First race reserved for Your race */
   g.num_races = 1;
   g.no_save = 0;
   zalloc_cpy(&g.racenames[0], YOUR); 

   read_group_data(SAVE_FILE);
}

/* clear all groups */
clear_groups(i_race)
int i_race;
{
    int id;

    for (id = 0; id < g.num_groups; id++)
      if ((i_race == RACE_ALL) || (g.group[id].race == i_race))
      {
	g.group[id].set = 0;
	g.group[id].race = -1;
	g.group[id].groupid = -1;
      }
}

save_groups()
{
   if (g.no_save)
   {
     printf("Not saving to %s due to modifications done by send orders.\n",
	SAVE_FILE);
     return;
   }
   do_more = 0;
   write_group_data(SAVE_FILE);
   do_more = 1;
}

write_group_data(cp_file)
char *cp_file;
{
  FILE *fp;

  if ((fp = fopen(cp_file, "w")) == NULL)
  {
    perror(cp_file);
    return;
  }
  write_all_groups(fp);

  printf("Wrote %s\n", cp_file);

  fclose(fp);
}


print_groups()
{
  write_all_groups(stdout);
}

write_all_groups(fp)
FILE *fp;
{
  line_num = 0;
  write_group(fp, RACE_ALL, PL_ALL, 1);
}

write_pl_groups(i_pl)
int i_pl;
{
  write_group(stdout, RACE_ALL, i_pl, 1);
}

write_group(fp, i_race, i_pl, i_dotitle)
FILE *fp;
int i_race;	/* may be RACE_ALL */
int i_pl;	/* may be PL_ALL */
int i_dotitle;
{
  int i;
  int r;
  int once;
  int printed_a_line;

  once = 0;
  printed_a_line = 0;

  if (i_race == RACE_ERR)
    return(0);
    
  if (i_race == RACE_ALL)
  {
    for (r = 0; r < g.num_races; r++)
      if (write_group(fp, r, i_pl, i_dotitle))
	printed_a_line = 1;
    return(printed_a_line);
  }
  for (i = 0; i < g.num_groups; i++)
  {
    if ((g.group[i].race == i_race) && g.group[i].set)
    {
      if (i_pl == PL_ALL || (i_pl == g.group[i].planet_id))
      {
	if (i_dotitle && !once)
	{
	  fprintf(fp, "\n"); chk_more(1);
	  fprintf(fp, "\t\t%s %s %s\n", ABBR_GROUPS, 
		g.racenames[i_race], GROUPS); 
	      chk_more(1);
	  fprintf(fp, "\n"); chk_more(1);

	  fprintf(fp, "G#  #   Type            Planet         Dist");

	  if (g_print_group_tech)
	    fprintf(fp, "  DRIVE WEAPON SHIELD CARGO\n");
	  else
	    fprintf(fp, "  LY  ATTACK DEF QUANTITY\n");
	  chk_more(1);

	  once = 1;
	}
	write_group_line(fp, i, i_race);
	printed_a_line = 1;
	chk_more(1);
      }
    }
  }
  return(printed_a_line);
}

write_group_line(fp, i, i_race)
FILE *fp;
int i;
int i_race;
{
      int ship_id;
      int tech_id;
      int print_tech;
      char line[80];

      fprintf(fp, "%-3d %-3d %-15s %-15s",
	g.group[i].groupid,
	g.group[i].num_ships,
	g.group[i].ship_name,
	g.group[i].planet_name);

      if (g.group[i].dist_from_planet != 0)
	fprintf(fp, "%-5.2f", g.group[i].dist_from_planet);
      else
	fprintf(fp, "  x  ");

      print_tech = 0;
      ship_id = get_ship_id(g.group[i].ship_name, i_race);
      tech_id = get_tech_id(i_race, 
		  GET_TECH(g.group[i].drive),
		  GET_TECH(g.group[i].weapon),
		  GET_TECH(g.group[i].shield),
		  GET_TECH(g.group[i].cargo));

      if ((fp != stdout) || g_print_group_tech)
	print_tech = 1;
      else if (ship_id == IS_ERR) 
      {
	/* printf("Warning: could not find ship %s for race %s.\n",
	 g.group[i].ship_name, get_race_name(i_race)); */
	print_tech = 1;
      }
      else if (tech_id == IS_ERR)
      {
	printf("Warning: could not tech level %f,%f,%f,%f for race %s.\n",
		  GET_TECH(g.group[i].drive),
		  GET_TECH(g.group[i].weapon),
		  GET_TECH(g.group[i].shield),
		  GET_TECH(g.group[i].cargo),
	 	  get_race_name(i_race));
	print_tech = 1;
      }
      if (print_tech)
      {
	fprintf(fp, " [%3.2f,%3.2f,%3.2f,%3.2f]", 
		  GET_TECH(g.group[i].drive),
		  GET_TECH(g.group[i].weapon),
		  GET_TECH(g.group[i].shield),
		  GET_TECH(g.group[i].cargo));
      }
      else
      {
	calc_ship_data(ship_id, tech_id);
	get_ship_brief_data(ship_id, line);
	fprintf(fp, " %s", line);
      }
      if ((g.group[i].what == 0) || (g.group[i].what[0] == '-'))
	fprintf(fp, "\n");
      else	
	fprintf(fp, " %s\n", g.group[i].what);
}

set_group_data(i_race, i_groupid, i_numships, cp_shipname, cp_planetname, f_dist, f_drive, f_weapon, f_shield, f_cargo, cp_what, f_quantity)
int i_race;
int i_groupid;
int i_numships;
char *cp_shipname;
char *cp_planetname;
char *cp_what;
float f_dist;
float f_drive, f_weapon, f_shield, f_cargo;
float f_quantity;
{
    int planet_id;
    int id;

    /* okay to have a unrecognized planet id */
    planet_id = decode_planet_name(cp_planetname);

    if ((i_groupid < 0) || (i_groupid >= MAX_GROUPS))
    {
      printf("set_group_data(), line %d : Illegal group number: %d\n",
	line_num, i_groupid);
      return(IS_ERR);
    }
    id = get_group_slot(i_race, i_groupid);

    /* add if this group does not exist */
    if (id == g.num_groups)
      id = g.num_groups++;

    g.group[id].set = 1;
    /* set groupid if current -1(unset) or > 0 */
    if ((g.group[id].groupid == -1) || (i_groupid > 0))
      g.group[id].groupid = i_groupid;
    g.group[id].race = i_race;
    g.group[id].num_ships = i_numships;
    g.group[id].planet_id = planet_id;
    g.group[id].drive = SET_TECH(f_drive);
    g.group[id].weapon = SET_TECH(f_weapon);
    g.group[id].shield = SET_TECH(f_shield);
    g.group[id].cargo = SET_TECH(f_cargo);
    zrealloc_cpy(&(g.group[id].planet_name), cp_planetname);
    g.group[id].dist_from_planet = f_dist;
    zrealloc_cpy(&(g.group[id].ship_name), cp_shipname);
    zrealloc_cpy(&(g.group[id].what), cp_what);
    if ((cp_what != 0) && strlen(g.group[id].what) > 3)
       g.group[id].what[3] = '\0';
    g.group[id].quantity = f_quantity;

    /* the tech level of the ship should be recalled for the race */
    new_tech_level(i_race, f_drive, f_weapon, f_shield, f_cargo);
}

new_alien_group_data(i_race, i_groupid, i_numships, cp_shipname, cp_planetname, f_dist, f_drive, f_weapon, f_shield, f_cargo, cp_what, f_quantity)
int i_race, i_groupid;
int i_numships;
char *cp_shipname;
char *cp_planetname;
char *cp_what;
float f_dist;
float f_drive, f_weapon, f_shield, f_cargo;
float f_quantity;
{
    int planet_id;
    int id;

    /* okay to have a unrecognized planet id */
    planet_id = decode_planet_name(cp_planetname);

    /* don't do anything if this group is already present */
    if ((id = group_exists(i_race, i_numships, cp_shipname, cp_planetname, f_dist,
      f_drive, f_weapon, f_shield, f_cargo)) >= 0)
    {
      return(id);
    }

    id = get_group_slot(i_race, -1);

    /* add if this group does not exist */
    if (id == g.num_groups)
      id = g.num_groups++;

    g.group[id].set = 1;
    g.group[id].groupid = i_groupid;
    g.group[id].race = i_race;
    g.group[id].num_ships = i_numships;
    g.group[id].planet_id = planet_id;
    g.group[id].drive = SET_TECH(f_drive);
    g.group[id].weapon = SET_TECH(f_weapon);
    g.group[id].shield = SET_TECH(f_shield);
    g.group[id].cargo = SET_TECH(f_cargo);
    zrealloc_cpy(&(g.group[id].planet_name), cp_planetname);
    g.group[id].dist_from_planet = f_dist;
    zrealloc_cpy(&(g.group[id].ship_name), cp_shipname);
    g.group[id].quantity = f_quantity;
    zrealloc_cpy(&(g.group[id].what), cp_what);
    if ((cp_what != 0) && strlen(g.group[id].what) > 3)
       g.group[id].what[3] = '\0';

    /* the tech level of the ship should be recalled for the race */
    new_tech_level(i_race, f_drive, f_weapon, f_shield, f_cargo);

    return(id);
}

get_group_slot(i_race, i_group)
int i_race;
int i_group;
{
    int id;

    if (i_group >= 0)
    {
      for (id = 0; id < g.num_groups; id++)
	if (g.group[id].set)
	  if ((g.group[id].race == i_race) && (g.group[id].groupid == i_group))
	    break;
    }
    else
      id = g.num_groups;

    /* if not found return unused group slot */
    if (id == g.num_groups)
    {
      for (id = 0; id < g.num_groups; id++)
	if (!g.group[id].set)
	  break;

      /* if no unused add new slot */
      if (id == g.num_groups)
	id = g.num_groups++;
    }
    return(id);
}

group_exists(i_race, i_numships, cp_shipname, cp_planetname, f_dist,
f_drive, f_weapon, f_shield, f_cargo)
int i_race;
int i_numships;
char *cp_shipname;
char *cp_planetname;
float f_dist;
float f_drive, f_weapon, f_shield, f_cargo;
{
    int id;

    for (id = 0; id < g.num_groups; id++)
      if (g.group[id].set && (g.group[id].race == i_race) &&
	 (g.group[id].num_ships == i_numships) && 
	 !strcmp(g.group[id].ship_name, cp_shipname) &&
	 !strcmp(g.group[id].planet_name, cp_planetname) &&
          (g.group[id].dist_from_planet == f_dist) &&
	  (g.group[id].drive == f_drive) &&
	  (g.group[id].weapon == f_weapon) &&
	  (g.group[id].shield == f_shield) &
	  (g.group[id].cargo == f_cargo))
	return(id);	/* good grief what a match! */
    
    return(-1);
}

print_sorted_groups(i_race)
int i_race;
{
    int pl[NUM_PLANETS]; 
    int i, dotitle;

    /* get list of planets first */
    for (i = 0; i < NUM_PLANETS; i++)
      pl[i] = 0;

    for (i = 0; i < g.num_groups; i++)
      pl[g.group[i].planet_id] = 1;

    dotitle = 1;
    line_num = 0;

    /* display each planet set in order of set */
    for (i = 0; i < NUM_PLANETS; i++)
    {
      if (pl[i])
      {
	if (write_group(stdout, i_race, i, dotitle))
	  dotitle = 0;
      }
    }
}

new_race_name(cp_race)
char *cp_race;
{
    int next;
    int r;

    if (cp_race[0] == '\0')
      return(RACE_ERR);
    /*
     *  Make sure this new race is not already in here or the
     *  name is part of another name.
     */
    for (r = 0; r < g.num_races; r++)
    {
      if (strLocate(g.racenames[r], cp_race) >= 0)
      {
	printf("Error: Already have race '%s' in race list.\n", 
		g.racenames[r]);
	return(RACE_ERR);
      }
    }
    if (g.num_races >= NUM_RACES)
    {
      printf("Too many races. Could not add %s\n", cp_race);
      return(RACE_ERR);
    }
    next = g.num_races++;
    zalloc_cpy(&g.racenames[next], cp_race);
    /* add basics tech level for race */
    new_tech_level(next, (float) 1.0, (float) 1.0, (float) 1.0, (float) 1.0);
    return(next);
}

/* allow user to select a race */
select_race_id(start)
char *start;
{
  int i, id;
  char line[80];
  char *s;

  if (g.num_races == 1)
    return(0);	/* only one race */

  if (start != 0)
  {
    strcpy(line, start);

    if (strIsInt(line))
    {
      id = atoi(line);
      if ((id >= 0) && (id < g.num_races))
	return(id);
    }
  }
  while(1)
  {
    for (i = 0; i < g.num_races; i++)
      printf(" %d - %s\n", i, g.racenames[i]);

    printf("> ");
    gets(line);

    id = atoi(line);
    if ((id >= 0) && (id < g.num_races))
      return(id);
  }
}



get_race_id(cp)
char *cp;
{
    int r;

    for (r = 0; r < g.num_races; r++)
      if (!strcmp(cp, g.racenames[r]))
	return(r);

    printf("Do not know race '%s'.\n", cp);
    return(RACE_ERR);
}

/* 'cp' will contain the race name within it */
/* It is okay for the race not to be known */
find_race_id(cp)
char *cp;
{
    extern int g_your_race_id;
    int r;

    for (r = 0; r < g.num_races; r++)
      if (strLocate(cp, g.racenames[r]) >= 0)
	return(r);

    if (strLocate(cp, YOUR) >= 0)
      return(g_your_race_id);

    return(RACE_ERR);
}

print_groups_brief(i_race, f_x, f_y, f_dist)
int i_race, f_x, f_y, f_dist;
{
    int pl[NUM_PLANETS]; 
    int i, dotitle;

    /* get list of planets first */
    for (i = 0; i < NUM_PLANETS; i++)
      pl[i] = 0;

    for (i = 0; i < g.num_groups; i++)
      pl[g.group[i].planet_id] = 1;

    /* reset to zero planets outside circle */
    if (f_dist > 0)
      for (i = 0; i < NUM_PLANETS; i++)
	if (pl[i])
	  if (planet_xy_dist(f_x, f_y, i) > f_dist)
	    pl[i] = 0;

    line_num = 0;

    /* display each planet set in order of set */
    for (i = 0; i < NUM_PLANETS; i++)
    {
      if (pl[i])
      {
	write_group_brief(stdout, i_race, i);
	dotitle = 0;
      }
    }
}

print_brief_groups()
{
  line_num = 0;
  write_group_brief(stdout, RACE_ALL, PL_ALL);
}

write_group_brief(fp, i_race, i_pl)
FILE *fp;
int i_race;	/* may be RACE_ALL */
int i_pl;	/* may be PL_ALL */
{
  char line[256];
  int i;
  int r;
  int p;
  int once;
  int numchrs, len;
  char *name;

  if (i_race == RACE_ERR)
    return;
    
  if (i_race == RACE_ALL)
  {
    for (r = 0; r < g.num_races; r++)
      write_group_brief(fp, r, i_pl);
    return;
  }
  if (i_pl == PL_ALL)
  {
    for (p = 0; p < NUM_PLANETS; p++)
      write_group_brief(fp, i_race, p);
    return;
  }
  once = 0;
  numchrs = 0;

  for (i = 0; i < g.num_groups; i++)
  {
    if ((g.group[i].race == i_race) && (i_pl == g.group[i].planet_id))
    {
      if (!once)
      {
	fprintf(fp, "%-11s: ", name = g.group[i].planet_name);
	once = 1;
	len = strlen(name) + 2;
	if (len < 13)
	  len = 13;
	numchrs += len;
      }
      else
      {
	fprintf(fp, ", ");
	numchrs += 2;
      }
      if (g.group[i].dist_from_planet != 0)
      {
	if (g.group[i].groupid != 0) {
	  sprintf(line, "%d %s (%d,%.2f)",
	    g.group[i].num_ships,
	    g.group[i].ship_name,
	    g.group[i].groupid,
	    g.group[i].dist_from_planet);
	} else {
	  sprintf(line, "%d %s (%.2f)",
	    g.group[i].num_ships,
	    g.group[i].ship_name,
	    g.group[i].dist_from_planet);
	}
      }
      else
      {
	if (g.group[i].groupid != 0) {
	  sprintf(line, "%d %s (%d)",
	    g.group[i].num_ships,
	    g.group[i].ship_name,
	    g.group[i].groupid);
	} else {
	  sprintf(line, "%d %s",
	    g.group[i].num_ships,
	    g.group[i].ship_name);
	}
      }
      chk_more(break_print(fp, line, &numchrs));
    }
  }
  if (once)
  {
    fprintf(fp, "\n");
    chk_more(1);
  }
}

#define LIMIT 72
#define INDENT "\t     "

/* returns number of returns done */
break_print(fp, cp_line, ip_numchrs)
FILE *fp;
char *cp_line;
int *ip_numchrs;
{
    int len;
    int num;
    int num_ret;
    char *s;

    num_ret = 0;
    len = strlen(cp_line);

    if (*ip_numchrs > LIMIT)
    {
      fprintf(fp, "\n%s%s", INDENT, cp_line);
      *ip_numchrs = strlen(INDENT) + strlen(cp_line);
      num_ret++;
    }
    else if (len + *ip_numchrs > LIMIT)
    {
      for (s = cp_line + len; s > cp_line; s--)
      {
	if (*s == ' ')
	{
	  num = s - cp_line;

	  if (len + *ip_numchrs - num < LIMIT)
	    break;
	}
      }
      if (s > cp_line)
      {
	*s = '\0';
	fprintf(fp, "%s", cp_line);
	fprintf(fp, "\n%s%s", INDENT, s+1);
	*ip_numchrs = strlen(s+1) + strlen(INDENT);
	num_ret++;
      }
      else
      {
	fprintf(fp, "\n%s%s", INDENT, cp_line);
	*ip_numchrs = strlen(INDENT) + strlen(cp_line);
	num_ret++;
      }
    }
    else
    {
      fprintf(fp, "%s", cp_line);
      *ip_numchrs += len;
    }
    return(num_ret);
}

/* send the group along its way.  If i_numships
 * is less than the number of ships in the group,
 * first break off the group, then send new group 
 * along it's way.
 */
send_group(i_groupid, i_numships, i_planet)
int i_groupid;
int i_numships;
int i_planet;
{
    float dist, ly;
    int shipid, id, new_id;
    int i;

    id = get_group_slot(i_groupid);

    if (id >= g.num_groups)
    {
      printf("send_group() : Illegal group number: %d\n", i_groupid);
      return(IS_ERR);
    }
    if (g.group[id].dist_from_planet > 0)
    {
      printf("send_group() : group number %d already in route to planet %s (%f)\n", 
	g.group[id].groupid, get_planet_name(g.group[id].planet_id,0),
	g.group[id].dist_from_planet);
      return(IS_ERR);
    }
    if (i_numships < g.group[id].num_ships)
    {
      if ((new_id = break_off_group(i_groupid, i_numships)) < 0)
      {
	printf("send_group() : could not break off group %d\n", i_groupid);
	return(IS_ERR);
      }
      id = new_id;
    }
    g.no_save = 1;

    dist = planet_dist(g.group[id].planet_id, i_planet);

    shipid = -1;

    for (i = 0; i < get_num_races(); i++)
      if ((shipid = get_ship_id(g.group[id].ship_name, i)) >= 0)
	break;

    if (shipid < 0)
    {
      printf("Could not find ship name '%d' in any ship list.", g.group[id].ship_name);
      return(IS_ERR);
    }
    ly = get_ship_ly(shipid);

    if (ly >= dist)
    {
      g.group[id].dist_from_planet = 0;
      g.group[id].planet_id = i_planet;
    }
    else
    {
      g.group[id].dist_from_planet = dist - ly;
      g.group[id].planet_id = i_planet;
    }
    return(0);
}

/* break off the number of ships specified in the
 * passed group and return new group id.
 */
break_off_group(i_groupid, i_numships)
int i_groupid;
int i_numships;
{
    int new;
    int id;

    id = get_group_slot(i_groupid);

    if (id >= g.num_groups)
    {
      printf("break_off_group() : Illegal group number: %d\n", i_groupid);
      return(IS_ERR);
    }
    if (i_numships >= g.group[id].num_ships)
    {
      printf("break_off_group() : specified too many ships, specified %d, max %d.\n",
	    i_numships, g.group[id].num_ships-1);
      return(IS_ERR);
    }
    g.no_save = 1;

    new = new_group_slot();

    g.group[id].num_ships -= i_numships;

    g.group[new].race = g.group[id].race;
    g.group[new].num_ships = i_numships;
    zrealloc_cpy(&(g.group[new].ship_name), g.group[id].ship_name);
    g.group[new].planet_id = g.group[id].planet_id;
    g.group[new].dist_from_planet = g.group[id].dist_from_planet;
    g.group[new].set = 1;

    return(new);
}

new_group_slot()
{
  int slot, group, i;
  int found;

  /* get an unused group id */
  for (group = 1, found = 0; !found && group < MAX_GROUPS; group++)
  {
    for (i = 0; i < g.num_groups; i++)
    {
      if (g.group[i].groupid == group)
      {
	if (!g.group[i].set)
	  found = 1;

	break;
      }
    }
    if (i == g.num_groups)
      found = 1;
  }
  if (group == MAX_GROUPS)
  {
    printf("No more room for groups (max %d)\n", MAX_GROUPS);
    return(-1);
  }
  /* get a new slot for our group */
  for (slot = 0; slot < MAX_GROUPS; slot++)
    if (!g.group[slot].set)
      break;

  if (slot == g.num_groups)
    g.num_groups++;
  
  g.group[slot].groupid = group;
  g.group[slot].race = 0;
  g.group[slot].num_ships = 0;
  zrealloc_cpy(&(g.group[slot].ship_name), 0);
  g.group[slot].planet_id = 0;
  g.group[slot].dist_from_planet = 0;
  g.group[slot].set = 1;

  return(slot);
}

char *getGroupSaveFile()
{
  return SAVE_FILE;
}

char *get_race_name(race)
{
  return (g.racenames[race]);
}

get_race_techs(race, tech_level, fp_drive, fp_weapon, fp_shield, fp_cargo)
int race;
int tech_level;
float *fp_drive;
float *fp_weapon;
float *fp_shield;
float *fp_cargo;
{
  /* if no techs add default tech */
  if (g.data[race].num_techs == 0)
  {
    new_tech_level(race, 1.0, 1.0, 1.0, 1.0);
  }
  assert (race >= 0 && race < g.num_races);
  assert (tech_level >= 0 && tech_level < g.data[race].num_techs);

  *fp_drive = g.data[race].tech_level[tech_level].drive;
  *fp_weapon = g.data[race].tech_level[tech_level].weapon;
  *fp_shield = g.data[race].tech_level[tech_level].shield;
  *fp_cargo = g.data[race].tech_level[tech_level].cargo;
}

get_num_races()
{
  return(g.num_races);
}
get_num_techs(race)
{
  return(g.data[race].num_techs);
}

chk_race_id(id)
int id;
{
  if ((id < 0) || (id >= g.num_races))
  {
    printf("race id %d out of range.\n", id);
    return(-1);
  }
  return(id);
}

chk_tech_level(race,lev)
int race, lev;
{
  if (chk_race_id(race) < 0)
    return(-1);

  if (lev < 0 || lev >= g.data[race].num_techs)
  {
    printf("tech level %d out of range.\n", lev);
    return(-1);
  }
  return(lev);
}

new_tech_level(race, drive, weapon, shield, cargo)
int race;
float drive, weapon, shield, cargo;
{
    int new;
    int i;

    if (chk_race_id(race) < 0)
      return;
    /* 
     * For simplicity tech level zero means use tech level one 
     */
    if (drive == 0.0)
      drive = 1.0;
    if (weapon == 0.0)
      weapon = 1.0;
    if (shield == 0.0)
      shield = 1.0;
    if (cargo == 0.0)
      cargo = 1.0;
    /*
     * Only add if we don't have it yet.
     */
    if (get_tech_id(race, drive, weapon, shield, cargo) >= 0)
      return;	/* already have this one! */

    new = g.data[race].num_techs++;

    if (g.data[race].tech_level == 0)
    {
      g.data[race].tech_level = (techPtr) calloc(1, 
	 sizeof(struct tech_level_def));
    }
    else
    {
      g.data[race].tech_level = (techPtr) realloc(g.data[race].tech_level,
		 g.data[race].num_techs * sizeof(struct tech_level_def));
    }
    g.data[race].tech_level[new].drive = drive;
    g.data[race].tech_level[new].weapon = weapon;
    g.data[race].tech_level[new].shield = shield;
    g.data[race].tech_level[new].cargo = cargo;
}


get_tech_id(i_race, drive, weapon, shield, cargo)
float drive, weapon, shield, cargo;
{
    int i;
    /* 
     * For simplicity tech level zero means use tech level one 
     */
    if (drive == 0.0)
      drive = 1.0;
    if (weapon == 0.0)
      weapon = 1.0;
    if (shield == 0.0)
      shield = 1.0;
    if (cargo == 0.0)
      cargo = 1.0;

    if (g.data[i_race].tech_level != 0)
    {
      for (i = 0; i < g.data[i_race].num_techs; i++)
	if ((g.data[i_race].tech_level[i].drive == drive) &&
	    (g.data[i_race].tech_level[i].weapon == weapon) &&
	    (g.data[i_race].tech_level[i].shield == shield) &&
	    (g.data[i_race].tech_level[i].cargo == cargo))
	return(i);
    }
    return(IS_ERR);
}
