/**************************************************************
 * PLANET.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroWar 1.00 - Empire database library.                   *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     28-Apr-97                                         *
 **************************************************************/


#define _PLANET_C_


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astrowar.h"
#include "planet.h"


/* some macros ************************************************/


/* record sizes */
#define DATASIZE 44  /* size of packed data record */
#define IND1SIZE 20  /* size of packed fleet number index */

/* flags */
#define PF_DELETED 0x01 /* deleted flag */
#define P_SCOUTED 0x01 /* scouted flag */


/* structures *************************************************/


/* hidden planet structure */
typedef struct {

    /* record fields */
    char  deleted,   /* 'deleted' flag */
          name[16],  /* planet name */
          owner[16]; /* planet's owner */
    pos   pp;        /* position */
    short prod;      /* planet's production */
    long  ships;     /* ships in orbit */
    int   scouted:1; /* 'scouted' flag */

    /* index maintenance fields */
    char  infile;    /* 0=new record, !0=somewhere in file */
    long  pos;       /* record position when last read */
    char  oname[16]; /* original planet name */

} planet;

/* indexes */
typedef struct {   /* planet number index */
    long pos;          /* position of record in data file */
    char name[16];     /* planet name */
} ind1;


/* global variables *******************************************/


FILE *planet_data, /* planet data file */
     *planet_ind1; /* planet number index file */


/* level 3 routines *******************************************/


/* planet_readind1() - read a number index record */
result planet_readind1(ind1 *i)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(planet_ind1) ) == EOF)
        return R_EOF;
    i->pos = pos;
    FREADLONGR(i->pos, planet_ind1);
    fread(i->name, 16, 1, planet_ind1);
    return R_OK;
}


/* level 2 routines *******************************************/


/* planet_findind1() - find a trans index record */
long planet_findind1(ind1 *i, char *name)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(planet_ind1); fseek(planet_ind1, 0, SEEK_END);
    if( ftell(planet_ind1) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(planet_ind1) - 8 ) / IND1SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(planet_ind1, 8 + IND1SIZE * middle, SEEK_SET);
    planet_readind1(i);
    while( stricmp(i->name, name) && start <= end )
    {
        if( stricmp(i->name, name) < 0 )
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(planet_ind1, 8 + IND1SIZE * middle, SEEK_SET);
        planet_readind1(i);
    }

    /* return value */
    if( !stricmp(i->name, name) )
        return 8 + middle * IND1SIZE;
    fseek(planet_ind1, pos, SEEK_SET);
    return 0;
}

/* planet_writeind1() - write a number index record */
result planet_writeind1(ind1 i)
{
    if( fputc(i.pos & 0xff, planet_ind1) == EOF ) return R_FILE;
    FWRITELONGR(i.pos,  planet_ind1);
    fwrite(i.name, 16, 1, planet_ind1);
    return R_OK;
}


/* level 1 routines *******************************************/


/* planet_readdata() - read a data record */
result planet_readdata(planet *p)
{
    int  deleted; /* deleted flag */
    long pos;     /* position in file */

    /* check for EOF */
    pos = ftell(planet_data);
    if(( deleted = fgetc(planet_data) ) == EOF)
        return R_EOF;

    /* read rest of record */
    p->deleted = deleted;
    fread(p->name,  16, 1, planet_data);
    fread(p->owner, 16, 1, planet_data);
    FREADSHORT(p->pp.x,    planet_data);
    FREADSHORT(p->pp.y,    planet_data);
    FREADSHORT(p->prod,    planet_data);
    FREADLONG(p->ships,    planet_data);
    p->scouted = ( fgetc(  planet_data) & P_SCOUTED ) / P_SCOUTED;

    /* set index maintenance fields and return */
    p->infile = 1;
    p->pos = pos;
    strcpy(p->oname, p->name);
    return R_OK;
}

/* planet_writedata() - write a data record */
result planet_writedata(planet *p)
{
    long pos; /* position in file */

    /* check that record can be written */
    pos = ftell(planet_data);
    if( fputc(p->deleted, planet_data) == EOF )
        return R_FILE;

    /* write rest of record */
    fwrite(p->name,  16, 1, planet_data);
    fwrite(p->owner, 16, 1, planet_data);
    FWRITESHORT(p->pp.x,    planet_data);
    FWRITESHORT(p->pp.y,    planet_data);
    FWRITESHORT(p->prod,    planet_data);
    FWRITELONG(p->ships,    planet_data);
    fputc(p->scouted & 1,   planet_data);

    /* set index maintenance fields and return */
    p->infile = 1;
    p->pos = pos;
    strcpy(p->oname, p->name);
    return R_OK;
}

/* planet_sortind1() - single bi-directional pass of sort */
int planet_sortind1(void)
{
    long   pos;      /* stored position in index */
    char   name[16]; /* stored planet name */
    ind1   i1,       /* first index record to compare */
           i2;       /* second index record to compare */
    result r;        /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(planet_ind1);
    if(pos > 8)
    {
        fseek(planet_ind1, -IND1SIZE, SEEK_CUR);
        planet_readind1(&i1);
        strcpy(name, i1.name);
    }
    else strcpy(name, "");

    /* ensure file is big enough to need sorting */
    fseek(planet_ind1, 0, SEEK_END);
    if( ftell(planet_ind1) <= 8 + IND1SIZE )
    {
        fseek(planet_ind1, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(planet_ind1, 8, SEEK_SET); planet_readind1(&i1);
    while( planet_readind1(&i2) == R_OK )
    {
        if( stricmp(i1.name, i2.name) > 0 )
        {
            fseek(planet_ind1, -2 * IND1SIZE, SEEK_CUR);
            planet_writeind1(i2); planet_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(planet_ind1, -IND1SIZE, SEEK_CUR);
        planet_readind1(&i1);
    }

    /* backward pass of sort */
    fseek(planet_ind1, -IND1SIZE, SEEK_END);
    while( ftell(planet_ind1) > 8 )
    {
        fseek(planet_ind1, -IND1SIZE, SEEK_CUR);
        planet_readind1(&i1); planet_readind1(&i2);
        if( stricmp(i1.name, i2.name) > 0 )
        {
            fseek(planet_ind1, -2 * IND1SIZE, SEEK_CUR);
            planet_writeind1(i2); planet_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(planet_ind1, -2 * IND1SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        planet_findind1(&i1, name);
    else
        fseek(planet_ind1, pos, SEEK_SET);
    return r;
}


/* level 0 routines *******************************************/


/* planet_open() - open planet files */
result planet_open(char *path)
{
    char dataname[128], /* name of data file */
         ind1name[128], /* name of planet number index file */
         dataheader[8], /* data file header */
         ind1header[8]; /* index 1 header */

    /* initialise data names */
    sprintf(dataname, "%splanet.data", path);
    sprintf(ind1name, "%splanet.index", path);

    /* attempt to open files */
    if(( planet_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    if(( planet_ind1 = fopen(ind1name, "r+b") ) == NULL)
    {
        fclose(planet_data);
        return R_FILE;
    }

    /* verify files */
    fread(dataheader, 8, 1, planet_data);
    fread(ind1header, 8, 1, planet_ind1);
    if(( strncmp(dataheader, "AST100P", 8) ) ||
       ( strncmp(ind1header, "AST100p", 8) ))
    {
        fclose(planet_data);
        fclose(planet_ind1);
        return R_HEADER;
    }

    /* files checked out OK */
    return R_OK;
}

/* planet_create() - create planet files */
result planet_create(char *path)
{
    char  dataname[128], /* name of data file */
          ind1name[128]; /* name of planet number index file */
    FILE *test;          /* used to test existence of files */

    /* initialise data names */
    sprintf(dataname, "%splanet.data", path);
    sprintf(ind1name, "%splanet.index", path);

    /* check for files' existence */
    if(( test = fopen(dataname, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1name, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }

    /* attempt to create files */
    if(( planet_data = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    if(( planet_ind1 = fopen(ind1name, "wb") ) == NULL)
    {
        fclose(planet_data); remove(dataname);
        return R_FILE;
    }

    /* write headers */
    fwrite("AST100P", 8, 1, planet_data);
    fwrite("AST100p", 8, 1, planet_ind1);

    /* close files and return */
    fclose(planet_data);
    fclose(planet_ind1);
    return R_OK;
}

/* planet_close() - close planet files */
result planet_close(void)
{
    fclose(planet_data);
    fclose(planet_ind1);
    return R_OK;
}

/* planet_new() - allocate memory for new planet record */
planet *planet_new(void)
{
    return calloc( 1, sizeof(planet) );
}

/* planet_old() - free memory for new planet record */
void planet_old(planet *p)
{
    free(p);
}

/* planet_clear() - clear details from existing planet record */
void planet_clear(planet *p)
{
    memset( p, 0, sizeof(planet) );
}

/* planet_setname() - set the planet name */
char *planet_setname(planet *p, char *name)
{
    return strcpy(p->name, name);
}

/* planet_getname() - return the planet name */
char *planet_getname(char *name, planet *p)
{
    return strcpy(name, p->name);
}

/* planet_setowner() - set the owner */
char *planet_setowner(planet *p, char *owner)
{
    return strcpy(p->owner, owner);
}

/* planet_getowner() - return the owner */
char *planet_getowner(char *owner, planet *p)
{
    return strcpy(owner, p->owner);
}

/* planet_setpos() - set planet position */
pos planet_setpos(planet *p, pos pp)
{
    return p->pp = pp;
}

/* planet_getpos() - return the planet position */
pos planet_getpos(planet *p)
{
    return p->pp;
}

/* planet_setprod() - set the planet production */
short planet_setprod(planet *p, short prod)
{
    return p->prod = prod;
}

/* planet_getprod() - return the planet production */
short planet_getprod(planet *p)
{
    return p->prod;
}

/* planet_setships() - set the ships in orbit */
long planet_setships(planet *p, long ships)
{
    return p->ships = ships;
}

/* planet_getships() - return the ships in orbit */
long planet_getships(planet *p)
{
    return p->ships;
}

/* planet_setscouted() - set the scouted flag */
int planet_setscouted(planet *p, int scouted)
{
    return p->scouted = scouted;
}

/* planet_getscouted() - return the scouted flag */
int planet_getscouted(planet *p)
{
    return p->scouted;
}

/* planet_write() - write a record */
result planet_write(planet *p)
{
    ind1   i;  /* trans index record */
    long   p1; /* position of index record 1 */

    /* write new record */
    if(!p->infile)
    {
        if( planet_findind1(&i, p->name) ) return R_DUPE;
        fseek(planet_data, 0, SEEK_END);
        fseek(planet_ind1, 0, SEEK_END);
        i.pos = ftell(planet_data);
        strcpy(i.name, p->name);
        if( planet_writedata(p) != R_OK )  return R_FILE;
        if( planet_writeind1(i) != R_OK )  return R_CORRUPT;
        planet_sortind1();
        return R_OK;
    }

    /* rewrite existing record */
    p1 = planet_findind1(&i, p->oname);
    if(!p1)                                return R_CORRUPT;
    fseek(planet_data, i.pos, SEEK_SET);
    if( planet_writedata(p) != R_OK )      return R_FILE;
    if( stricmp(i.name, p->name) )
    {
        strcpy(i.name, p->name);
        fseek(planet_ind1, p1, SEEK_SET);
        if( planet_writeind1(i) != R_OK )  return R_CORRUPT;
        planet_sortind1();
    }
    return R_OK;
}

/* planet_first() - read the first record */
result planet_first(planet *p, pindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case P_NONE:
            fseek(planet_data, 8, SEEK_SET);
            return planet_readdata(p);
        case P_NAME:
            fseek(planet_ind1, 8, SEEK_SET);
            if(( r = planet_readind1(&i) ) != R_OK) return r;
            fseek(planet_data, i.pos, SEEK_SET);
            return planet_readdata(p);
        default:
            return R_INDEX;
    }
}

/* planet_next() - read the next record */
result planet_next(planet *p, pindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case P_NONE:
            return planet_readdata(p);
        case P_NAME:
            if(( r = planet_readind1(&i) ) != R_OK) return r;
            fseek(planet_data, i.pos, SEEK_SET);
            return planet_readdata(p);
        default:
            return R_INDEX;
    }
}

/* planet_prev() - read the previous record */
result planet_prev(planet *p, pindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case P_NONE:
            if(( ftell(planet_data) - 2 * DATASIZE ) < 8)
                return R_EOF;
            fseek(planet_data, -2 * DATASIZE, SEEK_CUR);
            return planet_readdata(p);
        case P_NAME:
            if(( ftell(planet_ind1) - 2 * IND1SIZE ) < 8)
                return R_EOF;
            fseek(planet_ind1, -2 * IND1SIZE, SEEK_CUR);
            if(( r = planet_readind1(&i) ) != R_OK) return r;
            fseek(planet_data, i.pos, SEEK_SET);
            return planet_readdata(p);
        default:
            return R_INDEX;
    }
}

/* planet_last() - read the last record */
result planet_last(planet *p, pindex inum)
{
    ind1   i;   /* number index record */
    result r;   /* result of index read */
    long   pos; /* current data/index file position */

    switch(inum)
    {
        case P_NONE:
            pos = ftell(planet_data);
            if( fseek(planet_data, -DATASIZE, SEEK_END) )
                return R_EOF;
            if( ftell(planet_data) < 8 )
            {
                fseek(planet_data, pos, SEEK_SET);
                return R_EOF;
            }
            return planet_readdata(p);
        case P_NAME:
            pos = ftell(planet_ind1);
            if( fseek(planet_ind1, -IND1SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(planet_ind1) < 8 )
            {
                fseek(planet_ind1, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = planet_readind1(&i) ) != R_OK) return r;
            fseek(planet_data, i.pos, SEEK_SET);
            return planet_readdata(p);
        default:
            return R_INDEX;
    }
}

/* planet_find() - find a record by name or name */
result planet_find(planet *p, char *name)
{
    ind1   i;      /* planet name index record */

    if( planet_findind1(&i, name) == 0 ) return R_EOF;
    fseek(planet_data, i.pos, SEEK_SET);
    return planet_readdata(p);

    return R_OK;
}

/* planet_delete() - mark a record as deleted */
result planet_delete(planet *p)
{
    if(p->pos == 0) return R_EOF;
    p->deleted ^= PF_DELETED;
    fseek(planet_data, p->pos, SEEK_SET);
    return planet_writedata(p);
}

/* planet_deleted() - return the deleted status of a record */
int planet_deleted(planet *p)
{
    if(p->pos == 0) return R_EOF;
    return p->deleted & PF_DELETED;
}

/* planet_pack() - pack a close data file */
result planet_pack(char *path)
{
    planet *p;             /* planet record */
    char    dataname[128], /* name of data file */
            ind1name[128], /* name of planet number index file */
            tempname[128], /* name of temporary file */
            dataheader[8]; /* data file header */
    FILE   *planet_temp;   /* temporary file */
    ind1    i;             /* number index record */

    /* initialise data names */
    sprintf(dataname, "%splanet.data", path);
    sprintf(tempname, "%splanet.temp", path);

    /* attempt to open data & temp file */
    if(( planet_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    fread(dataheader, 8, 1, planet_data);
    if(( strncmp(dataheader, "AST100P", 8) ) ||
       ( planet_temp = fopen(tempname, "w+b") ) == NULL)
    {
        fclose(planet_data);
        return R_HEADER;
    }

    /* allocate memory */
    if( (p = malloc( sizeof(planet) )) == NULL )
    {
        fclose(planet_data);
        fclose(planet_temp);
        remove(tempname);
        return R_MEMORY;
    }

    /* copy non-deleted records to temporary file and back */
    while( planet_readdata(p) == R_OK )
        if( !(p->deleted & PF_DELETED) )
            fwrite(p, sizeof(planet), 1, planet_temp);
    fclose(planet_data);
    remove(dataname);
    if(( planet_data = fopen(dataname, "w+b") ) == NULL)
    {
        free(p);
        fclose(planet_temp);
        remove(tempname);
        return R_FILE;
    }
    fwrite("AST100P", 8, 1, planet_data);
    fseek(planet_temp, 0, SEEK_SET);
    while( fread(p, sizeof(planet), 1, planet_temp) == 1 )
        planet_writedata(p);
    fclose(planet_temp);
    remove(tempname);

    /* recreate fleet number index */
    sprintf(ind1name, "%splanet.index", path);
    remove(ind1name);
    if(( planet_ind1 = fopen(ind1name, "w+b") ) == NULL)
    {
        free(p);
        fclose(planet_data);
        return R_FILE;
    }
    fwrite("AST100p", 8, 1, planet_ind1);
    fseek(planet_data, 8, SEEK_SET); i.pos = ftell(planet_data);
    while( planet_readdata(p) == R_OK )
    {
        strcpy(i.name, p->name);
        planet_writeind1(i);
        i.pos = ftell(planet_data);
    }
    while( planet_sortind1() == R_SWAPS );
    fclose(planet_ind1);

    /* clean up */
    free(p);
    fclose(planet_data);
    return R_OK;
}

