/**************************************************************
 * ORDERS.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroWar 1.00 - Orders file library                        *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     30-Jan-97                                         *
 **************************************************************/


/* included files *********************************************/


#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include "astrowar.h"


/* structures *************************************************/


/* join order parameters */
typedef struct {
    long trans;      /* transaction number */
    char empire[16]; /* desired empire name */
} jorder;

/* send order parameters */
typedef struct {
    char empire[16], /* empire giving the order */
         origin[16], /* planet of origin */
         dest[16];   /* destination planet */
    long ships;      /* ships to send */
} sorder;

/* planets order parameters */
typedef struct {
    char empire[16]; /* empire giving the order */
} porder;

/* fleets order parameters */
typedef struct {
    char empire[16]; /* empire giving the order */
} forder;

/* map order parameters */
typedef struct {
    char empire[16], /* empire giving the order */
         centre[16]; /* planet to be the centre of the map */
} morder;

/* write order parameters */
typedef struct {
    char empire[16], /* sender of message */
         recip[16],  /* recipient of message */
        *text;       /* text of message */
} worder;

/* resign order parameters */
typedef struct {
    char empire[16]; /* empire giving the order */
} rorder;

/* general order parameters */
typedef union {
    jorder j; /* join order */
    sorder s; /* send order */
    porder p; /* planets order */
    forder f; /* fleets order */
    morder m; /* map order */
    worder w; /* write order */
    rorder r; /* resign order */
} oparam;

/* order record */
typedef struct {
    char type; /* order type */
    oparam op; /* order parameters */
} order;


/* global variables *******************************************/


FILE *ordfile;  /* data file handle */


/* level 2 routines *******************************************/


/* orders_readj() - read 'join' order */
void orders_readj(order *o)
{
    FREADLONG(o->op.j.trans, ordfile);
    fread(o->op.j.empire, 16, 1, ordfile);
}

/* orders_reads() - read 'send' order */
void orders_reads(order *o)
{
    fread(o->op.s.empire, 16, 1, ordfile);
    fread(o->op.s.origin, 16, 1, ordfile);
    fread(o->op.s.dest,   16, 1, ordfile);
    FREADLONG(o->op.s.ships, ordfile);
}

/* orders_readp() - read 'planets' order */
void orders_readp(order *o)
{
    fread(o->op.p.empire, 16, 1, ordfile);
}

/* orders_readf() - read 'fleets' order */
void orders_readf(order *o)
{
    fread(o->op.f.empire, 16, 1, ordfile);
}

/* orders_readm() - read 'map' order */
void orders_readm(order *o)
{
    fread(o->op.m.empire, 16, 1, ordfile);
    fread(o->op.m.centre, 16, 1, ordfile);
}

/* orders_readw() - read 'write' order */
void orders_readw(order *o)
{
    unsigned int msglen; /* message length */

    fread(o->op.w.empire, 16, 1, ordfile);
    fread(o->op.w.recip,  16, 1, ordfile);
    FREADSHORT(msglen, ordfile);
    o->op.w.text = malloc(msglen + 1);
    fread(o->op.w.text, msglen, 1, ordfile);
    o->op.w.text[msglen] = '\0'; /* NULL not stored in file */
}

/* orders_readr() - read 'resign' order */
void orders_readr(order *o)
{
    fread(o->op.r.empire, 16, 1, ordfile);
}

/* orders_writej() - write 'join' order */
void orders_writej(order *o)
{
    fputc(o->type,                            ordfile);
    fputc(o->op.j.trans & 255,                ordfile);
    fputc(( o->op.j.trans / 256 ) & 255,      ordfile);
    fputc(( o->op.j.trans / 65536 ) & 255,    ordfile);
    fputc(( o->op.j.trans / 16777216 ) & 255, ordfile);
    fwrite(o->op.j.empire, 16, 1,             ordfile);
}

/* orders_writes() - write 'send' order */
void orders_writes(order *o)
{
    fputc(o->type,                            ordfile);
    fwrite(o->op.s.empire, 16, 1,             ordfile);
    fwrite(o->op.s.origin, 16, 1,             ordfile);
    fwrite(o->op.s.dest,   16, 1,             ordfile);
    fputc(o->op.s.ships & 255,                ordfile);
    fputc(( o->op.s.ships / 256 ) & 255,      ordfile);
    fputc(( o->op.s.ships / 65536 ) & 255,    ordfile);
    fputc(( o->op.s.ships / 16777216 ) & 255, ordfile);
}

/* orders_writep() - write 'planets' order */
void orders_writep(order *o)
{
    fputc(o->type,                ordfile);
    fwrite(o->op.p.empire, 16, 1, ordfile);
}

/* orders_writef() - write 'fleets' order */
void orders_writef(order *o)
{
    fputc(o->type,                ordfile);
    fwrite(o->op.f.empire, 16, 1, ordfile);
}

/* orders_writem() - write 'map' order */
void orders_writem(order *o)
{
    fputc(o->type,                ordfile);
    fwrite(o->op.m.empire, 16, 1, ordfile);
    fwrite(o->op.m.centre, 16, 1, ordfile);
}

/* orders_writew() - write 'write' order */
void orders_writew(order *o)
{
    unsigned int msglen; /* message length */

    msglen = strlen(o->op.w.text);
    fputc(o->type,                  ordfile);
    fwrite(o->op.w.empire, 16, 1,   ordfile);
    fwrite(o->op.w.recip,  16, 1,   ordfile);
    fputc(msglen & 255,             ordfile);
    fputc(msglen / 256,             ordfile);
    fwrite(o->op.w.text, msglen, 1, ordfile);
}

/* orders_writer() - write 'resign' order */
void orders_writer(order *o)
{
    fputc(o->type,                ordfile);
    fwrite(o->op.r.empire, 16, 1, ordfile);
}


/* level 1 routines *******************************************/


/* orders_readrec() - generic read record function */
result orders_readrec(order *o)
{
    if( (o->type == 'W') && (o->op.w.text != NULL) )
    {
        free(o->op.w.text);
        o->op.w.text = NULL;
    }
    if( fread(&o->type, 1, 1, ordfile) == 0 )
        return R_EOF;
    switch(o->type)
    {
        case 'J': orders_readj(o); break;
        case 'S': orders_reads(o); break;
        case 'P': orders_readp(o); break;
        case 'F': orders_readf(o); break;
        case 'M': orders_readm(o); break;
        case 'W': orders_readw(o); break;
        case 'R': orders_readr(o); break;
        default:  return R_HEADER;
    }
    return R_OK;
}

/* orders_writerec() - generic write record function */
result orders_writerec(order *o)
{
    switch(o->type)
    {
        case 'J': orders_writej(o); break;
        case 'S': orders_writes(o); break;
        case 'P': orders_writep(o); break;
        case 'F': orders_writef(o); break;
        case 'M': orders_writem(o); break;
        case 'W': orders_writew(o); break;
        case 'R': orders_writer(o); break;
        default:  return R_HEADER;
    }
    return R_OK;
}


/* main function implementation *******************************/


/* orders_open() - open the order file */
result orders_open(char *path)
{
    char listname[128],  /* full filename of data file */
         header[8];      /* header stored for comparison */

    /* open list file */
    sprintf(listname, "%sorders.data", path);
    if(( ordfile = fopen(listname, "r+b") ) == NULL)
        return R_FILE;
    fread(header, 8, 1, ordfile);
    if( strcmp(header, "AST100O") == 0 )
        return R_OK;
    fclose(ordfile);
    return R_HEADER;
}

/* orders_create() - create a new order file */
result orders_create(char *path)
{
    char dataname[128];  /* full name of data file */

    /* test to see if the DAT file already exists */
    sprintf(dataname, "%sorders.data", path);
    if(( ordfile = fopen(dataname, "rb") ) != NULL)
    {
        fclose(ordfile);
        return R_EXISTS;
    }

    /* create DAT file if not already present */
    if(( ordfile = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    fwrite("AST100O\0", 8, 1, ordfile);
    fclose(ordfile);
    return R_OK;
}

/* orders_first() - read the first order */
result orders_first(order *o)
{
    fseek(ordfile, 8, SEEK_SET);
    return orders_readrec(o);
}

/* orders_next() - read the next order */
result orders_next(order *o)
{
    return orders_readrec(o);
}

/* orders_write() - add or update an order */
result orders_write(order *o)
{
    long filepos; /* current file position */

    filepos = ftell(ordfile);

    /* add new record */
    fseek(ordfile, 0, SEEK_END);
    orders_writerec(o);
    fseek(ordfile, filepos, SEEK_SET);
    return R_OK;
}

/* orders_new() - allocate memory for a new order */
order *orders_new()
{
    order *o; /* order record */

    /* allocate and initialise order record */
    o = malloc( sizeof(order) );
    if(o) memset( o, 0, sizeof(order) );

    return o;
}

/* orders_old() - free memory from an old order */
void orders_old(order *o)
{
    if( (o->type == 'W') && (o->op.w.text != NULL) )
        free(o->op.w.text);
    free(o);
}

/* orders_settype() - set the 'type' character */
void orders_settype(order *o, char type)
{
    if( (o->type == 'W') && (o->op.w.text != NULL) )
    {
        free(o->op.w.text);
        o->op.w.text = NULL;
    }
    o->type = type;
}

/* orders_settrans() - set the transaction number */
void orders_settrans(order *o, long number)
{
    if(o->type == 'J')
        o->op.j.trans = number;
}

/* orders_setempire() - set the empire name */
void orders_setempire(order *o, char *name)
{
    switch(o->type)
    {
        case 'J': strcpy(o->op.j.empire, name); break;
        case 'S': strcpy(o->op.s.empire, name); break;
        case 'P': strcpy(o->op.p.empire, name); break;
        case 'F': strcpy(o->op.f.empire, name); break;
        case 'M': strcpy(o->op.m.empire, name); break;
        case 'W': strcpy(o->op.w.empire, name); break;
        case 'R': strcpy(o->op.r.empire, name); break;
    }
}

/* orders_setorigin() - set the origin for a send order */
void orders_setorigin(order *o, char *name)
{
    if(o->type == 'S')
        strcpy(o->op.s.origin, name);
}

/* orders_setdest() - set the destination for a send order */
void orders_setdest(order *o, char *name)
{
    if(o->type == 'S')
        strcpy(o->op.s.dest, name);
}

/* orders_setships() - set the ships in a send order */
void orders_setships(order *o, long ships)
{
    if(o->type == 'S')
        o->op.s.ships = ships;
}

/* orders_setcentre() - set the centre in a map order */
void orders_setcentre(order *o, char *name)
{
    if(o->type == 'M')
        strcpy(o->op.m.centre, name);
}

/* orders_setrecip() - set the recipient in a write order */
void orders_setrecip(order *o, char *name)
{
    if(o->type == 'W')
        strcpy(o->op.w.recip, name);
}

/* orders_settext() - set the message text in a write order */
void orders_settext(order *o, char *text)
{
    if(o->type != 'W')
        return;
    if(o->op.w.text != NULL)
        free(o->op.w.text);
    if( (o->op.w.text = malloc( strlen(text) + 1 )) != NULL )
        strcpy(o->op.w.text, text);
}

/* orders_gettype() - return the order type */
char orders_gettype(order *o)
{
    return o->type;
}

/* orders_gettrans() - return the transaction number */
long orders_gettrans(order *o)
{
    if(o->type == 'J')
        return o->op.j.trans;
    else
        return -1;
}

/* orders_getempire() - return the empire name */
char *orders_getempire(char *name, order *o)
{
    switch(o->type)
    {
        case 'J': return strcpy(name, o->op.j.empire); break;
        case 'S': return strcpy(name, o->op.s.empire); break;
        case 'P': return strcpy(name, o->op.p.empire); break;
        case 'F': return strcpy(name, o->op.f.empire); break;
        case 'M': return strcpy(name, o->op.m.empire); break;
        case 'W': return strcpy(name, o->op.w.empire); break;
        case 'R': return strcpy(name, o->op.r.empire); break;
        default:  return NULL;
    }
}

/* orders_getorigin() - return the origin planet in a send */
char *orders_getorigin(char *name, order *o)
{
    if(o->type == 'S')
        return strcpy(name, o->op.s.origin);
    else
        return NULL;
}

/* orders_getdest() - return the destination planet in a send */
char *orders_getdest(char *name, order *o)
{
    if(o->type == 'S')
        return strcpy(name, o->op.s.dest);
    else
        return NULL;
}

/* orders_getships() - return the number of ships in a send */
long orders_getships(order *o)
{
    if(o->type == 'S')
        return o->op.s.ships;
    else
        return -1;
}

/* orders_getcentre() - return the centre planet of a map */
char *orders_getcentre(char *name, order *o)
{
    if(o->type == 'M')
        return strcpy(name, o->op.m.centre);
    else
        return NULL;
}

/* orders_getrecip() - return the recipient of a written msg */
char *orders_getrecip(char *name, order *o)
{
    if(o->type == 'W')
        return strcpy(name, o->op.w.recip);
    else
        return NULL;
}

/* orders_gettext() - return the text of a written msg */
char *orders_gettext(order *o)
{
    char *text; /* newly allocated var for msg text */

    if(o->type != 'W')
        return NULL;
    text = malloc( strlen(o->op.w.text) + 1 );
    if(text != NULL) strcpy(text, o->op.w.text);

    return text;
}

/* orders_close() - close orders file/library */
void orders_close(void)
{
    fclose(ordfile);
}

