/**************************************************************
 * FLEETS.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroWar 1.00 - Empire database library.                   *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     28-Apr-97                                         *
 **************************************************************/


#define _FLEETS_C_


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astrowar.h"
#include "fleets.h"


/* some macros ************************************************/


/* record sizes */
#define DATASIZE 59  /* size of packed data record */
#define IND1SIZE  8  /* size of packed fleet number index */

/* flags */
#define FF_DELETED 0x01 /* deleted flag */


/* structures *************************************************/


/* hidden fleet structure */
typedef struct {

    /* record fields */
    char deleted;    /* deleted flag */
    long number;     /* fleet number */
    char owner[16],  /* owner of fleet */
         origin[16], /* planet of origin */
         dest[16];   /* destination planet */
    long ships;      /* ships in fleet */
    int  dist;       /* distance to travel */

    /* index maintenance fields */
    char infile;     /* 0=new record, !0=somewhere in file */
    long pos;        /* record position when last read */
    long onumber;    /* original fleet number */

} fleet;

/* indexes */
typedef struct {   /* fleet number index */
    long pos;          /* position of record in data file */
    long number;       /* fleet number */
} ind1;


/* global variables *******************************************/


FILE *fleets_data, /* fleet data file */
     *fleets_ind1; /* fleet number index file */


/* level 3 routines *******************************************/


/* fleets_readind1() - read a number index record */
result fleets_readind1(ind1 *i)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(fleets_ind1) ) == EOF)
        return R_EOF;
    i->pos = pos;
    FREADLONGR(i->pos, fleets_ind1);
    FREADLONG(i->number, fleets_ind1);
    return R_OK;
}


/* level 2 routines *******************************************/


/* fleets_findind1() - find a trans index record */
long fleets_findind1(ind1 *i, long number)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(fleets_ind1); fseek(fleets_ind1, 0, SEEK_END);
    if( ftell(fleets_ind1) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(fleets_ind1) - 8 ) / IND1SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(fleets_ind1, 8 + IND1SIZE * middle, SEEK_SET);
    fleets_readind1(i);
    while(i->number != number && start <= end)
    {
        if(i->number < number)
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(fleets_ind1, 8 + IND1SIZE * middle, SEEK_SET);
        fleets_readind1(i);
    }

    /* return value */
    if(i->number == number)
        return 8 + middle * IND1SIZE;
    fseek(fleets_ind1, pos, SEEK_SET);
    return 0;
}

/* fleets_writeind1() - write a number index record */
result fleets_writeind1(ind1 i)
{
    if( fputc(i.pos & 0xff, fleets_ind1) == EOF ) return R_FILE;
    FWRITELONGR(i.pos,   fleets_ind1);
    FWRITELONG(i.number, fleets_ind1);
    return R_OK;
}


/* level 1 routines *******************************************/


/* fleets_readdata() - read a data record */
result fleets_readdata(fleet *f)
{
    int  deleted; /* deleted flag */
    long pos;     /* position in file */

    /* check for EOF */
    pos = ftell(fleets_data);
    if(( deleted = fgetc(fleets_data) ) == EOF)
        return R_EOF;

    /* read rest of record */
    f->deleted = deleted;
    FREADLONG(f->number,    fleets_data);
    fread(f->owner,  16, 1, fleets_data);
    fread(f->origin, 16, 1, fleets_data);
    fread(f->dest,   16, 1, fleets_data);
    FREADLONG(f->ships,     fleets_data);
    FREADSHORT(f->dist,     fleets_data);

    /* set index maintenance fields and return */
    f->infile = 1;
    f->pos = pos;
    f->onumber = f->number;
    return R_OK;
}

/* fleets_writedata() - write a data record */
result fleets_writedata(fleet *f)
{
    long pos; /* position in file */

    /* check that record can be written */
    pos = ftell(fleets_data);
    if( fputc(f->deleted, fleets_data) == EOF )
        return R_FILE;

    /* write rest of record */
    FWRITELONG(f->number,    fleets_data);
    fwrite(f->owner,  16, 1, fleets_data);
    fwrite(f->origin, 16, 1, fleets_data);
    fwrite(f->dest,   16, 1, fleets_data);
    FWRITELONG(f->ships,     fleets_data);
    FWRITESHORT(f->dist,     fleets_data);

    /* set index maintenance fields and return */
    f->infile = 1;
    f->pos = pos;
    f->onumber = f->number;
    return R_OK;
}

/* fleets_sortind1() - single bi-directional pass of sort */
int fleets_sortind1(void)
{
    long   pos,      /* stored position in index */
           number;   /* stored fleet number */
    ind1   i1,       /* first index record to compare */
           i2;       /* second index record to compare */
    result r;        /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(fleets_ind1);
    if(pos > 8)
    {
        fseek(fleets_ind1, -IND1SIZE, SEEK_CUR);
        fleets_readind1(&i1);
        number = i1.number;
    }
    else number = 0;

    /* ensure file is big enough to need sorting */
    fseek(fleets_ind1, 0, SEEK_END);
    if( ftell(fleets_ind1) <= 8 + IND1SIZE )
    {
        fseek(fleets_ind1, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(fleets_ind1, 8, SEEK_SET); fleets_readind1(&i1);
    while( fleets_readind1(&i2) == R_OK )
    {
        if( i1.number > i2.number )
        {
            fseek(fleets_ind1, -2 * IND1SIZE, SEEK_CUR);
            fleets_writeind1(i2); fleets_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(fleets_ind1, -IND1SIZE, SEEK_CUR);
        fleets_readind1(&i1);
    }

    /* backward pass of sort */
    fseek(fleets_ind1, -IND1SIZE, SEEK_END);
    while( ftell(fleets_ind1) > 8 )
    {
        fseek(fleets_ind1, -IND1SIZE, SEEK_CUR);
        fleets_readind1(&i1); fleets_readind1(&i2);
        if( i1.number > i2.number )
        {
            fseek(fleets_ind1, -2 * IND1SIZE, SEEK_CUR);
            fleets_writeind1(i2); fleets_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(fleets_ind1, -2 * IND1SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        fleets_findind1(&i1, number);
    else
        fseek(fleets_ind1, pos, SEEK_SET);
    return r;
}


/* level 0 routines *******************************************/


/* fleets_open() - open fleet files */
result fleets_open(char *path)
{
    char dataname[128], /* number of data file */
         ind1name[128], /* number of fleet number index file */
         dataheader[8], /* data file header */
         ind1header[8]; /* index 1 header */

    /* initialise data numbers */
    sprintf(dataname, "%sfleets.data", path);
    sprintf(ind1name, "%sfleets.index", path);

    /* attempt to open files */
    if(( fleets_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    if(( fleets_ind1 = fopen(ind1name, "r+b") ) == NULL)
    {
        fclose(fleets_data);
        return R_FILE;
    }

    /* verify files */
    fread(dataheader, 8, 1, fleets_data);
    fread(ind1header, 8, 1, fleets_ind1);
    if(( strncmp(dataheader, "AST100F", 8) ) ||
       ( strncmp(ind1header, "AST100f", 8) ))
    {
        fclose(fleets_data);
        fclose(fleets_ind1);
        return R_HEADER;
    }

    /* files checked out OK */
    return R_OK;
}

/* fleets_create() - create fleet files */
result fleets_create(char *path)
{
    char  dataname[128], /* number of data file */
          ind1name[128]; /* number of fleet number index file */
    FILE *test;          /* used to test existence of files */

    /* initialise data numbers */
    sprintf(dataname, "%sfleets.data", path);
    sprintf(ind1name, "%sfleets.index", path);

    /* check for files' existence */
    if(( test = fopen(dataname, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1name, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }

    /* attempt to create files */
    if(( fleets_data = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    if(( fleets_ind1 = fopen(ind1name, "wb") ) == NULL)
    {
        fclose(fleets_data); remove(dataname);
        return R_FILE;
    }

    /* write headers */
    fwrite("AST100F", 8, 1, fleets_data);
    fwrite("AST100f", 8, 1, fleets_ind1);

    /* close files and return */
    fclose(fleets_data);
    fclose(fleets_ind1);
    return R_OK;
}

/* fleets_close() - close fleet files */
result fleets_close(void)
{
    fclose(fleets_data);
    fclose(fleets_ind1);
    return R_OK;
}

/* fleets_new() - allocate memory for new fleet record */
fleet *fleets_new(void)
{
    return calloc( 1, sizeof(fleet) );
}

/* fleets_old() - free memory for new fleet record */
void fleets_old(fleet *f)
{
    free(f);
}

/* fleets_clear() - clear details from existing fleet record */
void fleets_clear(fleet *f)
{
    memset( f, 0, sizeof(fleet) );
}

/* fleets_setnumber() - set the fleet number */
long fleets_setnumber(fleet *f, long number)
{
    return f->number = number;
}

/* fleets_getnumber() - return the fleet number */
long fleets_getnumber(fleet *f)
{
    return f->number;
}

/* fleets_setdest() - set the homeworld number */
char *fleets_setdest(fleet *f, char *dest)
{
    return strcpy(f->dest, dest);
}

/* fleets_getdest() - return the homeworld */
char *fleets_getdest(char *dest, fleet *f)
{
    return strcpy(dest, f->dest);
}

/* fleets_setorigin() - set the homeworld number */
char *fleets_setorigin(fleet *f, char *origin)
{
    return strcpy(f->origin, origin);
}

/* fleets_getorigin() - return the homeworld */
char *fleets_getorigin(char *origin, fleet *f)
{
    return strcpy(origin, f->origin);
}

/* fleets_setowner() - set the homeworld number */
char *fleets_setowner(fleet *f, char *owner)
{
    return strcpy(f->owner, owner);
}

/* fleets_getowner() - return the homeworld */
char *fleets_getowner(char *owner, fleet *f)
{
    return strcpy(owner, f->owner);
}

/* fleets_setships() - set ship count */
long fleets_setships(fleet *f, long ships)
{
    return f->ships = ships;
}

/* fleets_getships() - return the fleet ships */
long fleets_getships(fleet *f)
{
    return f->ships;
}

/* fleets_setdist() - set distance */
int fleets_setdist(fleet *f, int dist)
{
    return f->dist = dist;
}

/* fleets_getdist() - return the fleet dist */
int fleets_getdist(fleet *f)
{
    return f->dist;
}

/* fleets_write() - write a record */
result fleets_write(fleet *f)
{
    ind1   i;   /* trans index record */
    long   p1;  /* position of index record 1 */
    static
    long   top; /* highest fleet number */

    /* write new record */
    if(!f->infile)
    {
        if( fleets_findind1(&i, f->number) ) return R_DUPE;
        fseek(fleets_data, 0, SEEK_END);
        fseek(fleets_ind1, 0, SEEK_END);
        i.pos = ftell(fleets_data);
        i.number = f->number;
        if( fleets_writedata(f) != R_OK )  return R_FILE;
        if( fleets_writeind1(i) != R_OK )  return R_CORRUPT;
        if(i.number <= top || top == 0)
        {
            fleets_sortind1();
            if(top == 0)
            {
                p1 = ftell(fleets_ind1);
                fseek(fleets_ind1, -8, SEEK_END);
                fleets_readind1(&i);
                top = i.number;
                fseek(fleets_ind1, p1, SEEK_SET);
            }
        }
        else top = i.number;
        return R_OK;
    }

    /* rewrite existing record */
    p1 = fleets_findind1(&i, f->onumber);
    if(!p1)                                return R_CORRUPT;
    fseek(fleets_data, i.pos, SEEK_SET);
    if( fleets_writedata(f) != R_OK )      return R_FILE;
    if(i.number != f->number)
    {
        i.number = f->number;
        fseek(fleets_ind1, p1, SEEK_SET);
        if( fleets_writeind1(i) != R_OK )  return R_CORRUPT;
        fleets_sortind1();
    }
    return R_OK;
}

/* fleets_first() - read the first record */
result fleets_first(fleet *f, findex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case F_NONE:
            fseek(fleets_data, 8, SEEK_SET);
            return fleets_readdata(f);
        case F_NUMBER:
            fseek(fleets_ind1, 8, SEEK_SET);
            if(( r = fleets_readind1(&i) ) != R_OK) return r;
            fseek(fleets_data, i.pos, SEEK_SET);
            return fleets_readdata(f);
        default:
            return R_INDEX;
    }
}

/* fleets_next() - read the next record */
result fleets_next(fleet *f, findex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case F_NONE:
            return fleets_readdata(f);
        case F_NUMBER:
            if(( r = fleets_readind1(&i) ) != R_OK) return r;
            fseek(fleets_data, i.pos, SEEK_SET);
            return fleets_readdata(f);
        default:
            return R_INDEX;
    }
}

/* fleets_prev() - read the previous record */
result fleets_prev(fleet *f, findex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case F_NONE:
            if(( ftell(fleets_data) - 2 * DATASIZE ) < 8)
                return R_EOF;
            fseek(fleets_data, -2 * DATASIZE, SEEK_CUR);
            return fleets_readdata(f);
        case F_NUMBER:
            if(( ftell(fleets_ind1) - 2 * IND1SIZE ) < 8)
                return R_EOF;
            fseek(fleets_ind1, -2 * IND1SIZE, SEEK_CUR);
            if(( r = fleets_readind1(&i) ) != R_OK) return r;
            fseek(fleets_data, i.pos, SEEK_SET);
            return fleets_readdata(f);
        default:
            return R_INDEX;
    }
}

/* fleets_last() - read the last record */
result fleets_last(fleet *f, findex inum)
{
    ind1   i;   /* number index record */
    result r;   /* result of index read */
    long   pos; /* current data/index file position */

    switch(inum)
    {
        case F_NONE:
            pos = ftell(fleets_data);
            if( fseek(fleets_data, -DATASIZE, SEEK_END) )
                return R_EOF;
            if( ftell(fleets_data) < 8 )
            {
                fseek(fleets_data, pos, SEEK_SET);
                return R_EOF;
            }
            return fleets_readdata(f);
        case F_NUMBER:
            pos = ftell(fleets_ind1);
            if( fseek(fleets_ind1, -IND1SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(fleets_ind1) < 8 )
            {
                fseek(fleets_ind1, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = fleets_readind1(&i) ) != R_OK) return r;
            fseek(fleets_data, i.pos, SEEK_SET);
            return fleets_readdata(f);
        default:
            return R_INDEX;
    }
}

/* fleets_find() - find a record by number or number */
result fleets_find(fleet *f, long number)
{
    ind1   i;      /* fleet number index record */

    if( fleets_findind1(&i, number) == 0 ) return R_EOF;
    fseek(fleets_data, i.pos, SEEK_SET);
    return fleets_readdata(f);

    return R_OK;
}

/* fleets_delete() - mark a record as deleted */
result fleets_delete(fleet *f)
{
    if(f->pos == 0) return R_EOF;
    f->deleted ^= FF_DELETED;
    fseek(fleets_data, f->pos, SEEK_SET);
    return fleets_writedata(f);
}

/* fleets_deleted() - return the deleted status of a record */
int fleets_deleted(fleet *f)
{
    if(f->pos == 0) return R_EOF;
    return f->deleted & FF_DELETED;
}

/* fleets_pack() - pack a close data file */
result fleets_pack(char *path)
{
    fleet *f;             /* fleet record */
    char   dataname[128], /* number of data file */
           ind1name[128], /* number of fleet number index file */
           tempnumber[128], /* number of temporary file */
           dataheader[8]; /* data file header */
    FILE  *fleets_temp;   /* temporary file */
    ind1   i;             /* number index record */

    /* initialise data numbers */
    sprintf(dataname, "%sfleets.data", path);
    sprintf(tempnumber, "%sfleets.temp", path);

    /* attempt to open data & temp file */
    if(( fleets_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    fread(dataheader, 8, 1, fleets_data);
    if(( strncmp(dataheader, "AST100F", 8) ) ||
       ( fleets_temp = fopen(tempnumber, "w+b") ) == NULL)
    {
        fclose(fleets_data);
        return R_HEADER;
    }

    /* allocate memory */
    if( (f = malloc( sizeof(fleet) )) == NULL )
    {
        fclose(fleets_data);
        fclose(fleets_temp);
        remove(tempnumber);
        return R_MEMORY;
    }

    /* copy non-deleted records to temporary file and back */
    while( fleets_readdata(f) == R_OK )
        if( !(f->deleted & FF_DELETED) )
            fwrite(f, sizeof(fleet), 1, fleets_temp);
    fclose(fleets_data);
    remove(dataname);
    if(( fleets_data = fopen(dataname, "w+b") ) == NULL)
    {
        free(f);
        fclose(fleets_temp);
        remove(tempnumber);
        return R_FILE;
    }
    fwrite("AST100F", 8, 1, fleets_data);
    fseek(fleets_temp, 0, SEEK_SET);
    while( fread(f, sizeof(fleet), 1, fleets_temp) == 1 )
        fleets_writedata(f);
    fclose(fleets_temp);
    remove(tempnumber);

    /* recreate fleet number index */
    sprintf(ind1name, "%sfleets.index", path);
    remove(ind1name);
    if(( fleets_ind1 = fopen(ind1name, "w+b") ) == NULL)
    {
        free(f);
        fclose(fleets_data);
        return R_FILE;
    }
    fwrite("AST100f", 8, 1, fleets_ind1);
    fseek(fleets_data, 8, SEEK_SET); i.pos = ftell(fleets_data);
    while( fleets_readdata(f) == R_OK )
    {
        i.number = f->number;
        fleets_writeind1(i);
        i.pos = ftell(fleets_data);
    }
    while( fleets_sortind1() == R_SWAPS );
    fclose(fleets_ind1);

    /* clean up */
    free(f);
    fclose(fleets_data);
    return R_OK;
}
