/**********************************************************************
 * DOC.C                             Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * Doc 1.00 - simple document processor.                              *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     05-May-97                                                 *
 **********************************************************************/


/* includes ***********************************************************/


#include <stdio.h>
#include <string.h>


/* enumerated types ***************************************************/


typedef enum {
    R_OK,      /* everything went OK */
    R_COMMAND, /* command line error */
    R_INPUT,   /* error reading from input file */
    R_OUTPUT,  /* error writing to output file */
    R_EOF,     /* eof met */
    R_LAST     /* placeholder */
} result;


/* global variables ***************************************************/


FILE *infile,        /* input file */
     *outfile;       /* output file */
char  inname[72],    /* input filename */
      outname[72],   /* output filename */
      pbuf[128][81], /* page buffer */
      header[81],    /* header text */
      footer[81];    /* footer text */
int   len = 60,      /* page length */
      page = 1,      /* page number */
      pstart = 0,    /* page start line */
      pend = 60;     /* page end line + 1 */


/* level 3 routines ***************************************************/


/* hfcpy() - header/footer copy */
char *hfcpy(char *dest, char *src)
{
    if( strchr(src, '%') )
        sprintf(dest, src, page);
    else
        strcpy(dest, src);
    return dest;
}

/* level 2 routines ***************************************************/


/* build() - add a line to the page */
void build(int line, char *buf)
{
    strcpy( pbuf[line], buf );
}

/* dot() - interpret dot command */
int dot(int line, char *buf)
{
    /* .h header command */
    if( !strncmp(buf, ".h", 2) )
    {
        strcpy( header, &buf[2] );
        if( !strcmp(header, "\n") )
            pstart = 0;
        else if(!line)
            line = pstart = 2;
    }

    /* .f footer command */
    else if( !strncmp(buf, ".f", 2) )
    {
        strcpy( footer, &buf[2] );
        if( !strcmp(footer, "\n") )
            pend = len;
        else if(line < len - 2)
            pend = len - 2;
    }

    return line;
}

/* print() - print current page */
int print(void)
{
    int lcount, /* line count */
        lastbl, /* last blank line on page */
        line;   /* current line after printing */

    /* put footer on page */
    if(pend < len)
    {
        hfcpy( pbuf[len - 1], footer );
        strcpy( pbuf[len - 2], "" );
    }
    if(pstart)
    {
        hfcpy( pbuf[0], header );
        strcpy( pbuf[1], "\n" );
    }
    page++;

    /* find last blank line on page */
    for(lastbl = lcount = pstart; lcount < pend; lcount++)
        if(!strcmp( pbuf[lcount], "\n" ) && (lcount - lastbl) != 2)
            lastbl = lcount;
    if(!lastbl) lastbl = pend;

    /* print page */
    for(lcount = 0; lcount < lastbl; lcount++)
        fputs( pbuf[lcount], outfile);
    for(;lcount < pend; lcount++)
        fputs("\n", outfile);
    if(lcount < len)
    {
        fputs("\n", outfile);
        fputs( pbuf[len - 1], outfile );
    }
    putc('\f', outfile);

    /* ensure there is room for the header on the new page */
    if( strcmp(header, "\n") ) pstart = 2;

    /* move unprinted material to top of page */
    for(lcount = pstart; lcount < pend - lastbl - 1 + pstart; lcount++)
        strcpy( pbuf[lcount], pbuf[lastbl + lcount - pstart + 1] );
    line = lcount;
    for(;lcount < pend; lcount++)
        strcpy( pbuf[lcount], "\n" );

    /* ensure there is room for the footer on the new page */
    if( strcmp(footer, "\n") ) pend = len - 2;

    /* return new line position */
    return line;
}


/* level 1 routines ***************************************************/


/* interpret() - interpret command line parameters */
result interpret(int argc, char **argv)
{
    int   curr = 1; /* current parameter */
    char *ca;       /* text of current parameter */

    /* initialise */
    strcpy(inname, "");
    strcpy(outname, "");

    /* scan parameters */
    while(curr < argc)
    {
        ca = argv[curr];
        if(!strncmp( ca, "-output" , strlen(ca) ))
        {
            if(++curr < argc) strcpy( outname, argv[curr] );
        }
        else if(!strncmp( ca, "-length", strlen(ca) ))
        {
            if(++curr < argc) sscanf( argv[curr], "%d", &len );
            pend = len;
        }
        else if(!strncmp( ca, "-page", strlen(ca) ))
        {
            if(++curr < argc) sscanf( argv[curr], "%d", &page );
        }
        else if( *ca == '-' )
            return R_COMMAND;
        else
            strcpy( inname, argv[curr] );
        ++curr;
    }

    if(*inname && *outname) return R_OK; else return R_COMMAND;
}

/* docopen() - open input and output files */
result docopen(void)
{
    int lcount; /* line counter to fill page buffer */

    /* open files */
    if(!( infile = fopen(inname, "r") )) return R_INPUT;
    if(!( outfile = fopen(outname, "w") ))
    {
        fclose(infile);
        return R_OUTPUT;
    }

    /* initialise and exist */
    strcpy(header, "\n");
    strcpy(footer, "\n");
    for(lcount = 0; lcount < 128; lcount++)
        strcpy( pbuf[lcount], "\n" );
    setbuf(stdout, NULL);
    printf("processing page %d ... ", page);
    return R_OK;
}

/* docprocess() - process a line of the document */
result docprocess(void)
{
    char       buf[81]; /* input line buffer */
    static int line;    /* line number */

    /* line input */
    if( !fgets(buf, 81, infile) )
    {
        print();
        return R_EOF;
    }

    /* line processing */
    while(*buf == '\f')
    {
        line = print();
        strcpy( buf, &buf[1] );
    }
    if(*buf == '.')
        line = dot(line, buf);
    else
        build(line++, buf);

    /* page output */
    if(line == pend)
    {
        line = print();
        printf("\rprocessing page %d ... ", page);
    }

    return R_OK;
}

/* docclose() - close document files */
result docclose(void)
{
    fclose(infile);
    fclose(outfile);
    puts("done.");
    return R_OK;
}


/* level 0 routine ****************************************************/


/* main() - main program */
int main(int argc, char **argv)
{
    result r; /* result of called process */

    if(( r = interpret(argc, argv) )) return r;
    if(( r = docopen() ))             return r;
    while( !docprocess() );           return docclose();
}
