/**************************************************************
 * user.c                    Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroMail 1.00 - User Database File Maintenance            *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     24-Mar-97                                         *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astroml.h"
#include "platform.h"


/* some macros ************************************************/


/* record sizes */
#define DATASIZE 50  /* size of packed data record */
#define IND1SIZE  8  /* size of packed number index record */
#define IND2SIZE 40  /* size of packed name index record */

/* flags */
#define UF_DELETED 0x01 /* deleted flag */


/* enumerated types *******************************************/


/* user index */
typedef enum {
    U_NONE,   /* read without index */
    U_NUMBER, /* read by number index */
    U_NAME,   /* read by name index */
    U_LAST    /* placeholder */
} uindex;


/* structures *************************************************/


/* hidden flags substructure */
typedef struct {
    int direct: 1; /* direct(1) or routed(0) responses */
} flags;

/* hidden user structure */
typedef struct {

    /* record fields */
    char  deleted;   /* deleted and other flags */
    long  number;    /* user number */
    char  name[36];  /* user name */
    fido  addr;      /* user's Fidonet address */
    flags f;         /* user flags */

    /* index maintenance fields */
    char  infile;    /* 0=new record, !0=somewhere in file */
    long  pos;       /* record position when last read */
    long  onumber;   /* original user number */
    char  oname[36]; /* original user name */

} user;

/* indexes */
typedef struct {   /* user number index */
    long pos,          /* position of record in data file */
         number;       /* user number */
} ind1;
typedef struct {   /* user name index */
    long pos;          /* position of record in data file */
    char name[36];     /* user name */
} ind2;


/* global variables *******************************************/


FILE *user_data, /* user data file */
     *user_ind1, /* user number index file */
     *user_ind2; /* user name index file */


/* level 3 routines *******************************************/


/* user_readind1() - read a number index record */
result user_readind1(ind1 *i)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(user_ind1) ) == EOF)
        return R_EOF;
    i->pos = pos;
    FREADLONGR(i->pos, user_ind1);
    FREADLONG(i->number, user_ind1);
    return R_OK;
}

/* user_readind2() - read a name index record */
result user_readind2(ind2 *j)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(user_ind2) ) == EOF)
        return R_EOF;
    j->pos = pos;
    FREADLONGR(j->pos, user_ind2);
    fread(j->name, 36, 1, user_ind2);
    return R_OK;
}


/* level 2 routines *******************************************/


/* user_findind1() - find a number index record */
long user_findind1(ind1 *i, long number)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(user_ind1); fseek(user_ind1, 0, SEEK_END);
    if( ftell(user_ind1) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(user_ind1) - 8 ) / IND1SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(user_ind1, 8 + IND1SIZE * middle, SEEK_SET);
    user_readind1(i);
    while(i->number != number && start <= end)
    {
        if(i->number < number)
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(user_ind1, 8 + IND1SIZE * middle, SEEK_SET);
        user_readind1(i);
    }

    /* return value */
    if(i->number == number)
        return 8 + middle * IND1SIZE;
    fseek(user_ind1, pos, SEEK_SET);
    return 0;
}

/* user_findind2() - find a name index record */
long user_findind2(ind2 *j, char *name)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(user_ind2); fseek(user_ind2, 0, SEEK_END);
    if( ftell(user_ind2) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(user_ind2) - 8 ) / IND2SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(user_ind2, 8 + IND2SIZE * middle, SEEK_SET);
    user_readind2(j);
    while( stricmp(j->name, name) && start <= end)
    {
        if( stricmp(j->name, name) < 0 )
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(user_ind2, 8 + IND2SIZE * middle, SEEK_SET);
        user_readind2(j);
    }

    if( !stricmp(j->name, name) )
        return 8 + middle * IND2SIZE;
    fseek(user_ind2, pos, SEEK_SET);
    return 0;
}

/* user_writeind1() - write a number index record */
result user_writeind1(ind1 i)
{
    if( fputc(i.pos & 0xff, user_ind1) == EOF ) return R_FILE;
    FWRITELONGR(i.pos,   user_ind1);
    FWRITELONG(i.number, user_ind1);
    return R_OK;
}

/* user_writeind2() - write a name index record */
result user_writeind2(ind2 j)
{
    if( fputc(j.pos & 0xff, user_ind2) == EOF ) return R_FILE;
    FWRITELONGR(j.pos, user_ind2);
    fwrite(j.name, 36, 1, user_ind2);
    return R_OK;
}


/* level 1 routines *******************************************/


/* user_readdata() - read a data record */
result user_readdata(user *u)
{
    int  deleted; /* deleted flag */
    long pos;     /* position in file */

    /* check for EOF */
    pos = ftell(user_data);
    if(( deleted = fgetc(user_data) ) == EOF)
        return R_EOF;

    /* read rest of record */
    u->deleted = deleted;
    FREADLONG(u->number,      user_data);
    fread(u->name, 36, 1,     user_data);
    FREADSHORT(u->addr.zone,  user_data);
    FREADSHORT(u->addr.net,   user_data);
    FREADSHORT(u->addr.node,  user_data);
    FREADSHORT(u->addr.point, user_data);
    u->f.direct = fgetc(user_data) & 1;

    /* set index maintenance fields and return */
    u->infile = 1;
    u->pos = pos;
    u->onumber = u->number;
    strcpy(u->oname, u->name);
    return R_OK;
}

/* user_writedata() - write a data record */
result user_writedata(user *u)
{
    long pos; /* position in file */

    /* check that record can be written */
    pos = ftell(user_data);
    if( fputc(u->deleted, user_data) == EOF )
        return R_FILE;

    /* write rest of record */
    FWRITELONG(u->number,      user_data);
    fwrite(u->name, 36, 1,     user_data);
    FWRITESHORT(u->addr.zone,  user_data);
    FWRITESHORT(u->addr.net,   user_data);
    FWRITESHORT(u->addr.node,  user_data);
    FWRITESHORT(u->addr.point, user_data);
    fputc(u->f.direct & 1, user_data);

    /* set index maintenance fields and return */
    u->infile = 1;
    u->pos = pos;
    u->onumber = u->number;
    strcpy(u->oname, u->name);
    return R_OK;
}

/* user_sortind1() - single bi-directional pass of sort */
int user_sortind1(void)
{
    long   pos,    /* stored position in index */
           number; /* original user number of index record */
    ind1   i1,     /* first index record to compare */
           i2;     /* second index record to compare */
    result r;      /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(user_ind1);
    if(pos > 8)
    {
        fseek(user_ind1, -IND1SIZE, SEEK_CUR);
        user_readind1(&i1);
        number = i1.number;
    }
    else number = 0;

    /* ensure file is big enough to need sorting */
    fseek(user_ind1, 0, SEEK_END);
    if( ftell(user_ind1) <= 8 + IND1SIZE )
    {
        fseek(user_ind1, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(user_ind1, 8, SEEK_SET); user_readind1(&i1);
    while( user_readind1(&i2) == R_OK )
    {
        if(i1.number > i2.number)
        {
            fseek(user_ind1, -2 * IND1SIZE, SEEK_CUR);
            user_writeind1(i2); user_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(user_ind1, -IND1SIZE, SEEK_CUR);
        user_readind1(&i1);
    }

    /* backward pass of sort */
    fseek(user_ind1, -IND1SIZE, SEEK_END);
    while( ftell(user_ind1) > 8 )
    {
        fseek(user_ind1, -IND1SIZE, SEEK_CUR);
        user_readind1(&i1); user_readind1(&i2);
        if(i1.number > i2.number)
        {
            fseek(user_ind1, -2 * IND1SIZE, SEEK_CUR);
            user_writeind1(i2); user_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(user_ind1, -2 * IND1SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        user_findind1(&i1, number);
    else
        fseek(user_ind1, pos, SEEK_SET);
    return r;
}

/* user_sortind2() - single bi-directional pass of sort */
int user_sortind2(void)
{
    long   pos;      /* stored position in index */
    ind2   j1,       /* first index record to compare */
           j2;       /* second index record to compare */
    char   name[36]; /* name field of current index record */
    result r;        /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(user_ind2);
    if(pos > 8)
    {
        fseek(user_ind2, -IND2SIZE, SEEK_CUR);
        user_readind2(&j1);
        strcpy(name, j1.name);
    }

    /* ensure file is big enough to need sorting */
    fseek(user_ind2, 0, SEEK_END);
    if( ftell(user_ind2) <= 8 + IND2SIZE )
    {
        fseek(user_ind2, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(user_ind2, 8, SEEK_SET); user_readind2(&j1);
    while( user_readind2(&j2) == R_OK )
    {
        if( stricmp(j1.name, j2.name) > 0 )
        {
            fseek(user_ind2, -2 * IND2SIZE, SEEK_CUR);
            user_writeind2(j2); user_writeind2(j1);
            r = R_SWAPS;
        }
        fseek(user_ind2, -IND2SIZE, SEEK_CUR);
        user_readind2(&j1);
    }

    /* backward pass of sort */
    fseek(user_ind2, -IND2SIZE, SEEK_END);
    while( ftell(user_ind2) > 8 )
    {
        fseek(user_ind2, -IND2SIZE, SEEK_CUR);
        user_readind2(&j1); user_readind2(&j2);
        if( stricmp(j1.name, j2.name) > 0 )
        {
            fseek(user_ind2, -2 * IND2SIZE, SEEK_CUR);
            user_writeind2(j2); user_writeind2(j1);
            r = R_SWAPS;
        }
        fseek(user_ind2, -2 * IND2SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        user_findind2(&j1, name);
    else
        fseek(user_ind2, pos, SEEK_SET);
    return r;
}


/* level 0 routines *******************************************/


/* user_open() - open user files */
result user_open(char *path)
{
    char dataname[128], /* name of data file */
         ind1name[128], /* name of user number index file */
         ind2name[128], /* name of user name index file */
         dataheader[8], /* data file header */
         ind1header[8], /* index 1 header */
         ind2header[8]; /* index 2 header */

    /* initialise data names */
    sprintf(dataname, "%suser.data", path);
    sprintf(ind1name, "%suser1.index", path);
    sprintf(ind2name, "%suser2.index", path);

    /* attempt to open files */
    if(( user_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    if(( user_ind1 = fopen(ind1name, "r+b") ) == NULL)
    {
        fclose(user_data);
        return R_FILE;
    }
    if(( user_ind2 = fopen(ind2name, "r+b") ) == NULL)
    {
        fclose(user_data);
        fclose(user_ind1);
        return R_FILE;
    }

    /* verify files */
    fread(dataheader, 8, 1, user_data);
    fread(ind1header, 8, 1, user_ind1);
    fread(ind2header, 8, 1, user_ind2);
    if(( strncmp(dataheader, "ASM10UF", 8) ) ||
       ( strncmp(ind1header, "ASM10U1", 8) ) ||
       ( strncmp(ind2header, "ASM10U2", 8) ))
    {
        fclose(user_data);
        fclose(user_ind1);
        fclose(user_ind2);
        return R_HEADER;
    }

    /* files checked out OK */
    return R_OK;
}

/* user_create() - create user files */
result user_create(char *path)
{
    char  dataname[128], /* name of data file */
          ind1name[128], /* name of user number index file */
          ind2name[128]; /* name of user name index file */
    FILE *test;          /* used to test existence of files */

    /* initialise data names */
    sprintf(dataname, "%suser.data", path);
    sprintf(ind1name, "%suser1.index", path);
    sprintf(ind2name, "%suser2.index", path);

    /* check for files' existence */
    if(( test = fopen(dataname, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1name, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1name, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }

    /* attempt to create files */
    if(( user_data = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    if(( user_ind1 = fopen(ind1name, "wb") ) == NULL)
    {
        fclose(user_data); remove(dataname);
        return R_FILE;
    }
    if(( user_ind2 = fopen(ind2name, "wb") ) == NULL)
    {
        fclose(user_data); remove(dataname);
        fclose(user_ind1); remove(ind1name);
        return R_FILE;
    }

    /* write headers */
    fwrite("ASM10UF", 8, 1, user_data);
    fwrite("ASM10U1", 8, 1, user_ind1);
    fwrite("ASM10U2", 8, 1, user_ind2);

    /* close files and return */
    fclose(user_data);
    fclose(user_ind1);
    fclose(user_ind2);
    return R_OK;
}

/* user_close() - close user files */
result user_close(void)
{
    fclose(user_data);
    fclose(user_ind1);
    fclose(user_ind2);
    return R_OK;
}

/* user_new() - allocate memory for new user record */
user *user_new(void)
{
    return calloc( 1, sizeof(user) );
}

/* user_old() - free memory for new user record */
void user_old(user *u)
{
    free(u);
}

/* user_clear() - clear details from existing user record */
void user_clear(user *u)
{
    memset( u, 0, sizeof(u) );
}

/* user_setnumber() - set the user number */
void user_setnumber(user *u, long number)
{
    u->number = number;
}

/* user_getnumber() - return the user number */
long user_getnumber(user *u)
{
    return u->number;
}

/* user_setname() - set the user name */
void user_setname(user *u, char *name)
{
    strcpy(u->name, name);
}

/* user_getname() - return the user name */
char *user_getname(char *name, user *u)
{
    return strcpy(name, u->name);
}

/* user_setaddr() - set the fidonet address */
void user_setaddr(user *u, fido addr)
{
    u->addr = addr;
}

/* user_getaddr() - return the user address */
fido user_getaddr(user *u)
{
    return u->addr;
}

/* user_setdirect() - set the direct flag */
void user_setdirect(user *u, int direct)
{
    u->f.direct = direct;
}

/* user_getdirect() - return the direct flag */
int user_getdirect(user *u)
{
    return u->f.direct;
}

/* user_write() - write a record */
result user_write(user *u)
{
    ind1   i;  /* number index record */
    ind2   j;  /* name index record */
    long   p1, /* position of index record 1 */
           p2; /* position of index record 2 */

    /* write new record */
    if(!u->infile)
    {
        if( user_findind1(&i, u->number) ) return R_DUPE;
        if( user_findind2(&j, u->name) )   return R_DUPE;
        fseek(user_data, 0, SEEK_END);
        fseek(user_ind1, 0, SEEK_END);
        fseek(user_ind2, 0, SEEK_END);
        i.pos = j.pos = ftell(user_data);
        i.number = u->number;
        strcpy(j.name, u->name);
        if( user_writedata(u) != R_OK )    return R_FILE;
        if( user_writeind1(i) != R_OK )    return R_CORRUPT;
        user_sortind1();
        if( user_writeind2(j) != R_OK )    return R_CORRUPT;
        user_sortind2();
        return R_OK;
    }

    /* rewrite existing record */
    p1 = user_findind1(&i, u->onumber);
    p2 = user_findind2(&j, u->oname);
    if( (!p1 || !p2) )                     return R_CORRUPT;
    if( (i.pos != j.pos) )                 return R_CORRUPT;
    fseek(user_data, i.pos, SEEK_SET);
    if( user_writedata(u) != R_OK )        return R_FILE;
    if(i.number != u->number)
    {
        fseek(user_ind1, p1, SEEK_SET);
        if( user_writeind1(i) != R_OK )    return R_CORRUPT;
        user_sortind1();
    }
    if( stricmp(j.name, u->name) )
    {
        fseek(user_ind2, p2, SEEK_SET);
        if( user_writeind2(j) != R_OK )    return R_CORRUPT;
        user_sortind2();
    }
    return R_OK;
}

/* user_first() - read the first record */
result user_first(user *u, uindex inum)
{
    ind1   i; /* number index record */
    ind2   j; /* name index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case U_NONE:
            fseek(user_data, 8, SEEK_SET);
            return user_readdata(u);
        case U_NUMBER:
            fseek(user_ind1, 8, SEEK_SET);
            if(( r = user_readind1(&i) ) != R_OK) return r;
            fseek(user_data, i.pos, SEEK_SET);
            return user_readdata(u);
        case U_NAME:
            fseek(user_ind2, 8, SEEK_SET);
            if(( r = user_readind2(&j) ) != R_OK) return r;
            fseek(user_data, j.pos, SEEK_SET);
            return user_readdata(u);
        default:
            return R_INDEX;
    }
}

/* user_next() - read the next record */
result user_next(user *u, uindex inum)
{
    ind1   i; /* number index record */
    ind2   j; /* name index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case U_NONE:
            return user_readdata(u);
        case U_NUMBER:
            if(( r = user_readind1(&i) ) != R_OK) return r;
            fseek(user_data, i.pos, SEEK_SET);
            return user_readdata(u);
        case U_NAME:
            if(( r = user_readind2(&j) ) != R_OK) return r;
            fseek(user_data, j.pos, SEEK_SET);
            return user_readdata(u);
        default:
            return R_INDEX;
    }
}

/* user_prev() - read the previous record */
result user_prev(user *u, uindex inum)
{
    ind1   i; /* number index record */
    ind2   j; /* name index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case U_NONE:
            if(( ftell(user_data) - 2 * DATASIZE ) < 8)
                return R_EOF;
            fseek(user_data, -2 * DATASIZE, SEEK_CUR);
            return user_readdata(u);
        case U_NUMBER:
            if(( ftell(user_ind1) - 2 * IND1SIZE ) < 8)
                return R_EOF;
            fseek(user_ind1, -2 * IND1SIZE, SEEK_CUR);
            if(( r = user_readind1(&i) ) != R_OK) return r;
            fseek(user_data, i.pos, SEEK_SET);
            return user_readdata(u);
        case U_NAME:
            if(( ftell(user_ind2) - 2 * IND2SIZE ) < 8)
                return R_EOF;
            fseek(user_ind2, -2 * IND2SIZE, SEEK_CUR);
            if(( r = user_readind2(&j) ) != R_OK) return r;
            fseek(user_data, j.pos, SEEK_SET);
            return user_readdata(u);
        default:
            return R_INDEX;
    }
}

/* user_last() - read the last record */
result user_last(user *u, uindex inum)
{
    ind1   i;   /* number index record */
    ind2   j;   /* name index record */
    result r;   /* result of index read */
    long   pos; /* current data/index file position */

    switch(inum)
    {
        case U_NONE:
            pos = ftell(user_data);
            if( fseek(user_data, -DATASIZE, SEEK_END) )
                return R_EOF;
            if( ftell(user_data) < 8 )
            {
                fseek(user_data, pos, SEEK_SET);
                return R_EOF;
            }
            return user_readdata(u);
        case U_NUMBER:
            pos = ftell(user_ind1);
            if( fseek(user_ind1, -IND1SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(user_ind1) < 8 )
            {
                fseek(user_ind1, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = user_readind1(&i) ) != R_OK) return r;
            fseek(user_data, i.pos, SEEK_SET);
            return user_readdata(u);
        case U_NAME:
            pos = ftell(user_ind2);
            if( fseek(user_ind2, -IND2SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(user_ind2) < 8 )
            {
                fseek(user_ind2, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = user_readind2(&j) ) != R_OK) return r;
            fseek(user_data, j.pos, SEEK_SET);
            return user_readdata(u);
        default:
            return R_INDEX;
    }
}

/* user_find() - find a record by number or name */
result user_find(user *u, char *searchtext)
{
    ind1   i;      /* user number index record */
    ind2   j;      /* user name index record */
    long   number; /* user number from search text */

    if( sscanf(searchtext, "%ld", &number) == 1 )
    {
        if( user_findind1(&i, number) == 0 ) return R_EOF;
        fseek(user_data, i.pos, SEEK_SET);
        return user_readdata(u);
    }
    if( user_findind2(&j, searchtext) == 0 ) return R_EOF;
    fseek(user_data, j.pos, SEEK_SET);
    return user_readdata(u);

    return R_OK;
}

/* user_delete() - mark a record as deleted */
result user_delete(user *u)
{
    if(u->pos == 0) return R_EOF;
    u->deleted ^= UF_DELETED;
    fseek(user_data, u->pos, SEEK_SET);
    return user_writedata(u);
}

/* user_deleted() - return the deleted status of a record */
int user_deleted(user *u)
{
    if(u->pos == 0) return R_EOF;
    return u->deleted & UF_DELETED;
}

/* user_pack() - pack a close data file */
result user_pack(char *path)
{
    user  *u;             /* user record */
    char   dataname[128], /* name of data file */
           ind1name[128], /* name of user number index file */
           ind2name[128], /* name of user name index file */
           tempname[128], /* name of temporary file */
           dataheader[8]; /* data file header */
    FILE  *user_temp;     /* temporary file */
    ind1   i;             /* number index record */
    ind2   j;             /* name index record */

    /* initialise data names */
    sprintf(dataname, "%suser.data", path);
    sprintf(tempname, "%suser.temp", path);

    /* attempt to open data & temp file */
    if(( user_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    fread(dataheader, 8, 1, user_data);
    if(( strncmp(dataheader, "ASM10UF", 8) ) ||
       ( user_temp = fopen(tempname, "w+b") ) == NULL)
    {
        fclose(user_data);
        return R_HEADER;
    }

    /* allocate memory */
    if( (u = malloc( sizeof(user) )) == NULL )
    {
        fclose(user_data);
        fclose(user_temp);
        remove(tempname);
        return R_MEMORY;
    }

    /* copy non-deleted records to temporary file and back */
    while( user_readdata(u) == R_OK )
        if( !(u->deleted & UF_DELETED) )
            fwrite(u, sizeof(user), 1, user_temp);
    fclose(user_data);
    remove(dataname);
    if(( user_data = fopen(dataname, "w+b") ) == NULL)
    {
        free(u);
        fclose(user_temp);
        remove(tempname);
        return R_FILE;
    }
    fwrite("ASM10UF", 8, 1, user_data);
    fseek(user_temp, 0, SEEK_SET);
    while( fread(u, sizeof(user), 1, user_temp) == 1 )
        user_writedata(u);
    fclose(user_temp);
    remove(tempname);

    /* recreate number index */
    sprintf(ind1name, "%suser1.index", path); remove(ind1name);
    if(( user_ind1 = fopen(ind1name, "w+b") ) == NULL)
    {
        free(u);
        fclose(user_data);
        return R_FILE;
    }
    fwrite("ASM10U1", 8, 1, user_ind1);
    fseek(user_data, 8, SEEK_SET); i.pos = ftell(user_data);
    while( user_readdata(u) == R_OK )
    {
        i.number = u->number;
        user_writeind1(i);
        i.pos = ftell(user_data);
    }
    while( user_sortind1() == R_SWAPS );
    fclose(user_ind1);

    /* recreate name index */
    sprintf(ind2name, "%suser2.index", path); remove(ind2name);
    if(( user_ind2 = fopen(ind2name, "w+b") ) == NULL)
    {
        free(u);
        fclose(user_data);
        return R_FILE;
    }
    fwrite("ASM10U2", 8, 1, user_ind2);
    fseek(user_data, 8, SEEK_SET); j.pos = ftell(user_data);
    while( user_readdata(u) == R_OK )
    {
        strcpy(j.name, u->name);
        user_writeind2(j);
        j.pos = ftell(user_data);
    }
    while( user_sortind2() == R_SWAPS );
    fclose(user_ind2);

    /* clean up */
    free(u);
    fclose(user_data);
    return R_OK;
}
