/**********************************************************************
 * SCRLOG.C                          Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * AstroMail 1.00 - Screen and Log Output Module.                     *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     09-Apr-97                                                 *
 **********************************************************************/


/* included headers ***************************************************/


#include <stdio.h>
#include "astroml.h"
#include "log.h"


/* global variables ***************************************************/


extern char pid[36];


/* main open/close routines *******************************************/


/* scrlog_open() - open screen and log */
result scrlog_open(void)
{
    setbuf(stdout, NULL);

    printf("\nCenturion %s\n", pid);
    puts(  "Copyright (C) Damian Walker 1997");
    puts(  "Fidonet 2:2502/666");
    puts(  "");

    return R_OK;
}

/* scrlog_openlog() - open the log file */
result scrlog_openlog(char *filename, short status)
{
    char fullpid[81];   /* "Centurion " + pid */

    sprintf(fullpid, "Centurion %s", pid);
    if( openlog(filename, fullpid) == LOGFILEERROR ) return R_FILE;
    setlogstatus(status);

    return R_OK;
}

/* scrlog_close() - close the library */
void scrlog_close(void)
{
    closelog();
}

/* void scrlog_cannotopenlog() - cannot open the log file */
void scrlog_cannotopenlog(char *filename)
{
    printf("! Cannot open log file %s\n", filename);
}


/* environment-related messages ***************************************/


/* scrlog_error() - command line error */
void scrlog_error(void)
{
    puts("! Command line error: type \'astromail ?\' for help");
}

/* scrlog_running() - program already running */
void scrlog_running(void)
{
    puts("! AstroMail already running");
}

/* scrlog_help() - program help output */
void scrlog_help(void)
{
    puts("usage: astroml <commands> [<switches>]");
    puts("");
    puts("commands =");
    puts("    in          process inbound order messages");
    puts("    out         generate outbound report messages");
    puts("    ?           request this help");
    puts("");
    puts("switches =");
    puts("    -path <p>   path to configs, semaphores etc.");
    puts("    -game <n>   name of game to process (turnout only)");
}


/* general messages ***************************************************/


/* scrlog_turnin() - running turn input module */
void scrlog_turnin(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Running turn input module");
    printf("* %s\n", entry);
    writelog(LOGMAJOR, entry);
}

/* scrlog_trnout() - running turn output module */
void scrlog_trnout(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Running turn output module");
    printf("* %s\n", entry);
    writelog(LOGMAJOR, entry);
}

/* scrlog_outofmemory() - general out of memory error */
void scrlog_outofmemory(char *act)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Out of memory %s", act);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badreturncode() - unexpected result code from function */
void scrlog_badreturncode(result r, char *function)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad return code %d from %s", r, function);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_finished() - finished */
void scrlog_finished(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Finished!");
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}


/* config-related messages ********************************************/


/* scrlog_badcommandinconfig() - bad command in config file */
void scrlog_badcommandinconfig(char *path, int line)
{
    printf("! Bad command in line %d of %sastroml.data", line, path);
}

/* scrlog_cannotreadconfig() - file error opening config */
void scrlog_cannotreadconfig(char *path)
{
    printf("! Cannot read config file %sastroml.data\n", path);
}

/* scrlog_cannotwriteconfig() - file error creating config */
void scrlog_cannotwriteconfig(char *path)
{
    printf("! Cannot write config file %sastroml.data\n", path);
}

/* scrlog_configread() - config read successfully */
void scrlog_configread(char *path)
{
    char entry[128]; /* line for log entry */

    sprintf(entry, "Config file %sastroml.data read successfully",
        path);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_noprimaryaddress() - no primary address given */
void scrlog_noprimaryaddress(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "No primary address given in %sastroml.data", path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}


/* detail-related errors **********************************************/


/* scrlog_cannotopendetail() - cannot read detail file */
void scrlog_cannotopendetail(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot open detail file %sdetail.data", path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badcommandindetail() - bad command in detail file */
void scrlog_badcommandindetail(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad command in detail file %sdetail.data", path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_detailread() - detail file read successfully */
void scrlog_detailread(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Detail file %sdetail.data read successfully", path);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}


/* game list file related messages ************************************/


/* scrlog_cannotopengamelist() - error opening game list */
void scrlog_cannotopengamelist(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot open game list %sgamelist.data", path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badfileingamelist() - bad command in game list */
void scrlog_badfileingamelist(char *path, int line)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad detail file (line %d in %sgamelist.data)", line,
        path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badlineingamelist() - bad command in game list */
void scrlog_badlineingamelist(char *path, int line)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad line %d in game line %sgamelist.data", line,
        path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_gamelistread() - game list read successfully */
void scrlog_gamelistread(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Game list %sgamelist.data read successfully", path);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_gamelistempty() - game list empty */
void scrlog_gamelistempty(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Game list %sgamelist.data is empty", path);
    printf("? %s\n", entry);
    writelog(LOGWARNING, entry);
}

/* scrlog_gamenotfound() - game not found */
void scrlog_gamenotfound(char *gamename)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Game \"%s\" not found", gamename);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_cannotwritegamelist() - cannot write game list */
void scrlog_cannotwritegamelist(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot write game list %sgamelist.data", path);
    printf("? %s\n", entry);
    writelog(LOGWARNING, entry);
}


/* msglib/iomsg errors ************************************************/


/* scrlog_cannotopennetmailarea() - cannot open netmail directory */
void scrlog_cannotopennetmailarea(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot open netmail directory %s", path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_netmailopened() - netmail area opened successfully */
void scrlog_netmailopened(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Netmail area %s opened successfully", path);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_scanning() - scanning inbound messages */
void scrlog_scanning(void)
{
    printf("Scanning...\r");
}

/* scrlog_msgread() - message read */
void scrlog_msgread(char *name, fido addr)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Processing netmail from %s, %d:%d/%d.%d", name,
        addr.zone, addr.net, addr.node, addr.point);
    printf("* %s\n", entry);
    writelog(LOGMAJOR, entry);
}

/* scrlog_processingorder() - processing order <whatever> */
void scrlog_processingorder(char *ordertype)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Processing \"%s\" order", ordertype);
    printf("- %s ...%15s\r", entry, "");
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_badorder() - bad order in message */
void scrlog_badorder(int line)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Rejecting bad order in line %d of message", line);
    printf("+ %s\n", entry);
    writelog(LOGMINOR, entry);
}

/* scrlog_messagedone() - finished processing message */
void scrlog_messagedone(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Done processing this message");
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_replyposted() - reply posted */
void scrlog_replyposted(char *name, fido addr)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Reply posted to %s, %d:%d/%d.%d", name, addr.zone,
        addr.net, addr.node, addr.point);
    printf("+ %s\n", entry);
    writelog(LOGMINOR, entry);
}

/* scrlog_reportposted() - report posted */
void scrlog_reportposted(char *name, fido addr)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Report posted to %s, %d:%d/%d.%d", name, addr.zone,
        addr.net, addr.node, addr.point);
    printf("* %s\n", entry);
    writelog(LOGMAJOR, entry);
}

/* scrlog_scancomplete() - netmail scan complete */
void scrlog_scancomplete(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Netmail scan complete");
    printf("* %-72s\n", entry);
    writelog(LOGMINOR, entry);
}


/* general file-related messages **************************************/


/* scrlog_cannotopenfile() - cannot open file.* */
void scrlog_cannotopenfile(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot open file %s%s", path, file);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badheaderinfile() - bad header in file.* */
void scrlog_badheaderinfile(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad header in file %s%s", path, file);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_fileopened() - join request file opened */
void scrlog_fileopened(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "File %s%s opened successfully", path, file);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_fileexists() - join request file exists */
void scrlog_fileexists(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "file %s%s already exists", path, file);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_cannotcreatefile() - cannot create join request file */
void scrlog_cannotcreatefile(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot create file %s%s", path, file);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_filecreated() - join request file created */
void scrlog_filecreated(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "File %s%s created successfully", path, file);
    printf("+ %s\n", entry);
    writelog(LOGMINOR, entry);
}

/* scrlog_purgedfile() - file purged */
void scrlog_purgedfile(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "File %s%s purged of deleted records", path, file);
    printf("+ %s\n", entry);
    writelog(LOGMAJOR, entry);
}


/* 'turnout' specific messages ****************************************/


/* scrlog_processingreport() - processing <whatever> report */
void scrlog_processingreport(char *name)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Processing \"%s\" report", name);
    printf("- %s... %15s\r", entry, "");
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_donereports() - done processing reports */
void scrlog_donereports(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "All reports have been processed");
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_readingorderfile() - reading order file */
void scrlog_readingorderfile(void)
{
    printf("- Reading order file ...\r");
}

/* scrlog_debug() - debug messages */
void scrlog_debug(char *msg)
{
    char entry[128]; /* line for screen and log */

    strcpy(entry, msg);
    printf("? %-72s\n", entry);
    writelog(LOGWARNING, entry);
}
