/**************************************************************
 * REPORT.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroWar 1.00 play-by-mail space conquest game host.       *
 * Report file access module.                                 *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     04-Feb-97                                         *
 **************************************************************/


/* included files *********************************************/


#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include "astrowar.h"


/* structures *************************************************/


/* planet list entry */
typedef struct {
    char  name[16]; /* name of planet */
    pos   pp;       /* position of planet */
    int   prod;     /* production rating */
    long  ships;    /* ships in orbit */
    void *next;     /* next entry */
} plist;

/* fleet list entry */
typedef struct {
    long  number;     /* fleet number */
    char  origin[16], /* origin planet */
          dest[16];   /* destination planet */
    long  ships;      /* ships in fleet */
    int   dist;       /* distance left to travel */
    void *next;       /* next entry */
} flist;

/* map image entry */
typedef struct {
    char  name[16]; /* planet name */
    pos   pp;       /* planet position */
    void *next;     /* next entry */
} mlist;

/* join report */
typedef struct {
    long trans;       /* transaction number */
    char homewld[16]; /* name of homeworld */
} jreport;

/* join error */
typedef struct {
    long trans; /* transaction number */
    char error; /* error code */
} jerror;

/* send report */
typedef struct {
    char empire[16]; /* empire name */
    long fleetno;    /* number for new fleet */
    char origin[16], /* planet of origin */
         dest[16];   /* destination planet */
    long ships;      /* ships in fleet */
} sreport;

/* send error */
typedef struct {
    char empire[16], /* empire name */
         origin[16], /* planet of origin */
         dest[16];   /* destination planet */
    long ships;      /* intended fleet size */
    char error;      /* error code */
} serror;

/* planet report */
typedef struct {
    char   empire[16]; /* empire name */
    plist *first,      /* planet list */
          *current;    /* current planet in list */
} preport;

/* fleet report */
typedef struct {
    char   empire[16]; /* empire name */
    flist *first,      /* fleet list */
          *current;    /* current fleet in list */
} freport;

/* map image */
typedef struct {
    char   empire[16], /* empire name */
           centre[16]; /* planet at map centre */
    mlist *first,      /* list of planet on map */
          *current;    /* current planet in list */
} mreport;

/* mapping error */
typedef struct {
    char empire[16], /* empire name */
         centre[16], /* intended map centre */
         error;      /* error code */
} merror;

/* written message */
typedef struct {
    char empire[16], /* recipient empire */
         sender[16], /* sender of message */
        *text;       /* message text */
} wreport;

/* bounced message */
typedef struct {
    char empire[16], /* sender empire */
         recip[16],  /* intended recipient of message */
         error;      /* error code */
} werror;

/* victory report */
typedef struct {
    char empire[16], /* empire name */
         victor[16]; /* name of winning empire */
} vreport;

/* fleet arrival */
typedef struct {
    char empire[16]; /* fleet owner */
    long fleetno;    /* fleet number */
} areport;

/* intelligence report */
typedef struct {
    char empire[16]; /* empire receiving report */
    long fleetno;    /* 'fleet' number */
    char owner[16];  /* owner of scouted planet */
    int  prod;       /* production rating of scouted planet */
    long ships;      /* ships at scouted planet */
} ireport;

/* battle report */
typedef struct {
    char empire[16]; /* attacking empire */
    long fleetno;    /* fleet number */
    char owner[16];  /* owner of defending planet */
    long ships,      /* ships left after attack (on success) */
         enemy;      /* enemy left after attack (on failure) */
} breport;

/* defence report */
typedef struct {
    char empire[16], /* defending empire */
         planet[16], /* planet attacked */
         owner[16];  /* owner of attacking fleet */
    long ships,      /* ships left after defence (on success) */
         enemy;      /* enemy left after defence (on failure) */
} dreport;

/* general report parameters */
typedef union {
    jreport j;  /* join report */
    jerror  je; /* join error */
    sreport s;  /* send report */
    serror  se; /* send error */
    preport p;  /* planets report */
    freport f;  /* fleets report */
    mreport m;  /* map report */
    merror  me; /* mapping error */
    wreport w;  /* write report */
    werror  we; /* write error */
    vreport v;  /* victory report */
    areport a;  /* arrival report */
    ireport i;  /* intelligence report */
    breport b;  /* battle report */
    dreport d;  /* defence report */
} rparam;

/* report record */
typedef struct {
    char type; /* report type */
    rparam rp; /* report parameters */
} report;


/* global variables *******************************************/


FILE *repfile;  /* data file handle */


/* level 2 routines *******************************************/


/* report_readj() - read 'join' report */
void report_readj(report *r)
{
    FREADLONG(r->rp.j.trans, repfile);
    fread(r->rp.j.homewld, 16, 1, repfile);
}

/* report_readje() - read 'join' error */
void report_readje(report *r)
{
    FREADLONG(r->rp.je.trans, repfile);
    r->rp.je.error = fgetc(repfile);
}

/* report_reads() - read 'send' report */
void report_reads(report *r)
{
    fread(r->rp.s.empire, 16, 1, repfile);
    FREADLONG(r->rp.s.fleetno, repfile);
    fread(r->rp.s.origin, 16, 1, repfile);
    fread(r->rp.s.dest,   16, 1, repfile);
    FREADLONG(r->rp.s.ships, repfile);
}

/* report_readse() - read 'send' error */
void report_readse(report *r)
{
    fread(r->rp.se.empire, 16, 1, repfile);
    fread(r->rp.se.origin, 16, 1, repfile);
    fread(r->rp.se.dest,   16, 1, repfile);
    FREADLONG(r->rp.se.ships, repfile);
    r->rp.se.error = fgetc(repfile);
}

/* report_readp() - read 'planets' report */
void report_readp(report *r)
{
    char   firstletter;
    plist *p = NULL;

    r->rp.p.first = NULL;
    fread(r->rp.p.empire, 16, 1, repfile);
    while( (firstletter = fgetc(repfile)) != ' ')
    {
        /* allocate memory */
        if(r->rp.p.first == NULL)
            p = r->rp.p.first = malloc( sizeof(plist) );
        else
            p = p->next = malloc( sizeof(plist) );

        /* read data */
        *(p->name) = firstletter;
        fread(&p->name[1], 15, 1, repfile);
        FREADSHORT(p->pp.x, repfile);
        FREADSHORT(p->pp.y, repfile);
        FREADSHORT(p->prod, repfile);
        FREADLONG(p->ships, repfile);
        p->next = NULL;
    }
}

/* report_readf() - read 'fleets' report */
void report_readf(report *r)
{
    long   fleetno;
    flist *f = NULL;

    r->rp.f.first = NULL;
    fread(r->rp.f.empire, 16, 1, repfile);
    FREADLONG(fleetno, repfile);
    while(fleetno != 0)
    {
        /* allocate memory */
        if(r->rp.f.first == NULL)
            f = r->rp.f.first = malloc( sizeof(flist) );
        else
            f = f->next = malloc( sizeof(flist) );

        /* read data */
        f->number = fleetno;
        fread(f->origin, 16, 1, repfile);
        fread(f->dest,   16, 1, repfile);
        FREADLONG(f->ships, repfile);
        FREADSHORT(f->dist, repfile);

        /* next record */
        FREADLONG(fleetno, repfile);
        f->next = NULL;
    }
}

/* report_readm() - read 'map' report */
void report_readm(report *r)
{
    char   firstletter;
    mlist *m = NULL;

    r->rp.m.first = NULL;
    fread(r->rp.m.empire, 16, 1, repfile);
    fread(r->rp.m.centre, 16, 1, repfile);
    while( (firstletter = fgetc(repfile)) != ' ')
    {
        /* allocate memory */
        if(r->rp.m.first == NULL)
            m = r->rp.m.first = malloc( sizeof(mlist) );
        else
            m = m->next = malloc( sizeof(mlist) );

        /* read data */
        *(m->name) = firstletter;
        fread(&m->name[1], 15, 1, repfile);
        FREADSHORT(m->pp.x, repfile);
        FREADSHORT(m->pp.y, repfile);
        m->next = NULL;
    }
}

/* report_readme() - read mapping error */
void report_readme(report *r)
{
    fread(r->rp.me.empire, 16, 1, repfile);
    fread(r->rp.me.centre, 16, 1, repfile);
    r->rp.me.error = fgetc(repfile);
}

/* report_readw() - read 'write' report */
void report_readw(report *r)
{
    unsigned int msglen; /* message length */

    fread(r->rp.w.empire, 16, 1, repfile);
    fread(r->rp.w.sender, 16, 1, repfile);
    FREADSHORT(msglen, repfile);
    r->rp.w.text = malloc(msglen + 1);
    fread(r->rp.w.text, msglen, 1, repfile);
    r->rp.w.text[msglen] = '\0'; /* NULL not stored in file */
}

/* report_readwe() - read bounced message */
void report_readwe(report *r)
{
    fread(r->rp.we.empire, 16, 1, repfile);
    fread(r->rp.we.recip,  16, 1, repfile);
    r->rp.we.error = fgetc(repfile);
}

/* report_readv() - read victory report */
void report_readv(report *r)
{
    fread(r->rp.v.empire, 16, 1, repfile);
    fread(r->rp.v.victor, 16, 1, repfile);
}

/* report_reada() - read arrival report */
void report_reada(report *r)
{
    fread(r->rp.a.empire, 16, 1, repfile);
    FREADLONG(r->rp.a.fleetno, repfile);
}

/* report_readi() - read intelligence report */
void report_readi(report *r)
{
    fread(r->rp.i.empire, 16, 1, repfile);
    FREADLONG(r->rp.i.fleetno, repfile);
    fread(r->rp.i.owner, 16, 1, repfile);
    FREADSHORT(r->rp.i.prod, repfile);
    FREADLONG(r->rp.i.ships, repfile);
}

/* report_readb() - read battle report */
void report_readb(report *r)
{
    fread(r->rp.b.empire, 16, 1, repfile);
    FREADLONG(r->rp.b.fleetno, repfile);
    fread(r->rp.b.owner, 16, 1, repfile);
    FREADLONG(r->rp.b.ships, repfile);
    FREADLONG(r->rp.b.enemy, repfile);
}

/* report_readd() - read defence report */
void report_readd(report *r)
{
    fread(r->rp.d.empire, 16, 1, repfile);
    fread(r->rp.d.planet, 16, 1, repfile);
    fread(r->rp.d.owner,  16, 1, repfile);
    FREADLONG(r->rp.d.ships, repfile);
    FREADLONG(r->rp.d.enemy, repfile);
}

/* report_writej() - write 'join' report */
void report_writej(report *r)
{
    fputc(r->type,                            repfile);
    fputc(r->rp.j.trans & 255,                repfile);
    fputc(( r->rp.j.trans / 256 ) & 255,      repfile);
    fputc(( r->rp.j.trans / 65536 ) & 255,    repfile);
    fputc(( r->rp.j.trans / 16777216 ) & 255, repfile);
    fwrite(r->rp.j.homewld, 16, 1,            repfile);
}

/* report_writeje() - write 'join' error */
void report_writeje(report *r)
{
    fputc(r->type,                             repfile);
    fputc(r->rp.je.trans & 255,                repfile);
    fputc(( r->rp.je.trans / 256 ) & 255,      repfile);
    fputc(( r->rp.je.trans / 65536 ) & 255,    repfile);
    fputc(( r->rp.je.trans / 16777216 ) & 255, repfile);
    fputc(r->rp.je.error,                      repfile);
}

/* report_writes() - write 'send' report */
void report_writes(report *r)
{
    fputc(r->type,                              repfile);
    fwrite(r->rp.s.empire, 16, 1,               repfile);
    fputc(r->rp.s.fleetno & 255,                repfile);
    fputc(( r->rp.s.fleetno / 256 ) & 255,      repfile);
    fputc(( r->rp.s.fleetno / 65536 ) & 255,    repfile);
    fputc(( r->rp.s.fleetno / 16777216 ) & 255, repfile);
    fwrite(r->rp.s.origin, 16, 1,               repfile);
    fwrite(r->rp.s.dest,   16, 1,               repfile);
    fputc(r->rp.s.ships & 255,                  repfile);
    fputc(( r->rp.s.ships / 256 ) & 255,        repfile);
    fputc(( r->rp.s.ships / 65536 ) & 255,      repfile);
    fputc(( r->rp.s.ships / 16777216 ) & 255,   repfile);
}

/* report_writese() - write 'send' error */
void report_writese(report *r)
{
    fputc(r->type,                             repfile);
    fwrite(r->rp.se.empire, 16, 1,             repfile);
    fwrite(r->rp.se.origin, 16, 1,             repfile);
    fwrite(r->rp.se.dest,   16, 1,             repfile);
    fputc(r->rp.se.ships & 255,                repfile);
    fputc(( r->rp.se.ships / 256 ) & 255,      repfile);
    fputc(( r->rp.se.ships / 65536 ) & 255,    repfile);
    fputc(( r->rp.se.ships / 16777216 ) & 255, repfile);
    fputc(r->rp.se.error,                      repfile);
}

/* report_writep() - write 'planets' report */
void report_writep(report *r)
{
    plist *p; /* pointer to current planet */

    fputc(r->type,                repfile);
    fwrite(r->rp.p.empire, 16, 1, repfile);
    p = r->rp.p.first;
    while(p != NULL)
    {
        fwrite(p->name, 16, 1,               repfile);
        fputc(p->pp.x & 255,                 repfile);
        fputc((unsigned)p->pp.x / 256,       repfile);
        fputc(p->pp.y & 255,                 repfile);
        fputc((unsigned)p->pp.y / 256,       repfile);
        fputc(p->prod & 255,                 repfile);
        fputc(p->prod / 256,                 repfile);
        fputc(p->ships & 255,                repfile);
        fputc(( p->ships / 256 ) & 255,      repfile);
        fputc(( p->ships / 65536 ) & 255,    repfile);
        fputc(( p->ships / 16777216 ) & 255, repfile);
        p = p->next;
    }
    fputc(' ', repfile);
}

/* report_writef() - write 'fleets' report */
void report_writef(report *r)
{
    flist *f; /* pointer to current planet */

    fputc(r->type,                repfile);
    fwrite(r->rp.f.empire, 16, 1, repfile);
    f = r->rp.f.first;
    while(f != NULL)
    {
        fputc(f->number & 255,                repfile);
        fputc(( f->number / 256 ) & 255,      repfile);
        fputc(( f->number / 65536 ) & 255,    repfile);
        fputc(( f->number / 16777216 ) & 255, repfile);
        fwrite(f->origin, 16, 1,              repfile);
        fwrite(f->dest,   16, 1,              repfile);
        fputc(f->ships & 255,                 repfile);
        fputc(( f->ships / 256 ) & 255,       repfile);
        fputc(( f->ships / 65536 ) & 255,     repfile);
        fputc(( f->ships / 16777216 ) & 255,  repfile);
        fputc(f->dist & 255,                  repfile);
        fputc(f->dist / 256,                  repfile);
        f = f->next;
    }
    fwrite("\0\0\0\0", 4, 1, repfile);
}

/* report_writem() - write 'map' report */
void report_writem(report *r)
{
    mlist *m; /* pointer to current planet */

    fputc(r->type,                repfile);
    fwrite(r->rp.m.empire, 16, 1, repfile);
    fwrite(r->rp.m.centre, 16, 1, repfile);
    m = r->rp.m.first;
    while(m != NULL)
    {
        fwrite(m->name, 16, 1,               repfile);
        fputc(m->pp.x & 255,                 repfile);
        fputc((unsigned)m->pp.x / 256,       repfile);
        fputc(m->pp.y & 255,                 repfile);
        fputc((unsigned)m->pp.y / 256,       repfile);
        m = m->next;
    }
    fputc(' ', repfile);
}

/* report_writeme() - write mapping error */
void report_writeme(report *r)
{
    fputc(r->type,                 repfile);
    fwrite(r->rp.me.empire, 16, 1, repfile);
    fwrite(r->rp.me.centre, 16, 1, repfile);
    fputc(r->rp.me.error,          repfile);
}

/* report_writew() - write 'write' report */
void report_writew(report *r)
{
    unsigned int msglen; /* message length */

    msglen = strlen(r->rp.w.text);
    fputc(r->type,                  repfile);
    fwrite(r->rp.w.empire, 16, 1,   repfile);
    fwrite(r->rp.w.sender, 16, 1,   repfile);
    fputc(msglen & 255,             repfile);
    fputc(msglen / 256,             repfile);
    fwrite(r->rp.w.text, msglen, 1, repfile);
}

/* report_writewe() - write bounced message */
void report_writewe(report *r)
{
    fputc(r->type,                 repfile);
    fwrite(r->rp.we.empire, 16, 1, repfile);
    fwrite(r->rp.we.recip, 16, 1,  repfile);
    fputc(r->rp.we.error,          repfile);
}

/* report_writev() - write the victory report */
void report_writev(report *r)
{
    fputc(r->type,                repfile);
    fwrite(r->rp.v.empire, 16, 1, repfile);
    fwrite(r->rp.v.victor, 16, 1, repfile);
}

/* report_writea() - write an fleet arrival report */
void report_writea(report *r)
{
    fputc(r->type,                              repfile);
    fwrite(r->rp.a.empire, 16, 1,               repfile);
    fputc(r->rp.a.fleetno & 255,                repfile);
    fputc(( r->rp.a.fleetno / 256 ) & 255,      repfile);
    fputc(( r->rp.a.fleetno / 65536 ) & 255,    repfile);
    fputc(( r->rp.a.fleetno / 16777216 ) & 255, repfile);
}

/* report_writei() - write an intelligence report */
void report_writei(report *r)
{
    fputc(r->type,                              repfile);
    fwrite(r->rp.i.empire, 16, 1,               repfile);
    fputc(r->rp.i.fleetno & 255,                repfile);
    fputc(( r->rp.i.fleetno / 256 ) & 255,      repfile);
    fputc(( r->rp.i.fleetno / 65536 ) & 255,    repfile);
    fputc(( r->rp.i.fleetno / 16777216 ) & 255, repfile);
    fwrite(r->rp.i.owner, 16, 1,                repfile);
    fputc(r->rp.i.prod & 255,                   repfile);
    fputc(r->rp.i.prod / 256,                   repfile);
    fputc(r->rp.i.ships & 255,                  repfile);
    fputc(( r->rp.i.ships / 256 ) & 255,        repfile);
    fputc(( r->rp.i.ships / 65536 ) & 255,      repfile);
    fputc(( r->rp.i.ships / 16777216 ) & 255,   repfile);
}

/* report_writeb() - write battle report */
void report_writeb(report *r)
{
    fputc(r->type,                              repfile);
    fwrite(r->rp.b.empire, 16, 1,               repfile);
    fputc(r->rp.b.fleetno & 255,                repfile);
    fputc(( r->rp.b.fleetno / 256 ) & 255,      repfile);
    fputc(( r->rp.b.fleetno / 65536 ) & 255,    repfile);
    fputc(( r->rp.b.fleetno / 16777216 ) & 255, repfile);
    fwrite(r->rp.b.owner, 16, 1,                repfile);
    fputc(r->rp.b.ships & 255,                  repfile);
    fputc(( r->rp.b.ships / 256 ) & 255,        repfile);
    fputc(( r->rp.b.ships / 65536 ) & 255,      repfile);
    fputc(( r->rp.b.ships / 16777216 ) & 255,   repfile);
    fputc(r->rp.b.enemy & 255,                  repfile);
    fputc(( r->rp.b.enemy / 256 ) & 255,        repfile);
    fputc(( r->rp.b.enemy / 65536 ) & 255,      repfile);
    fputc(( r->rp.b.enemy / 16777216 ) & 255,   repfile);
}

/* report_writed() - write defence report */
void report_writed(report *r)
{
    fputc(r->type,                              repfile);
    fwrite(r->rp.d.empire, 16, 1,               repfile);
    fwrite(r->rp.d.planet, 16, 1,               repfile);
    fwrite(r->rp.d.owner,  16, 1,               repfile);
    fputc(r->rp.d.ships & 255,                  repfile);
    fputc(( r->rp.d.ships / 256 ) & 255,        repfile);
    fputc(( r->rp.d.ships / 65536 ) & 255,      repfile);
    fputc(( r->rp.d.ships / 16777216 ) & 255,   repfile);
    fputc(r->rp.d.enemy & 255,                  repfile);
    fputc(( r->rp.d.enemy / 256 ) & 255,        repfile);
    fputc(( r->rp.d.enemy / 65536 ) & 255,      repfile);
    fputc(( r->rp.d.enemy / 16777216 ) & 255,   repfile);
}

/* report_clearp() - clear planet list */
void report_clearp(report *r)
{
    plist *p, /* current entry */
          *n; /* next entry */

    p = r->rp.p.first;
    while(p != NULL)
    {
        n = p->next; free(p); p = n;
    }
    r->rp.p.first = NULL;
}

/* report_clearf() - clear fleet list */
void report_clearf(report *r)
{
    flist *f, /* current entry */
          *n; /* next entry */

    f = r->rp.f.first;
    while(f != NULL)
    {
        n = f->next; free(f); f = n;
    }
    r->rp.f.first = NULL;
}

/* report_clearm() - clear map image */
void report_clearm(report *r)
{
    mlist *m, /* current entry */
          *n; /* next entry */

    m = r->rp.m.first;
    while(m != NULL)
    {
        n = m->next; free(m); m = n;
    }
    r->rp.m.first = NULL;
}


/* level 1 routines *******************************************/


/* report_readrec() - generic read record function */
result report_readrec(report *r)
{
    switch(r->type)
    {
        case 'W':
            if(r->rp.w.text != NULL) free(r->rp.w.text);
            break;
        case 'P': report_clearp(r); break;
        case 'F': report_clearf(r); break;
        case 'M': report_clearm(r); break;
    }
    if( fread(&r->type, 1, 1, repfile) == 0 )
        return R_EOF;
    switch(r->type)
    {
        case 'J': report_readj(r);  break;
        case 'j': report_readje(r); break;
        case 'S': report_reads(r);  break;
        case 's': report_readse(r); break;
        case 'P': report_readp(r);  break;
        case 'F': report_readf(r);  break;
        case 'M': report_readm(r);  break;
        case 'm': report_readme(r); break;
        case 'W': report_readw(r);  break;
        case 'w': report_readwe(r); break;
        case 'V': report_readv(r);  break;
        case 'A': report_reada(r);  break;
        case 'I': report_readi(r);  break;
        case 'B': report_readb(r);  break;
        case 'D': report_readd(r);  break;
        default:  return R_HEADER;
    }
    return R_OK;
}

/* report_writerec() - generic write record function */
result report_writerec(report *r)
{
    switch(r->type)
    {
        case 'J': report_writej(r);  break;
        case 'j': report_writeje(r); break;
        case 'S': report_writes(r);  break;
        case 's': report_writese(r); break;
        case 'P': report_writep(r);  break;
        case 'F': report_writef(r);  break;
        case 'M': report_writem(r);  break;
        case 'm': report_writeme(r); break;
        case 'W': report_writew(r);  break;
        case 'w': report_writewe(r); break;
        case 'V': report_writev(r);  break;
        case 'A': report_writea(r);  break;
        case 'I': report_writei(r);  break;
        case 'B': report_writeb(r);  break;
        case 'D': report_writed(r);  break;
        default:  return R_HEADER;
    }
    return R_OK;
}


/* main function implementation *******************************/


/* report_open() - open the report file */
result report_open(char *path)
{
    char listname[128],  /* full filename of data file */
         header[8];      /* header stored for comparison */

    /* open list file */
    sprintf(listname, "%sreport.data", path);
    if(( repfile = fopen(listname, "r+b") ) == NULL)
        return R_FILE;
    fread(header, 8, 1, repfile);
    if( strcmp(header, "AST100R") == 0 )
        return R_OK;
    fclose(repfile);
    return R_HEADER;
}

/* report_create() - create a new report file */
result report_create(char *path)
{
    char dataname[128];  /* full name of data file */

    /* test to see if the DAT file already exists */
    sprintf(dataname, "%sreport.data", path);
    if(( repfile = fopen(dataname, "rb") ) != NULL)
    {
        fclose(repfile);
        return R_EXISTS;
    }

    /* create DAT file if not already present */
    if(( repfile = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    fwrite("AST100R\0", 8, 1, repfile);
    fclose(repfile);
    return R_OK;
}

/* report_first() - read the first report */
result report_first(report *r)
{
    fseek(repfile, 8, SEEK_SET);
    return report_readrec(r);
}

/* report_next() - read the next report */
result report_next(report *r)
{
    return report_readrec(r);
}

/* report_write() - add or update an report */
result report_write(report *r)
{
    long filepos; /* current file position */

    filepos = ftell(repfile);

    /* add new record */
    fseek(repfile, 0, SEEK_END);
    report_writerec(r);
    fseek(repfile, filepos, SEEK_SET);
    return R_OK;
}

/* report_addentry() - add an entry to the linked list */
result report_addentry(report *r)
{
    switch(r->type)
    {
        case 'P':
            if(r->rp.p.first == NULL)
                r->rp.p.first = r->rp.p.current =
                    malloc( sizeof(plist) );
            else
            {
                r->rp.p.current = r->rp.p.first;
                while(r->rp.p.current->next != NULL)
                    r->rp.p.current = r->rp.p.current->next;
                r->rp.p.current = r->rp.p.current->next =
                    malloc( sizeof(plist) );
            }
            if(r->rp.p.current != NULL)
            {
                r->rp.p.current->next = NULL;
                return R_OK;
            }
            else return R_MEMORY;
        case 'F':
            if(r->rp.f.first == NULL)
                r->rp.f.first = r->rp.f.current =
                    malloc( sizeof(flist) );
            else
            {
                r->rp.f.current = r->rp.f.first;
                while(r->rp.f.current->next != NULL)
                    r->rp.f.current = r->rp.f.current->next;
                r->rp.f.current = r->rp.f.current->next =
                    malloc( sizeof(flist) );
            }
            if(r->rp.f.current != NULL)
            {
                r->rp.f.current->next = NULL;
                return R_OK;
            }
            else return R_MEMORY;
        case 'M':
            if(r->rp.m.first == NULL)
                r->rp.m.first = r->rp.m.current =
                    malloc( sizeof(mlist) );
            else
            {
                r->rp.m.current = r->rp.m.first;
                while(r->rp.m.current->next != NULL)
                    r->rp.m.current = r->rp.m.current->next;
                r->rp.m.current = r->rp.m.current->next =
                    malloc( sizeof(mlist) );
            }
            if(r->rp.m.current != NULL)
            {
                r->rp.m.current->next = NULL;
                return R_OK;
            }
            else return R_MEMORY;
        default:
            return R_HEADER;
    }
}

/* report_firstentry() - choose first entry in linked list */
result report_firstentry(report *r)
{
    switch(r->type)
    {
        case 'P':
            if( (r->rp.p.current = r->rp.p.first) == NULL )
                return R_EOF;
            else
                return R_OK;
        case 'F':
            if( (r->rp.f.current = r->rp.f.first) == NULL )
                return R_EOF;
            else
                return R_OK;
        case 'M':
            if( (r->rp.m.current = r->rp.m.first) == NULL )
                return R_EOF;
            else
                return R_OK;
        default:
            return R_HEADER;
    }
}

/* report_nextentry() - choose next entry in linked list */
result report_nextentry(report *r)
{
    switch(r->type)
    {
        case 'P':
            if((r->rp.p.current = r->rp.p.current->next) == NULL)
                return R_EOF;
            else
                return R_OK;
        case 'F':
            if((r->rp.f.current = r->rp.f.current->next) == NULL)
                return R_EOF;
            else
                return R_OK;
        case 'M':
            if((r->rp.m.current = r->rp.m.current->next) == NULL)
                return R_EOF;
            else
                return R_OK;
        default:
            return R_HEADER;
    }
}

/* report_new() - allocate memory for a new report */
report *report_new()
{
    report *r; /* report record */

    /* allocate and initialise report record */
    r = malloc( sizeof(report) );
    if(r) memset( r, 0, sizeof(report) );

    return r;
}

/* report_old() - free memory from an old report */
void report_old(report *r)
{
    switch(r->type)
    {
        case 'W':
            if(r->rp.w.text != NULL) free(r->rp.w.text);
            break;
        case 'P': report_clearp(r); break;
        case 'F': report_clearf(r); break;
        case 'M': report_clearm(r); break;
    }
    free(r);
}

/* report_settype() - set the 'type' character */
void report_settype(report *r, char type)
{
    switch(r->type)
    {
        case 'W':
            if(r->rp.w.text != NULL)
            {
                free(r->rp.w.text);
                r->rp.w.text = NULL;
            }
            break;
        case 'P': report_clearp(r); break;
        case 'F': report_clearf(r); break;
        case 'M': report_clearm(r); break;
    }
    r->type = type;
    switch(r->type)
    {
        case 'P': r->rp.p.first = NULL; break;
        case 'F': r->rp.f.first = NULL; break;
        case 'M': r->rp.m.first = NULL; break;
    }
}

/* report_settrans() - set the transaction number */
void report_settrans(report *r, long number)
{
    if(r->type == 'J')
        r->rp.j.trans = number;
    else if(r->type == 'j')
        r->rp.je.trans = number;
}

/* report_sethomewld() - set the homeworld */
void report_sethomewld(report *r, char *name)
{
    if(r->type == 'J')
        strcpy(r->rp.j.homewld, name);
}

/* report_setempire() - set the empire name */
void report_setempire(report *r, char *name)
{
    switch(r->type)
    {
        case 'S': strcpy(r->rp.s.empire, name);  break;
        case 's': strcpy(r->rp.se.empire, name); break;
        case 'P': strcpy(r->rp.p.empire, name);  break;
        case 'F': strcpy(r->rp.f.empire, name);  break;
        case 'M': strcpy(r->rp.m.empire, name);  break;
        case 'm': strcpy(r->rp.me.empire, name); break;
        case 'W': strcpy(r->rp.w.empire, name);  break;
        case 'w': strcpy(r->rp.we.empire, name); break;
        case 'V': strcpy(r->rp.v.empire, name);  break;
        case 'A': strcpy(r->rp.a.empire, name);  break;
        case 'I': strcpy(r->rp.i.empire, name);  break;
        case 'B': strcpy(r->rp.b.empire, name);  break;
        case 'D': strcpy(r->rp.d.empire, name);  break;
    }
}

/* report_setfleetno() - set the fleet number */
void report_setfleetno(report *r, long number)
{
    switch(r->type)
    {
        case 'S': r->rp.s.fleetno         = number; break;
        case 'F': r->rp.f.current->number = number; break;
        case 'A': r->rp.a.fleetno         = number; break;
        case 'I': r->rp.i.fleetno         = number; break;
        case 'B': r->rp.b.fleetno         = number; break;
    }
}

/* report_setorigin() - set the origin for a send report */
void report_setorigin(report *r, char *name)
{
    switch(r->type)
    {
        case 'S': strcpy(r->rp.s.origin, name);          break;
        case 's': strcpy(r->rp.se.origin, name);         break;
        case 'F': strcpy(r->rp.f.current->origin, name); break;
    }
}

/* report_setdest() - set the destination for a send report */
void report_setdest(report *r, char *name)
{
    switch(r->type)
    {
        case 'S': strcpy(r->rp.s.dest, name);          break;
        case 's': strcpy(r->rp.se.dest, name);         break;
        case 'F': strcpy(r->rp.f.current->dest, name); break;
    }
}

/* report_setships() - set the ships in a send report */
void report_setships(report *r, long ships)
{
    switch(r->type)
    {
        case 'S': r->rp.s.ships          = ships; break;
        case 's': r->rp.se.ships         = ships; break;
        case 'P': r->rp.p.current->ships = ships; break;
        case 'F': r->rp.f.current->ships = ships; break;
        case 'I': r->rp.i.ships          = ships; break;
        case 'B': r->rp.b.ships          = ships; break;
        case 'D': r->rp.d.ships          = ships; break;
    }
}

/* report_setcentre() - set the centre in a map report */
void report_setcentre(report *r, char *name)
{
    if(r->type == 'M')
        strcpy(r->rp.m.centre, name);
    else if(r->type == 'm')
        strcpy(r->rp.me.centre, name);
}

/* report_setsender() - set the sender in a write report */
void report_setsender(report *r, char *name)
{
    if(r->type == 'W')
        strcpy(r->rp.w.sender, name);
}

/* report_setrecip() - set the recipient in a write report */
void report_setrecip(report *r, char *name)
{
    if(r->type == 'w')
        strcpy(r->rp.we.recip, name);
}

/* report_setname() - set the planet name */
void report_setname(report *r, char *name)
{
    switch(r->type)
    {
        case 'P': strcpy(r->rp.p.current->name, name); break;
        case 'M': strcpy(r->rp.m.current->name, name); break;
    }
}

/* report_settext() - set the message text in a write report */
void report_settext(report *r, char *text)
{
    if(r->type != 'W')
        return;
    if(r->rp.w.text != NULL)
        free(r->rp.w.text);
    if( (r->rp.w.text = malloc( strlen(text) + 1 )) != NULL )
        strcpy(r->rp.w.text, text);
}

/* report_setvictor() - set the victor in a victory message */
void report_setvictor(report *r, char *name)
{
    if(r->type == 'V')
        strcpy(r->rp.v.victor, name);
}

/* report_setowner() - set the owner of a fleet/planet */
void report_setowner(report *r, char *name)
{
    switch(r->type)
    {
        case 'I': strcpy(r->rp.i.owner, name); break;
        case 'B': strcpy(r->rp.b.owner, name); break;
        case 'D': strcpy(r->rp.d.owner, name); break;
    }
}

/* report_setenemy() - set the number of enemy ships */
void report_setenemy(report *r, long number)
{
    switch(r->type)
    {
        case 'B': r->rp.b.enemy = number; break;
        case 'D': r->rp.d.enemy = number; break;
    }
}

/* report_setpos() - set planet position */
void report_setpos(report *r, pos pp)
{
    switch(r->type)
    {
        case 'P': r->rp.p.current->pp = pp; break;
        case 'M': r->rp.m.current->pp = pp; break;
    }
}

/* report_setprod() - set production figures */
void report_setprod(report *r, int prod)
{
    switch(r->type)
    {
        case 'I': r->rp.i.prod = prod;          break;
        case 'P': r->rp.p.current->prod = prod; break;
    }
}

/* report_setdist() - set the fleet distance */
void report_setdist(report *r, int dist)
{
    if(r->type == 'F')
        r->rp.f.current->dist = dist;
}

/* report_setplanet() - set the planet defended */
void report_setplanet(report *r, char *planet)
{
    if(r->type == 'D')
        strcpy(r->rp.d.planet, planet);
}

/* report_seterror() - set the error code */
void report_seterror(report *r, char error)
{
    switch(r->type)
    {
        case 'j': r->rp.je.error = error; break;
        case 's': r->rp.se.error = error; break;
        case 'm': r->rp.me.error = error; break;
        case 'w': r->rp.we.error = error; break;
    }
}

/* report_gettype() - return the report type */
char report_gettype(report *r)
{
    return r->type;
}

/* report_gettrans() - return the transaction number */
long report_gettrans(report *r)
{
    if(r->type == 'J')
        return r->rp.j.trans;
    else if(r->type == 'j')
        return r->rp.je.trans;
    else
        return -1;
}

/* report_gethomewld() - return the homeworld */
char *report_gethomewld(char *homewld, report *r)
{
    if(r->type == 'J')
        return strcpy(homewld, r->rp.j.homewld);
    else
        return NULL;
}

/* report_getempire() - return the empire name */
char *report_getempire(char *name, report *r)
{
    switch(r->type)
    {
        case 'S': return strcpy(name, r->rp.s.empire);  break;
        case 's': return strcpy(name, r->rp.se.empire); break;
        case 'P': return strcpy(name, r->rp.p.empire);  break;
        case 'F': return strcpy(name, r->rp.f.empire);  break;
        case 'M': return strcpy(name, r->rp.m.empire);  break;
        case 'm': return strcpy(name, r->rp.me.empire); break;
        case 'W': return strcpy(name, r->rp.w.empire);  break;
        case 'w': return strcpy(name, r->rp.we.empire); break;
        case 'V': return strcpy(name, r->rp.v.empire);  break;
        case 'A': return strcpy(name, r->rp.a.empire);  break;
        case 'I': return strcpy(name, r->rp.i.empire);  break;
        case 'B': return strcpy(name, r->rp.b.empire);  break;
        case 'D': return strcpy(name, r->rp.d.empire);  break;
        default:  return NULL;
    }
}

/* report_getfleetno() - return the fleet number */
long report_getfleetno(report *r)
{
    switch(r->type)
    {
        case 'S': return r->rp.s.fleetno;         break;
        case 'F': return r->rp.f.current->number; break;
        case 'A': return r->rp.a.fleetno;         break;
        case 'I': return r->rp.i.fleetno;         break;
        case 'B': return r->rp.b.fleetno;         break;
        default:  return -1;
    }
}

/* report_getorigin() - return the origin planet in a send */
char *report_getorigin(char *name, report *r)
{
    switch(r->type)
    {
        case 'S': return strcpy(name, r->rp.s.origin);
        case 's': return strcpy(name, r->rp.se.origin);
        case 'F': return strcpy(name, r->rp.f.current->origin);
        default:  return NULL;
    }
}

/* report_getdest() - return the destination planet in a send */
char *report_getdest(char *name, report *r)
{
    switch(r->type)
    {
        case 'S': return strcpy(name, r->rp.s.dest);
        case 's': return strcpy(name, r->rp.se.dest);
        case 'F': return strcpy(name, r->rp.f.current->dest);
        default:  return NULL;
    }
}

/* report_getships() - return the number of ships in a send */
long report_getships(report *r)
{
    switch(r->type)
    {
        case 'S': return r->rp.s.ships;          break;
        case 's': return r->rp.se.ships;         break;
        case 'P': return r->rp.p.current->ships; break;
        case 'F': return r->rp.f.current->ships; break;
        case 'I': return r->rp.i.ships;          break;
        case 'B': return r->rp.b.ships;          break;
        case 'D': return r->rp.d.ships;          break;
        default:  return -1;
    }
}

/* report_getcentre() - return the centre planet of a map */
char *report_getcentre(char *name, report *r)
{
    if(r->type == 'M')
        return strcpy(name, r->rp.m.centre);
    else if(r->type == 'm')
        return strcpy(name, r->rp.me.centre);
    else
        return NULL;
}

/* report_getsender() - return the sender of a written msg */
char *report_getsender(char *name, report *r)
{
    if(r->type == 'W')
        return strcpy(name, r->rp.w.sender);
    else
        return NULL;
}

/* report_getrecip() - return the recipient of a written msg */
char *report_getrecip(char *name, report *r)
{
    if(r->type == 'w')
        return strcpy(name, r->rp.we.recip);
    else
        return NULL;
}

/* report_getname() - return the planet name */
char *report_getname(char *name, report *r)
{
    switch(r->type)
    {
        case 'P': return strcpy(name, r->rp.p.current->name);
        case 'M': return strcpy(name, r->rp.m.current->name);
        default:  return NULL;
    }
}

/* report_gettext() - return the text of a written msg */
char *report_gettext(report *r)
{
    char *text; /* newly allocated var for msg text */

    if(r->type != 'W')
        return NULL;
    text = malloc( strlen(r->rp.w.text) + 1 );
    if(text != NULL) strcpy(text, r->rp.w.text);

    return text;
}

/* report_getvictor() - return the victor in a victory rpt */
char *report_getvictor(char *name, report *r)
{
    if(r->type == 'V')
        return strcpy(name, r->rp.v.victor);
    else
        return NULL;
}

/* report_getowner() - return the owner of a planet/fleet */
char *report_getowner(char *name, report *r)
{
    switch(r->type)
    {
        case 'I': return strcpy(name, r->rp.i.owner);
        case 'B': return strcpy(name, r->rp.b.owner);
        case 'D': return strcpy(name, r->rp.d.owner);
        default:  return NULL;
    }
}

/* report_getenemy() - return the number of enemy ships */
long report_getenemy(report *r)
{
    switch(r->type)
    {
        case 'B': return r->rp.b.enemy;
        case 'D': return r->rp.d.enemy;
        default:  return -1;
    }
}

/* report_getpos() - return a planet's position */
pos report_getpos(report *r)
{
    pos pp = {0, 0}; /* default in case of error */

    switch(r->type)
    {
        case 'P': return r->rp.p.current->pp;
        case 'M': return r->rp.m.current->pp;
        default:  return pp;
    }
}

/* report_getprod() - return a planet's production */
int report_getprod(report *r)
{
    switch(r->type)
    {
        case 'I': return r->rp.i.prod;
        case 'P': return r->rp.p.current->prod;
        default:  return -1;
    }
}

/* report_getdist() - return a planet's distuction */
int report_getdist(report *r)
{
    if(r->type == 'F')
        return r->rp.f.current->dist;
    else
        return -1;
}

/* report_getplanet() - return the planet defended */
char *report_getplanet(char *planet, report *r)
{
    if(r->type == 'D')
        return strcpy(planet, r->rp.d.planet);
    else
        return NULL;
}

/* report_geterror() - get the error code */
char report_geterror(report *r)
{
    switch(r->type)
    {
        case 'j': return r->rp.je.error; break;
        case 's': return r->rp.se.error; break;
        case 'm': return r->rp.me.error; break;
        case 'w': return r->rp.we.error; break;
        default:  return '\0';
    }
}

/* report_close() - close report file/library */
void report_close(void)
{
    fclose(repfile);
}
