/**************************************************************
 * MSGLIB.C                  Copyright (C) Damian Walker 1996 *
 *------------------------------------------------------------*
 * MsgLib 1.01.  A generic message library shell, expandable  *
 * by adding modules to handle specific message bases.  ANSI  *
 * C version.                                                 *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     31-Dec-96 (msglib 1.00)                           *
 *          09-Apr-97 (msglib 1.01)                           *
 *          13-May-97 (msglib/ANSI 1.01)                      *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include "dirent.h"
#include "platform.h"


/* some macros ************************************************/


/* read binary number fields */
#define FREADSHORT(s,d) {                          \
                        s = fgetc(d);              \
                        s += 0x100 * fgetc(d);     \
                        }

/* write binary number fields */
#define FWRITESHORT(s,d) {                                           \
                         fputc((unsigned short)s & 0xff,         d); \
                         fputc((unsigned short)s / 0x100 & 0xff, d); \
                         }


/* defined constants ******************************************/


/* message flags */
#define MSGPVT      0x0001    /* Private */
#define MSGCRASH    0x0002    /* Crash message */
#define MSGRECD     0x0004    /* Message received */
#define MSGSENT     0x0008    /* Message sent */
#define MSGFILE     0x0010    /* File attached */
#define MSGTRANSIT  0x0020    /* In transit */
#define MSGORPHAN   0x0040    /* Orphan */
#define MSGKILL     0x0080    /* Kill/sent */
#define MSGLOCAL    0x0100    /* Local */
#define MSGHOLD     0x0200    /* Hold for pickup */
#define MSGFREQ     0x0800    /* File request */
#define MSGRRR      0x1000    /* Return receipt request */
#define MSGIRR      0x2000    /* Is return receipt */
#define MSGAUDIT    0x4000    /* Audit request */
#define MSGUPDATE   0x8000    /* File update request */

#define MSGMAX      1000      /* maximum message number */


/* enumerated types *******************************************/


/* result - return codes for various actions */
enum result {
    M_OK,   /* everything went fine */
    M_EOF,  /* no (more) messages */
    M_FILE, /* file I/O error */
    M_NEW,  /* cannot rewrite new message */
    M_LAST  /* place marker */
};


/* structure definitions **************************************/


/* fido - fidonet address */
typedef struct {
    int zone,  /* zone number */
        net,   /* net number */
        node,  /* node number */
        point; /* point number */
} fido;

/* attr - attribute 'word' */
typedef struct {
    unsigned int crash: 1,  /* crash message */
                 attach: 1, /* file attach */
                 hold: 1,   /* held message */
                 freq: 1,   /* file request */
                 update: 1, /* update request */
                 kill: 1,   /* kill when sent */
                 pvt: 1,    /* private */
                 local: 1,  /* local */
                 direct: 1, /* direct */
                 erase: 1,  /* erase file when sent */
                 rcvd: 1,   /* message received */
                 sent: 1;   /* message sent */
} attr;

/* msg - message record */
typedef struct {
    int    number;       /* message number */
    char   fromname[36]; /* sender's name */
    fido   fromaddr;     /* sender's address */
    char   toname[36];   /* recipient's name */
    fido   toaddr;       /* recipient's address */
    time_t datetime;     /* date/time posted */
    attr   flags;        /* flags */
    char   subject[72],  /* subject line */
          *text;         /* pointer to message text */
} msg;

/* msgnode - node line message list */
typedef struct {
    int   number; /* message number */
    void *n,      /* next message node */
         *p;      /* previous message node */
} msgnode;

/* area - *.MSG-type message area */
typedef struct {
    char     dir[128]; /* directory of *.msg files */
    msgnode *first,    /* pointer to first message */
            *curr;     /* pointer to current message */
} area;

/* mbase - message base type */
typedef struct {
    area *(*mopen)(char *);
    void  (*mclose)(area *);
    int   (*mfirst)(msg *, area *),
          (*mlast)(msg *, area *),
          (*mnext)(msg *, area *),
          (*mprev)(msg *, area *),
          (*mgoto)(msg *, area *, int),
          (*mpost)(area *, msg *),
          (*mkill)(area *, msg *),
          (*mrewrite)(area *, msg *);
} mbase;


/* global variables *******************************************/


int  myzone;  /* zone for *.MSG-based messages */


/* individual message related routines ************************/


/* msg_new() - allocate memory for new message */
msg *msg_new(void)
{
    msg *m; /* pointer to new message */

    m = calloc( 1, sizeof(msg) );
    if(m != NULL) m->text = NULL;

    return m;
}

/* msg_old() - free memory used by old message */
void msg_old(msg *m)
{
    if(m->text != NULL) free(m->text);
    free(m);
}

/* msg_setnumber() - set the message number */
void msg_setnumber(msg *m, int number)
{
    m->number = number;
}

/* msg_setfromname() - set the sender's name */
void msg_setfromname(msg *m, char *fromname)
{
    strcpy(m->fromname, fromname);
}

/* msg_setfromaddr() - set the sender's address */
void msg_setfromaddr(msg *m, fido fromaddr)
{
    m->fromaddr = fromaddr;
}

/* msg_settoname() - set the sender's name */
void msg_settoname(msg *m, char *toname)
{
    strcpy(m->toname, toname);
}

/* msg_settoaddr() - set the sender's address */
void msg_settoaddr(msg *m, fido toaddr)
{
    m->toaddr = toaddr;
}

/* msg_setdatetime() - set the date and time */
void msg_setdatetime(msg *m, time_t datetime)
{
    m->datetime = datetime;
}

/* msg_setcrash() - set crash flag */
void msg_setcrash(msg *m, int crash)
{
    m->flags.crash = crash;
}

/* msg_setattach() - set file attach flag */
void msg_setattach(msg *m, int attach)
{
    m->flags.attach = attach;
}

/* msg_sethold() - set hold flag */
void msg_sethold(msg *m, int hold)
{
    m->flags.hold = hold;
}

/* msg_setfreq() - set freq flag */
void msg_setfreq(msg *m, int freq)
{
    m->flags.freq = freq;
}

/* msg_setupdate() - set update flag */
void msg_setupdate(msg *m, int update)
{
    m->flags.update = update;
}

/* msg_setkill() - set kill flag */
void msg_setkill(msg *m, int kill)
{
    m->flags.kill = kill;
}

/* msg_setpvt() - set pvt flag */
void msg_setpvt(msg *m, int pvt)
{
    m->flags.pvt = pvt;
}

/* msg_setlocal() - set local flag */
void msg_setlocal(msg *m, int local)
{
    m->flags.local = local;
}

/* msg_setdirect() - set direct flag */
void msg_setdirect(msg *m, int direct)
{
    m->flags.direct = direct;
}

/* msg_seterase() - set erase flag */
void msg_seterase(msg *m, int erase)
{
    m->flags.erase = erase;
}

/* msg_setrcvd() - set rcvd flag */
void msg_setrcvd(msg *m, int rcvd)
{
    m->flags.rcvd = rcvd;
}

/* msg_setsent() - set sent flag */
void msg_setsent(msg *m, int sent)
{
    m->flags.sent = sent;
}

/* msg_setsubject() - set subject line */
void msg_setsubject(msg *m, char *subject)
{
    strcpy(m->subject, subject);
}

/* msg_settext() - set message text */
void msg_settext(msg *m, char *text)
{
    if(m->text != NULL)
        free(m->text);
    m->text = malloc( strlen(text) + 1 );
    strcpy(m->text, text);
}

/* msg_getnumber() - return message number */
int msg_getnumber(msg *m)
{
    return m->number;
}

/* msg_getfromname() - return the from name */
char *msg_getfromname(char *fromname, msg *m)
{
    return strcpy(fromname, m->fromname);
}

/* msg_getfromaddr() - return the from address */
fido msg_getfromaddr(msg *m)
{
    return m->fromaddr;
}

/* msg_gettoname() - return the to name */
char *msg_gettoname(char *toname, msg *m)
{
    return strcpy(toname, m->toname);
}

/* msg_gettoaddr() - return the to address */
fido msg_gettoaddr(msg *m)
{
    return m->toaddr;
}

/* msg_getdatetime() - return the date and time sent */
time_t msg_getdatetime(msg *m)
{
    return m->datetime;
}

/* msg_getcrash() - return the crash bit */
int msg_getcrash(msg *m)
{
    return m->flags.crash;
}

/* msg_getattach() - return the attach bit */
int msg_getattach(msg *m)
{
    return m->flags.attach;
}

/* msg_gethold() - return the hold bit */
int msg_gethold(msg *m)
{
    return m->flags.hold;
}

/* msg_getfreq() - return the freq bit */
int msg_getfreq(msg *m)
{
    return m->flags.freq;
}

/* msg_getupdate() - return the update bit */
int msg_getupdate(msg *m)
{
    return m->flags.update;
}

/* msg_getkill() - return the kill bit */
int msg_getkill(msg *m)
{
    return m->flags.kill;
}

/* msg_getpvt() - return the pvt bit */
int msg_getpvt(msg *m)
{
    return m->flags.pvt;
}

/* msg_getlocal() - return the local bit */
int msg_getlocal(msg *m)
{
    return m->flags.local;
}

/* msg_getdirect() - return the direct bit */
int msg_getdirect(msg *m)
{
    return m->flags.direct;
}

/* msg_geterase() - return the erase bit */
int msg_geterase(msg *m)
{
    return m->flags.erase;
}

/* msg_getrcvd() - return the rcvd bit */
int msg_getrcvd(msg *m)
{
    return m->flags.rcvd;
}

/* msg_getsent() - return the sent bit */
int msg_getsent(msg *m)
{
    return m->flags.sent;
}

/* msg_getsubject() - return the subject */
char *msg_getsubject(char *subject, msg *m)
{
    return strcpy(subject, m->subject);
}

/* msg_gettext() - return the message text */
char *msg_gettext(msg *m)
{
    return strcpy( malloc(strlen( m->text ) + 1), m->text );
}


/* area-related routines for *.MSG level 2 ********************/


/* msg_attrin() - sort out message attributes */
attr msg_attrin(int flags, char *text)
{
    attr  rtn;     /* return value */
    char *flagpos, /* position of ^AFLAGS kludge */
         *retpos,  /* position of \r after ^AFLAGS */
         *pos;     /* position of specific flag kludge */

    rtn.crash  = (flags & MSGCRASH)  / MSGCRASH;
    rtn.attach = (flags & MSGFILE)   / MSGFILE;
    rtn.hold   = (flags & MSGHOLD)   / MSGHOLD;
    rtn.freq   = (flags & MSGFREQ)   / MSGFREQ;
    rtn.update = (flags & MSGUPDATE) / MSGUPDATE;
    rtn.kill   = (flags & MSGKILL)   / MSGKILL;
    rtn.pvt    = (flags & MSGPVT)    / MSGPVT;
    rtn.local  = (flags & MSGLOCAL)  / MSGLOCAL;
    rtn.rcvd   = (flags & MSGRECD)   / MSGRECD;
    rtn.sent   = (flags & MSGSENT)   / MSGSENT;
    flagpos = strstr(text, "\1FLAGS");
    if(flagpos != NULL)
    {
        retpos = strchr(flagpos, '\r');
        pos = strstr(flagpos, " KFS");
        rtn.erase = ( (pos != NULL) && (pos < retpos) );
        pos = strstr(flagpos, " DIR");
        rtn.direct = ((pos != NULL) && (pos < retpos));
    }
    else rtn.erase = rtn.direct = 0;

    return rtn;
}

/* checkintl() - check a ^AINTL kludge for addressing info */
void msg_checkintl(int *fmzone, int *tozone, char *text)
{
    char *kludgepos, *desttext, *origtext;

    kludgepos = strstr(text, "\1INTL");
    if(kludgepos != NULL)
    {
        desttext = &kludgepos[6];
        origtext = strstr(desttext, " ") + 1;
        *tozone = atoi(desttext);
        *fmzone = atoi(origtext);
    }
}

/* checkfmpt() - check ^AFMPT kludge for addressing info */
int msg_checkfmpt(char *text)
{
    char *kludgepos;

    kludgepos = strstr(text, "\1FMPT");
    if(kludgepos != NULL)
        return atoi( &kludgepos[6] );
    else
        return 0;
}

/* checktopt() - check ^ATOPT kludge for addressing info */
int msg_checktopt(char *text)
{
    char *kludgepos;

    kludgepos = strstr(text, "\1TOPT");
    if(kludgepos != NULL)
        return atoi( &kludgepos[6] );
    else
        return 0;
}

/* msg_parsedate() - parse brain-dead *.MSG date format */
time_t msg_parsedate(char *datetime)
{
    struct tm df;      /* individual date fields */
    char      mon[20]; /* month name */
    int       items;   /* items 'read' */

    items = sscanf(datetime, "%d%s%d%d:%d:%d", &df.tm_mday, mon,
        &df.tm_year, &df.tm_hour, &df.tm_min, &df.tm_sec);
    if(items == 6)
    {
        df.tm_isdst = 0;
             if( strcmp(mon, "Jan") == 0 ) df.tm_mon = 0;
        else if( strcmp(mon, "Feb") == 0 ) df.tm_mon = 1;
        else if( strcmp(mon, "Mar") == 0 ) df.tm_mon = 2;
        else if( strcmp(mon, "Apr") == 0 ) df.tm_mon = 3;
        else if( strcmp(mon, "May") == 0 ) df.tm_mon = 4;
        else if( strcmp(mon, "Jun") == 0 ) df.tm_mon = 5;
        else if( strcmp(mon, "Jul") == 0 ) df.tm_mon = 6;
        else if( strcmp(mon, "Aug") == 0 ) df.tm_mon = 7;
        else if( strcmp(mon, "Sep") == 0 ) df.tm_mon = 8;
        else if( strcmp(mon, "Oct") == 0 ) df.tm_mon = 9;
        else if( strcmp(mon, "Nov") == 0 ) df.tm_mon = 10;
        else if( strcmp(mon, "Dec") == 0 ) df.tm_mon = 11;
        return mktime(&df);
    }
    else return time(NULL);
}

/* msg_attrout() - convert to fido attributes */
int msg_attrout(attr flags)
{
    int fidoflags; /* fidonet flags word */

    fidoflags = 0;
    if(flags.crash)  fidoflags |= MSGCRASH;
    if(flags.attach) fidoflags |= MSGFILE;
    if(flags.hold)   fidoflags |= MSGHOLD;
    if(flags.freq)   fidoflags |= MSGFREQ;
    if(flags.update) fidoflags |= MSGUPDATE;
    if(flags.kill)   fidoflags |= MSGKILL;
    if(flags.pvt)    fidoflags |= MSGPVT;
    if(flags.local)  fidoflags |= MSGLOCAL;
    if(flags.rcvd)   fidoflags |= MSGRECD;
    if(flags.sent)   fidoflags |= MSGSENT;

    return fidoflags;
}

/* makedate() - create a FTS-0001 compatible date */
void msg_makedate(char *msgdate)
{
    time_t timer;

    time(&timer);
    strftime(msgdate, 20, "%d %b %y  %H:%M:%S",
        localtime(&timer));
}

/* msg_stripline() - strip conflicting control line */
int msg_stripline(char *text, char *word)
{
    char *textpos, *cr;

    textpos = strstr(text, word);
    if(textpos != NULL)
    {
        cr = strchr(textpos, '\r');
        if(cr != NULL)
            strcpy( textpos, &cr[1] );
        else
            *textpos = '\0';
    }

    return(textpos != NULL);
}

/* addkludges() - add kludges for extra addressing info */
void msg_addkludges(FILE *msgfile, msg *m)
{
    fprintf(msgfile, "\1MSGID: %d:%d/%d.%d %08lx\r",
        m->fromaddr.zone, m->fromaddr.net, m->fromaddr.node,
        m->fromaddr.point, (long)random() );
    if(m->fromaddr.zone != m->toaddr.zone)
        fprintf(msgfile, "\1INTL %d:%d/%d %d:%d/%d\r",
            m->toaddr.zone, m->toaddr.net, m->toaddr.node,
            m->fromaddr.zone, m->fromaddr.net,
            m->fromaddr.node);
    if(m->fromaddr.point != 0)
        fprintf(msgfile, "\1FMPT %d\r", m->fromaddr.point);
    if(m->toaddr.point != 0)
        fprintf(msgfile, "\1TOPT %d\r", m->toaddr.point);
    if(m->flags.direct || m->flags.erase)
    {
        fprintf(msgfile, "\1FLAGS");
        if(m->flags.direct) fprintf(msgfile, " DIR");
        if(m->flags.erase)  fprintf(msgfile, " KFS");
        fprintf(msgfile, "\r");
    }
}


/* area-related routines for *.MSG level 1 ********************/


/* msg_insert() - insert message into linked list */
msgnode *msg_insert(msgnode *first, int number)
{
    msgnode *prev,    /* pointer to previous node */
            *newnode, /* pointer to inserted node */
            *node;    /* pointer to next node */

    /* add first message to list */
    if(first == NULL)
    {
        first = malloc( sizeof(msgnode) );
        if(first != NULL)
        {
            first->number = number;
            first->n = first->p = NULL;
        }
    }

    /* add subsequent messages to list */
    else
    {
        /* scan for correct position */
        prev = NULL;
        node = first;
        while( (node->number < number) &&
               (node->n != NULL) )
        {
            prev = node;
            node = node->n;
        }

        /* insert message ... */
        if(node->number > number)
        {

            /* ... at start */
            if(prev == NULL)
            {
                first = malloc( sizeof(msgnode) );
                if(first != NULL)
                {
                    first->number = number;
                    first->n = node; node->p = first;
                    first->p = NULL;
                }
            }

            /* ... in middle */
            else
            {
                newnode = malloc( sizeof(msgnode) );
                if(newnode != NULL)
                {
                    newnode->number = number;
                    newnode->n = node; node->p = newnode;
                    newnode->p = prev; prev->n = newnode;
                }
            }
        }

        /* ... at end */
        else
        {
            newnode = malloc( sizeof(msgnode) );
            if(newnode != NULL)
            {
                newnode->number = number;
                newnode->n = NULL;
                newnode->p = node; node->n = newnode;
            }
        }
    }

    return first;
}


/* msg_read() - read a message */
int msg_read(msg *m, area *a)
{
    char  msgname[128], /* filename of message */
          datetime[20]; /* date and time in Fido format */
    FILE *msgfile;      /* message file handle */
    int   resultcode,   /* returned to calling process */
          skip,         /* skip byte counter */
          flags,        /* attribute word as read */
          hdrsize,      /* size of message header */
          txtsize;      /* size of message text */

    sprintf(msgname, "%s%s%d.msg", a->dir, DIRSEP,
        a->curr->number);
    msgfile = fopen(msgname, "rb");
    if(msgfile != NULL)
    {
        /* read in message header */
        m->number = a->curr->number;
        fread(m->fromname, 36, 1, msgfile);
        fread(m->toname, 36, 1, msgfile);
        fread(m->subject, 72, 1, msgfile);
        fread(datetime, 20, 1, msgfile);
        fgetc(msgfile); fgetc(msgfile); /* skip times read */
        FREADSHORT(m->toaddr.node, msgfile);
        FREADSHORT(m->fromaddr.node, msgfile);
        fgetc(msgfile); fgetc(msgfile); /* skip cost */
        FREADSHORT(m->fromaddr.net, msgfile);
        FREADSHORT(m->toaddr.net, msgfile);
        skip = 10; while(skip--) fgetc(msgfile); /* skip */
        FREADSHORT(flags, msgfile);
        fgetc(msgfile); fgetc(msgfile); /* skip nextreply */

        /* read in message text */
        hdrsize = ftell(msgfile);
        fseek(msgfile, 0, SEEK_END);
        txtsize = ftell(msgfile) - hdrsize;
        fseek(msgfile, hdrsize, SEEK_SET);
        if(m->text != NULL) free(m->text);
        m->text = malloc(txtsize + 1);
        fread(m->text, txtsize, 1, msgfile);
        m->text[txtsize] = '\0';

        /* establish more accurate msg info */
        m->fromaddr.zone = m->toaddr.zone = myzone;
        m->flags = msg_attrin(flags, m->text);
        msg_checkintl(&m->fromaddr.zone, &m->toaddr.zone,
            m->text);
        m->fromaddr.point = msg_checkfmpt(m->text);
        m->toaddr.point = msg_checktopt(m->text);
        m->datetime = msg_parsedate(datetime);

        fclose(msgfile);
        resultcode = M_OK;
    }
    else resultcode = M_FILE;

    return resultcode;
}

/* msg_write() - write a message */
int msg_write(area *a, msg *m)
{
    char  msgname[128], /* name of message file */
          datetime[20]; /* date in fido format */
    FILE *msgfile;      /* message file handle */
    int   flags,        /* attribute to write */
          skip,         /* used to skip bytes */
          resultcode;   /* returned to calling process */

    sprintf(msgname, "%s%s%d.msg", a->dir, DIRSEP, m->number);
    msgfile = fopen(msgname, "wb");
    if(msgfile != NULL)
    {
        /* establish some header fields */
        flags = msg_attrout(m->flags);
        msg_makedate(datetime);

        /* write message header */
        fwrite(m->fromname, 36, 1, msgfile);
        fwrite(m->toname, 36, 1, msgfile);
        fwrite(m->subject, 72, 1, msgfile);
        fwrite(datetime, 20, 1, msgfile);
        fputc('\0', msgfile);                   /* times read */
        fputc('\0', msgfile);
        fputc(m->toaddr.node & 0xff, msgfile);    /* destnode */
        fputc(m->toaddr.node / 0x100, msgfile);
        fputc(m->fromaddr.node & 0xff, msgfile);  /* orignode */
        fputc(m->fromaddr.node / 0x100, msgfile);
        fputc('\0', msgfile);                         /* cost */
        fputc('\0', msgfile);
        fputc(m->fromaddr.net & 0xff, msgfile);    /* orignet */
        fputc(m->fromaddr.net / 0x100, msgfile);
        fputc(m->toaddr.net & 0xff, msgfile);      /* destnet */
        fputc(m->toaddr.net / 0x100, msgfile);
        skip = 10;                    /* skip numerous fields */
        while(skip--) fputc('\0', msgfile);
        fputc(flags & 0xff, msgfile);                 /* attr */
        fputc(flags / 0x100, msgfile);
        fputc('\0', msgfile);                   /* next reply */
        fputc('\0', msgfile);

        /* write message text */
        msg_stripline(m->text, "\1MSGID");
        msg_stripline(m->text, "\1INTL");
        msg_stripline(m->text, "\1FMPT");
        msg_stripline(m->text, "\1TOPT");
        msg_stripline(m->text, "\1PID");
        while( msg_stripline(m->text, "\1Via") );
        msg_addkludges(msgfile, m);
        fwrite(m->text, strlen(m->text), 1, msgfile);
        fputc('\0', msgfile);
        fclose(msgfile);

        resultcode = M_OK;
    }
    else resultcode = M_FILE;

    return resultcode;
}


/* area-related routines for *.MSG level 0 ********************/


/* msg_open() - open a *.MSG area */
area *msg_open(char *dir)
{
    area    *a;             /* new area variable */
    FILE    *testfile;      /* test pointer for message file */
    char     filename[128]; /* filename of message */
    int      t;             /* test message number for building list */
    msgnode *node;          /* message node for searching */

    /* allocate memory for area */
    a = malloc( sizeof(area) );
    if(a != NULL)
    {
        /* create message node list */
        a->first = NULL;
        strcpy(a->dir, dir);
        for(t = 1; t < MSGMAX; ++t)
        {
            sprintf(filename, "%s%s%d.msg", dir, DIRSEP, t);
            if(( testfile = fopen(filename, "rb") ) != NULL)
            {
                fclose(testfile);
                a->first = msg_insert(a->first, t);
                dp = readdir(dfd);
            }
        }
        a->curr = a->first;
    }
    return a;
}

/* msg_close() - close a *.MSG area */
void msg_close(area *a)
{
    msgnode *node;          /* used for freeing msgnode memory */

    /* clear out message list */
    if(a->first != NULL)
    {
        node = a->first;
        while(node->n != NULL)
        {
            node = node->n;
            free(node->p);
        }
        free(node);
    }

    /* clear out area variable */
    free(a);
}

/* msg_first() - return the first message */
int msg_first(msg *m, area *a)
{
    int resultcode;

    if(a->first != NULL)
    {
        a->curr = a->first;
        resultcode = msg_read(m, a);
    }
    else resultcode = M_EOF;

    return resultcode;
}

/* msg_last() - return the last message */
int msg_last(msg *m, area *a)
{
    int resultcode;

    /* quit if no messages to search */
    if(a->curr == NULL) a->curr = a->first;
    if(a->first == NULL)
        return M_EOF;

    /* find and read last message */
    a->curr = a->first;
    while(a->curr->n != NULL) a->curr = a->curr->n;
    resultcode = msg_read(m, a);

    return resultcode;
}

/* msg_next() - return the next message */
int msg_next(msg *m, area *a)
{
    int resultcode;

    resultcode = M_EOF;
    if(a->curr != NULL)
        if(a->curr->n != NULL)
        {
            a->curr = a->curr->n;
            resultcode = msg_read(m, a);
        }

    return resultcode;
}

/* msg_prev() - return the previous message */
int msg_prev(msg *m, area *a)
{
    int resultcode;

    resultcode = M_EOF;
    if(a->curr != NULL)
        if(a->curr->p != NULL)
        {
            a->curr = a->curr->p;
            resultcode = msg_read(m, a);
        }

    return resultcode;
}

/* msg_goto() - goto a particular message */
int msg_goto(msg *m, area *a, int number)
{
    int resultcode;

    if(a->curr != NULL)
    {
        while( (a->curr->number > number) &&
               (a->curr->p != NULL) )
            a->curr = a->curr->p;
        while( (a->curr->number < number) &&
               (a->curr->n != NULL) )
            a->curr = a->curr->n;
        if(a->curr->number == number)
            resultcode = msg_read(m, a);
        else
            resultcode = M_EOF;
    }
    else resultcode = M_EOF;

    return resultcode;
}

/* msg_post() - post a message */
int msg_post(area *a, msg *m)
{
    int      resultcode; /* returned to calling process */
    msgnode *search;     /* message node for scanning */

    /* check to see if this is first message */
    if(a->first == NULL)
        m->number = 1;
    else
    {
        search = a->first;
        while(search->n != NULL) search = search->n;
        m->number = search->number + 1;
    }

    /* write last message */
    resultcode = msg_write(a, m);
    if(resultcode == M_OK) msg_insert(a->first, m->number);

    return resultcode;
}

/* msg_kill() - kill a message */
int msg_kill(area *a, msg *m)
{
    char     msgname[128]; /* filename of message */
    int      resultcode;   /* returned to calling process */
    msgnode *prev,         /* previous node */
            *this,         /* this node to delete */
            *next;         /* next node */

    /* can only delete if there are some messages */
    if( a->first != NULL )
    {
        /* find message in list */
        this = a->first;
        while( (this->number > m->number) &&
               (this->p != NULL) )
            this = this->p;
        while( (this->number < m->number) &&
               (this->n != NULL) )
            this = this->n;

        /* found - attempt to delete */
        if(this->number == m->number)
        {
            sprintf(msgname, "%s%s%d.msg", a->dir, DIRSEP,
                m->number);
            if( remove(msgname) )
                resultcode = M_FILE;
            else
            {
                next = this->n; prev = this->p;
                if(prev != NULL) prev->n = next;
                if(next != NULL) next->p = prev;
                free(this);
                resultcode = M_OK;
            }
        }
        else resultcode = M_EOF;
    }
    else resultcode = M_EOF;

    return resultcode;
}

/* msg_rewrite() - rewrite an existing message */
int msg_rewrite(area *a, msg *m)
{
    int resultcode;

    if(m->number != 0)
        resultcode = msg_write(a, m);
    else
        resultcode = M_NEW;

    return resultcode;
}

/* msg_initmsg() - initialise the *.MSG messagebase type */
void msg_init(mbase *b, int zone)
{
    b->mopen = msg_open;
    b->mclose = msg_close;
    b->mfirst = msg_first;
    b->mlast = msg_last;
    b->mnext = msg_next;
    b->mprev = msg_prev;
    b->mgoto = msg_goto;
    b->mpost = msg_post;
    b->mkill = msg_kill;
    b->mrewrite = msg_rewrite;
    myzone = zone;
}

