/**********************************************************************
 * IOMSG.C                           Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * AstroMail 1.00 - input message processing via MSGLIB.              *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     09-Apr-97                                                 *
 **********************************************************************/


/* included headers ***************************************************/


#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include "msglib.h"
#include "astroml.h"
#include "config.h"


/* global variables ***************************************************/


mbase iomsg_b;       /* message base type */
area *iomsg_area;    /* message area */
msg  *iomsg_m;       /* an input message */
long  iomsg_last;    /* last read message */
char *iomsg_text,    /* the text of the inbound message */
     *iomsg_pos;     /* current position in above */

extern char pid[36]; /* program ID */


/* level 0 routines ***************************************************/


/* iomsg_open() - open the netmail area */
result iomsg_open(void)
{
    fido f;           /* address of installation */
    char netmail[72]; /* netmail directory */

    /* ascertain zone & initialise messagebase */
    if( config_firstaka(&f) != R_OK )
        return R_EOF;
    while( config_nextaka(&f) == R_OK );
    msg_init(&iomsg_b, f.zone);

    /* open message area */
    config_getnetmail(netmail);
    if(( iomsg_area = iomsg_b.mopen(netmail) ) == NULL)
        return R_FILE;
    if(( iomsg_m = msg_new() ) == NULL)
    {
        iomsg_b.mclose(iomsg_area);
        return R_MEMORY;
    }
    if( iomsg_b.mfirst(iomsg_m, iomsg_area) == R_OK )
        iomsg_last = msg_getnumber(iomsg_m);
    else
        iomsg_last = 0;

    return R_OK;
}

/* iomsg_close() - close the netmail area */
result iomsg_close(void)
{
    if(iomsg_last)
        iomsg_b.mgoto(iomsg_m, iomsg_area, iomsg_last);
    else
        iomsg_b.mfirst(iomsg_m, iomsg_area);
    iomsg_b.mclose(iomsg_area);
    if(iomsg_text != NULL) free(iomsg_text);

    return R_OK;
}

/* iomsg_first() - find the first message to AstroMail */
result iomsg_first(void)
{
    char toname[36],  /* 'to' name from message */
         astname[36]; /* astromail's name */
    fido toaddr;      /* to-address */

    /* return if no netmails in area */
    if( iomsg_b.mfirst(iomsg_m, iomsg_area) != M_OK ) return R_EOF;

    /* get astromail's name & first message name */
    config_getname(astname);
    msg_gettoname(toname, iomsg_m);
    toaddr = msg_gettoaddr(iomsg_m);

    /* look for messages addressed to AstroMail */
    while( stricmp(astname, toname) ||
           config_findaka(toaddr) ||
           msg_getrcvd(iomsg_m) )
    {
        if( iomsg_b.mnext(iomsg_m, iomsg_area) != M_OK ) return R_EOF;
        msg_gettoname(toname, iomsg_m);
        toaddr = msg_gettoaddr(iomsg_m);
    }

    return R_OK;
}

/* iomsg_next() - find the next message to AstroMail */
result iomsg_next(void)
{
    char toname[36],  /* 'to' name from message */
         astname[36]; /* astromail's name */
    fido toaddr;      /* to-address of msg */

    /* return if no netmails in area */
    if( iomsg_b.mnext(iomsg_m, iomsg_area) != M_OK ) return R_EOF;

    /* get astromail's name & next message name */
    config_getname(astname);
    msg_gettoname(toname, iomsg_m);
    toaddr = msg_gettoaddr(iomsg_m);

    /* look for messages addressed to AstroMail */
    while( stricmp(astname, toname) ||
           config_findaka(toaddr) ||
           msg_getrcvd(iomsg_m) )
    {
        if( iomsg_b.mnext(iomsg_m, iomsg_area) != M_OK ) return R_EOF;
        msg_gettoname(toname, iomsg_m);
        toaddr = msg_gettoaddr(iomsg_m);
    }

    return R_OK;
}

/* iomsg_get...() - return stuff about message */
char *iomsg_getfromname(char *fromname)
    { return msg_getfromname(fromname, iomsg_m); }
char *iomsg_getgame(char *game)
    { return msg_getsubject(game, iomsg_m); }
fido iomsg_getfromaddr(void)
    { return msg_getfromaddr(iomsg_m); }
fido iomsg_gettoaddr(void)
    { return msg_gettoaddr(iomsg_m); }

/* iomsg_firstline() - first line of message text */
char *iomsg_firstline(void)
{
    char *line;   /* line of text to return */
    int   length; /* length of line of message text */

    /* grab message text */
    if(iomsg_text != NULL) free(iomsg_text);
    if(( iomsg_pos = iomsg_text = msg_gettext(iomsg_m) ) == NULL)
        return NULL;
    while(( line = strstr(iomsg_text, "\r\n") ))
        strcpy( &line[1], &line[2] );
    while(( line = strchr(iomsg_text, '\n') )) *line = '\r';

    /* find first line */
    if(*iomsg_pos == '\0') return NULL;
    if( strchr(iomsg_pos, '\r') == NULL )
        length = strlen(iomsg_pos);
    else
        length = strchr(iomsg_pos, '\r') - iomsg_pos;
    if(( line = malloc(length + 1) ) == NULL) return NULL;
    strncpy(line, iomsg_pos, length);
    line[length] = '\0';
    if(iomsg_pos[length] == '\0')
        iomsg_pos += length;
    else
        iomsg_pos += length + 1;

    return line;
}

/* iomsg_nextline() - next line of message text */
char *iomsg_nextline(void)
{
    char *line;   /* line of text to return */
    int   length; /* length of line of message text */

    /* grab message text */
    if(iomsg_pos == NULL) return NULL;
    if(*iomsg_pos == '\0') return NULL;

    /* find next line */
    if( strchr(iomsg_pos, '\r') == NULL )
        length = strlen(iomsg_pos);
    else
        length = strchr(iomsg_pos, '\r') - iomsg_pos;
    if(( line = malloc(length + 1) ) == NULL) return NULL;
    strncpy(line, iomsg_pos, length);
    line[length] = '\0';
    if(iomsg_pos[length] == '\0')
        iomsg_pos += length;
    else
        iomsg_pos += length + 1;

    return line;
}

/* iomsg_post() - post an outgoing message */
result iomsg_post(fido fromaddr, char *toname, fido toaddr,
    char *subject, int attr, char *text)
{
    msg *outm; /* output message */

    /* initialise message */
    if(( outm = msg_new() ) == NULL) return R_MEMORY;

    /* set fields */
    msg_setfromname(outm, pid);
    msg_setfromaddr(outm, fromaddr);
    msg_settoname(  outm, toname);
    msg_settoaddr(  outm, toaddr);
    msg_setsubject( outm, subject);
    msg_setcrash(   outm, (attr & A_CRA) / A_CRA );
    msg_setattach(  outm, (attr & A_ATT) / A_ATT );
    msg_sethold(    outm, (attr & A_HLD) / A_HLD );
    msg_setfreq(    outm, (attr & A_FRQ) / A_FRQ );
    msg_setupdate(  outm, (attr & A_UPD) / A_UPD );
    msg_setkill(    outm, (attr & A_KIL) / A_KIL );
    msg_setpvt(     outm, (attr & A_PVT) / A_PVT );
    msg_setlocal(   outm, (attr & A_LOC) / A_LOC );
    msg_setdirect(  outm, (attr & A_DIR) / A_DIR );
    msg_seterase(   outm, (attr & A_ERA) / A_ERA );
    msg_setrcvd(    outm, (attr & A_RCV) / A_RCV );
    msg_setsent(    outm, (attr & A_SNT) / A_SNT );
    msg_settext(    outm, text);

    /* post and clear */
    iomsg_b.mpost(iomsg_area, outm);
    msg_old(outm);

    return R_OK;
}

/* iomsg_recd() - mark read message as received */
result iomsg_rcvd(void)
{
    msg_setrcvd(iomsg_m, 1);
    return iomsg_b.mrewrite(iomsg_area, iomsg_m);
}
