/**********************************************************************
 * ASTROML.C                         Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * AstroMail 1.00 - main program                                      *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     13-Apr-97                                                 *
 **********************************************************************/


/* included headers ***************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astroml.h"
#include "turnin.h"
#include "trnout.h"
#include "scrlog.h"


/* enumerated types ***************************************************/


typedef enum {
    S_NONE,    /* no service selected */
    S_TURNIN,  /* turn input selected */
    S_TURNOUT, /* turn output selected */
    S_HELP,    /* program help selected */
    S_ERROR    /* error on command line */
} service;


/* global variables ***************************************************/


char pid[36] = PROGNAME;


/* level 1 routines ***************************************************/


/* astroml_args() - parse command line */
service astroml_args(char *path, char **argp, int argc, char **argv)
{
    int     argi, /* current input parameter (argv) */
            argo; /* current output parameter (argp) */
    service s;    /* service to return */
    char   *env;  /* pointer to environment variable */

    /* initialise */
    strcpy(path, "");
    if(( env = getenv("ASTROML") ) != NULL) strcpy(path, env);
    argi = 1; argo = 0; s = S_NONE;

    /* loop through parameters */
    while(argi < argc)
    {
        if( !stricmp(argv[argi], "in") && s == S_NONE )
            s = S_TURNIN;
        else if( !stricmp(argv[argi], "out") && s == S_NONE )
            s = S_TURNOUT;
        else if( !stricmp(argv[argi], "?") )
            s = S_HELP;
        else if( !stricmp(argv[argi], "-path") )
        {
            if(++argi < argc)
                strcpy(path, argv[argi]);
            else
                s = S_ERROR;
        }
        else if(argo < 10)
            argp[argo++] = argv[argi];
        else s = S_ERROR;
        argi++;
    }

    /* fill rest of output parameters and return */
    while(argo < 10) argp[argo++] = NULL;
    return s;
}

/* astroml_drop() - drop semaphore */
result astroml_drop(char *path)
{
    char  filename[128]; /* full name of file */
    FILE *sem;           /* file handle for semaphore */

    sprintf(filename, "%sastroml.sem", path);
    if(( sem = fopen(filename, "r") ) != NULL)
    {
        fclose(sem);
        return R_EXISTS;
    }
    if(( sem = fopen(filename, "w") ) == NULL) return R_FILE;
    fclose(sem);

    return R_OK;
}

/* astroml_lift() - lift a semaphore */
void astroml_lift(char *path)
{
    char filename[128]; /* full name of file */

    sprintf(filename, "%sastroml.sem", path);
    remove(filename);
}


/* level 0 routines ***************************************************/


/* main program */
int main(int argc, char **argv)
{
    service s;        /* service requested on command line */
    char   *argp[10], /* process-specific parameters */
            path[72]; /* path to configs, semaphores etc. */
    result  r;        /* returned to calling process */

    scrlog_open();
    s = astroml_args(path, argp, argc, argv);
    if(( r = astroml_drop(path) ) == R_OK)
    {
        switch(s)
        {
            case S_TURNIN:  r = turnin_main(path, argp);
                            break;
            case S_TURNOUT: r = trnout_main(path, argp);
                            break;
            case S_ERROR:   r = R_BADCMD;
                            scrlog_error();
                            break;
            default:        r = R_OK;
                            scrlog_help();
        }
        astroml_lift(path);
    }
    scrlog_close();

    return r;
}
