/**********************************************************************
 * M2F.C                             Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * Ash 1.00 - Message-to-Files conversion program.                    *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     20-Jul-97                                                 *
 **********************************************************************/


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "ash.h"
#include "forecs.h"
#include "fleets.h"
#include "planet.h"
#include "detail.h"
#include "orders.h"


/* level 2 routines ***************************************************/


result readword(char *word, FILE *textfile)
{
    static char ic[2];

    /* initialise */
    strcpy(word, "");

    /* main loop */
    while( !isalnum( *ic = fgetc(textfile) ) )
        if(*ic == EOF) return R_EOF;
    do {
        strcat(word, ic);
    } while(isalnum( *ic = fgetc(textfile) ));

    return R_OK;
}


/* level 1 routines ***************************************************/


/* m2f_openfiles() - open all files */
FILE *m2f_openfiles(char *gamepath)
{
    char   filename[128]; /* name of message file */
    FILE  *textfile;      /* message file */

    /* attempt to open message file */
    sprintf(filename, "%sreport-file", gamepath);
    return textfile = fopen(filename, "r");
}

/* m2f_saveforecasts() - load forecast details into planet records */
void m2f_saveforecasts(detail *d, char *ename)
{
    /* explanation: before message processing, planet forecasts (for
       last turn) are committed to the forecast file, and are
       simultaneously copied over the 'live' planet data in order that
       planets for which no data is received this turn adopt their
       forecasted details (in the absence of anything more accurate). */

    forecs *z = forecs_new();
    planet *p = planet_new();
    order  *o = orders_new();
    fleet  *f = fleets_new();
    result  r;
    char    name[16],
            owner[16],
            origin[16],
            dest[16];
    long    ships,
            minships;
    int     mine,
            dist,
            speed;

    /* initialise */
    puts("reading forecasts...");
    speed = detail_getspeed(d);

    /* load forecast details */
    r = planet_first(p, P_NAME);
    while(!r)
    {
        /* ascertain current planet details */
        planet_getname(name, p);
        planet_getowner(owner, p);
        if(*owner)
        {
            ships = planet_getships(p);
            if( strcmp(owner, ename) ) mine = 0; else mine = 1;
            if( strcmp(owner, "neutral") ) ships += planet_getprod(p);
            minships = 2 - mine;

            /* read through fleets to do forecast... */
            r = fleets_first(f, F_NONE);
            while(!r)
            {
                fleets_getdest(dest, f);
                dist = fleets_getdist(f);
                if( !strcmp(name, dest) &&
		    fleets_getships(f) >= minships &&
		    dist <= speed)
                {
                    if(mine)
                        ships += fleets_getships(f);
                    else
                        ships -= fleets_getships(f);
                    if(ships < 0)
                    {
                        mine = 1;
                        ships = -ships;
                    }
                }
                r = fleets_next(f, F_NONE);
            }

            /* ... and read through orders */
            if(mine)
            {
                r = orders_first(o);
                while(!r)
                {
                    if( orders_gettype(o) == 'S' )
                    {
                        orders_getorigin(origin, o);
                        if( !strcmp(origin, name) )
                            ships -= orders_getships(o);
                    }
                    r = orders_next(o);
                }
            }

	    /* save new forecast to planet and forecast files */
            if(mine) strcpy(owner, ename);
	    planet_setowner(p, ename);
	    planet_setowner(p, owner);
	    planet_setships(p, ships);
	    planet_write(p);
            if( forecs_find(z, name) ) forecs_clear(z);
            forecs_setname(z, name);
            forecs_setowner(z, ename);
            forecs_setowner(z, owner);
            forecs_setships(z, ships);
            forecs_write(z);
        }
        r = planet_next(p, P_NAME);
    }

    /* advance fleets */
    r = fleets_first(f, F_NUMBER);
    while(!r)
    {
        fleets_setdist( f, fleets_getdist(f) - speed );
        if( fleets_getdist(f) <= 0 )
            fleets_delete(f);
        else
            fleets_write(f);
        r = fleets_next(f, F_NUMBER);
    }

    /* clean up */
    fleets_old(f);
    orders_old(o);
    forecs_old(z);
    planet_old(p);
}

/* m2f_findfleets() - find the 'your fleets are' report */
result m2f_findfleets(FILE *textfile)
{
    char word[72];
    int  state = 0;

    /* initialise */
    puts("finding fleet list...");
    fseek(textfile, 0, SEEK_SET);

    /* do the search */
    while( !readword(word, textfile) )
    {
        switch(state)
        {
            case 0:
                if( !stricmp(word, "your") ) state++; else state = 0;
                break;
            case 1:
                if( !stricmp(word, "fleets") ) state++; else state = 0;
                break;
            case 2:
                if( !stricmp(word, "are") ) return R_OK; else state = 0;
        }
    }
    return R_EOF;
}

/* m2f_readfleets() */
void m2f_readfleets(FILE *textfile, char *ename)
{
    char   line[128],
           origin[16],
           dest[16];
    long   fleetno,
           ships;
    int    dist,
           temp;
    fleet *f = fleets_new();
    result r;

    /* initialise */
    puts("reading fleet list...");
    r = fleets_first(f, F_NUMBER);
    while(!r)
    {
        if( !fleets_deleted(f) ) fleets_delete(f);
        r = fleets_next(f, F_NUMBER);
    }

    /* find first actual fleet line */
    do {
        if( !fgets(line, 128, textfile) ) goto rfreturn;
    } while(( temp = sscanf(line, "%ld%s%s%ld%d", &fleetno, origin, dest,
        &ships, &dist) ) != 5);

    /* read through fleet lines */
    while(( temp = sscanf(line, "%ld%s%s%ld%d", &fleetno, origin, dest,
        &ships, &dist) ) == 5 )
    {
        if( fleets_find(f, fleetno) )
            fleets_clear(f);
        else if( fleets_deleted(f) )
            fleets_delete(f);
        fleets_setnumber(f, fleetno);
        fleets_setowner(f, ename);
        fleets_setorigin(f, origin);
        fleets_setdest(f, dest);
        fleets_setships(f, ships);
        fleets_setdist(f, dist);
        fleets_write(f);
        if( !fgets(line, 128, textfile) ) goto rfreturn;
    }

rfreturn:
    fleets_old(f);
}

/* m2f_findplanets() - find the 'your planets are' report */
result m2f_findplanets(FILE *textfile)
{
    char word[72];
    int  state = 0;

    /* initialise */
    puts("finding planet list...");
    fseek(textfile, 0, SEEK_SET);

    /* do the search */
    while( !readword(word, textfile) )
    {
        switch(state)
        {
            case 0:
                if( !stricmp(word, "your") ) state++; else state = 0;
                break;
            case 1:
                if( !stricmp(word, "planets") ) state++; else state = 0;
                break;
            case 2:
                if( !stricmp(word, "are") ) return R_OK; else state = 0;
        }
    }
    return R_EOF;
}

/* m2f_readplanets() */
void m2f_readplanets(FILE *textfile, char *ename)
{
    char   line[128],
           name[128];
    pos    pp;
    int    prod;
    long   ships;
    planet *p = planet_new();

    puts("reading planet list...");
    do {
        if( !fgets(line, 128, textfile) ) goto rfreturn;
    } while( sscanf(line, "%s%hd,%hd%d%ld", name, &pp.x, &pp.y, &prod,
        &ships) != 5 );
    while( sscanf(line, "%s%hd,%hd%d%ld", name, &pp.x, &pp.y, &prod,
        &ships) == 5 )
    {
        if( planet_find(p, name) )
            planet_clear(p);
        planet_setname(p, name);
        planet_setowner(p, ename);
        planet_setpos(p, pp);
        planet_setprod(p, prod);
        planet_setships(p, ships);
        planet_write(p)
        ;
        if( !fgets(line, 128, textfile) ) goto rfreturn;
    }

rfreturn:
    planet_old(p);
}

/* m2f_scaninteletc() - find intelligence/attack reports */
void m2f_scaninteletc(FILE *textfile)
{
    char    word[72];
    int     state = 0;
    planet *p = planet_new();

    /* initialise */
    puts("finding intelligence reports etc...");
    fseek(textfile, 0, SEEK_SET);

    /* do the scan */
    while( !readword(word, textfile) )
    {
        switch(state)
        {
            case -7:
                planet_setowner(p, word);
                planet_write(p);
                state = 0;
                break;
            case -6:
                if( !stricmp(word, "of") ) state--;
                break;
            case -5:
                planet_setships( p, atoi(word) );
                state--;
                break;
            case -4:
                if( !stricmp(word, "with") ) state--;
                break;
            case -3:
                if( !stricmp(word, "failed") )
                    state--;
                else if( !stricmp(word, "successful") )
                    state = 0;
                break;
            case -2:
                if( planet_find(p, word) )
                {
                    planet_clear(p);
                    planet_setname(p, word);
                }
                state--;
                break;
            case -1:
                if( !stricmp(word, "of") ) state--;
                break;
            case 0:
                if( !stricmp(word, "intelligence") )
                    state++;
                else if( !stricmp(word, "invasion") )
                    state--;
                else state = 0;
                break;
            case 1:
                if( !stricmp(word, "at") ) state++;
                break;
            case 2:
                if( planet_find(p, word) )
                {
                    planet_clear(p);
                    planet_setname(p, word);
                }
                state++;
                break;
            case 3:
                if( !stricmp(word, "owner") ) state++;
                break;
            case 4:
                planet_setowner(p, word);
                state++;
                break;
            case 5:
                if( !stricmp(word, "production") ) state++;
                break;
            case 6:
                planet_setprod( p, atoi(word) );
                state++;
                break;
            case 7:
                if( !stricmp(word, "ships") ) state++;
                break;
            case 8:
                planet_setships( p, atoi(word) );
                planet_write(p);
                state = 0;
                break;
        }
    }

    planet_old(p);
}

/* m2f_scanmap() - scan maps */
void m2f_scanmap(FILE *textfile)
{
    int     state = 0,
            items;
    char    line[128],
            name[128],
           *a;
    static
    char    ch[2];
    planet *p = planet_new();
    pos     pp;

    /* initialise */
    puts("scanning maps...");
    fseek(textfile, 0, SEEK_SET);

    /* scan maps */
    a = fgets(line, 128, textfile);
    while( a )
    {
        switch(state)
        {
            case 0: /* looking for 'map of planets around' */
                a = fgets(line, 128, textfile);
                if( !strcmp(line,
                    "------+------------------------------\n") )
                    state++;
                break;
            case 1: /* reading until blank line */
                a = fgets(line, 128, textfile);
                if(*line == '\n')
                    state++;
                break;
            case 2: /* reading planet co-ordinates */
                do {
                    *line = '\0';
                    *ch = fgetc(textfile);
                    while(*ch != EOF && *ch != '.' && *ch != ':')
                    {
                        if(*ch != '\n')
                            strcat(line, ch);
                        else
                            strcat(line, " ");
                        *ch = fgetc(textfile);
                    }
                    if(*ch == '.') fgetc(textfile);
                    items = sscanf(line, "%c%c=%s (%hd,%hd)", ch, ch,
                        name, &pp.x, &pp.y);
                    if(items == 5)
                    {
                        if( planet_find(p, name) ) planet_clear(p);
                        planet_setname(p, name);
                        planet_setpos(p, pp);
                        planet_write(p);
                    }
                    else state = 0;
                } while(items == 5 && *ch != EOF);
                if(*ch == EOF) a = NULL;
                break;
        }
    }

    /* clean up */
    planet_old(p);
}

/* m2f_testreport() - output some reports */
void m2f_testreport(void)
{
    planet *p = planet_new();
    fleet  *f = fleets_new();
    char    name[16],
            owner[16],
            origin[16],
            dest[16];
    pos     pp;
    long    ships,
            number;
    int     prod,
            dist;

    if( !planet_first(p, P_NAME) )
    {
        puts("--- Planets:");
        do {
            planet_getname(name, p);
            planet_getowner(owner, p);
            pp = planet_getpos(p);
            prod = planet_getprod(p);
            ships = planet_getships(p);
            printf("%-15s (%-15s) %5d,%5d %5d %10ld\n", name, owner,
                pp.x, pp.y, prod, ships);
        } while ( !planet_next(p, P_NAME) );
    }
    if( !fleets_first(f, F_NUMBER) )
    {
        puts("--- Fleets:");
        do {
            number = fleets_getnumber(f);
            fleets_getowner(owner, f);
            fleets_getorigin(origin, f);
            fleets_getdest(dest, f);
            ships = fleets_getships(f);
            dist = fleets_getdist(f);
            printf("%10ld (%-15s) %-15s %-15s %10ld %5d\n", number,
                owner, origin, dest, ships, dist);
        } while ( !fleets_next(f, F_NUMBER) );
    }

    /* clean up */
    planet_old(p);
    fleets_old(f);
}

/* m2f_closefiles() - close all files */
void m2f_closefiles(FILE *textfile, char *gamepath)
{
    char filename[128];

    /* close and remove text file */
    fclose(textfile);
    sprintf(filename, "%sreport-file", gamepath);
    remove(filename);

    /* close, remove and renew orders */
    orders_close();
    sprintf(filename, "%sorders.data", gamepath);
    remove(filename);
    orders_create(gamepath);
    orders_open(gamepath);
}


/* level 0 routines ***************************************************/


/* m2f_main() - main program */
result m2f_main(char *gamepath, char *ename, detail *d)
{
    FILE *textfile; /* message file */

    if(( textfile = m2f_openfiles(gamepath) ))
    {
        m2f_saveforecasts(d, ename);
        if( !m2f_findfleets(textfile) )
            m2f_readfleets(textfile, ename);
        m2f_scaninteletc(textfile);
        if( !m2f_findplanets(textfile) )
            m2f_readplanets(textfile, ename);
        m2f_scanmap(textfile);
        m2f_closefiles(textfile, gamepath);
    }

    return R_OK;
}
