/* 
 *    Programmed By: Mohammed Isam Mohammed [mohammed_isam1984@yahoo.com]
 *    Copyright 2014, 2015, 2016, 2017, 2018 (c)
 * 
 *    file: init.c
 *    This file is part of mino.
 *
 *    mino is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    mino is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with mino.  If not, see <http://www.gnu.org/licenses/>.
 */    
#include <pwd.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <asm/types.h>
#include "defs.h"
#include "kbd.h"
#include "edit.h"
#include "options.h"
#include "help.h"

int     NEW_GNU_DOS_LEVEL;
struct  passwd *pass;//will be used to find the home dir
char   *STARTUP_FILE_NAME;
char   *mino_ver = "2.0";
char   *copyrightNotice = "Mino for GNU/Linux is developed by Mohammed Isam\n"
                          "Copyright (C) Mohammed Isam 2014, 2015, 2016, 2017, 2018\n"
                          "This is a GNU software, part of the GnuDOS package\n";
int  SHOW_README;


void outOfMemory()
{
    fprintf(stderr, "Fatal error: Insufficient memory\n");
    exit(1);
}

void exit_gracefully(int exit_code, char *open_file_name)
{
    if(FILE_STATE == MODIFIED) 
    {
        int i = msgBox("File has been changed. Save changes?", YES|NO, INFO);
      if(i == YES) 
      {
	if(NEW_FILE) 
	{
	  if(!openSaveFile(SAVE, YES, open_file_name)) 
	  {
	    msgBox("Failed to save file.", OK, ERROR);
	  } else { FILE_STATE = SAVED; }
	} 
	else 
	{
	  if(!openSaveFile(SAVE, NO, open_file_name)) 
	  {
	    msgBox("Failed to save file.", OK, ERROR);
	  } else { FILE_STATE = SAVED; }
	}
      } 
      else if(i == CANCEL) 
      { 
	FILE_STATE = MODIFIED;
      } else { FILE_STATE = IDLE; }
    if(FILE_STATE == SAVED || FILE_STATE == IDLE)
      goto good_to_go;
  } else goto good_to_go;
  refreshView();
  return;

good_to_go:
  fcloseall();
  setScreenColors(WHITE, BGBLACK);
  clearScreen(); 
  restoreTerminal(); 
  setScreenColors(WHITE, BGBLACK);
  fprintf(stdout, "\x1b[24m");
  fflush(stdout);
  exit(exit_code); 
}

void parseLineArgs(int argc, char **argv) 
{
    GNU_DOS_LEVEL = 1;
    NEW_FILE = 1;
    FILE_STATE = NEW;
    open_file_at_startup = 0;
    AUTO_HIGHLIGHTING = 0;
    NEW_GNU_DOS_LEVEL = 0;
    ///////////////////////////////////////
    //parse command line arguments
    ///////////////////////////////////////
    int c;
    static struct option long_options[] =
    {
	 {"reset-config", no_argument,            0,  'r'},
	 {"help",         no_argument,            0,  'h'},
	 {"level",  required_argument,            0,  'l'},
	 {"version",      no_argument,            0,  'v'},
	 {0, 0, 0, 0}
    };

    while(1)
    {
        int option_index=0;
        c = getopt_long(argc, argv, "rhl:v", long_options, &option_index);
        if(c==-1) break;	//end of options

        switch(c)
        {
            case 0:
                break;
            case 'r':	//reset config file
                if(!(pass = getpwuid(geteuid()))) 
                {
		    printf("Error: couldn't open home directory to write "
			   "configuration file.\n");
		    printf("Aborting.\n");
		    exit(1);
                }
                config_file_name = (char *) malloc(strlen(pass->pw_dir)+12);
                if(!config_file_name) { fprintf(stderr, "Insufficient memory\n"); exit(1); }
                strcpy(config_file_name, pass->pw_dir);
                strcat(config_file_name, "/");
                strcat(config_file_name, ".mino.conf");
                if(!(config_file = fopen(config_file_name, "w"))) 
                {
		    printf("Error: couldn't write to configuration file in "
		           "your home directory.\n");
		    printf("Aborting.\n");
		    exit(1);
                }
                printf("Resetting program configuration..\n");
                _write_config_file(config_file, 1);
                fclose(config_file);
                printf("Finished writing default values to ~/.mino.conf\n");
                exit(0);
                break;
            case 'h':	//show program help
                printf("\nCommand line help for mino\n");
                printf("%s\n", copyrightNotice);
                printf("Usage: %s [filename][options]\n\n", argv[0]);
                printf("Options:\n");
                printf("\tfilename: File to be opened by mino (optional)\n");
                printf("\t--reset-config, -r\tReset program configuration. "
                        "Writes default\n");
                printf("\t                  \tvalues to .mino.conf file\n");
                printf("\t--help, -h        \tShow this command line help\n");
                printf("\t--version, -v     \tShow program version\n");
                printf("\t--levelX, -lX     \tSet the experience level, where X is 1-6\n");
                printf("For more information, see 'info mino' or the README"
                        " file in mino help menu\n");
                exit(0);
                break;
            case 'v':	//show program version
                printf("%s\n", mino_ver);
                exit(0);
                break;
            case 'l':	//set GNU_DOS level
                ;
                int i = atoi(optarg);
                if(i < 1 || i > 6)
                {
                    printf("Unrecognised level. See 'man mino' or "
                            "'info mino' for information about possible"
                            " levels.\n");
                    exit(1); break;
                }
                NEW_GNU_DOS_LEVEL = i;
                break;
            case '?':
                break;
            default:
                abort();
            }
    }
    ///////////////////////////////////////
    //parse the remaining arguments
    ///////////////////////////////////////
    STARTUP_FILE_NAME = NULL;
    if(optind < argc)
    {
	  NEW_FILE = 0; FILE_STATE = OPENED;
	  open_file_at_startup = 1;
	  STARTUP_FILE_NAME = argv[optind];
    }
}


void init_error() 
{
      printf("Error: couldn't open home directory and read configuration file.\n");
      printf("Will use built-in defaults. This means if you have set any\n");
      printf("preferences (for example, display colors), they will be of\n");
      printf("no effect.\n");
      printf("You can reset your config file to program defaults by invoking:\n");
      printf("mino --reset-config\n\nPress any key to continue..\n");
      getchar();
      //init_built_in_defaults();
}

int __initNewDocument()
{
    if(totalLines)
    {
	int i;
        for(i = 0; i < totalLines; i++) freeLineStruct(lines[i]);
        totalLines = 0;
    }
    lines[0] = allocLineStructB(maxLen);
    if(!lines[0]) return 0;
    totalLines = 1;
    initEdit();
    return 1;
}

int initNewDocument()
{
    if(documentTitle) free(documentTitle);
    documentTitle = (char *) malloc(strlen(DEFAULT_TITLE)+1);
    if(!documentTitle) return 0;
    strcpy(documentTitle, DEFAULT_TITLE);
    return __initNewDocument();
}

void resetLineCounters()
{
    selectedLine       = 0;
    selectedChar       = 0;
    selectedCharCarry  = 0;
    totalVisLines      = SCREEN_H-4;
    firstVisLine       = 0;
}

void __init()
{
    getScreenSize();
    //1. Main Menu//
    menu[0] = "&File";
    menu[1] = "&Edit";
    menu[2] = "&Options";
    menu[3] = "&Help";
    //2. File Menu//
    fileMenu[0] = "New..         ^N";
    fileMenu[1] = "Open file     ^O";
    fileMenu[2] = "Save file     ^S";
    fileMenu[3] = "Save as..       ";
    fileMenu[4] = "Print         ^P";
    fileMenu[5] = "Exit          ^Q";
    //3. Edit Menu//
    editMenu[0] = "Cut                ^X";
    editMenu[1] = "Copy               ^C";
    editMenu[2] = "Paste              ^V";
    editMenu[3] = "Select all         ^A";
    editMenu[4] = "Undo               ^Z";
    editMenu[5] = "Redo               ^Y";
    editMenu[6] = "Delete Line        ^D";
    editMenu[7] = "Find..             ^F";
    editMenu[8] = "Replace..          ^R";
    editMenu[9] = "Toggle select mode ^E";
    //4. Options Menu//
    optionsMenu[0] = "Change colors  ";
    optionsMenu[1] = "Tab spaces     ";
    optionsMenu[2] = "Autoindent     ";
    optionsMenu[3] = "Reset config   ";
    //5. Help Menu//
    helpMenu[0] = "View README    ";
    helpMenu[1] = "GNU Keybindings";
    helpMenu[2] = "Quick reference";
    helpMenu[3] = "About Mino..   ";
    //initiate color arrays
    FG_COLOR_ARRAY[0] = 30;
    FG_COLOR_ARRAY[1] = 31;
    FG_COLOR_ARRAY[2] = 32;
    FG_COLOR_ARRAY[3] = 33;
    FG_COLOR_ARRAY[4] = 34;
    FG_COLOR_ARRAY[5] = 35;
    FG_COLOR_ARRAY[6] = 36;
    FG_COLOR_ARRAY[7] = 37;
    BG_COLOR_ARRAY[0] = 40;
    BG_COLOR_ARRAY[1] = 41;
    BG_COLOR_ARRAY[2] = 42;
    BG_COLOR_ARRAY[3] = 43;
    BG_COLOR_ARRAY[4] = 44;
    BG_COLOR_ARRAY[5] = 45;
    BG_COLOR_ARRAY[6] = 46;
    BG_COLOR_ARRAY[7] = 47;
    
    //MAX_FILE_NAME_LEN = 100;
    AUTO_INDENT = 1;
    autoIndentStr = (char *)malloc(512);
    if(!autoIndentStr) outOfMemory();
    MAX_MSG_BOX_W      = SCREEN_W - 4;
    MAX_MSG_BOX_H      = SCREEN_H - 4;
    MAX_CHARS_PER_LINE = SCREEN_W - 2;
    maxLen             = MAX_CHARS_PER_LINE*4;
    CAPS               = 0;
    INSERT             = 0;
    SELECTING          = 0;
    CLIPBOARD_IS_EMPTY = 1;
    resetLineCounters();

    //initialize color arrays
    FG_COLOR[COLOR_WINDOW]         = 37;
    FG_COLOR[COLOR_HIGHLIGHT_TEXT] = 34;
    FG_COLOR[COLOR_MENU_BAR]       = 34;
    FG_COLOR[COLOR_STATUS_BAR]     = 34;
    FG_COLOR[COLOR_BUTTONS]        = 37;
    FG_COLOR[COLOR_HBUTTONS]       = 32;
    BG_COLOR[COLOR_WINDOW]         = 44;
    BG_COLOR[COLOR_HIGHLIGHT_TEXT] = 47;
    BG_COLOR[COLOR_MENU_BAR]       = 47;
    BG_COLOR[COLOR_STATUS_BAR]     = 47;
    BG_COLOR[COLOR_BUTTONS]        = 41;
    BG_COLOR[COLOR_HBUTTONS]       = 41;
    old_window_color = BG_COLOR[COLOR_WINDOW];

    GNU_DOS_LEVEL = 1;
    WRAP_LINES    = 1;//TRUE -- restric line width to (screen width-2)
    TAB_CHARS     = 8;	//define 8 spaces in a tab
    SHOW_README   = 0;
}
/***************************************
 * init(): 
 * Procedure containing code that init-
 * ializes various data structures.
 * **************************************/
void init(char *open_file_name)
{
    //int i;
    DEFAULT_TITLE = __DEFAULT_TITLE;
    //initiate color arrays
    COLOR_STR[0] = "BLACK";
    COLOR_STR[1] = "RED";
    COLOR_STR[2] = "GREEN";
    COLOR_STR[3] = "BROWN";
    COLOR_STR[4] = "BLUE";
    COLOR_STR[5] = "MAGENTA";
    COLOR_STR[6] = "CYAN";
    COLOR_STR[7] = "WHITE";
    __init();
    /////////////////////////////////////
  
    if(!(pass = getpwuid(geteuid()))) 
    {
        init_error();
        goto post_config_file;
    }
    config_file_name = (char *) malloc(strlen(pass->pw_dir)+11);
    if(!config_file_name) outOfMemory();
    strcpy(config_file_name, pass->pw_dir);
    strcat(config_file_name, "/");
    strcat(config_file_name, ".mino.conf");
    if((config_file = fopen(config_file_name, "r"))) 
    {
        char buf[100];
        //read configuration file
        while(fgets(buf, sizeof(buf), config_file)) 
        {
            if(buf[0] == '#' || buf[0] == '\n') continue;
            /*
            if(strstr(buf, "MAXDIRS")) {
                MAXDIRS = atoi((strchr(buf, "=")+2));
            } else if (strstr(buf, "MAXFILES")) {
                MAXFILES = atoi((strchr(buf, "=")+2));
            } else if (strstr(buf, "MAXLINES")) {
                MAX_LINES = atoi((strchr(buf, "=")+2));
            } else if (strstr(buf, "MAX_CHARS_PER_LINE")) 
                MAX_CHARS_PER_LINE = atoi((strchr(buf, '=')+2));
            */
            if (strstr(buf, "GNU_DOS_LEVEL")) 
                GNU_DOS_LEVEL = atoi((strchr(buf, '=')+2));
            else if (strstr(buf, "TAB_CHARS")) 
            {
                TAB_CHARS = atoi(strchr(buf, '=')+2);
                if(TAB_CHARS < 1 || TAB_CHARS > SCREEN_W-3) TAB_CHARS = 8;
            }
            else if (strstr(buf, "DEFAULT_TITLE")) 
            {
                char *title = strchr(buf, '=')+2;
                int titlelen = strlen(title);
                DEFAULT_TITLE = (char *)malloc(titlelen+1);
                if(!DEFAULT_TITLE) goto memerr;
                strcpy(DEFAULT_TITLE, title);
                DEFAULT_TITLE[titlelen-1] = '\0';
            }
            else if (strstr(buf, "WRAP_LINES")) 
            {
                if(strstr(buf, "TRUE")) WRAP_LINES = 1;
                else WRAP_LINES = 0;
            }
            else if (strstr(buf, "CAPS")) 
            {
                if(strstr(buf, "OFF")) CAPS = 0;
                else CAPS = 1;
            }
            else if (strstr(buf, "INSERT")) 
            {
                if(strstr(buf, "OFF")) INSERT = 0;
                else INSERT = 1;
            }
            else if (strstr(buf, "FG_COLOR_WIN")) 
                FG_COLOR[COLOR_WINDOW] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "FG_COLOR_HLT")) 
                FG_COLOR[COLOR_HIGHLIGHT_TEXT] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "FG_COLOR_MBAR")) 
                FG_COLOR[COLOR_MENU_BAR] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "FG_COLOR_SBAR")) 
                FG_COLOR[COLOR_STATUS_BAR] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "FG_COLOR_HBUT")) 
                FG_COLOR[COLOR_HBUTTONS] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "FG_COLOR_BUT")) 
                FG_COLOR[COLOR_BUTTONS] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "BG_COLOR_WIN"))
            {
                BG_COLOR[COLOR_WINDOW] = atoi(strchr(buf, '=')+2);
                old_window_color = BG_COLOR[COLOR_WINDOW];
            }
            else if (strstr(buf, "BG_COLOR_HLT")) 
                BG_COLOR[COLOR_HIGHLIGHT_TEXT] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "BG_COLOR_MBAR")) 
                BG_COLOR[COLOR_MENU_BAR] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "BG_COLOR_SBAR")) 
                BG_COLOR[COLOR_STATUS_BAR] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "BG_COLOR_HBUT")) 
                BG_COLOR[COLOR_HBUTTONS] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "BG_COLOR_BUT")) 
                BG_COLOR[COLOR_BUTTONS] = atoi(strchr(buf, '=')+2);
            else if (strstr(buf, "SHOW_README")) 
                SHOW_README = 1;
            else if (strstr(buf, "AUTO_INDENT")) 
                AUTO_INDENT = atoi(strchr(buf, '=')+2);
        }
        fclose(config_file);
        free(config_file_name); 
    }
    else write_config_file();
    
post_config_file:
    if(AUTO_INDENT) optionsMenu[2] = strdup("Autoindent    *");
    else optionsMenu[2] = strdup("Autoindent     ");
 
    if(NEW_GNU_DOS_LEVEL) GNU_DOS_LEVEL = NEW_GNU_DOS_LEVEL;
    if(GNU_DOS_LEVEL > 5 || GNU_DOS_LEVEL < 1) GNU_DOS_LEVEL = 1;
 
    //set defaults for Dir view//
    initDirView();
    totalLines = 0;

    if(NEW_FILE) 
    {
        if(!initNewDocument()) goto memerr;
    }
    else
    {
        int len;
        char *slash = strrchr(open_file_name, '/');
        if(slash) len = strlen(slash);
        else      len = strlen(open_file_name);
        documentTitle = (char *) malloc(len+1);
        if(!documentTitle) goto memerr;
        if(slash) strcpy(documentTitle, slash+1);
        else      strcpy(documentTitle, open_file_name);
    }
 
    if(!initTerminal()) 
    {
        fprintf(stderr, "Fatal error: Failed to initialize the terminal.\r\nAborting.\n");
        exit(1);
    }
    catchSignals();
    return;
 
memerr:
    fprintf(stderr, "Fatal error: Insufficient memory\n");
    exit(1);
}


void showREADMEOnStartup() 
{
  int x = SCREEN_H/2-3;
  int y = SCREEN_W/2-20;
  int w = y+40;
  int h = x+6;
  
  setScreenColorsI(COLOR_WINDOW);
  drawBox(x, y, h, w, " Welcome to mino ", YES, 0);
  printf("\e[%d;%dH", x+1, y+2);
  printf("Welcome to mino!");
  printf("\e[%d;%dH", x+2, y+2);
  printf("README file will be shown to help you");
  printf("\e[%d;%dH", x+3, y+2);
  printf("start using mino.");
  setScreenColorsI(COLOR_HBUTTONS);
  printf("\e[%d;%dH", x+5, y+2);
  printf("  OK  ");
  setScreenColorsI(COLOR_BUTTONS);
  printf("\e[%d;%dH", x+5, y+12);
  printf(" Don't show README again ");
  printf("\e[%d;%dH", x+5, y+2);
  fflush(stdout);
  int sel = 0;
  char *c = (char *)malloc(5);
  while((c = getKey())) 
  {
    switch(c[0]) 
    {
      case(RIGHT_KEY):
      case(LEFT_KEY):
      case(TAB_KEY):
	if(sel == 0) 
	{
	  setScreenColors(FG_COLOR[COLOR_BUTTONS], BG_COLOR[COLOR_BUTTONS]);
	  printf("\e[%d;%dH", x+5, y+2);
	  printf("  OK  ");
	  setScreenColors(FG_COLOR[COLOR_HBUTTONS], BG_COLOR[COLOR_HBUTTONS]);
	  printf("\e[%d;%dH", x+5, y+12);
	  printf(" Don't show README again ");
	  printf("\e[%d;%dH", x+5, y+12);
	  sel = 1;
	} 
	else 
	{
	  setScreenColors(FG_COLOR[COLOR_HBUTTONS], BG_COLOR[COLOR_HBUTTONS]);
	  printf("\e[%d;%dH", x+5, y+2);
	  printf("  OK  ");
	  setScreenColors(FG_COLOR[COLOR_BUTTONS], BG_COLOR[COLOR_BUTTONS]);
	  printf("\e[%d;%dH", x+5, y+12);
	  printf(" Don't show README again ");
	  printf("\e[%d;%dH", x+5, y+2);
	  sel = 0;
	} break;
      case(ENTER_KEY):
      case(SPACE_KEY):
	if(sel == 0) SHOW_README = 1;
	else SHOW_README = 0;
	showReadMe();
	write_config_file();
	refreshView();
	return;
    }//end switch
    fflush(stdout);
  }//end while
}
