/*****************************************************************************\
 * Copyright 2005, 2006 Niels Lohmann, Christian Gierds, Dennis Reinert      *
 *                                                                           *
 * This file is part of BPEL2oWFN.                                           *
 *                                                                           *
 * BPEL2oWFN is free software; you can redistribute it and/or modify it      *
 * under the terms of the GNU General Public License as published by the     *
 * Free Software Foundation; either version 2 of the License, or (at your    *
 * option) any later version.                                                *
 *                                                                           *
 * BPEL2oWFN is distributed in the hope that it will be useful, but WITHOUT  *
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     *
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  *
 * more details.                                                             *
 *                                                                           *
 * You should have received a copy of the GNU General Public License along   *
 * with BPEL2oWFN; if not, write to the Free Software Foundation, Inc., 51   *
 * Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.                      *
\****************************************************************************/

%{
/*!
 * \file bpel-syntax.c
 *
 * \brief BPEL grammar (implementation)
 *
 * This file defines and implements the grammar of BPEL using standard 
 * BNF-rules to describe the originally XML-based syntax as it is specified in
 * the BPEL4WS 1.1 specification. All terminals are passed from the lexer
 * (implemented in \ref bpel-lexic.c). Besides simple syntax-checking the
 * grammar is used to build the abstract syntax tree as it is defined in
 * bpel-abstract.k and implemented in \ref bpel-abstract.cc and
 * \ref bpel-abstract.h.
 * 
 * \author  
 *          - responsible: Niels Lohmann <nlohmann@informatik.hu-berlin.de>
 *          - last changes of: \$Author: nlohmann $
 *          
 * \date 
 *          - created: 2005/11/10
 *          - last changed: \$Date: 2006/01/13 15:13:41 $
 * 
 * \note    This file is part of the tool BPEL2oWFN and was created during the
 *          project "Tools4BPEL" at the Humboldt-Universit� zu Berlin. See
 *          http://www.informatik.hu-berlin.de/top/forschung/projekte/tools4bpel
 *          for details.
 *
 * \note    This file was created using GNU Bison 1.875b reading file
 *          bpel-syntax.yy. Copyright (C) 1984, 1989, 1990, 2000, 2001, 2002,
 *          2003 Free Software Foundation, Inc.
 *          See http://www.gnu.org/software/bison/bison.html for details
 *
 * \version \$Revision: 1.94 $
 * 
 * \todo
 *          - add rules to ignored everything non-BPEL
 *          - inProcess can be replaced by "(currentScopeId->value == 1)"
 */
%}


%{	
/*!
 * \file bpel-syntax.h
 * \brief BPEL grammar (interface)
 *
 * See \ref bpel-syntax.c for more information.
 */
%}

// the terminal symbols (tokens)
%token K_ASSIGN K_CASE K_CATCH K_CATCHALL K_COMPENSATE K_COMPENSATIONHANDLER
%token K_COPY K_CORRELATION K_CORRELATIONS K_CORRELATIONSET K_CORRELATIONSETS
%token K_EMPTY K_EVENTHANDLERS K_FAULTHANDLERS K_FLOW K_FROM K_IMPORT K_INVOKE
%token K_LINK K_LINKS K_ONALARM K_ONMESSAGE K_OTHERWISE K_PARTNER K_PARTNERLINK
%token K_PARTNERLINKS K_PARTNERS K_PICK K_PROCESS K_RECEIVE K_REPLY K_SCOPE
%token K_SEQUENCE K_SOURCE K_SWITCH K_TARGET K_TERMINATE K_THROW K_TO
%token K_VARIABLE K_VARIABLES K_WAIT K_WHILE
%token X_OPEN X_SLASH X_CLOSE X_NEXT X_EQUALS
%token <yt_casestring> X_NAME
%token <yt_casestring> X_STRING


// the start symbol of the grammar
%start tProcess


// Bison generates a list of all used tokens in file "bpel-syntax.h" (for flex)
%token_table



%{



// options for Bison
#define YYDEBUG 1
#define YYERROR_VERBOSE 1  // for verbose error messages


// to avoid the message "parser stack overflow"
#define YYMAXDEPTH 1000000
#define YYINITDEPTH 10000


// generated by Kimwitu++
#include "bpel-kc-k.h" // phylum definitions
#include "bpel-kc-yystype.h" // data types for tokens and non-terminals

#include <map>

// from flex
extern char* yytext;
extern int yylex();


// defined in "debug.h"
extern int yyerror(const char *);


using namespace kc;
using namespace std;

// manage attributes
#include "bpel-attributes.h"

/// an instance of the attribute manager
attributeManager att = attributeManager();

#include "check-symbols.h"

/// from check-symbols
SymbolManager symMan = SymbolManager();

/// needed to distinguish context of tPartnerLink
bool inPartners = false;

/// needed to check occurrence of links within whiles
bool inWhile = false;

/// needed to distinguish context of the Fault Handler
bool inProcess = true;

/// needed to tag handlers
integer currentScopeId;

/// needed to tag scopes
map <integer, integer> parent;

/// the root of the abstract syntax tree
tProcess TheProcess;

/// stack for checking for FaultHandler
stack<bool> isInFH;
/// stack for checking for CompensationHandler
stack< pair<bool,int> > isInCH;
int hasCompensate;

%}



/* the types of the non-terminal symbols */
%type <yt_activity_list> activity_list
%type <yt_activity> activity
%type <yt_integer> arbitraryAttributes
%type <yt_standardElements> standardElements
%type <yt_tAssign> tAssign
%type <yt_tCase_list> tCase_list
%type <yt_tCase> tCase
%type <yt_tCatch_list> tCatch_list
%type <yt_tCatch> tCatch
%type <yt_tCatchAll> tCatchAll
%type <yt_tCompensate> tCompensate
%type <yt_tCompensationHandler> tCompensationHandler
%type <yt_tCopy_list> tCopy_list
%type <yt_tCopy> tCopy
%type <yt_tCorrelation_list> tCorrelation_list
%type <yt_tCorrelation_list> tCorrelations
%type <yt_tCorrelation> tCorrelation
%type <yt_tCorrelationSet_list> tCorrelationSet_list
%type <yt_tCorrelationSet_list> tCorrelationSets
%type <yt_tCorrelationSet> tCorrelationSet
%type <yt_tEmpty> tEmpty
%type <yt_tEventHandlers> tEventHandlers
%type <yt_tFaultHandlers> tFaultHandlers
%type <yt_tFlow> tFlow
%type <yt_tFrom> tFrom
%type <yt_activity> tInvoke
%type <yt_tLink_list> tLink_list
%type <yt_tLink_list> tLinks
%type <yt_tLink> tLink
%type <yt_tOnAlarm_list> tOnAlarm_list
%type <yt_tOnAlarm> tOnAlarm
%type <yt_tOnMessage_list> tOnMessage_list
%type <yt_tOnMessage> tOnMessage
%type <yt_tOtherwise> tOtherwise
%type <yt_tPartner_list> tPartner_list
%type <yt_tPartner_list> tPartners
%type <yt_tPartner> tPartner
%type <yt_tPartnerLink_list> tPartnerLink_list
%type <yt_tPartnerLink_list> tPartnerLinks
%type <yt_tPartnerLink> tPartnerLink
%type <yt_tPick> tPick
%type <yt_tProcess> tProcess
%type <yt_tReceive> tReceive
%type <yt_tReply> tReply
%type <yt_tScope> tScope
%type <yt_tSequence> tSequence
%type <yt_tSource_list> tSource_list
%type <yt_tSource> tSource
%type <yt_tSwitch> tSwitch
%type <yt_tTarget_list> tTarget_list
%type <yt_tTarget> tTarget
%type <yt_tTerminate> tTerminate
%type <yt_tThrow> tThrow
%type <yt_tTo> tTo
%type <yt_tVariable_list> tVariable_list
%type <yt_tVariable_list> tVariables
%type <yt_tVariable> tVariable
%type <yt_tWait> tWait
%type <yt_tWhile> tWhile




%%




/*
  <process name="ncname" targetNamespace="uri" 
           queryLanguage="anyURI"?
           expressionLanguage="anyURI"?
           suppressJoinFailure="yes|no"?
           enableInstanceCompensation="yes|no"?
           abstractProcess="yes|no"?
           xmlns="http://schemas.xmlsoap.org/ws/2003/03/business-process/">

    <partnerLinks>?
    <partners>?
    <variables>?
    <correlationSets>?
    <faultHandlers>?
    <compensationHandler>?
    <eventHandlers>?
    
    activity
  </process>

  The top-level attributes are as follows:

   * queryLanguage. This attribute specifies the XML query language used for
     selection of nodes in assignment, property definition, and other uses.
     The default for this attribute is XPath 1.0, represented by the URI of the
     XPath 1.0 specification: http://www.w3.org/TR/1999/REC-xpath-19991116.

   * expressionLanguage. This attribute specifies the expression language used
     in the process. The default for this attribute is XPath 1.0, represented
     by the URI of the XPath 1.0 specification:
     http://www.w3.org/TR/1999/REC-xpath-19991116.

  * suppressJoinFailure. This attribute determines whether the joinFailure
     fault will be suppressed for all activities in the process. The effect of
     the attribute at the process level can be overridden by an activity using
     a different value for the attribute. The default for this attribute is
     "no".

  * enableInstanceCompensation. This attribute determines whether the process
    instance as a whole can be compensated by platform-specific means. The
    default for this attribute is "no".

  * abstractProcess. This attribute specifies whether the process being defined
    is abstract (rather than executable). The default for this attribute is
    "no".
*/

tProcess:
  X_OPEN K_PROCESS arbitraryAttributes
    { att.check($3, K_PROCESS);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// default attribute value
      	att.pushSJFStack($3, mkcasestring("no"));
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }      
      symMan.initialiseProcessScope($3);
      currentScopeId = $3;
      isInFH.push(false);
      isInCH.push(pair<bool,int>(false,0));
      hasCompensate = 0;
   }
  X_NEXT imports tPartnerLinks tPartners tVariables tCorrelationSets tFaultHandlers tCompensationHandler tEventHandlers
    { inProcess = false; }
  activity
  X_NEXT X_SLASH K_PROCESS X_CLOSE
    { TheProcess = $$ = Process($7, $8, $9, $10, $11, $12, $13, StopInProcess(), $15);
      symMan.quitScope();
      $$->name = att.read($3, "name");
      $$->targetNamespace = att.read($3, "targetNamespace");
      $$->queryLanguage = att.read($3, "queryLanguage", $$->queryLanguage);
      $$->expressionLanguage = att.read($3, "expressionLanguage", $$->expressionLanguage);
      $$->suppressJoinFailure = att.read($3, "suppressJoinFailure", $$->suppressJoinFailure);
      att.traceAM(string("tProcess: ") + ($$->suppressJoinFailure)->name + string("\n"));      
      att.popSJFStack();
      $$->enableInstanceCompensation = att.read($3, "enableInstanceCompensation", $$->enableInstanceCompensation);
      $$->abstractProcess = att.read($3, "abstractProcess", $$->abstractProcess);
      $$->xmlns = att.read($3, "xmlns", $$->xmlns);
      isInFH.pop();
      isInCH.pop();
      $$->id = $3; }
;

/* import other namespaces */
imports:
  /* empty */
| K_IMPORT arbitraryAttributes X_SLASH X_NEXT imports
;

/*---------------------------------------------------------------------------*/


/*
  The token "activity" can be any of the following:

  Basic activities:
    * <receive>
    * <reply>
    * <invoke>
    * <assign>
    * <throw>
    * <wait>
    * <empty>
    * <terminate>

  Structured activities:
    * <sequence>
    * <switch>
    * <while>
    * <pick>
    * <flow>

  Other activities:
    * <scope>
    * <compensate>
*/

activity:
  tEmpty
    { $$ = activityEmpty($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tInvoke
    { $$ = $1;  }
/*
| tInvoke
    { $$ = activityInvoke($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
*/
| tReceive
    { $$ = activityReceive($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tReply
    { $$ = activityReply($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tAssign
    { $$ = activityAssign($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tWait
    { $$ = activityWait($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tThrow
    { $$ = activityThrow($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tTerminate
    { $$ = activityTerminate($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tFlow
    { $$ = activityFlow($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tSwitch
    { $$ = activitySwitch($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tWhile
    { $$ = activityWhile($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tSequence
    { $$ = activitySequence($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tPick
    { $$ = activityPick($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tScope
    { $$ = activityScope($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tCompensate
    { $$ = activityCompensate($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
;


/******************************************************************************
  PARTNER LINKS
******************************************************************************/

/*
  The services with which a business process interacts are modeled as partner
  links in BPEL4WS. Each partner link is characterized by a partnerLinkType.
  More than one partner link can be characterized by the same partnerLinkType.
  For example, a certain procurement process might use more than one vendor
  for its transactions, but might use the same partnerLinkType for all vendors.

  <partnerLinks>
    <partnerLink name="ncname" partnerLinkType="qname" 
             myRole="ncname"? partnerRole="ncname"?>+
    </partnerLink>
  </partnerLinks>
*/

tPartnerLinks:
  /* empty */
    { $$ = NiltPartnerLink_list(); }
| K_PARTNERLINKS X_NEXT tPartnerLink_list X_SLASH K_PARTNERLINKS X_NEXT
    { $$ = $3; } 
;

tPartnerLink_list:
  tPartnerLink X_NEXT
    { $$ = ConstPartnerLink_list($1, NiltPartnerLink_list()); }
| tPartnerLink X_NEXT tPartnerLink_list
    { $$ = ConstPartnerLink_list($1, $3); }
;

tPartnerLink:
  K_PARTNERLINK arbitraryAttributes X_NEXT X_SLASH K_PARTNERLINK
    { $$ = PartnerLink();
      $$->name = att.read($2, "name");
      $$->partnerLinkType = att.read($2, "partnerLinkType");
      $$->myRole = att.read($2, "myRole");
      $$->partnerRole = att.read($2, "partnerRole"); 
      if (inPartners) {
        symMan.checkPartnerLink($$->name->name);
      }
      else
      {
        symMan.addPartnerLink(new csPartnerLink($$->name->name, $$->partnerLinkType->name, 
	  		                        $$->myRole->name, $$->partnerRole->name)); 
      }
    }
| K_PARTNERLINK arbitraryAttributes X_SLASH
    { $$ = PartnerLink();
      $$->name = att.read($2, "name");
      $$->partnerLinkType = att.read($2, "partnerLinkType");
      $$->myRole = att.read($2, "myRole");
      $$->partnerRole = att.read($2, "partnerRole");
      if (inPartners) {
        symMan.checkPartnerLink($$->name->name);
      }
      else
      {
        symMan.addPartnerLink(new csPartnerLink($$->name->name, $$->partnerLinkType->name, 
	  		                        $$->myRole->name, $$->partnerRole->name)); 
      }
    }
;


/******************************************************************************
  PARTNERS
******************************************************************************/

/*
  While a partner link represents a conversational relationship between two
  partner processes, relationships with a business partner in general require
  more than a single conversational relationship to be established. To
  represent the capabilities required from a business partner, BPEL4WS uses the
  partner element. 

  <partners>
    <partner name="ncname">+
      <partnerLink name="ncname"/>+
    </partner>
  </partners>
*/

tPartners:
  /* empty */
    { $$ = NiltPartner_list(); }
| K_PARTNERS X_NEXT
    { inPartners = true; }
  tPartner_list X_SLASH K_PARTNERS X_NEXT
    { $$ = $4;
      inPartners = false; }
;

tPartner_list:
  tPartner X_NEXT
    { $$ = ConstPartner_list($1, NiltPartner_list()); }
| tPartner X_NEXT tPartner_list
    { $$ = ConstPartner_list($1, $3); }
;

tPartner:
  K_PARTNER arbitraryAttributes X_NEXT tPartnerLink_list X_SLASH K_PARTNER
    { $$ = Partner($4);
      $$->name = att.read($2, "name"); }
| K_PARTNER arbitraryAttributes X_SLASH
    { $$ = Partner(NiltPartnerLink_list());
      $$->name = att.read($2, "name"); }
;







/******************************************************************************
  FAULT HANDLERS
******************************************************************************/

/*
  Fault handling in a business process can be thought of as a mode switch from
  the normal processing in a scope. Fault handling in BPEL4WS is always treated
  as "reverse work" in that its sole aim is to undo the partial and
  unsuccessful work of a scope in which a fault has occurred. The completion of
  the activity of a fault handler, even when it does not rethrow the fault
  handled, is never considered successful completion of the attached scope and
  compensation is never enabled for a scope that has had an associated fault
  handler invoked.

  <faultHandlers>?
    <!-- there must be at least one fault handler or default -->
    <catch faultName="qname"? faultVariable="ncname"?>*
      activity
    </catch>
    <catchAll>?
      activity
    </catchAll>
  </faultHandlers>
*/

tFaultHandlers:
  /* empty */
    { $$ = implicitFaultHandler();
      $$->inProcess = inProcess;
      $$->parentScopeId = currentScopeId; 
    }
| K_FAULTHANDLERS X_NEXT 
    {
      symMan.startDPEinWhile();
      isInFH.push(true);
      hasCompensate = 0;
      inProcess = false;
    }
  tCatch_list  
  tCatchAll 
  X_SLASH K_FAULTHANDLERS X_NEXT
    { $$ = userDefinedFaultHandler($4, $5);
      $$->inProcess = (currentScopeId->value == 1);
      inProcess = false; // hack!
      $$->parentScopeId = currentScopeId;
      isInFH.pop();
      hasCompensate = 0;
      symMan.endDPEinWhile();
    }
;

tCatch_list:
  /* empty */
    { $$ = NiltCatch_list(); }
| tCatch X_NEXT tCatch_list
    { $$ = ConstCatch_list($1, $3); }
;

tCatch:
  K_CATCH arbitraryAttributes X_NEXT activity X_NEXT X_SLASH K_CATCH
    { $$ = Catch($4);
      $$->faultName = att.read($2, "faultName");
      $$->faultVariable = att.read($2, "faultVariable"); 
      $$->variableID = symMan.checkVariable(att.read($2, "faultVariable")->name, true); 
    }
;

tCatchAll:
  /* empty */
    { $$ = NoCatchAll(); }
| K_CATCHALL arbitraryAttributes X_NEXT activity X_NEXT X_SLASH K_CATCHALL X_NEXT
    { $$ = CatchAll($4);
      $$->faultName = att.read($2, "faultName");
      $$->faultVariable = att.read($2, "faultVariable");
      // $$->variableID = symMan.checkVariable(att.read($2, "faultVariable")->name, true); 
    }
;






/******************************************************************************
  COMPENSATION HANDLERS
******************************************************************************/

/*
  Scopes can delineate a part of the behavior that is meant to be reversible
  in an application-defined way by a compensation handler. Scopes with
  compensation and fault handlers can be nested without constraint to arbitrary
  depth.

  <compensationHandler>?
    activity
  </compensationHandler>
*/

tCompensationHandler:
  /* empty */
    { $$ = implicitCompensationHandler();
      $$->parentScopeId = currentScopeId; }
| K_COMPENSATIONHANDLER X_NEXT 
    {
      symMan.startDPEinWhile();
      isInCH.push(pair<bool,int>(true,hasCompensate));
      hasCompensate = 0;
    }
  activity 
  X_NEXT X_SLASH K_COMPENSATIONHANDLER X_NEXT
    { $$ = userDefinedCompensationHandler($4);
      switch ( hasCompensate ) {
        case 1 : $$->compensateWithoutScope = true;
                 $$->compensateWithScope = false;
                 break;
        case 2 : $$->compensateWithoutScope = false;
                 $$->compensateWithScope = true;
                 break;
        default: $$->compensateWithoutScope = false;
                 $$->compensateWithScope = false;
                 break;
      }
      hasCompensate = isInCH.top().second;
      isInCH.pop();

      $$->parentScopeId = currentScopeId; 
      symMan.endDPEinWhile();
    }
;






/******************************************************************************
  EVENT HANDLERS
******************************************************************************/

/*
  The whole process as well as each scope can be associated with a set of event
  handlers that are invoked concurrently if the corresponding event occurs. The
  actions taken within an event handler can be any type of activity, such as
  sequence or flow, but invocation of compensation handlers using the
  <compensate/> activity is not permitted.

  <eventHandlers>?
    <!-- Note: There must be at least one onMessage or onAlarm handler. -->
    <onMessage partnerLink="ncname" portType="qname"
               operation="ncname" variable="ncname"?>
      <correlations>?
        <correlation set="ncname" initiate="yes|no"?>+
      <correlations>
      activity
    </onMessage>
    <onAlarm for="duration-expr"? until="deadline-expr"?>*
      activity
    </onAlarm>
  </eventHandlers>
*/

tEventHandlers:
  /* empty */
    { $$ = implicitEventHandler();
      $$->parentScopeId = currentScopeId; }
| K_EVENTHANDLERS X_NEXT 
    {
      symMan.startDPEinWhile();
    }
  tOnMessage_list 
  tOnAlarm_list 
  X_SLASH K_EVENTHANDLERS X_NEXT
    { $$ = EventHandler($4, $5);
      $$->parentScopeId = currentScopeId; 
      symMan.endDPEinWhile();
    }
;

tOnMessage_list:
  /* empty */
    { $$ = NiltOnMessage_list(); 
      $$->dpe = kc::mkinteger(0);
    }
| tOnMessage X_NEXT tOnMessage_list
    { $$ = ConstOnMessage_list($1, $3); 
      $$->dpe = kc::mkinteger($1->dpe->value + $3->dpe->value);
    }
;

tOnAlarm_list:
  /* empty */
    { $$ = NiltOnAlarm_list(); 
      $$->dpe = kc::mkinteger(0);
    }
| tOnAlarm X_NEXT tOnAlarm_list
    { $$ = ConstOnAlarm_list($1, $3); 
      $$->dpe = kc::mkinteger($1->dpe->value + $3->dpe->value);
    }
;

tOnMessage:
  K_ONMESSAGE arbitraryAttributes X_NEXT
    {
      symMan.checkPartnerLink(att.read($2, "partnerLink")->name);
      symMan.resetDPEend();
    }
  tCorrelations activity X_NEXT X_SLASH K_ONMESSAGE
    { att.check($2, K_ONMESSAGE);
      $$ = OnMessage($6);
      $$->partnerLink = att.read($2, "partnerLink");
      $$->portType = att.read($2, "portType");
      $$->operation = att.read($2, "operation");
      $$->variableID  = symMan.checkVariable(att.read($2, "variable")->name);
      $$->variable = att.read($2, "variable"); 
      $$->channelID = symMan.addChannel(new csChannel($$->portType->name, 
				      $$->operation->name, 
				      $$->partnerLink->name), true); 
      $$->dpe = symMan.needsDPE();
    }
;

tOnAlarm:
  K_ONALARM arbitraryAttributes X_NEXT 
    {
      symMan.resetDPEend();
    }
  activity X_NEXT X_SLASH K_ONALARM 
    { att.check($2, K_ONALARM);
      $$ = OnAlarm($5);
      $$->For = att.read($2, "for");  // "for" is a keyword
      $$->until = att.read($2, "until");
      $$->dpe = symMan.needsDPE();
    }
;







/******************************************************************************
  VARIABLES
******************************************************************************/

/*
  Business processes specify stateful interactions involving the exchange of
  messages between partners. The state of a business process includes the
  messages that are exchanged as well as intermediate data used in business
  logic and in composing messages sent to partners.
  Variables provide the means for holding messages that constitute the state
  of a business process. The messages held are often those that have been
  received from partners or are to be sent to partners.Variables can also hold
  data that are needed for holding state related to the process and never
  exchanged with partners.

  <variables>
    <variable name="ncname" messageType="qname"?
                type="qname"? element="qname"?/>+
  </variables>
*/

tVariables:
  /* empty */
    { $$ = NiltVariable_list(); }
| K_VARIABLES X_NEXT tVariable_list X_SLASH K_VARIABLES X_NEXT
    { $$ = $3; }
;

tVariable_list:
  tVariable X_NEXT
    { $$ = ConstVariable_list($1, NiltVariable_list()); }
| tVariable X_NEXT tVariable_list
    { $$ = ConstVariable_list($1, $3); }
;

tVariable:
  K_VARIABLE arbitraryAttributes X_NEXT X_SLASH K_VARIABLE
    { $$ = Variable();
      $$->name = att.read($2, "name");
      $$->messageType = att.read($2, "messageType");
      $$->type = att.read($2, "type");
      $$->element = att.read($2, "element"); 
      $$->variableID = symMan.addVariable( 
	              new csVariable($$->name->name, 
				     $$->messageType->name, 
				     $$->type->name, 
				     $$->element->name)); }
| K_VARIABLE arbitraryAttributes X_SLASH
    { $$ = Variable();
      $$->name = att.read($2, "name");
      $$->messageType = att.read($2, "messageType");
      $$->type = att.read($2, "type");
      $$->element = att.read($2, "element"); 
      $$->variableID = symMan.addVariable( 
	              new csVariable($$->name->name, 
				     $$->messageType->name, 
				     $$->type->name, 
				     $$->element->name)); }
;


/******************************************************************************
  CORRELATION SETS
******************************************************************************/

/*
  Each correlation set in BPEL4WS is a named group of properties that, taken
  together, serve to define a way of identifying an application-level
  conversation within a business protocol instance. A given message can carry
  multiple correlation sets. After a correlation set is initiated, the values
  of the properties for a correlation set must be identical for all the
  messages in all the operations that carry the correlation set and occur
  within the corresponding scope until its completion. The semantics of a
  process in which this consistency constraint is violated is undefined.
  Similarly undefined is the semantics of a process in which an activity
  with the initiate attribute set to no attempts to use a correlation set
  that has not been previously initiated.

  <correlationSets>?
    <correlationSet name="ncname" properties="qname-list"/>+
  </correlationSets>
*/

tCorrelationSets:
  /* empty */
    { $$ = NiltCorrelationSet_list(); }
| K_CORRELATIONSETS X_NEXT tCorrelationSet_list X_SLASH K_CORRELATIONSETS X_NEXT
    { $$ = $3; }
;

tCorrelationSet_list:
  tCorrelationSet X_NEXT
    { $$ = ConstCorrelationSet_list($1, NiltCorrelationSet_list()); }
| tCorrelationSet X_NEXT tCorrelationSet_list
    { $$ = ConstCorrelationSet_list($1, $3); }
;

tCorrelationSet:
  K_CORRELATIONSET arbitraryAttributes X_NEXT X_SLASH K_CORRELATIONSET
    { $$ = CorrelationSet(); }
| K_CORRELATIONSET arbitraryAttributes X_SLASH
    { $$ = CorrelationSet();
      $$->properties = att.read($2, "properties");
      $$->name = att.read($2, "name"); }
;


/******************************************************************************
  CORRELATIONS
******************************************************************************/

tCorrelations:
  /* empty */
    { $$ = NiltCorrelation_list(); }
| K_CORRELATIONS X_NEXT tCorrelation_list X_SLASH K_CORRELATIONS X_NEXT
    { $$ = $3; }
;

tCorrelation_list:
  tCorrelation X_NEXT
    { $$ = ConstCorrelation_list($1, NiltCorrelation_list());
      if ($1->initiate = mkcasestring("yes"))
        $$->initiateCorrelationSet = true; }
| tCorrelation X_NEXT tCorrelation_list
    { $$ = ConstCorrelation_list($1, $3);
      if ($1->initiate = mkcasestring("yes"))
        $$->initiateCorrelationSet = true; }
;

tCorrelation:
  K_CORRELATION arbitraryAttributes X_NEXT X_SLASH K_CORRELATION
    { $$ = Correlation();
      $$->set = att.read($2, "set");
      $$->initiate = att.read($2, "initiate", $$->initiate);
      $$->pattern = att.read($2, "pattern"); }
| K_CORRELATION arbitraryAttributes X_SLASH
    { $$ = Correlation();
      $$->set = att.read($2, "set");
      $$->initiate = att.read($2, "initiate", $$->initiate);
      $$->pattern = att.read($2, "pattern"); }
;







/******************************************************************************
  EMPTY
******************************************************************************/

/*
  The <empty> construct allows you to insert a "no-op" instruction into a
  business process. This is useful for synchronization of concurrent
  activities, for instance.

  <empty standard-attributes>
    standard-elements
  </empty>
*/

tEmpty:
  K_EMPTY 
  arbitraryAttributes 
    { 
      att.check($2, K_EMPTY);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_EMPTY
    { $$ = Empty($5);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");
      att.traceAM(string("tEmpty: ") + (att.read($2, "suppressJoinFailure"))->name + string("\n"));
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tEmpty: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; 
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
    }
| K_EMPTY 
  arbitraryAttributes 
    { 
      att.check($2, K_EMPTY);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Empty(noLinks);
      $$->name = att.read($2, "name");
      $$->joinCondition = att.read($2, "joinCondition");
      $$->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tEmpty: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;


/******************************************************************************
  INVOKE
******************************************************************************/

/*
  The <invoke> construct allows the business process to invoke a one-way or
  request-response operation on a portType offered by a partner.

  <invoke partnerLink="ncname" portType="qname" operation="ncname"
          inputVariable="ncname"? outputVariable="ncname"?
          standard-attributes>
    standard-elements
    <correlations>?
    <catch>*
    <catchAll>?
    <compensationHandler>?
  </invoke>

  Attributes:
  * partnerLink: name of a partner declared in the script to who send a message
    and optionally receive a response.
  * portType: name of the 'port' as declared in corresponding WSDL file.
  * operation: name of the operation to invoke
  * inputVariable: name of the variable whose value will be used as the message
    to the partner.
  * outputVariable: name of the variable to which the response will be assigned. 
*/

tInvoke:
  K_INVOKE 
  arbitraryAttributes 
    { 
      att.check($2, K_INVOKE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT
    {
      // automatically create scope?
      symMan.checkPartnerLink(att.read($2, "partnerLink")->name);
      isInFH.push(false);
      isInCH.push(pair<bool,int>(false,hasCompensate));
      parent[$2] = currentScopeId;
      currentScopeId = $2; 
    }
  standardElements 
  tCorrelations tCatch_list  tCatchAll tCompensationHandler X_SLASH K_INVOKE
    { 
      isInFH.pop();
      hasCompensate = isInCH.top().second;
      isInCH.pop();
      if ($8->length() > 0 
           || (string($9->op_name()) != "NoCatchAll") 
           || string($10->op_name()) != "implicitCompensationHandler")
      {
        //cerr << "embed in scope" << endl;
        standardElements se =  StandardElements(NiltTarget_list(), NiltSource_list());
        tInvoke invoke = Invoke(se, $7);
        activity ai = activityInvoke(invoke);
        tFaultHandlers fh = userDefinedFaultHandler($8, $9);
        tEventHandlers eh = implicitEventHandler();
        tScope scope = Scope($6, NiltVariable_list(), fh, $10, eh, StopInScope(), ai);

        fh->inProcess = false;
        fh->parentScopeId = $2;
        $10->parentScopeId = $2;
        eh->parentScopeId = $2;

        symMan.newScopeScope($2);
        symMan.quitScope();

	invoke->name = att.read($2, "name");
        scope->name = att.read($2, "name");
        scope->joinCondition = invoke->joinCondition = $6->joinCondition = att.read($2, "joinCondition");
        scope->suppressJoinFailure = invoke->suppressJoinFailure = $6->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
        scope->variableAccessSerializable = mkcasestring("");
        att.traceAM(string("tInvoke: ") + (invoke->suppressJoinFailure)->name + string("\n"));
	att.popSJFStack();
	invoke->partnerLink = att.read($2, "partnerLink");
        invoke->portType = att.read($2, "portType");
        invoke->operation = att.read($2, "operation");
        invoke->inputVariable = att.read($2, "inputVariable");
        invoke->outputVariable = att.read($2, "outputVariable");
        invoke->variableIDin  = symMan.checkVariable(att.read($2, "inputVariable")->name);
        invoke->variableIDout = symMan.checkVariable(att.read($2, "outputVariable")->name);
        if (string(invoke->variableIDin->name) != "")
        {
          invoke->channelID = symMan.addChannel(new csChannel(invoke->portType->name, 
					  invoke->operation->name, 
					  invoke->partnerLink->name), true);
        }
        if (string(invoke->variableIDout->name) != "")
        {
          invoke->channelID = symMan.addChannel(new csChannel(invoke->portType->name, 
					  invoke->operation->name, 
					  invoke->partnerLink->name), false);
        }
	scope->dpe = invoke->dpe = mkinteger(0);
        // symMan.needsDPE();
	if ($6->dpe->value > 0)
        {
          symMan.addDPEend();
        }
        scope->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
        invoke->negativeControlFlow = mkinteger(0);
        scope->id = $6->parentId = $2; 
        invoke->id = ai->id = se->parentId = att.nextId();
        currentScopeId = scope->parentScopeId = parent[$2];

        $$ = activity(activityScope(scope));

        $$->id = scope->id;
        $$->negativeControlFlow = scope->negativeControlFlow;
      }
      else
      { 
        //cerr << "don't embed" << endl;
        tInvoke invoke = Invoke($6, $7);

	invoke->name = att.read($2, "name");
        invoke->joinCondition = $6->joinCondition = att.read($2, "joinCondition");
        invoke->suppressJoinFailure = $6->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
        att.traceAM(string("tInvoke: ") + (invoke->suppressJoinFailure)->name + string("\n"));
        att.popSJFStack();
        invoke->partnerLink = att.read($2, "partnerLink");
        invoke->portType = att.read($2, "portType");
        invoke->operation = att.read($2, "operation");
        invoke->inputVariable = att.read($2, "inputVariable");
        invoke->outputVariable = att.read($2, "outputVariable");
        invoke->variableIDin  = symMan.checkVariable(att.read($2, "inputVariable")->name);
        invoke->variableIDout = symMan.checkVariable(att.read($2, "outputVariable")->name);
        if (string(invoke->variableIDin->name) != "")
        {
          invoke->channelID = symMan.addChannel(new csChannel(invoke->portType->name, 
					  invoke->operation->name, 
					  invoke->partnerLink->name), true);
        }
        if (string(invoke->variableIDout->name) != "")
        {
          invoke->channelID = symMan.addChannel(new csChannel(invoke->portType->name, 
					  invoke->operation->name, 
					  invoke->partnerLink->name), false);
        }
        invoke->dpe = symMan.needsDPE();
        if ($6->dpe->value > 0)
        {
          symMan.addDPEend();
        }
        invoke->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
        invoke->id = $6->parentId = $2; 

        $$ = activity(activityInvoke(invoke));
        $$->id = invoke->id;
        $$->negativeControlFlow = invoke->negativeControlFlow;
      }    
    }
| K_INVOKE 
  arbitraryAttributes 
    { 
      att.check($2, K_INVOKE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { 
      impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      tInvoke invoke = Invoke(noLinks, NiltCorrelation_list());
      invoke->name = att.read($2, "name");
      invoke->joinCondition = att.read($2, "joinCondition");    
      invoke->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tInvoke: ") + (invoke->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      invoke->partnerLink = att.read($2, "partnerLink");
      invoke->portType = att.read($2, "portType");
      invoke->operation = att.read($2, "operation");
      invoke->inputVariable = att.read($2, "inputVariable");
      invoke->outputVariable = att.read($2, "outputVariable"); 
      invoke->variableIDin  = symMan.checkVariable(att.read($2, "inputVariable")->name);
      invoke->variableIDout = symMan.checkVariable(att.read($2, "outputVariable")->name);
      symMan.checkPartnerLink(invoke->partnerLink->name); 
      if (string(invoke->variableIDin->name) != "")
      {
        invoke->channelID = symMan.addChannel(new csChannel(invoke->portType->name, 
					invoke->operation->name, 
					invoke->partnerLink->name), true);
      }
      if (string(invoke->variableIDout->name) != "")
      {
        invoke->channelID = symMan.addChannel(new csChannel(invoke->portType->name, 
					invoke->operation->name, 
					invoke->partnerLink->name), false);
      }
      invoke->dpe = kc::mkinteger(0);
      invoke->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      invoke->id = $2; 

      $$ = activity(activityInvoke(invoke));
      $$->id = invoke->id;
      $$->negativeControlFlow = invoke->negativeControlFlow;
}
;


/******************************************************************************
  RECEIVE
******************************************************************************/

/*
  The <receive> construct allows the business process to do a blocking wait for
  a matching message to arrive.

  <receive partnerLink="ncname" portType="qname" operation="ncname"
           variable="ncname"? createInstance="yes|no"?
           standard-attributes>
    standard-elements
    <correlations>?
      <correlation set="ncname" initiate="yes|no"?>+
    </correlations>
  </receive>

  Attributes:
  * partnerLink: name of a partner declared in the script from which the
    process is to receive a message.
  * portType: name of the 'port' as declared in corresponding WSDL file.
  * operation: name of the operation
  * variable: name of the variable to which the received message will be
    assigned.
  * createInstance: used to make instance of the BPEL process and start its
    execution. 
*/

tReceive:
  K_RECEIVE
  arbitraryAttributes
    { 
      att.check($2, K_RECEIVE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT
    { symMan.checkPartnerLink(att.read($2, "partnerLink")->name); }
  standardElements tCorrelations X_SLASH K_RECEIVE
    { $$ = Receive($6, $7);
      $$->name = att.read($2, "name");
      $$->joinCondition = $6->joinCondition = att.read($2, "joinCondition");    
      $$->suppressJoinFailure = $6->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tReceive: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->partnerLink = att.read($2, "partnerLink");
      $$->portType = att.read($2, "portType");
      $$->operation = att.read($2, "operation");
      $$->variable = att.read($2, "variable");
      $$->createInstance = att.read($2, "createInstance", $$->createInstance); 
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      $$->channelID = symMan.addChannel(new csChannel($$->portType->name, 
				      $$->operation->name, 
				      $$->partnerLink->name), true);
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; }
| K_RECEIVE
  arbitraryAttributes
    { 
      att.check($2, K_RECEIVE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Receive(noLinks, NiltCorrelation_list());
      $$->name = att.read($2, "name");
      $$->joinCondition = att.read($2, "joinCondition");     
      $$->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tReceive: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->partnerLink = att.read($2, "partnerLink");
      $$->portType = att.read($2, "portType");
      $$->operation = att.read($2, "operation");
      $$->variable = att.read($2, "variable");
      $$->createInstance = att.read($2, "createInstance", $$->createInstance);
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; 
      symMan.checkPartnerLink($$->partnerLink->name); 
      $$->channelID = symMan.addChannel(new csChannel($$->portType->name, 
				      $$->operation->name, 
				      $$->partnerLink->name), true); 
    }
;


/******************************************************************************
  REPLY
******************************************************************************/

/*
  The <reply> construct allows the business process to send a message in reply
  to a message that was received through a <receive>. The combination of a
  <receive> and a <reply> forms a request-response operation on the WSDL
  portType for the process.

  <reply partnerLink="ncname" portType="qname" operation="ncname"
         variable="ncname"? faultName="qname"?
         standard-attributes>
    standard-elements
    <correlations>?
       <correlation set="ncname" initiate="yes|no"?>+
    </correlations>
  </reply>

  Attributes:
  * partnerLink: name of a partner declared in the script to which to send a
    message.
  * portType: name of the 'port' as declared in corresponding WSDL file.
  * operation: name of the operation
  * variable: name of the variable whose value will be used as output message.
  * faultName

*/

tReply:
  K_REPLY
  arbitraryAttributes
    { 
      att.check($2, K_REPLY);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT 
  standardElements 
  tCorrelations
  X_SLASH K_REPLY
    { $$ = Reply($5, $6);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tReply: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->partnerLink = att.read($2, "partnerLink");
      $$->portType = att.read($2, "portType");
      $$->operation = att.read($2, "operation");
      $$->variable = att.read($2, "variable");
      $$->faultName = att.read($2, "faultName");
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      symMan.checkPartnerLink($$->partnerLink->name); 
      $$->channelID = symMan.addChannel(new csChannel($$->portType->name, 
				      $$->operation->name, 
				      $$->partnerLink->name), false);
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; }
| K_REPLY
  arbitraryAttributes
    { 
      att.check($2, K_REPLY);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Reply(noLinks, NiltCorrelation_list());
      $$->name = att.read($2, "name");
      $$->joinCondition = att.read($2, "joinCondition");    
      $$->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tReply: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->partnerLink = att.read($2, "partnerLink");
      $$->portType = att.read($2, "portType");
      $$->operation = att.read($2, "operation");
      $$->variable = att.read($2, "variable");
      $$->faultName = att.read($2, "faultName");
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      symMan.checkPartnerLink($$->partnerLink->name); 
      $$->channelID = symMan.addChannel(new csChannel($$->portType->name, 
				      $$->operation->name, 
				      $$->partnerLink->name), false);
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;


/******************************************************************************
  ASSIGN
******************************************************************************/

/*
  The <assign> construct can be used to update the values of variables with new
  data. An <assign> construct can contain any number of elementary assignments.
  The syntax of the assignment activity is:

  <assign standard-attributes>
    standard-elements
    <copy>+
      from-spec
      to-spec
    </copy>
  </assign>
*/

tAssign:
  K_ASSIGN 
  arbitraryAttributes
    { 
      att.check($2, K_ASSIGN);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements tCopy_list  X_SLASH K_ASSIGN
    { $$ = Assign($5, $6);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");     
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tAssign: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; }
;

tCopy_list:
  tCopy X_NEXT
    { $$ = ConstCopy_list($1, NiltCopy_list()); }
| tCopy X_NEXT tCopy_list
    { $$ = ConstCopy_list($1, $3); }
;

tCopy:
  K_COPY X_NEXT tFrom X_NEXT tTo X_NEXT X_SLASH K_COPY
    { $$ = Copy($3, $5); }
; 

/*
  The from-spec MUST be one of the following forms except for the opaque form
  available in abstract processes:

  <from variable="ncname" part="ncname"?/>
  <from partnerLink="ncname" endpointReference="myRole|partnerRole"/>
  <from variable="ncname" property="qname"/>
  <from expression="general-expr"/>
  <from> ... literal value ... </from>
*/

tFrom:
  K_FROM arbitraryAttributes X_NEXT X_SLASH K_FROM
    { att.check($2, K_FROM);
      $$ = From();
      $$->variable = att.read($2, "variable");
      $$->part = att.read($2, "part");
      $$->query = att.read($2, "query");
      $$->partnerLink = att.read($2, "partnerLink");
      $$->endpointReference = att.read($2, "endpointReference");
      $$->property = att.read($2, "property");
      $$->expression = att.read($2, "expression");
      $$->opaque = att.read($2, "opaque"); 
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      symMan.checkPartnerLink($$->partnerLink->name); }
| K_FROM arbitraryAttributes X_CLOSE X_NAME X_OPEN X_SLASH K_FROM
    { att.check($2, $4, K_FROM);
      $$ = From();
      $$->variable = att.read($2, "variable");
      $$->part = att.read($2, "part");
      $$->query = att.read($2, "query");
      $$->partnerLink = att.read($2, "partnerLink");
      $$->endpointReference = att.read($2, "endpointReference");
      $$->property = att.read($2, "property");
      $$->expression = att.read($2, "expression");
      $$->opaque = att.read($2, "opaque");
      $$->literalValue = $4;
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      symMan.checkPartnerLink($$->partnerLink->name); }
| K_FROM arbitraryAttributes X_SLASH
    { att.check($2, K_FROM);
      $$ = From();
      $$->variable = att.read($2, "variable");
      $$->part = att.read($2, "part");
      $$->query = att.read($2, "query");
      $$->partnerLink = att.read($2, "partnerLink");
      $$->endpointReference = att.read($2, "endpointReference");
      $$->property = att.read($2, "property");
      $$->expression = att.read($2, "expression");
      $$->opaque = att.read($2, "opaque"); 
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      symMan.checkPartnerLink($$->partnerLink->name); }
;

/*
  The to-spec MUST be one of the following forms:

  <to variable="ncname" part="ncname"?/>
  <to partnerLink="ncname"/>
  <to variable="ncname" property="qname"/>  
*/

tTo:
  K_TO arbitraryAttributes X_NEXT X_SLASH K_TO
    { att.check($2, K_TO);
      $$ = To();
      $$->variable = att.read($2, "variable");
      $$->part = att.read($2, "part");
      $$->partnerLink = att.read($2, "partnerLink");
      $$->property = att.read($2, "property"); 
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      symMan.checkPartnerLink($$->partnerLink->name); }
| K_TO arbitraryAttributes X_SLASH
    { att.check($2, K_TO);
      $$ = To();
      $$->variable = att.read($2, "variable");
      $$->part = att.read($2, "part");
      $$->partnerLink = att.read($2, "partnerLink");
      $$->property = att.read($2, "property"); 
      $$->variableID = symMan.checkVariable(att.read($2, "variable")->name);
      symMan.checkPartnerLink($$->partnerLink->name); }
;


/******************************************************************************
  WAIT
******************************************************************************/

/*
  The <wait> construct allows you to wait for a given time period or until a
  certain time has passed. Exactly one of the expiration criteria must be
  specified.

  <wait (for="duration-expr" | until="deadline-expr") standard-attributes>
    standard-elements
  </wait>

  Attributes:
  * for: a duration expression as defined in XMLSchema (for example PT10S)
  * until: a date time expression as defined in XMLSchema
*/

tWait:
  K_WAIT
  arbitraryAttributes
    { 
      att.check($2, K_WAIT);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_WAIT
    { $$ = Wait($5);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");     
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tWait: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();      
      $$->For = att.read($2, "for"); // "for" is a keyword
      $$->until = att.read($2, "until");
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; }
| K_WAIT
  arbitraryAttributes
    { 
      att.check($2, K_WAIT);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Wait(noLinks);
      $$->name = att.read($2, "name");
      $$->joinCondition = att.read($2, "joinCondition");    
      $$->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tWait: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();      
      $$->For = att.read($2, "for"); // "for" is a keyword
      $$->until = att.read($2, "until");
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;


/******************************************************************************
  THROW
******************************************************************************/

/*
  The <throw> construct generates a fault from inside the business process.

  <throw faultName="qname" faultVariable="ncname"? standard-attributes>
    standard-elements
  </throw>

  Attributes:
  * faultName: the fault code to be thrown.
  * faultVariable
*/

tThrow:
  K_THROW
  arbitraryAttributes
    { 
      att.check($2, K_THROW);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_THROW
    { att.check($2, K_THROW);
      $$ = Throw($5);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");    
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tThrow: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();      
      $$->faultName = att.read($2, "faultName");
      $$->faultVariable = att.read($2, "faultVariable");
      $$->variableID = symMan.checkVariable(att.read($2, "faultVariable")->name);
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; }
| K_THROW
  arbitraryAttributes
    { 
      att.check($2, K_THROW);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Throw(noLinks);
      $$->name = att.read($2, "name");
      $$->joinCondition = att.read($2, "joinCondition");   
      $$->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tThrow: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->faultName = att.read($2, "faultName");
      $$->faultVariable = att.read($2, "faultVariable");
      $$->variableID = symMan.checkVariable(att.read($2, "faultVariable")->name);
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;


/******************************************************************************
  COMPENSATE
******************************************************************************/

/*
  The <compensate> construct is used to invoke compensation on an inner scope
  that has already completed normally. This construct can be invoked only from
  within a fault handler or another compensation handler.

  <compensate scope="ncname"? standard-attributes>
    standard-elements
  </compensate>
*/

tCompensate:
  K_COMPENSATE
  arbitraryAttributes
    { 
      att.check($2, K_COMPENSATE);

      // compensate only is allowed within Compensation- or FaultHandler
      if ( ! (isInCH.top().first || isInFH.top()) )
      {
        yyerror(string("The activity <compensate> is only allowed with in a Compensation of FaultHandler.\n").c_str());
      }
      if(string(att.read($2, "scope")->name) == "")
      {
        if ( hasCompensate > 0)
        {
          yyerror(string("You may only define one <compensate> if you're using it without attributes.\n").c_str());
        }
        hasCompensate = 1;
      }
      else
      {
        if ( hasCompensate == 1)
        {
          yyerror(string("You may only define one <compensate> if you're using it without attributes.\n").c_str());
        }
        hasCompensate = 2;
      }

      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_COMPENSATE
    { $$ = Compensate($5);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");   
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tCompensate: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->scope = att.read($2, "scope");
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; }
| K_COMPENSATE
  arbitraryAttributes
    { 
      att.check($2, K_COMPENSATE);

      // compensate only is allowed within Compensation- or FaultHandler
      if ( ! (isInCH.top().first || isInFH.top()) )
      {
        yyerror(string("The activity <compensate> is only allowed with in a Compensation of FaultHandler.\n").c_str());
      }
      if(string(att.read($2, "scope")->name) == "")
      {
        if ( hasCompensate > 0)
        {
          yyerror(string("You may only define one <compensate> if you're using it without attributes.\n").c_str());
        }
        hasCompensate = 1;
      }
      else
      {
        if ( hasCompensate == 1)
        {
          yyerror(string("You may only define one <compensate> if you're using it without attributes.\n").c_str());
        }
        hasCompensate = 2;
      }

      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Compensate(noLinks);
      $$->name = att.read($2, "name");
      $$->joinCondition = att.read($2, "joinCondition");     
      $$->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tCompensate: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->scope = att.read($2, "scope");
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;


/******************************************************************************
  TERMINATE
******************************************************************************/

/*
  The terminate activity can be used to immediately terminate the behavior of a
  business process instance within which the terminate activity is performed.
  All currently running activities MUST be terminated as soon as possible
  without any fault handling or compensation behavior.

  <terminate standard-attributes>
    standard-elements
  </terminate>
*/

tTerminate:
  K_TERMINATE
  arbitraryAttributes
    { 
      att.check($2, K_TERMINATE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_TERMINATE
    { $$ = Terminate($5);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");    
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tTerminate: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; }
| K_TERMINATE
  arbitraryAttributes
    { 
      att.check($2, K_TERMINATE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Terminate(noLinks);
      $$->name = att.read($2, "name");
      $$->joinCondition = att.read($2, "joinCondition");     
      $$->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tTerminate: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;


/******************************************************************************
  FLOW
******************************************************************************/

/*
  The <flow> construct allows you to specify one or more activities to be
  performed concurrently. Links can be used within concurrent activities to
  define arbitrary control structures.

  <flow standard-attributes>
    standard-elements
    <links>?
      <link name="ncname">+
    </links>
    activity+
  </flow>
*/

tFlow:
  K_FLOW
  arbitraryAttributes
    { 
      att.check($2, K_FLOW);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }   
  X_NEXT
    {
      symMan.newFlowScope($2);
    } 
  standardElements tLinks activity_list X_SLASH K_FLOW
    { $$ = Flow($6, $7, $8);
      $$->name = att.read($2, "name");
      $$->joinCondition = $6->joinCondition = att.read($2, "joinCondition");     
      $$->suppressJoinFailure = $6->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tFlow: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->dpe = symMan.needsDPE();
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2;
      symMan.checkLinks();
      symMan.quitScope(); }
;

activity_list:
  activity X_NEXT
    { $$ = Consactivity_list($1, Nilactivity_list()); }
| activity X_NEXT activity_list
    { $$ = Consactivity_list($1, $3); }
;

tLinks:
  /* empty */
    { $$ = NiltLink_list(); }
| K_LINKS arbitraryAttributes X_NEXT tLink_list X_SLASH K_LINKS X_NEXT
    { $$ = $4; }
;

tLink_list:
  tLink X_NEXT
    { $$ = ConstLink_list($1, NiltLink_list()); }
| tLink X_NEXT tLink_list
    { $$ = ConstLink_list($1, $3); }
;

tLink:
  K_LINK arbitraryAttributes X_NEXT X_SLASH K_LINK
    { $$ = Link();
      $$->name = att.read($2, "name"); 
      $$->linkID = symMan.addLink(new csLink($$->name->name)); }
| K_LINK arbitraryAttributes X_SLASH
    { $$ = Link();
      $$->name = att.read($2, "name"); 
      $$->linkID = symMan.addLink(new csLink($$->name->name)); }
;


/******************************************************************************
  SWITCH
******************************************************************************/

/*
  The <switch> construct allows you to select exactly one branch of activity
  from a set of choices.

  <switch standard-attributes>
    standard-elements
    <case condition="bool-expr">+
      activity
    </case>
    <otherwise>?
      activity
    </otherwise>
  </switch>
*/

tSwitch:
  K_SWITCH
  arbitraryAttributes
    { 
      att.check($2, K_SWITCH);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    } 
  X_NEXT 
  standardElements 
    {
      symMan.addDPEstart();
    }
  tCase_list 
  tOtherwise 
  X_SLASH K_SWITCH
    { $$ = Switch($5, $7, $8);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tSwitch: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      symMan.remDPEstart();
      $$->dpe = symMan.needsDPE();
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2;
    }
;

tCase_list:
  tCase X_NEXT
    { $$ = ConstCase_list($1, NiltCase_list()); 
      $$->dpe = $1->dpe;
    }
| tCase X_NEXT tCase_list
    { $$ = ConstCase_list($1, $3); 
      $$->dpe = kc::mkinteger($1->dpe->value + $3->dpe->value);
    }
;

tCase:
  K_CASE arbitraryAttributes X_NEXT 
    {
      // since we descend, set DPE ends to 0
      symMan.resetDPEend();
    }
  activity 
  X_NEXT X_SLASH K_CASE
    { att.check($2, K_CASE);
      $$ = Case($5);
      $$->condition = att.read($2, "condition"); 
      $$->dpe = symMan.needsDPE();
    }
;



tOtherwise:
  /* If the otherwise branch is not explicitly specified, then an otherwise
     branch with an empty activity is deemed to be present. */
    { // creaty empty activit with id, without links etc.
      integer id = att.nextId();
      impl_standardElements_StandardElements* noLinks = StandardElements(NiltTarget_list(),NiltSource_list());
      noLinks->parentId = id;
      impl_tEmpty_Empty* implicitEmpty = Empty(noLinks);
      implicitEmpty->id = id;
      implicitEmpty->negativeControlFlow = noLinks->negativeControlFlow = mkinteger(0);
//      implicitEmpty->dpe = mkinteger(0);
      implicitEmpty->name = mkcasestring("implicit empty in otherwise");
      impl_activity *otherwiseActivity = activityEmpty(implicitEmpty);
      otherwiseActivity->id = id;

      $$ = Otherwise(otherwiseActivity);
      $$->dpe = kc::mkinteger(0);
    }
| K_OTHERWISE X_NEXT activity X_NEXT X_SLASH K_OTHERWISE X_NEXT
    { $$ = Otherwise($3); 
      $$->dpe = symMan.needsDPE();
    }
;


/******************************************************************************
  WHILE
******************************************************************************/

/*
  The <while> construct allows you to indicate that an activity is to be
  repeated until a certain success criteria has been met.

  <while condition="bool-expr" standard-attributes>
     standard-elements
     activity
  </while>

  Attributes:
  * condition: an XPath expression which will be evaluated every time before
    contained activities. If this evaluates to false the loop finishes.
*/

tWhile:
  K_WHILE
  arbitraryAttributes
    { 
      att.check($2, K_WHILE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }    
  X_NEXT 
  standardElements 
    { 
      symMan.startDPEinWhile();
    }
  activity 
  X_NEXT X_SLASH K_WHILE
    { att.check($2, K_WHILE);
      $$ = While($5, $7);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");     
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure", (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tWhile: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->condition = att.read($2, "condition");
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; 
      symMan.endDPEinWhile();
    }
;


/******************************************************************************
  SEQUENCE
******************************************************************************/

/*
  The <sequence> construct allows you to define a collection of activities to
  be performed sequentially in lexical order.

  <sequence standard-attributes>
    standard-elements
    activity+
  </sequence>
*/

tSequence:
  K_SEQUENCE
  arbitraryAttributes 
    { 
      att.check($2, K_SEQUENCE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }
  X_NEXT 
  standardElements 
  activity_list 
  X_SLASH 
  K_SEQUENCE
    { $$ = Sequence($5, $6);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure", (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tSequence: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $5->parentId = $2; 
      $$->dpe = symMan.needsDPE();
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
    }
;


/******************************************************************************
  PICK
******************************************************************************/

/*
  The <pick> construct allows you to block and wait for a suitable message to
  arrive or for a time-out alarm to go off. When one of these triggers occurs,
  the associated activity is performed and the pick completes.

  <pick createInstance="yes|no"? standard-attributes>
    standard-elements
    <onMessage partnerLink="ncname" portType="qname"
               operation="ncname" variable="ncname"?>+
      <correlations>?
         <correlation set="ncname" initiate="yes|no"?>+
      </correlations>
      activity
    </onMessage>
    <onAlarm (for="duration-expr" | until="deadline-expr")>*
      activity
    </onAlarm>
  </pick>

  Attributes:
  * createInstance: This is an alternative of the 'receive' to make a new
    process instance. And can be expressed as: 'pick' plus 'onMessage' equals to
    'receive' activity.
*/

tPick:
  K_PICK
  arbitraryAttributes
    { 
      att.check($2, K_PICK);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }
  X_NEXT 
  standardElements 
    {
      symMan.addDPEstart();
    }
  tOnMessage X_NEXT 
  tOnMessage_list 
  tOnAlarm_list 
  X_SLASH K_PICK
    { $$ = Pick($5, ConstOnMessage_list($7, $9), $10);
      $$->name = att.read($2, "name");
      $$->joinCondition = $5->joinCondition = att.read($2, "joinCondition");
      $$->suppressJoinFailure = $5->suppressJoinFailure = att.read($2, "suppressJoinFailure", (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tPick: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->createInstance = att.read($2, "createInstance", $$->createInstance);
      symMan.remDPEstart();
      $$->dpe = symMan.needsDPE();
      if ($5->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->id = $5->parentId = $2;
      $$->negativeControlFlow = $5->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
    }
;





/******************************************************************************
  SCOPE
******************************************************************************/

/*
  The <scope> construct allows you to define a nested activity with its own
  associated variables, fault handlers, and compensation handler.

  <scope variableAccessSerializable="yes|no" standard-attributes>
    standard-elements

    <variables>?
    <correlationSets>?
    <faultHandlers>?
    <compensationHandler>?
    <eventHandlers>?

    activity
  </scope>
*/

tScope:
  K_SCOPE
  arbitraryAttributes
    { 
      att.check($2, K_SCOPE);
      if(att.isAttributeValueEmpty($2, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($2, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($2, att.read($2, "suppressJoinFailure"));      
      }
    }  
  X_NEXT
    { symMan.newScopeScope($2);
      symMan.setBlackListMode(true);
      isInFH.push(false);
      isInCH.push(pair<bool,int>(false,hasCompensate));
      parent[$2] = currentScopeId;
      currentScopeId = $2; }
  standardElements 
    {
      symMan.setBlackListMode(false);
    }
  tVariables 
  tCorrelationSets 
  tFaultHandlers 
  tCompensationHandler 
  tEventHandlers 
  activity 
  X_NEXT X_SLASH K_SCOPE
    { att.check($2, K_SCOPE);
      isInFH.pop();
      hasCompensate = isInCH.top().second;
      isInCH.pop();
      $$ = Scope($6, $8, $10, $11, $12, StopInScope(), $13);
      $$->name = att.read($2, "name");
      $$->joinCondition = $6->joinCondition = att.read($2, "joinCondition");
      $$->suppressJoinFailure = $6->suppressJoinFailure = att.read($2, "suppressJoinFailure", (att.topSJFStack()).getSJFValue());
      att.traceAM(string("tScope: ") + ($$->suppressJoinFailure)->name + string("\n"));
      att.popSJFStack();
      $$->variableAccessSerializable = att.read($2, "variableAccessSerializable", $$->variableAccessSerializable);
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2;
      $$->parentScopeId = currentScopeId = parent[$2];
      $$->dpe = symMan.needsDPE();
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      symMan.quitScope(); }
;





/******************************************************************************
  STANDARD ELEMENTS
******************************************************************************/

/*
  Note that the "standard-elements" referred to above are:

  <target linkName="ncname"/>*
  <source linkName="ncname" transitionCondition="bool-expr"?/>*

  where the default value of the "transitionCondition" attribute is "true()",
  the truth-value function from the default expression language XPath 1.0.
*/

standardElements:
  tTarget_list tSource_list
    { $$ = StandardElements($1, $2); 
      $$->dpe = $2->dpe;
      $$->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
    }
;

tTarget_list:
  /* empty */
    { $$ = NiltTarget_list(); 
    }
| tTarget X_NEXT tTarget_list
    { $$ = ConstTarget_list($1, $3); 
    }
;

tTarget:
  K_TARGET arbitraryAttributes X_NEXT X_SLASH K_TARGET
    { att.check($2, K_TARGET);
      $$ = Target();
      $$->linkName = att.read($2, "linkName"); 
      $$->linkID = symMan.checkLink($$->linkName->name, false); }
| K_TARGET arbitraryAttributes X_SLASH
    { att.check($2, K_TARGET);
      $$ = Target();
      $$->linkName = att.read($2, "linkName"); 
      $$->linkID = symMan.checkLink($$->linkName->name, false); }
;

tSource_list:
  /* empty */
    { $$ = NiltSource_list(); 
      $$->dpe = kc::mkinteger(0);
    }
| tSource X_NEXT tSource_list
    { $$ = ConstSource_list($1, $3);
      $$->dpe = kc::mkinteger($1->dpe->value + $3->dpe->value);
    }
;

tSource:
  K_SOURCE arbitraryAttributes X_NEXT X_SLASH K_SOURCE
    { att.check($2, K_SOURCE);
      $$ = Source();
      $$->linkName = att.read($2, "linkName");
      $$->transitionCondition = att.read($2, "transitionCondition", $$->transitionCondition); 
      $$->linkID = symMan.checkLink($$->linkName->name, true); 
      symMan.addDPEend();
      $$->dpe = symMan.needsDPE();
      symMan.remDPEend();
    }
| K_SOURCE arbitraryAttributes X_SLASH
    { att.check($2, K_SOURCE);
      $$ = Source();
      $$->linkName = att.read($2, "linkName");
      $$->transitionCondition = att.read($2, "transitionCondition", $$->transitionCondition); 
      $$->linkID = symMan.checkLink($$->linkName->name, true);
      symMan.addDPEend();
      $$->dpe = symMan.needsDPE();
      symMan.remDPEend();
    }
;


/*---------------------------------------------------------------------------*/

arbitraryAttributes:
  /* empty */
    { $$ = att.nextId(); }
| X_NAME X_EQUALS X_STRING arbitraryAttributes
    { $$ = $4;
      att.define($1, $3); }
;
