/*
	Chaf and Wino utilities for UNIX, by Michael Graffam (mgraffam@mhv.net)
	Released under the GNU public license.
	
	SHA code is in the public domain (see shs.c)
	MD5 code is copyright 1990 by RSA Data Security Inc.
	RC4 code is from the public domain implementation of the algorithm
	    this code is not santioned or licensed by RSADSI.
*/

#include <stdio.h>
#include "shs/shs.h"
#include "md2/md2.h"
#include "md5/md5.h"
#include "rc4/rc4.h"
#include "include/cw.h"
#include "tiger/tiger.h"
#include "haval/haval.h"
#include "md4/md4.h"

void hash(CW_OPTS *opts, char *buf, char byte);
int parseopts(int, char **, CW_OPTS *);
void help(void);
void keysetup(CW_OPTS *opts, rc4_key *key);

int main (int argc, char **argv)
{
	FILE *fp, *randnums, *tty;
	CW_OPTS opts;
	char buffer[1024];
	char *junk;
	char junk1[1024];
	char byte;
	rc4_key rc4key;
	int i,j,r;

	opts.seq=1;

        if ( parseopts(argc, argv, &opts) )
                return 0; 

        if (opts.passphrase[0]==0)
        {
		if ( (tty=fopen("/dev/tty", "r+")) == 0)
		{
	                fprintf(stderr,"Requires passphrase.\n");
			help();
		        return 0;
		}
		fclose(tty);
		junk=(char *)getpass("Enter passphrase: ");
		strcpy(junk1, junk);
		junk=(char *)getpass("Confirm: ");
		if ( strcmp(junk, junk1) )
		{
			fprintf(stderr,"\nPassphrase mismatch.");
			return 0;
		}
		strcpy(opts.passphrase, junk1);
		opts.passlen=strlen(junk1);
		i=strlen(junk);
		for (j=0; j<=i; j++)
		{
			junk[j]=0;
			junk1[j]=0;
		}
        }
	if (opts.filename[0] != 0)
	{
		if ( (fp=fopen(opts.filename, "r")) == 0)
		{
			fprintf(stderr,"Error opening chaf file (%s)\n", opts.filename);
			return 0;
		}
	}
	if (opts.crypto == '4')
		keysetup(&opts, &rc4key); 

	if (opts.ratio != 0 )
	{
		if ( (randnums=fopen(opts.randpath, "r")) == 0)
		{
			fprintf(stderr,"Error opening random numbers (%s)\n", opts.randpath);
			return 0;
		}
	}

	while ( !feof(stdin) || !feof(fp) )
	{

		if ( fread(&byte, 1, 1, stdin) )
		{
			if ( opts.crypto == '4' )
				rc4(&byte, 1, &rc4key);		

			hash(&opts, buffer, byte);
			fwrite(&byte, 1,1,stdout);
			fwrite(buffer, 1, opts.hashsize, stdout);
			opts.seq++;
		}
		if ( opts.ratio != 0 )
		{
			fread(&r, 1, sizeof(int), randnums);
			r=r%opts.ratio;
			for (i=0; i<=r; i++)
			{
				fread(buffer, 1, opts.hashsize+1, randnums);
				fwrite(buffer, 1, opts.hashsize+1, stdout);
			}
		}
	
		if ( (fread(buffer, 1, opts.hashsize+1, fp)) )
			fwrite(buffer,1,opts.hashsize+1, stdout);
	}
}
void hash(CW_OPTS *opts, char *buf, char byte)
{
	SHS_INFO shsInfo;
	MD5_CTX md5Info;
	MD2_CTX md2Info;
	MD4_CTX md4Info;
	word64 Tiger_Digest[3];
        haval_state Haval_State;

	char buffer[1024];
	buffer[0]=byte;
	memcpy(buffer+1, opts->passphrase, opts->passlen);
	memcpy(buffer+1+opts->passlen, &opts->seq, sizeof(long)); 
	switch (opts->hash_function)
	{
		case 's':
			shsInit(&shsInfo);
			shsUpdate(&shsInfo, buffer, opts->passlen+1+sizeof(long));
			shsFinal(&shsInfo);
			memcpy(buf,shsInfo.digest, opts->hashsize);
			break;
		case '2':
			MD2Init(&md2Info);
			MD2Update(&md2Info, buffer, opts->passlen+1+sizeof(long)); 
			MD2Final(buf, &md2Info);
			break;			
		case '4':
			MD4Init(&md4Info);
			MD4Update(&md4Info, buffer, opts->passlen+1+sizeof(long)); 
			MD4Final(buf, &md4Info);
			break;			
		case '5':
			MD5Init(&md5Info);
			MD5Update(&md5Info, buffer, opts->passlen+1+sizeof(long));
			MD5Final(&md5Info);
			memcpy(buf,md5Info.digest, opts->hashsize);
			break;
		case 't':
			tiger(buffer, opts->passlen+1+sizeof(long), Tiger_Digest);
			memcpy(buf,Tiger_Digest, opts->hashsize);
			break;
		case 'h':
			haval_start (&Haval_State);
			haval_hash  (&Haval_State, buffer, opts->passlen+1+sizeof(long));
			haval_end   (&Haval_State, buf);
	}
}
void help(void)
{
        fprintf(stderr,"Valid options:\n");
        fprintf(stderr,"     -hN             selects hash to use (s=SHA, 2=MD2, 4=MD4, 5=MD5, t=Tiger,\n");
	fprintf(stderr,"                           h=Haval).\n");
	fprintf(stderr,"                     Default: SHA\n");
        fprintf(stderr,"     -cN             selects encryption algorithm(4 = RC4)\n");
        fprintf(stderr,"     -a phrase       Defines passphrase when using valid MACs\n");
        fprintf(stderr,"     -f filename     Defines the input file to use. (Chaf packets)\n");
        fprintf(stderr,"                     Default: none.\n");
	fprintf(stderr,"     -r filename     Defines the path to a stream of random numbers.\n");
	fprintf(stderr,"                     Default: /dev/urandom\n");
	fprintf(stderr,"     -m ratio        Defines the modulus for generating random chaf packets.\n");
	fprintf(stderr,"                     Default: 0 (no chaf)\n");
	fprintf(stderr,"     -t N            Truncate hash to N bytes length. (Default: off)\n");
        fprintf(stderr,"     Input is from stdin or a file, Output is always stdout.\n\n");
        fprintf(stderr,"Example: chaf -hs -a password -f secret.field > new.field\n");
        fprintf(stderr,"This will use SHA  with 'password' as the MAC phrase.\n");
        fprintf(stderr,"secret.txt  will get decoded from secret.field\n");
 }
int parseopts(int argc, char **argv, CW_OPTS *opts)
{
        int i;
	char trunc = 0;

        opts->hash_function='s';
        opts->crypto='0';
	opts->ratio=0;
        strcpy(opts->randpath, "/dev/urandom");
        strcpy(opts->filename, "");

        for (i=1; i<=argc-1; i++)
        {
                if ( argv[i][0]=='-' )
                {
                        switch (argv[i][1])
                        {
                                case 'h':
					if ( strchr("s245th", argv[i][2]))
						opts->hash_function=argv[i][2];
                                        break;
                                case 'c':
                                        if ( (argv[i][2]=='4') )
                                                opts->crypto=argv[i][2];
                                        break;
                                case 'a':
					strcpy(opts->passphrase,argv[i+1]);
					opts->passlen=strlen(opts->passphrase);
                                        break;
                                case 'f':
                                        strcpy(opts->filename,argv[i+1]);
                                        break;
				case 'm':
					opts->ratio=atoi(argv[i+1]);
					break;
				case 'r':
					strcpy(opts->randpath,argv[i+1]);
					break;
				case 't':
					opts->hashsize=atoi(argv[i+1]);
					trunc=1;
					break;
                                default:
                                        fprintf(stderr,"Bad option: %s\n",argv[i]);
                                        help();
                                        return -1;
                        }
                }
        }

	if (!trunc)
	{

		switch (opts->hash_function)
		{
			case 's':
		                opts->hashsize=20;
				break;
			case '2':
				opts->hashsize=16;
				break;
			case '4':
				opts->hashsize=16;
				break;
			case '5':
	        	        opts->hashsize=16;
				break;
			case 't':
				opts->hashsize=24;
				break;
			case 'h':
				opts->hashsize=32;
				break;
		}	
	}

        return 0;                           
}
void keysetup(CW_OPTS *opts, rc4_key *key)
{
        unsigned char p[1024];
        int i;
        SHS_INFO shsInfo; /* data for SHA */
        memcpy(p, opts->passphrase, opts->passlen);
        i=opts->passlen;
        shsInit(&shsInfo);
        shsUpdate(&shsInfo, p, i);
	shsFinal(&shsInfo);
        memcpy(opts->passphrase, &shsInfo, 20);
	shsInit(&shsInfo);
        shsUpdate(&shsInfo, opts->passphrase, 20);
	shsUpdate(&shsInfo, p, i);
	shsFinal(&shsInfo);
	opts->passlen=20;
        memcpy(p, &shsInfo, 20);
        prepare_key((unsigned char *)p,20,key);
} 
