/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997 Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/file.h>
#include <sys/wait.h>
#include "freedom.h"

int pgpdecrypt(const char *encrypted, char *decrypted)
{
  int lockfd, status;
  pid_t pid = getpid();
  char pathenv[BUFSIZ], passenv[BUFSIZ];
  FILE *infile, *outfile;

  /* PGP needs exclusive access to randseed.bin */
  lockfd = open("pgp.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));
  sprintf(passenv, "PGPPASS=%s", PASS_PHRASE);
  putenv(passenv);
  sprintf(pathenv, "PGPPATH=%s", PGP_DIR);
  putenv(pathenv);

  /* call PGP to do decryption */
  if (!fork()) { /* child */
    infile = fopen(encrypted, "r");
    dup2(fileno(infile), STDIN_FILENO);
    fclose(infile);
    outfile = fopen(decrypted, "w");
    dup2(fileno(outfile), STDOUT_FILENO);
    fclose(outfile);
    fclose(stderr); /* shut up! */
    if (USE_SYSLOG)
      syslog(LOG_DEBUG, "Decrypting message");
    execl(PGP, basename(PGP), "+batchmode", "+verbose=0", "-f", NULL);
  }
  else /* parent */
    wait(&status);

  putenv("PGPPASS=");
  putenv("PGPPATH=");
  flock(lockfd, LOCK_UN);
  close(lockfd);
  return status;
}

int encrypt_key(const char *toencrypt, char *toappendto, const char *key)
{
  int lockfd, status;
  pid_t pid = getpid();
  char tempfilename[BUFSIZ], line[BUFSIZ], pathenv[BUFSIZ];
  FILE *infile, *outfile;

  /* PGP needs exclusive access to randseed.bin */
  lockfd = open("pgp.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));
  sprintf(tempfilename, "%s.asc", toencrypt);
  sprintf(pathenv, "PGPPATH=%s", PGP_DIR);
  putenv(pathenv);

  /* call PGP to do encryption */
  if (!fork()) { /* child */
    infile = fopen(toencrypt, "r");
    dup2(fileno(infile), STDIN_FILENO);
    fclose(infile);
    outfile = fopen(tempfilename, "w");
    dup2(fileno(outfile), STDOUT_FILENO);
    fclose(outfile);
    fclose(stderr); /* shut up! */
    if (USE_SYSLOG)
      syslog(LOG_DEBUG, "Encrypting message");
    execl(PGP, basename(PGP), "-fact", "+batchmode", "+armorlines=0",
	  "+verbose=0", "-z", key, NULL);
  }
  else /* parent */
    wait(&status);

  putenv("PGPPATH=");
  flock(lockfd, LOCK_UN);
  close(lockfd);

  /* append encrypted text to end of message */
  if ((infile = fopen(tempfilename, "r"))) {
    outfile = fopen(toappendto, "a");
    fputs("**\n", outfile);
    while (fgets(line, sizeof(line), infile))
      fputs(line, outfile);
    fclose(outfile);
    fclose(infile);
  }
  unlink(tempfilename);
  unlink(toencrypt);
  return(status);
}

int geturl(const char *url, char *urlfilename, const int type)
{
  FILE *outfile;
  int status;

  outfile = fopen(urlfilename, "w");

  /* URL is blocked; write error message to outfile */
  if (blocked(url, URL_BLOCK)) {
    if (USE_SYSLOG) {
      syslog(LOG_NOTICE, "Blocked URL: %s", url);
      syslog(LOG_NOTICE, "Ignoring request");
    }
    fprintf(outfile, "You are not allowed to download %s\n"
	             "through the %s.\n", url, REMAILER_NAME);
    fclose(outfile);
    return 0;
  }

  /* call Lynx to retrieve the requested URL */
  else {
    if (!fork()) { /* child */
      dup2(fileno(outfile), STDOUT_FILENO);
      fclose(outfile);
      if (type == URL_REQUEST) {
	if (USE_SYSLOG)
	  syslog(LOG_INFO, "Downloading page %s", url);
	execl(LYNX, basename(LYNX), "-dump", url, NULL);
      }
      else if (type == HTML_REQUEST) {
	if (USE_SYSLOG)
	  syslog(LOG_INFO, "Downloading HTML %s", url);
	execl(LYNX, basename(LYNX), "-source", url, NULL);
      }
    }
    else /* parent */
      wait(&status);

    if (USE_SYSLOG)
      syslog(LOG_INFO, "Download finished");
    return status;
  }
}
