/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997 Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <syslog.h>
#include "freedom.h"

enum message_type
scan_message(const char *infilename, char *to, char *from)
{
  int headers = 1, colhead = 0, linenum = 0, replyto = 0,
      ctflag = 0, esflag = 0, ekflag = 0, ltflag = 0,
      anflag = 0, ptflag = 0, urlflag = 0, htmlflag = 0, rrflag = 0;
  FILE *infile;
  char line[BUFSIZ], tmpline[BUFSIZ], cutmarks[BUFSIZ], encryptsub[BUFSIZ],
       encryptkey[BUFSIZ], latency[BUFSIZ], url[BUFSIZ];
  const char *TO = (USE_MIX == 1) ? "" : "To: ";
  const char *POST = (USE_MIX == 1) ? "post: " : "Newsgroups: ";

  to[0] = '\0'; from[0] = '\0';

  if (!(infile = fopen(infilename, "r")))
    return NON_ANON;
  if (USE_SYSLOG)
    syslog(LOG_DEBUG, "Scanning message");
  while (fgets(line, sizeof(line), infile)) {
    if (!headers) {
      if ((linenum > colhead) || (strlen(line) > 1))
	linenum++;
      if ((linenum == 1) && strstr(line, "::")) {
	headers = 1;
	colhead = 1;
      }
    }
    else {
      if (strlen(line) <= 1)
	headers = 0;
      else {
	if (linenum == 0) { /* real headers */
	  if (strileft(line, "reply-to:")) {
	    strcpy(from, line + sizeof("reply-to:") - 1);
	    chop(from);
	    replyto = 1;
	  }
	  else if (strileft(line, "from:") && !replyto) {
	    strcpy(from, line + sizeof("from:") - 1);
	    chop(from);
	  }
	  else if (strileft(line, "subject:")) {
	    strcpy(tmpline, line + sizeof("subject:") - 1);
	    chop(tmpline);
	    if (strieq(tmpline, "help")) {
	      strcpy(to, from);
	      return HELP_REQ;
	    }
	    else if (strieq(tmpline, "remailer-help")) {
	      strcpy(to, from);
	      return HELP_REQ;
	    }
	    else if (strieq(tmpline, "freedom-help")) {
	      strcpy(to, from);
	      return HELP_REQ;
	    }
	    else if (USE_STATS && strieq(tmpline, "remailer-stats")) {
	      strcpy(to, from);
	      return STATS_REQ;
	    }
	    else if (USE_STATS && strieq(tmpline, "freedom-stats")) {
	      strcpy(to, from);
	      return STATS_REQ;
	    }
	    else if (ALLOW_PGP && strieq(tmpline, "remailer-key")) {
	      strcpy(to, from);
	      return KEY_REQ;
	    }
	    else if (ALLOW_PGP && strieq(tmpline, "freedom-key")) {
	      strcpy(to, from);
	      return KEY_REQ;
	    }
	  }
	}
	if (ALLOW_PGP && strileft(line, "encrypted: pgp")) {
	  chop(line);
	  if (USE_SYSLOG)
	    syslog(LOG_DEBUG, "%s detected", line);
	  return PGP_MESSAGE;
	}
	else if (strileft(line, "anon-to:")) {
	  strcpy(tmpline, line + sizeof("anon-to:") - 1);
	  chop(tmpline);
	  if (!blocked(tmpline, DEST_BLOCK))
	    strcat(to, TO), strcat(to, tmpline), strcat(to, "\n");
	  anflag = 1;
	}
	else if (strileft(line, "request-remailing-to:")) {
	  strcpy(tmpline, line + sizeof("request-remailing-to:") - 1);
	  chop(tmpline);
	  if (!blocked(tmpline, DEST_BLOCK))
	    strcat(to, TO), strcat(to, tmpline), strcat(to, "\n");
	  anflag = 1;
	}
	else if (strileft(line, "remail-to:")) {
	  strcpy(tmpline, line + sizeof("remail-to:") - 1);
	  chop(tmpline);
	  if (!blocked(tmpline, DEST_BLOCK))
	    strcat(to, TO), strcat(to, tmpline), strcat(to, "\n");
	  anflag = 1;
	}
	else if (strileft(line, "anon-send-to:")) {
	  strcpy(tmpline, line + sizeof("anon-send-to:") - 1);
	  chop(tmpline);
	  if (!blocked(tmpline, DEST_BLOCK))
	    strcat(to, TO), strcat(to, tmpline), strcat(to, "\n");
	  anflag = 1;
	}
	else if (ALLOW_POST && strileft(line, "anon-post-to:")) {
	  strcpy(tmpline, line + sizeof("anon-post-to:") - 1);
	  chop(tmpline);
	  if (!blocked(tmpline, GROUP_BLOCK))
	    strcat(to, POST), strcat(to, tmpline), strcat(to, "\n");
	  ptflag = 1;
	}
	else if (strileft(line, "cutmarks:")) {
	  strcpy(cutmarks, line + sizeof("cutmarks:") - 1);
	  chop(cutmarks);
	  ctflag = 1;
	}
	else if (strileft(line, "encrypt-subject:")) {
	  strcpy(encryptsub, line + sizeof("encrypt-subject:") - 1);
	  chop(encryptsub);
	  esflag = 1;
	}
	else if (ALLOW_PGP && strileft(line, "encrypt-key:")) {
	  strcpy(encryptkey, line + sizeof("encrypt-key:") - 1);
	  chop(encryptkey);
	  ekflag = 1;
	}
	else if (ALLOW_LTIME && strileft(line, "latent-time:")) {
	  strcpy(latency, line + sizeof("latent-time:") - 1);
	  chop(latency);
	  ltflag = 1;
	}
	/* The following directives are only evaluated in the ::'ed headers
	   to make sure that everything below the next blank line will be
	   remailed to the Return-To: address */
	else if (ALLOW_WWW && strileft(line, "get-url:") && colhead) {
	  strcpy(url, line + sizeof("get-url:") - 1);
	  chop(url);
	  urlflag = 1;
	}
	else if (ALLOW_WWW && strileft(line, "get-html-source:") && colhead) {
	  strcpy(url, line + sizeof("get-html-source:") - 1);
	  chop(url);
	  htmlflag = 1;
	}
	else if (ALLOW_WWW && strileft(line, "return-to:") && colhead) {
	  strcpy(tmpline, line + sizeof("return-to:") - 1);
	  chop(tmpline);
	  if (!blocked(tmpline, DEST_BLOCK))
	    strcpy(to, TO), strcat(to, tmpline), strcat(to, "\n");
	  else
	    to[0] = '\0';
	  rrflag = 1;
	}
	/* The Null: directive for bit-bucket mail makes sense only
	   in the ::'ed headers of encrypted cover traffic messages */
	else if (strileft(line, "null:") && colhead) {
	  chop(line);
	  if (USE_SYSLOG)
	    syslog(LOG_DEBUG, "%s detected", line);
	  return NULL_MESSAGE;
	}
      }
    }
  }
  fclose(infile);

  if (urlflag || htmlflag) {
    if (!rrflag) {
      if (!blocked(from, DEST_BLOCK))
	strcpy(to, TO), strcat(to, from), strcat(to, "\n");
      else
	to[0] = '\0';
    }
    strcpy(from, url);
    if (urlflag)
      return URL_REQUEST;
    if (htmlflag)
      return HTML_REQUEST;
  }

  /* process headers in order: cutmarks, encrypt-subject, encrypt-key,
     latent-time */
  /* if these don't exist, then remail or post the message if that is where
     it's headed; else, return NON_ANON to send it to the spool or bucket */

  if (ctflag) {
    strcpy(to, cutmarks);
    return CUTMARKS;
  }
  if (esflag) {
    strcpy(to, encryptsub);
    return ENCRYPT_SUB;
  }
  if (ekflag) {
    strcpy(to, encryptkey);
    return ENCRYPT_KEY;
  }
  if (ltflag) {
    strcpy(to, latency);
    return LATENT_TIME;
  }
  if (ptflag)
    return ANON_POST;
  if (anflag)
    return ANON_MESSAGE;

  return NON_ANON;
}

void
remove_headers(const char *infilename, const char *outfilename, const char *to)
{
  FILE *infile, *outfile;
  int headers = 1, colhead = 0, addhead = 0, linenum = 0;
  char line[BUFSIZ], tmpline[BUFSIZ];

  if (!(infile = fopen(infilename, "r"))) {
    outfile = fopen(outfilename, "w");
    fclose(outfile);
    return;
  }
  outfile = fopen(outfilename, "w");

  if (USE_MIX == 1)
    fputs("-----Final Hop-----\n", outfile);
  fputs(to, outfile);
  if (USE_MIX == 1)
    fputs("END\n", outfile);

  while (fgets(line, sizeof(line), infile)) {
    if (!headers) {
      if ((linenum > colhead) || (strlen(line) > 1))
	linenum++;
      if ((linenum == 1) && strstr(line, "::")) {
	headers = 1;
	colhead = 1;
      }
      else if (linenum == (colhead + 1)) {
	if (strstr(line, "##"))
	  addhead = 1;
	else {
	  fputs("\n", outfile);
	  fputs(line, outfile);
	}
      }
      else {
	if (addhead) {
	  if (strlen(line) <= 1) {
	    fputs(line, outfile);
	    addhead = 0;
	  }
	  else if (!blocked(line, HDRFILTER)) {
	    if (strileft(line, "to:") || strileft(line, "cc:")) {
	      strcpy(tmpline, line + sizeof("to:") - 1);
	      chop(tmpline);
	      if (!blocked(tmpline, DEST_BLOCK))
		fputs(line, outfile);
	    }
	    else if (strileft(line, "bcc:")) {
	      strcpy(tmpline, line + sizeof("bcc:") - 1);
	      chop(tmpline);
	      if (!blocked(tmpline, DEST_BLOCK))
		fputs(line, outfile);
	    }
	    else if (strileft(line, "reply-to:")) {
	      strcpy(tmpline, line + sizeof("reply-to:") - 1);
	      chop(tmpline);
	      if (!blocked(tmpline, DEST_BLOCK))
		fputs(line, outfile);
	    }
	    else if (strileft(line, "newsgroups:")) {
	      strcpy(tmpline, line + sizeof("newsgroups:") - 1);
	      chop(tmpline);
	      if (!blocked(tmpline, GROUP_BLOCK))
		fputs(line, outfile);
	    }
	    else
	      fputs(line, outfile);
	  }
	}
	else {
	  if ((linenum > colhead) || (strlen(line) > 1))
	    fputs(line, outfile);
	}
      }
    }
    else {
      if (strlen(line) <= 1)
	headers = 0;
    }
  }
  fclose(infile);
  fclose(outfile);
}
