/* pggalgo.c - settings for algorithm usage
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <includes.h>
#include <pgg.h>
#include <pggalgo.h>
#include <pggdebug.h>


#define algo		((PggAlgoPtr)(_algo))
#define old_algo	((PggAlgoPtr)(_old_algo))


PggAlgo pgg_algo_new(PggErrenv errenv)
{
    PggAlgoPtr	new_algo;
    
    PGG_CHECK_SKIP_ARG(NULL);
    
    if (!( new_algo = _malloc(PggAlgo) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    new_algo->magic      = PggAlgoMAGIC;
    new_algo->refcounter = 1;
    new_algo->cipher     = PGG_ALGO_CIPHER_USEPREFS;
    new_algo->digest     = PGG_ALGO_DIGEST_USEPREFS;
    new_algo->compress   = PGG_ALGO_COMPRESS_USEPREFS;
    new_algo->packet     = PGG_ALGO_PACKET_OPENPGP;
    new_algo->s2k_mode   = PGG_ALGO_S2K_MODE_ITERATEDSALTED;
    new_algo->s2k_cipher = PGG_ALGO_CIPHER_BLOWFISH;
    new_algo->s2k_digest = PGG_ALGO_DIGEST_RIPEMD160;
    
    return _hd(PggAlgo, new_algo);
}


PggAlgo pgg_algo_clone(PggAlgo _old_algo, PggErrenv errenv)
{
    PggAlgoPtr	new_algo;
    
    PGG_STD_ASSERT_ARG(PggAlgo, old_algo, NULL);
    
    if (!( new_algo = _malloc(PggAlgo) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    memcpy(new_algo, old_algo, _size(PggAlgo));
    
    new_algo->refcounter = 1;
    
    return _hd(PggAlgo, new_algo);
}


void pgg_algo_addref(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    algo->refcounter++;
}


void pgg_algo_release(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    
    if (!--algo->refcounter)
        free(algo);
}


void pgg_algo_set_cipher(PggAlgo _algo, PggAlgo_Cipher cipher, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(cipher >= PGG_ALGO_CIPHER_USEPREFS, ARGUMENT, VALUE);
    PGG_ASSERT(cipher <= PGG_ALGO_CIPHER_TWOFISH, ARGUMENT, VALUE);
    algo->cipher = cipher;
}


void pgg_algo_set_digest(PggAlgo _algo, PggAlgo_Digest digest, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(digest >= PGG_ALGO_DIGEST_USEPREFS, ARGUMENT, VALUE);
    PGG_ASSERT(digest <= PGG_ALGO_DIGEST_RIPEMD160, ARGUMENT, VALUE);
    algo->digest = digest;
}


void pgg_algo_set_compress(PggAlgo _algo, PggAlgo_Compress compress, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(compress >= PGG_ALGO_COMPRESS_USEPREFS, ARGUMENT, VALUE);
    PGG_ASSERT(compress <= PGG_ALGO_COMPRESS_RFCZLIB, ARGUMENT, VALUE);
    algo->compress = compress;
}


void pgg_algo_set_packet(PggAlgo _algo, PggAlgo_Packet packet, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(packet >= PGG_ALGO_PACKET_OPENPGP, ARGUMENT, VALUE);
    PGG_ASSERT(packet <= PGG_ALGO_PACKET_RFC1991, ARGUMENT, VALUE);
    algo->packet = packet;
}


void pgg_algo_set_s2k_mode(PggAlgo _algo, PggAlgo_S2kMode mode, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(mode >= PGG_ALGO_S2K_MODE_SIMPLE, ARGUMENT, VALUE);
    PGG_ASSERT(mode <= PGG_ALGO_S2K_MODE_ITERATEDSALTED, ARGUMENT, VALUE);
    algo->s2k_mode = mode;
}


void pgg_algo_set_s2k_cipher(PggAlgo _algo, PggAlgo_Cipher cipher, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(cipher >= PGG_ALGO_CIPHER_IDEA, ARGUMENT, VALUE);
    PGG_ASSERT(cipher <= PGG_ALGO_CIPHER_TWOFISH, ARGUMENT, VALUE);
    algo->s2k_cipher = cipher;
}


void pgg_algo_set_s2k_digest(PggAlgo _algo, PggAlgo_Digest digest, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(digest >= PGG_ALGO_DIGEST_MD5, ARGUMENT, VALUE);
    PGG_ASSERT(digest <= PGG_ALGO_DIGEST_RIPEMD160, ARGUMENT, VALUE);
    algo->s2k_digest = digest;
}


PggAlgo_Cipher pgg_algo_get_cipher(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggAlgo, algo, PGG_ALGO_CIPHER_ERROR);
    return algo->cipher;
}


PggAlgo_Digest pgg_algo_get_digest(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggAlgo, algo, PGG_ALGO_DIGEST_ERROR);
    return algo->digest;
}


PggAlgo_Compress pgg_algo_get_compress(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggAlgo, algo, PGG_ALGO_COMPRESS_ERROR);
    return algo->compress;
}


PggAlgo_S2kMode pgg_algo_get_s2k_mode(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggAlgo, algo, PGG_ALGO_S2K_MODE_ERROR);
    return algo->s2k_mode;
}


PggAlgo_Cipher pgg_algo_get_s2k_cipher(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggAlgo, algo, PGG_ALGO_CIPHER_ERROR);
    return algo->s2k_cipher;
}


PggAlgo_Digest pgg_algo_get_s2k_digest(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggAlgo, algo, PGG_ALGO_DIGEST_ERROR);
    return algo->s2k_digest;
}


PggAlgo_Packet pgg_algo_get_packet(PggAlgo _algo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggAlgo, algo, PGG_ALGO_PACKET_ERROR);
    return algo->packet;
}


static void build_digest(PggAlgo_Digest digest, PggExe exe, PggErrenv errenv)
{
    switch (digest) {
        case PGG_ALGO_DIGEST_MD5:
            pgg_exe_add_arg(exe, "md5", errenv);
            break;
        
        case PGG_ALGO_DIGEST_SHA1:
            pgg_exe_add_arg(exe, "sha1", errenv);
            break;
        
        case PGG_ALGO_DIGEST_RIPEMD160:
            pgg_exe_add_arg(exe, "ripemd160", errenv);
            break;
        
        default:
            break;
    }
}


static void build_cipher(PggAlgo_Cipher cipher, PggExe exe, PggErrenv errenv)
{
    switch (cipher) {
        case PGG_ALGO_CIPHER_IDEA:
            pgg_exe_add_arg(exe, "idea", errenv);
            break;
    
        case PGG_ALGO_CIPHER_3DES:
            pgg_exe_add_arg(exe, "3des", errenv);
            break;

        case PGG_ALGO_CIPHER_CAST5:
            pgg_exe_add_arg(exe, "cast5", errenv);
            break;

        case PGG_ALGO_CIPHER_BLOWFISH:
            pgg_exe_add_arg(exe, "blowfish", errenv);
            break;

        case PGG_ALGO_CIPHER_TWOFISH:
            pgg_exe_add_arg(exe, "twofish", errenv);
            break;

        default:
            break;
    }
}


void pgg_algo_setup_exe(PggAlgo _algo, PggExe exe, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(algo, ARGUMENT, NULLPTR);
    
    pgg_errenv_reset(local_errenv);
    
    pgg_exe_add_arg(exe, "--openpgp", local_errenv);
    
    switch (algo->packet) {
        case PGG_ALGO_PACKET_V3SIGS:
            pgg_exe_add_arg(exe, "--force-v3-sigs", local_errenv);
            break;
        
        case PGG_ALGO_PACKET_RFC1991:
            pgg_exe_add_arg(exe, "--rfc1991", local_errenv);
            break;
        
        default:
            break;
    }
    
    switch (algo->compress) {
        case PGG_ALGO_COMPRESS_NONE:
            pgg_exe_add_arg(exe, "--compress-algo", local_errenv);
            pgg_exe_add_arg(exe, "0", local_errenv);
            break;
        
        case PGG_ALGO_COMPRESS_PGPZLIB:
            pgg_exe_add_arg(exe, "--compress-algo", local_errenv);
            pgg_exe_add_arg(exe, "1", local_errenv);
            break;
        
        case PGG_ALGO_COMPRESS_RFCZLIB:
            pgg_exe_add_arg(exe, "--compress-algo", local_errenv);
            pgg_exe_add_arg(exe, "2", local_errenv);
            break;
        
        default:
            break;
    }
    
    if (algo->cipher > PGG_ALGO_CIPHER_USEPREFS) {
        pgg_exe_add_arg(exe, "--cipher-algo", local_errenv);
        build_cipher(algo->cipher, exe, local_errenv);
    }
    
    if (algo->digest > PGG_ALGO_DIGEST_USEPREFS) {
        pgg_exe_add_arg(exe, "--digest-algo", local_errenv);
        build_digest(algo->digest, exe, local_errenv);
    }
    
    pgg_exe_add_arg(exe, "--s2k-cipher-algo", local_errenv);
    build_cipher(algo->s2k_cipher, exe, local_errenv);
    
    pgg_exe_add_arg(exe, "--s2k-digest-algo", local_errenv);
    build_digest(algo->s2k_digest, exe, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        PGG_DEBUG(("problems on setting up exe"));
        pgg_errenv_copy(errenv, local_errenv);
    }
}





