
/* ====================================================================
 * Copyright (c) 1995 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * IT'S CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */


/*
 * http_auth_system.c
 *
 * By Lou Langholtz <http://www.cs.utah.edu/~ldl/>
 *
 * $Id: mod_auth_system.c,v 1.6 1997/07/02 19:30:31 sysldl Exp $
 *
 * This module provides user authentication using the host's authentication
 * system. The code is derived from Dirk.vanGulik@jrc.it's mod_auth_nis.c
 * code.
 *
 * mod_auth_nis.c really just asked the system for the user's password
 * despite having "nis" in various names. I've mostly just renamed things
 * here just so it isn't as misleading (at least to me ;-). I also fixed
 * a few code bugs he had so the code uses log_reason() properly.
 */

#include <pwd.h>

/*
 * Define shadow if your system uses shadow passwords. Of course
 * you'd then need to run the web server as root or whatever it'd
 * take such that it had permission to access the system's shadow
 * password file.
 */
/* #define SHADOW */

/* #define MULTITREADING */


#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"

typedef struct auth_system_config_struct {
    int auth_system;	/* contains true/false */
} auth_config_rec;

static void *create_system_auth_dir_config (pool *p, char *d)
{
    return ap_pcalloc (p, sizeof(auth_config_rec));
}

#if MODULE_MAGIC_NUMBER < 19961125
static char *system_auth_set_flag_slot(cmd_parms *cmd, char *struct_ptr, int arg)
#else
static const char *system_auth_set_flag_slot(cmd_parms *cmd, char *struct_ptr, int arg)
#endif
{
    (int) *(char **) (struct_ptr + ((int) cmd->info)) = arg;
    return NULL;
}

static command_rec auth_system_cmds[] = {
{ "AuthUserSystem", system_auth_set_flag_slot,
    (void*)XtOffsetOf(auth_config_rec,auth_system), OR_AUTHCFG, FLAG, 
	"authenticate user using host's authentication system"
},
{ NULL }
};

module auth_system_module;

static char *get_system_pw(request_rec *r, char *user)
{
    char buff[MAX_STRING_LEN];
#ifdef MULTITREADING
    struct passwd *result;
    struct passwd tmp;

#ifdef SHADOW
    result=getspnam_r(user,tmp,buff,MAX_STRING_LEN);
#else
    result=getpwnam_r(user,tmp,buff,MAX_STRING_LEN);
#endif

#else /* !MULTITREADING */
    struct passwd *result;

#ifdef SHADOW
    result=getspnam(user);
#else /* !SHADOW */
    result=getpwnam(user);
#endif /* SHADOW */

#endif /* MULTITREADING */
    /* success ? */

    if (!result) {
	return NULL;
    };
    if (!result->pw_passwd) {
#ifndef SHADOW
	sprintf(buff,"no system password for %s; is system using shadow passwords?", user);
	ap_log_reason (buff, r->uri, r);
#endif
	return NULL;
    };

    return ap_pstrdup (r->pool, result->pw_passwd);
}


/* These functions return 0 if client is OK, and proper error status
 * if not... either AUTH_REQUIRED, if we made a check, and it failed, or
 * SERVER_ERROR, if things are so totally confused that we couldn't
 * figure out how to tell if the client is authorized or not.
 *
 * If they return DECLINED, and all other modules also decline, that's
 * treated by the server core as a configuration error, logged and
 * reported as such.
 */

/* Determine user ID, and check if it really is that user, for HTTP
 * basic authentication...
 */

static int authenticate_system_basic_user (request_rec *r)
{
    auth_config_rec *sec =
      (auth_config_rec *)ap_get_module_config (r->per_dir_config, &auth_system_module);
    conn_rec *c = r->connection;
    const char *sent_pw, *real_pw;
    char errstr[MAX_STRING_LEN];
    int res;

    if ((res = ap_get_basic_auth_pw (r, &sent_pw))) return res;
    
    if(!sec->auth_system) 
        return DECLINED;

    errno = 0;	/* clear errno just in case */
    if (!(real_pw = get_system_pw(r, c->user))) {
	if (errno)
	    sprintf(errstr,"can't get system password for user %s: %s",
		c->user, strerror(errno));
	else
	    sprintf(errstr,"user %s not found",c->user);
	ap_log_reason (errstr, r->uri, r);
	ap_note_basic_auth_failure (r);
	return AUTH_REQUIRED;
    }
    /* anyone know where the prototype for crypt is? */
    if(strcmp(real_pw,(char *)crypt(sent_pw,real_pw))) {
        sprintf(errstr,"user %s: password mismatch",c->user);
	ap_log_reason (errstr, r->uri, r);
	ap_note_basic_auth_failure (r);
	return AUTH_REQUIRED;
    }
    return OK;
}

module auth_system_module = {
   STANDARD_MODULE_STUFF,
   NULL,			/* initializer */
   create_system_auth_dir_config,	/* dir config creater */
   NULL,			/* dir merger --- default is to override */
   NULL,			/* server config */
   NULL,			/* merge server config */
   auth_system_cmds,		/* command table */
   NULL,			/* handlers */
   NULL,			/* filename translation */
   authenticate_system_basic_user,	/* check_user_id */
   NULL,			/* check auth */
   NULL,			/* check access */
   NULL,			/* type_checker */
   NULL,			/* fixups */
   NULL				/* logger */
};
