/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>

RCSID("$Id: xfs_vfsops.c,v 1.3 1998/12/22 13:16:08 lha Exp $");

/*
 * XFS vfs operations.
 */

#include <xfs/xfs_common.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_deb.h>
#include <xfs/nxfs.h>

static struct vnode *make_dead_vnode (struct vfs *vfsp);

struct xfs xfs[NXFS];

static int xfsfstype;

static int
xfs_mount(struct vfs *vfsp,
	  struct ucred *cred)
{
  char *devpath;
  struct vnode *devvp;
  dev_t dev;
  int error;
  struct vmount *vmnt;

  XFSDEB(XDEBVFOPS, ("xfs_mount vfsp = 0x%x\n"));

  vmnt = vfsp->vfs_mdata;

  devpath = (char *)vmnt + vmnt->vmt_data[VMT_OBJECT].vmt_off;

  error = lookupvp (devpath, L_NOFOLLOW, &devvp, U.U_cred);
  if (error)
      return error;

  if (devvp->v_type != VCHR) {
      VNOP_RELE(devvp);
      return ENXIO;
  }
  dev = devvp->v_rdev;
  VNOP_RELE(devvp);

  /* Check that this device really is an xfs_dev */

#if 0
  XFSDEB(XDEBVFOPS, ("xfs_mount dev = %x, minor = %x, major = %x,"
		     "ops = %x, "
		     "cb_ops = %x, "
		     "open = %x, "
		     "(xfs_devopen = %x)\n",
		     (unsigned)dev,
		     (unsigned)minor(dev),
		     (unsigned)major(dev),
		     devopsp[major(dev)],
		     devopsp[major(dev)] ? devopsp[major(dev)]->devo_cb_ops : 0,
		     (devopsp[major(dev)] && devopsp[major(dev)]->devo_cb_ops) ? devopsp[major(dev)]->devo_cb_ops->cb_open : 0,
		     xfs_devopen));
#endif

  if (minor(dev) < 0 || NXFS < minor(dev))
    return ENXIO;

#if 0 /* XXX - It doesn't seem we can perform this test */
  if (devsw[major(dev)].d_open != xfs_devopen)
      return ENXIO;
#endif

  if (xfs[minor(dev)].status & XFS_MOUNTED)
    return EBUSY;

  xfs[minor(dev)].status = XFS_MOUNTED;
  xfs[minor(dev)].vfsp = vfsp;
  xfs[minor(dev)].root = 0;
  xfs[minor(dev)].nnodes = 0;
  xfs[minor(dev)].nodes = 0;
  xfs[minor(dev)].fd = minor(dev);

  SET_VFS_TO_XFS(vfsp, &xfs[minor(dev)]);

  vfsp->vfs_bsize = PAGESIZE;

#if 0
  vfsp->vfs_fstype = xfsfstype;
  vfsp->vfs_dev    = minor(dev);
  vfsp->vfs_bsize  = PAGESIZE;
  vfsp->vfs_flag  |= VFS_NOTRUNC;
  vfsp->vfs_fsid.val[0] = minor(dev);
  vfsp->vfs_fsid.val[1] = major(dev); /* What is this good for */
#endif

  return 0;
}

static int
xfs_unmount(struct vfs *vfsp,
	    int flag,
	    struct ucred *cred)
{
  struct xfs *xfsp = VFS_TO_XFS(vfsp);

  XFSDEB(XDEBVFOPS, ("xfs_umount vfsp = 0x%x\n", (u_int) vfsp));

  free_all_xfs_nodes(xfsp);
  xfsp->status = 0;
  return 0;			/* Always allow umount to succed */
}

static int
xfs_root(struct vfs *vfsp,
	 struct vnode **vpp,
	 struct ucred *cred)
{
  struct xfs *xfsp = VFS_TO_XFS(vfsp);
  struct xfs_message_getroot msg;
  int error;
  
  XFSDEB(XDEBVFOPS, ("xfs_root vfsp = 0x%x\n", (u_int) vfsp));

  do {
    if (xfsp->root != 0) {
	*vpp = XNODE_TO_VNODE(xfsp->root);
	VNOP_HOLD(*vpp);
	XFSDEB(XDEBVFOPS, ("xfs_root: returning real vnode\n"));
	return 0;
    }

    msg.header.opcode = XFS_MSG_GETROOT;
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = cred->cr_pag;
    error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
    if (error == 0)
      error = ((struct xfs_message_wakeup *) &msg)->error;
  } while (error == 0);

  XFSDEB(XDEBVFOPS, ("xfs_root: returning dead vnode\n"));

  /*
   * Failed to get message through, need to pretend that all went well
   * and return a fake dead vnode to be able to unmount.
   */
  *vpp = make_dead_vnode(vfsp);
  VNOP_HOLD(*vpp);
  return 0;
}

static int
xfs_statfs(struct vfs *vfsp,
	   struct statfs *statfs,
	   struct ucred *cred)
{
    XFSDEB(XDEBVFOPS, ("xfs_statvfs\n"));

    statfs->f_version   = 0;
    statfs->f_type      = 0;
    statfs->f_bsize     = 8192;
    statfs->f_blocks    = 4711 * 4711;
    statfs->f_bfree     = 4711 * 4711;
    statfs->f_bavail    = 4711 * 4711;
    statfs->f_files     = 4711;
    statfs->f_ffree     = 4711;
    
    statfs->f_vfstype   = vfsp->vfs_type;
    statfs->f_fsid      = vfsp->vfs_fsid;
    statfs->f_fsize     = 8192;
    statfs->f_vfsnumber = 17;	/* XXX */
    strcpy(statfs->f_fname, "arla");
    strcpy(statfs->f_fpack, "arla");

    return 0;
}

static int
xfs_sync(struct gfs *gfsp)
{
  XFSDEB(XDEBVFOPS, ("xfs_sync\n"));
  return 0;
}

static int
xfs_vget(struct vfs *vfsp,
	 struct vnode **vpp,
	 struct fileid *fidp,
	 struct ucred *cred)
{
  XFSDEB(XDEBVFOPS, ("xfs_vget\n"));
  return ENOSYS;
}

static int
xfs_cntl (struct vfs *vfsp,
	  int cmd,
	  caddr_t arg,
	  unsigned long argsize,
	  struct ucred *cred)
{
  XFSDEB(XDEBVFOPS, ("xfs_cntl\n"));
  return ENOSYS;
}

static int
xfs_quotactl (struct vfs *vfsp,
	      int foo,
	      uid_t bar,
	      caddr_t baz,
	      struct ucred *cred)
{
  XFSDEB(XDEBVFOPS, ("xfs_quotactl\n"));
  return ENOSYS;
}

/*
 * To be able to unmount when the XFS daemon is not
 * responding we need a root vnode, use a dead vnode!
 */

struct dead_node {
    struct vnode vn;
    struct gnode gn;
};

static int
dead_hold (struct vnode *vp)
{
    XFSDEB(XDEBVFOPS, ("dead_hold\n"));
    ++vp->v_count;
    return 0;
}

static int
dead_rele (struct vnode *vp)
{
    XFSDEB(XDEBVFOPS, ("dead_rele\n"));
    if (--vp->v_count == 0)
	xfs_free(vp, sizeof(*vp));
    return 0;
}

extern int nodev();

struct vnodeops dead_vnodeops = {
    nodev /* link */,
    nodev /* mkdir */,
    nodev /* mknod */,
    nodev /* remove */,
    nodev /* rename */,
    nodev /* rmdir */,
    nodev /* lookup */,
    nodev /* fid */,
    nodev /* open */,
    nodev /* create */,
    dead_hold /* hold */,
    dead_rele /* rele */,
    nodev /* close */,
    nodev /* map */,
    nodev /* unmap */,
    nodev /* access */,
    nodev /* getattr */,
    nodev /* setattr */,
    nodev /* fclear */,
    nodev /* fsync */,
    nodev /* ftrunc */,
    nodev /* rdwr */,
    nodev /* lockctl */,
    nodev /* ioctl */,
    nodev /* readlink */,
    (int (*)(struct vnode *,int,u_short,u_short*,void(*)(),char*,struct ucred* ))nodev /* select */,
    nodev /* symlink */,
    nodev /* readdir */,
    nodev /* strategy */,
    nodev /* revoke */,
    nodev /* getacl */,
    nodev /* setacl */,
    nodev /* getpcl */,
    nodev /* setpcl */,
    nodev /* seek */
};

static struct vnode *
make_dead_vnode(struct vfs *vfsp)
{
  struct dead_node *dead;

  XFSDEB(XDEBNODE, ("make_dead_vnode vfsp = 0x%x\n", (u_int) vfsp));

  dead = xfs_alloc(sizeof(*dead));
  bzero((caddr_t)dead, sizeof(*dead));

  dead->vn.v_flag    = V_ROOT;
  dead->vn.v_count   = 1;
  dead->vn.v_vfsgen  = 0;
  dead->vn.v_vfsp    = vfsp;
  dead->vn.v_mvfsp   = NULL;
  dead->vn.v_gnode   = &dead->gn;
  dead->vn.v_next    = NULL;
  dead->vn.v_socket  = NULL;
  dead->vn.v_audit   = NULL;

  dead->gn.gn_type   = VDIR;
  dead->gn.gn_flags  = 0;
  dead->gn.gn_seg    = 0;
  dead->gn.gn_mwrcnt = 0;
  dead->gn.gn_mrdcnt = 0;
  dead->gn.gn_rdcnt  = 0;
  dead->gn.gn_wrcnt  = 0;
  dead->gn.gn_excnt  = 0;
  dead->gn.gn_rshcnt = 0;
  dead->gn.gn_ops    = &dead_vnodeops;
  dead->gn.gn_vnode  = &dead->vn;
  dead->gn.gn_rdev   = 0;
  dead->gn.gn_chan   = 0;

  return &dead->vn;
}

static int
xfs_init (struct gfs *gfs)
{
  XFSDEB(XDEBNODE, ("xfs_init\n"));
  return 0;
}

static int
xfs_rinit (void)
{
  XFSDEB(XDEBNODE, ("xfs_rinit\n"));
  return 0;
}

static struct vfsops xfs_vfsops = {
    xfs_mount,			/* mount */
    xfs_unmount,		/* unmount */
    xfs_root,			/* root */
    xfs_statfs,			/* statfs */
    xfs_sync,			/* sync */
    xfs_vget,			/* vget */
    xfs_cntl,			/* cntl */
    xfs_quotactl		/* quoatctl */
};

static struct gfs xfs_gfs = {
    &xfs_vfsops,		/* gfs_ops */
    &xfs_vnodeops,		/* gn_ops */
    MNT_USRVFS,			/* gfs_type */
    "xfs",			/* gfs_name */
    xfs_init,			/* gfs_init */
    GFS_VERSION4,		/* flags */
    NULL,			/* gfs_data */
    xfs_rinit,			/* gfs_rinit */
    0,				/* gfs_hold */
};

int
xfs_install_filesys (void)
{
    return gfsadd (MNT_USRVFS, &xfs_gfs);
}

int
xfs_uninstall_filesys (void)
{
    return gfsdel (MNT_USRVFS);
}
