/*
 * Copyright (c) 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sl.h>
#include <roken.h>
#include <err.h>

/*
 * Test various features.
 */ 

static struct timeval starttime;
static struct timeval endtime;

static void
timer_start(void)
{
    gettimeofday(&starttime, NULL);
}

static void 
timer_end(void)
{
    gettimeofday(&endtime, NULL);
    
    endtime.tv_sec -= starttime.tv_sec;
    endtime.tv_usec -= starttime.tv_usec;
    if (endtime.tv_usec < 0) {
	endtime.tv_usec += 1000000;
	endtime.tv_sec -= 1;
    }

    printf("Time: %d.%06d\n", (int) endtime.tv_sec, (int) endtime.tv_usec);
}

int 
null_cmd(int argc, char **argv)
{
    printf("I do nothing\n");
    timer_start();
    timer_end();

    return 0;
}

static int
fstat_or_lseek(int argc, char **argv)
{
    struct stat sb;
    char tmp[MAXPATHLEN] = "/tmp/benchXXXXX";
    int fd = mkstemp(tmp);
    int times;
    int rtimes = 100;

    if (argc > 1)
	rtimes = atoi(argv[1]);
	
    if (fd == -1)
	errx(1, "mkstemp");

    times = rtimes;
    printf("fstat: times %d\n", times);
    timer_start();
    while (times--)
	fstat(fd, &sb);
    
    timer_end();

    times = rtimes;
    printf("lseek: times %d\n", times);
    timer_start();
    while(times--)
	lseek(fd, SEEK_END, 0L);

    timer_end();

    close(fd);
    unlink(tmp);

    return 0;
}

#define LSP 10000L

static void
create_large_file(int fd, int filesize)
{
    if (lseek(fd, filesize, SEEK_SET) == -1)
	err(1, "lseek");
    write (fd, "foo", 3);
}

#ifdef HAVE_MMAP
static int
fetch_data_mmap(int fd)
{
    char *foo = mmap(NULL, LSP+4000, PROT_READ, MAP_PRIVATE, fd, 0);
    
    if (foo == (void *) -1)
	err(1, "mmap");

    if (strncmp(foo + LSP, "foo", 3))
	errx(1, "mmap: not correct content");

    munmap(foo, LSP+4000);

    return 0;
}
#endif

static int
fetch_data_read(int fd)
{
    char foo[100];

    lseek(fd, LSP, SEEK_SET);
    if (read(fd, foo, 3) != 3)
	err(1, "read");
    
    if (strncmp(foo, "foo", 3))
	err(1, "read: not correct content");

    return 0;
}



static int
fetch_data(int argc, char **argv)
{
    char tmp[MAXPATHLEN] = "/tmp/benchXXXXX";
    int fd = mkstemp(tmp);
    int times;
    int rtimes = 100;

    if (argc > 1)
	rtimes = atoi(argv[1]);
	
    if (fd == -1)
	errx(1, "mkstemp");

    create_large_file(fd, LSP);

#ifdef HAVE_MMAP
    times = rtimes;
    printf("mmap: times %d\n", times);
    timer_start();
    while (times--)
	fetch_data_mmap(fd);
    
    timer_end();
#endif /* HAVE_MMAP */

    times = rtimes;
    printf("read: times %d\n", times);
    timer_start();
    while(times--)
	fetch_data_read(fd);

    timer_end();

    close(fd);
    unlink(tmp);

    return 0;
}

/*
 *
 */

#ifdef HAVE_MMAP
static int
read_data_mmap(int fd, int filesize)
{
    char *foo;
    long *bar, kaka;
    int size = filesize; 

    foo = mmap(NULL, filesize, PROT_READ, MAP_PRIVATE, fd, 0);

    if (foo == (void *) -1)
	err(1, "mmap");

    bar = (long *) foo;
    filesize -= filesize % sizeof(*bar);

    while (filesize > 0) { 
	kaka = *bar; 
	filesize -= sizeof(*bar); 
    }

    munmap(foo, size);

    return 0;
}
#endif


static int
read_data_read(int fd, int filesize)
{
    /* XXX what is the optimal size */
    char foo[5*1024];
    int error;

    int chunk = sizeof(foo);

    error = lseek(fd, 0, SEEK_SET);
    if (error)
	err(1, "lseek");

    while (filesize > 0) {
	if (chunk > filesize)
	    chunk = filesize;
	error = read(fd, foo, chunk);
	if (error != chunk)
	    err(1, "read: %d  chunk: %d", error, chunk);
	filesize -= chunk;
    }
    
    return 0;
}

static int
read_data(int argc, char **argv)
{
    char tmp[MAXPATHLEN] = "/tmp/benchXXXXX";
    int fd = mkstemp(tmp);
    int times;
    int filesize = 1000;
    int rtimes = 100;

    if (fd == -1)
	errx(1, "mkstemp");

    if (argc > 1)
	rtimes = atoi(argv[1]);

    if (argc > 2)
	filesize = atoi(argv[2]);
    if (argc < 1 && argc > 3)
	errx(1, "read_data times [size]");

    create_large_file(fd, filesize);

    printf("doing the iterations %d times using filesize: %d\n", 
	   rtimes, filesize);

#ifdef HAVE_MMAP
    times = rtimes;
    printf("mmap: times %d\n", times);
    timer_start();
    while (times--)
	read_data_mmap(fd, filesize);
    
    timer_end();
#endif /* HAVE_MMAP */

    times = rtimes;
    printf("read: times %d\n", times);
    timer_start();
    while(times--)
	read_data_read(fd, filesize);

    timer_end();

    close(fd);
    unlink(tmp);

    return 0;
}



/*
 *
 */

static SL_cmd cmds[] = {
    {"null",		
     null_cmd,	
     "do nothing"},
    {"fstat_or_lseek",
     fstat_or_lseek,
     "fast_or_lseek"},
    {"fetch_data",
     fetch_data,
     "fetch data"},
    {"read_data",
     read_data,
     "read large chucks of a file"},
    { NULL, NULL }
};

/*
 *
 */

int 
main(int argc, char **argv)
{
    set_progname(argv[0]);
    
    if (argc > 1)
	sl_command(cmds, argc - 1, argv + 1);
    else
	sl_help(cmds, 1, NULL);
    
    return 0;
}

