/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define __NO_VERSION__

#include <xfs/xfs_locl.h>
#include <xfs/xfs_cache.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_syscalls.h>
#include <xfs/nxfs.h>

RCSID("$Id: xfs_vfsops.c,v 1.40 1999/02/06 00:40:53 map Exp $");

struct xfs xfs[NXFS];

static void xfs_read_inode(struct inode *inode);
static void xfs_delete_inode(struct inode *inode);
static void xfs_put_super(struct super_block *sb);
#ifndef LINUX2_1
void xfs_statfs(struct super_block *, struct statfs *, int);
static int xfs_notify_change(struct inode *inode, struct iattr *attr);
#else
static void xfs_put_inode(struct inode *inode);
static void xfs_write_super(struct super_block * sb);
int xfs_statfs(struct super_block *, struct statfs *, int);
static int xfs_notify_change(struct dentry *dentry, struct iattr *attr);
#endif /* LINUX2_1 */

static struct inode *
make_dead_vnode(struct super_block *sb);

extern struct inode_operations xfs_dead_inode_operations;

static struct super_operations xfs_sops = { 
#ifndef LINUX2_1
    xfs_read_inode,	/* read_inode */
    xfs_notify_change,	/* notify_change */
    NULL,		/* write_inode */
    xfs_delete_inode,	/* put_inode */
    xfs_put_super,	/* put_super */
    NULL,		/* write_super */
    xfs_statfs,		/* xfs_statfs */
    NULL		/* remount */
#else
    xfs_read_inode,	/* read_inode */
    NULL,		/* write_inode */
    xfs_put_inode,	/* put_inode */
    xfs_delete_inode,	/* delete_inode */
    xfs_notify_change,	/* notify_change */
    xfs_put_super,	/* put_super */
    xfs_write_super,	/* write_super */
    xfs_statfs,		/* statfs */
    NULL,		/* remount_fs */
    NULL		/* clear_inode */
#endif /* LINUX2_1 */
};

int
#ifndef LINUX2_1
xfs_root(struct super_block *sb, struct inode **vpp)
#else
xfs_root(struct super_block *sb, struct dentry **dp)
#endif /* LINUX2_1 */
{
    struct xfs *xfsp = VFS_TO_XFS(sb);
    struct xfs_message_getroot msg;
    int error;
    
    XFSDEB(XDEBVFOPS, ("xfs_root sp = 0x%p\n", sb));
    
    do {
	if (xfsp->root != NULL) {
#ifndef LINUX2_1
	    *vpp = XNODE_TO_VNODE(xfsp->root);
#else
	    *dp = d_alloc_root(XNODE_TO_VNODE(xfsp->root), NULL);
	    xfs_d_init(*dp);
	    DENTRY_TO_XDENTRY(*dp)->valid = 1;
#endif /* LINUX2_1 */
	    xfs[0].status |= XFS_ROOTINSTALLED;
	    return 0;
	}
	
	msg.header.opcode = XFS_MSG_GETROOT;
	
	/*
	 * Mounting should done by root, so get the root node with
	 * root's priviliges (usually none, and none is needed).
	 */
	
	msg.cred.uid = 0;
	msg.cred.pag = 0;
	
	error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
	XFSDEB(XDEBVFOPS, ("xfs_root xfs_message_rpc error = %d\n", error));
	if (error == 0)
	    error = ((struct xfs_message_wakeup *) &msg)->error;
	XFSDEB(XDEBVFOPS, ("xfs_root xfs_message_wakeup error = %d\n", error));
    } while (error == 0);
    
    /*
     * Failed to get message through, need to pretend that all went well
     * and return a fake dead vnode to be able to unmount.
     */

#ifndef LINUX2_1
    *vpp = make_dead_vnode(sb);
#else
    *dp = d_alloc_root(make_dead_vnode(sb), NULL);
    xfs_d_init(*dp);
    DENTRY_TO_XDENTRY(*dp)->valid = 1;
#endif /* LINUX2_1 */

    return 0;
}

static struct inode *
make_dead_vnode(struct super_block *sb)
{
    struct inode *dead;
    
    XFSDEB(XDEBNODE, ("make_dead_vnode sb = 0x%p\n", sb));
    
    dead = iget (sb,0);
    if(!dead) { /* XXX - ugly.. better handling */
	panic("make_dead_vnode: iget failed\n");
    }
    
    dead->i_mode = 0555|S_IFDIR;
    dead->i_op = &xfs_dead_inode_operations;
    
    return dead;
}

#ifndef LINUX2_1
struct super_block * xfs_read_super (struct super_block * sb, void * data,
				      int silent)
#else
struct super_block * xfs_read_super (struct super_block * sb, void * data,
				      int silent, struct dentry *dir_d)
#endif
{
    int minordevice=0;
    
    XFSDEB(XDEBVFOPS, ("xfs_read_super starting\n"));
    XFSDEB(XDEBVFOPS, ("xfs_read_super: sb: %p data: %p silent: %d\n",
		       sb, data, silent));
    XFSDEB(XDEBVFOPS, ("xfs_read_super: %d:%d\n",
		       (int) MAJOR(sb->s_dev),
		       (int) MINOR(sb->s_dev)));
    
    XFSDEB(XDEBVFOPS, ("xfs_read_super: begin setting variables\n"));
    MOD_INC_USE_COUNT;
    
    xfs[minordevice].status |= XFS_MOUNTED;
    xfs[minordevice].sb = sb;
    xfs[minordevice].root = 0;
    xfs[minordevice].nnodes = 0;
    xfs[minordevice].nodes = 0;
    xfs[minordevice].fd = minordevice;
    VFS_TO_XFS(sb) = &xfs[minordevice];
    
    sb->s_op = &xfs_sops;
#ifndef LINUX2_1
    if (xfs_root(sb,&sb->s_mounted)) {
	XFSDEB(XDEBVFOPS, ("xfs_read_super: get root inode failed\n"));
	sb->s_dev=0;
	MOD_DEC_USE_COUNT;
	return NULL;
    }
#else
    if (xfs_root(sb,&sb->s_root) || !sb->s_root) {
	XFSDEB(XDEBVFOPS, ("xfs_read_super: get root inode failed\n"));
	sb->s_dev=0;
	MOD_DEC_USE_COUNT;
	return NULL;
    }
#endif /* LINUX2_1 */

    sb->s_blocksize = 1024;
    sb->s_blocksize_bits = 10;
    
    XFSDEB(XDEBVFOPS, ("xfs_read_super: returning\n"));

    return sb;
}

#ifdef LINUX2_1
static void xfs_write_super(struct super_block * sb)
{
    sb->s_dirt = 0;
}
#endif /* LINUX2_1 */

static void xfs_put_super(struct super_block *sb)
{
    int minordevice = 0;
    XFSDEB(XDEBVFOPS, ("xfs_put_super starting\n"));
    xfs[minordevice].status &= ~XFS_MOUNTED;
    xfs[minordevice].status &= ~XFS_ROOTINSTALLED;
    sb->s_dev = 0;
    xfs[minordevice].root = NULL;
    XFSDEB(XDEBVFOPS, ("xfs_put_super exiting\n"));
    MOD_DEC_USE_COUNT;
}

static void xfs_read_inode(struct inode *inode)
{
    XFSDEB(XDEBVFOPS,("xfs_read_inode starting inode: %p\n",inode));
    XFSDEB(XDEBVFOPS,("xfs_read_inode inode->i_ino: %ld\n",inode->i_ino));
    inode->i_blksize = 1024;
    inode->i_mode = 0;
    inode->i_op = NULL;
    XFSDEB(XDEBVFOPS,("xfs_read_inode exiting\n"));
}

#ifdef LINUX2_1
static void xfs_put_inode(struct inode *inode)
{
    XFSDEB(XDEBVFOPS,("xfs_put_inode: inode: %p count: %d aliases:",
		      inode, inode->i_count));
    print_aliases(inode);
    if (inode->i_count == 1)
	inode->i_nlink = 0;
}
#endif

static void xfs_delete_inode(struct inode *inode)
{
    struct xfs_message_inactivenode msg;
    struct xfs *xfsp = XFS_FROM_VNODE(inode);
    struct xfs_node *xn = VNODE_TO_XNODE(inode);

    XFSDEB(XDEBNODE,
	   ("xfs_delete_inode inode = 0x%p i_ino:0x%lx i_count:%u\n",
	    inode, inode->i_ino, inode->i_count));

    if (xn == NULL)
	return;

    msg.header.opcode = XFS_MSG_INACTIVENODE;
    msg.handle = xn->handle;
    msg.flag   = XFS_NOREFS | XFS_DELETE;
    xfs_message_send(xfsp->fd, &msg.header, sizeof(msg));

#ifndef LINUX2_1
    xfs_cache_delete(xn);
#endif /* LINUX2_1 */

    free_xfs_node(xn);
    clear_inode(inode);
}

void
vattr2xfs_attr(struct iattr *iattr, struct xfs_attr *attr)
{
    XA_CLEAR(attr);
    if (iattr->ia_valid & ATTR_MODE)
	XA_SET_MODE(attr,iattr->ia_mode);
    if (iattr->ia_valid & ATTR_UID)
	XA_SET_UID(attr,iattr->ia_uid);
    if (iattr->ia_valid & ATTR_GID)
	XA_SET_GID(attr,iattr->ia_gid);
    if (iattr->ia_valid & ATTR_ATIME)
	XA_SET_ATIME(attr,iattr->ia_atime);
    if (iattr->ia_valid & ATTR_MTIME)
	XA_SET_MTIME(attr,iattr->ia_mtime);
    if (iattr->ia_valid & ATTR_CTIME)
	XA_SET_CTIME(attr,iattr->ia_ctime);
    if (iattr->ia_valid & ATTR_SIZE)
	XA_SET_SIZE(attr,iattr->ia_size);
}

static int
#ifndef LINUX2_1
xfs_notify_change(struct inode *inode, struct iattr *attr)
#else
xfs_notify_change(struct dentry *dentry, struct iattr *attr)
#endif
{
#ifdef LINUX2_1
    struct inode *inode = dentry->d_inode;
#endif
    struct xfs *xfsp = XFS_FROM_VNODE(inode);
    struct xfs_node *xn = VNODE_TO_XNODE(inode);
    int error = 0;
    
    XFSDEB(XDEBNODE, ("xfs_notify_change\n"));
    
    if (XFS_TOKEN_GOT(xn, XFS_ATTR_W)) {
        /* Update attributes and mark them dirty. */
        VNODE_TO_XNODE(inode)->flags |= XFS_ATTR_DIRTY;
	return -EINVAL;                /* XXX not yet implemented */
    } else {
        struct xfs_message_putattr msg;

        msg.header.opcode = XFS_MSG_PUTATTR;
	msg.cred.uid = current->uid;
	msg.cred.pag = xfs_get_pag();
        msg.handle = xn->handle;
        vattr2xfs_attr(attr, &msg.attr);
	inode_setattr(inode, attr);
        error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
        if (error == 0) {
            error = ((struct xfs_message_wakeup *) & msg)->error;
	    
#ifndef LINUX2_1
	    inode->i_dirt = 0;
#endif /* LINUX2_1 */
	}
        XFS_TOKEN_CLEAR(xn, XFS_ATTR_VALID, XFS_ATTR_MASK);
    }
    
    return error;
}

#ifndef LINUX2_1
void
#else
int
#endif /* LINUX2_1 */
xfs_statfs(struct super_block *sb, struct statfs *buf, int bufsiz)
{
    struct statfs tmp;
    
    tmp.f_type = 0x47114711;
    tmp.f_bsize = sb->s_blocksize;
    tmp.f_blocks = 1024*1024;
    tmp.f_bfree = 1024*1024;
    tmp.f_bavail = 1024*1024;
    tmp.f_files = 0;
    tmp.f_ffree = 1024*1024;
    tmp.f_namelen = NAME_MAX;
#ifdef LINUX2_1
    return
#endif
    copy_to_user(buf, &tmp, bufsiz) ? -EFAULT : 0;
}
