/*
Copyright (C) 1992,1993,1994,1995 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/MOSS source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/


/*
 * validate_cert: validate the given certificate according to the issuer
 *		  certificate and crl given
 *
 * It is valid to pass a NULL issuer certificate or a NULL crl.  In this
 * case the signature or crl or both tests are not done, respectively.
 *
 * The status bit flag in the certificate wrapper is set according to
 * the result of performing the validation tests.  Specifically, as each
 * test is performed, if it is successful the bit is turned off,
 * otherwise the bit is set.  Bits are *not* changed unless the test is
 * actually performed.  Thus, for related bits which represent the
 * success or failure of subtests of a test, only the *first* set bit is
 * significant.  See disp_cert_status() for more information.
 *
 * Three "suites" of validation tests are performed:
 *
 *	check the validity period
 *	check the signature
 *	check the crl to see if the certificate is a member
 *
 * In addition, the certificate is checked to see if the subject and
 * issuer names are equal.  If they are, the status of the certificate
 * is set to indicate the certificate is self-signed.
 *
 * Finally, a significant speedup in the validation process is realized
 * by honoring the BBADSIG bit if it is turned off.
 *
 * RETURN CODES
 *	OK	- certificate status has been set accordingly
 *
 *	EINVAL	- an invalid argument was received; processing stops
 *
 * RETURN VALUES
 *	certificate status - if OK is returned then the status of the
 *		certificate parameter is guaranteed to be set accordingly
 */

#include "config.h"
#include "general.h"

#include "bbuf.h"
#include "cert.h"
#include "crypto.h"
#include "dname.h"
#include "util.h"

int validate_cert(cert, icert, crl)
struct bbuf	*cert, *icert, *crl;
{
    struct bbuf		*ikey = NULLBB;

    struct tobesigned	*tobesigned = NULL_TOBESIGNED;
    struct bbuf		*etobesigned = NULLBB;

    struct bbuf		*mic = NULLBB;
    struct bbuf		*emic = NULLBB;
    struct bbuf		*sig = NULLBB;

    struct bbuf		*subject = NULLBB;
    struct bbuf		*issuer = NULLBB;

    struct algid	*i_sig_aid = NULL_ALGID;
    struct algid	*o_sig_aid = NULL_ALGID;
    struct algid	*mic_aid = NULL_ALGID;

    int			ret = OK, slen;
    long		begin, end, clock;

    /* check to see that we got a pointer to a certificate */

    if (cert == NULLBB) 
	return(EINVAL);

    /* NULL icert is ok */
    /* NULL crl is ok */

    /* Validation testing begins here.  It is invalid for this routine
     * to return until all tests have been completed.
     */

    /* get and check the certificate validity period */
    if (valid_cert(&cert, &begin, &end, READ) != OK) 
	cert->status |= BNOVAL;
    else {
	cert->status &= ~BNOVAL;
	/* check begin and end times (begin <= clock <= end) */
	clock = TIME();
	if (begin >= clock || clock >= end) 
	    cert->status |= BBADVAL;
	else
	    cert->status &= ~BBADVAL;
    }

    /* make sure subject and issuer names exist */
    if (subject_cert(&cert, &subject, READ) == OK && subject != NULLBB) {

	cert->status &= ~BNOSDN;

	if (issuer_cert(&cert, &issuer, READ) == OK && issuer != NULLBB) {

	    cert->status &= ~BNOIDN;

	    /* as long as we're here, check to see if this certificate
	     * is self-signed, ie, the subject and issuer names are equal
	     */
	    if (dncmp(subject, issuer))
		cert->status &= ~BSLFSGN;
	    else
		cert->status |= BSLFSGN;
	}
	else
	    cert->status |= BNOIDN;
    }
    else
	cert->status |= BNOSDN;

    /* we set BNOSIG now instead of for each goto below */
    cert->status |= BNOSIG;

    /* get the signature from the certificate */
    sig = alloc_bbuf();
    if (sig_cert(&cert, &(sig->data), &slen, READ) != OK) {
	sig->length = (short)slen;
	goto skipsig;
    }
    sig->length = (short)slen;
    cert->status &= ~BNOSIG;

    /* we set BNOISS now instead of for each goto below */
    cert->status |= BNOISS;

    if (icert == NULLBB) 
	goto skipsig;

    /* get the issuer's public key */
    if (key_cert(&icert, &ikey, READ) != OK) 
	goto skipsig;

    cert->status &= ~BNOISS;

    /* we set BBADSIG now instead of for each goto below */
    cert->status |= BBADSIG;

    /* get the inner signature algorithm identifier */
    if (sig_alg_cert(&cert, &i_sig_aid, INNER, READ) != OK) 
	goto skipsig;

    /* get the outer signature algorithm identifier */
    if (sig_alg_cert(&cert, &o_sig_aid, OUTER, READ) != OK) 
	goto skipsig;

    /* TBD: check parameters too? */

    if (i_sig_aid->alg != o_sig_aid->alg) 
	goto skipsig;

    mic_aid = alloc_algid(); 

    mic_aid->alg = i_sig_aid->alg & HASH_MASK;

    /* get the tobesigned field */
    if (tobesigned_cert(&cert, &tobesigned, READ) != OK
	|| tobesigned == NULL_TOBESIGNED) {
	goto skipsig;
    }

    /* make sure subject and issuer names in TBS are DER */
    if (subject != NULLBB
	&& (tobesigned->subject = der_dn(subject)) != NULLBB)
	cert->status &= ~BNOSDN;
    else
	cert->status |= BNOSDN;

    if (issuer != NULLBB
	&& (tobesigned->issuer = der_dn(issuer)) != NULLBB)
	cert->status &= ~BNOIDN;
    else
        cert->status |= BNOIDN;

    /* encode the certificate.tobesigned field */
    if (encode_ctbs(tobesigned, &etobesigned) != OK) 
	goto skipsig;

    /* generate a mic for the encoded tobesigned field */
    if (gen_mic(mic_aid->alg, NULLBB, etobesigned, &mic) != OK) 
	goto skipsig;

    if (encode_mic(mic, mic_aid, &emic) != OK) 
	goto skipsig;

    if (set_key(ikey) != OK) 
	goto skipsig;

    if (verify(sig, emic) != OK) 
	goto skipsig;

    cert->status &= ~BBADSIG;

 skipsig:

    /* check for crl membership */
    if (crl != NULLBB) {
	cert->status &= ~BNOCRL;
	if (check_crl(cert, crl) == BBADCRL)
	    cert->status |= BBADCRL;
	else
	    cert->status &= ~BBADCRL;
    }
    else
	cert->status |= BNOCRL;

    /* ret = OK; */

    cert->status &= BUNKNOWN;	/* zero out unused bits */

    FREE_BBUF(ikey);

    FREE_TOBESIGNED(tobesigned);
    FREE_BBUF(etobesigned);

    FREE_BBUF(mic);
    FREE_BBUF(emic);
    FREE_BBUF(sig);

    FREE_BBUF(subject);
    FREE_BBUF(issuer);

    FREE_ALGID(i_sig_aid);
    FREE_ALGID(o_sig_aid);
    FREE_ALGID(mic_aid);

    return(ret);
}
