// winkey.cpp
// Windows OPIE key calculator
// David Aylesworth
// July, 1995

#include "stdafx.h"
extern "C"
{
#include "opie.h"
}
#include "winkey.h"
#include "optdlg.h"


CKeyApp NEAR keyApp;

BOOL CKeyApp::InitInstance()
{
   m_dialog.DoModal();
   return FALSE;
}

   
/////////////////////////////////////////////////////////////////////////////
// CKeyDialog dialog


CKeyDialog::CKeyDialog(CWnd* pParent /*=NULL*/)
   : CDialog(CKeyDialog::IDD, pParent)
{
   //{{AFX_DATA_INIT(CKeyDialog)
   m_password = _T("");
   //}}AFX_DATA_INIT
}


void CKeyDialog::DoDataExchange(CDataExchange* pDX)
{
   CDialog::DoDataExchange(pDX);
   //{{AFX_DATA_MAP(CKeyDialog)
   DDX_Control(pDX, IDC_EDIT1, m_challenge);
   DDX_Control(pDX, IDC_EDIT3, m_response);
   DDX_Text(pDX, IDC_EDIT2, m_password);
   //}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CKeyDialog, CDialog)
   //{{AFX_MSG_MAP(CKeyDialog)
   ON_BN_CLICKED(IDOK, OnCompute)
   ON_BN_CLICKED(ID_OPTIONS, OnOptions)
   ON_WM_DESTROY()
   //}}AFX_MSG_MAP
   ON_WM_PAINT()
   ON_WM_ACTIVATE()
   ON_WM_ERASEBKGND()
END_MESSAGE_MAP()


BOOL CKeyDialog::OnInitDialog() 
{
   CDialog::OnInitDialog();

   // edit menu
   CMenu *menu = GetSystemMenu(FALSE);
   menu->DeleteMenu(SC_MAXIMIZE, MF_BYCOMMAND);
   menu->DeleteMenu(SC_SIZE, MF_BYCOMMAND);

   // set window position
   int x = keyApp.GetProfileInt("Defaults", "X", 20);
   int y = keyApp.GetProfileInt("Defaults", "Y", 20);
   SetWindowPos(NULL, x, y, 0, 0, SWP_NOZORDER | SWP_NOSIZE);
   
   // get autopaste flag
   m_autopaste = keyApp.GetProfileInt("Defaults", "AutoPaste", 1);
   
   // get autocopy flag   
   m_autocopy = keyApp.GetProfileInt("Defaults", "AutoCopy", 1);
   
   // get autodecrement flag
   m_autodecrement = keyApp.GetProfileInt("Defaults", "AutoDecrement", 0);
   
   // get autoclear flag
   m_autoclear = keyApp.GetProfileInt("Defaults", "AutoClear", 0);
   
   // get algorithm
   CString alg = keyApp.GetProfileString("Defaults", "Algorithm", "MD5");
   m_algorithm = (alg == "MD5");
   
   if (!m_autopaste)
      m_challenge.SetWindowText(keyApp.GetProfileString("Defaults", "Challenge", ""));

   //UpdateData(FALSE);

   return TRUE;
}


void CKeyDialog::OnActivate(UINT nState, CWnd *pWndOther, BOOL bMinimized)
{
   if (nState != WA_INACTIVE) {
      // set focus to challenge field
      m_challenge.SetFocus();
      m_challenge.SetSel(0, -1, TRUE);

      if (m_autopaste) {
         // paste clipboard text into challenge
         OpenClipboard();
         HANDLE handle = GetClipboardData(CF_TEXT);
         if (handle) {
            CString s = (char FAR *) GlobalLock(handle);
            GlobalUnlock(handle);
            CloseClipboard();
            int algorithm, sequence;
            CString seed;
            if (ParseChallenge(s, algorithm, sequence, seed))
               m_challenge.Paste();
               m_challenge.SetSel(0, -1, TRUE);
         }
         else {
            CloseClipboard();
         }
      }
   }
}


void CKeyDialog::OnPaint()
{
   if (IsIconic()) {           
      // draw icon if minimized
      PAINTSTRUCT ps;
      CDC *dc = BeginPaint(&ps);
      HICON icon = keyApp.LoadIcon(IDR_MAINFRAME);
      DefWindowProc(WM_ICONERASEBKGND, (WORD)ps.hdc, 0L);
      dc->DrawIcon(0, 0, icon);
      EndPaint(&ps);
   }
   else
      CDialog::OnPaint();
}
   

BOOL CKeyDialog::OnEraseBkgnd(CDC *dc)
{
   // don't erase background if minimized
   if (IsIconic())
      return TRUE;
   else
      return CWnd::OnEraseBkgnd(dc);
}

         
void CKeyDialog::OnCompute() 
{
   HCURSOR oldCursor = SetCursor(keyApp.LoadStandardCursor(IDC_WAIT));
   UpdateData(TRUE);

   /* parse challenge */
   CString challenge;
   m_challenge.GetWindowText(challenge);
   int algorithm;
   int keynum;
   CString seed;
   
   if (!ParseChallenge(challenge, algorithm, keynum, seed)) {
      MessageBox("Invalid OTP challenge.", "Error", MB_ICONEXCLAMATION);
      if (m_autoclear) {
         m_password.Empty();
         UpdateData(FALSE);
      }
      SetCursor(oldCursor);
      return;
   }
   
   /* Crunch seed and secret password into starting key normally */
   if (algorithm == -1)
      algorithm = m_algorithm + 4;
   char key[8];   
   if (opiekeycrunch((unsigned)algorithm, key, (char *)(const char *)seed,
                     (char *)(const char *)m_password) != 0) {
      MessageBox("Key crunch failed.", "Error", MB_ICONEXCLAMATION);
      if (m_autoclear) {
         m_password.Empty();
         UpdateData(FALSE);
      }
      SetCursor(oldCursor);
      return;
   }

   // clear password
   if (m_autoclear) {
      m_password.Empty();
      UpdateData(FALSE);
   }
      
   char buf[33];      

   if (m_autodecrement && keynum>0) {
      // decrement sequence #
      sprintf(buf, "%s %d %s", (algorithm == 4) ? "otp-md4" : "otp-md5",
              keynum-1, seed);
      challenge = buf;
      m_challenge.SetWindowText(challenge);
   }
   
   keyApp.WriteProfileString("Defaults", "Challenge", challenge);

   while (keynum-- != 0)
      opiehash(key, algorithm);

   CString response = opiebtoe(buf, key);
   //response += "\n";
   m_response.SetWindowText(response);
   m_response.SetFocus();
   m_response.SetSel(0, -1, TRUE);
   //m_response.LineScroll(0);
   if (m_autocopy)
      m_response.Copy();
      
   //UpdateData(FALSE);
   SetCursor(oldCursor);
}


void CKeyDialog::OnOptions() 
{
   // create options dialog
   COptionsDlg *dlg;
   dlg = new COptionsDlg(this);
   if (dlg)
      dlg->DoModal();
   delete dlg;
}


BOOL CKeyDialog::ParseChallenge(CString challenge,
                                int &algorithm, int &sequence, CString &seed)
{
   char *s = challenge.GetBuffer(0);
   if (s == NULL || *s == '\0')
      return FALSE;

   // parse algorithm      
   char *t = strtok(s, " ");
   if (t == NULL) {
      challenge.ReleaseBuffer();
      return FALSE;
   }
   if (!stricmp(t, "otp-md4")) {
      algorithm = 4;
      t = strtok(NULL, " ");
   }
   else if (!stricmp(t, "otp-md5")) {
      algorithm = 5;
      t = strtok(NULL, " ");
   }
   else
      algorithm = -1;

   // parse sequence number
   if (t == NULL) {
      challenge.ReleaseBuffer();
      return FALSE;
   }
   sequence = atoi(t);
   if (sequence < 1) {
      challenge.ReleaseBuffer();
      return FALSE;
   }
   
   // parse seed
   t = strtok(NULL, " ");
   if (t == NULL) {
      challenge.ReleaseBuffer();
      return FALSE;
   }
   seed = t;   

   challenge.ReleaseBuffer();   
   return TRUE;
}

void CKeyDialog::OnDestroy()
{
   CDialog::OnDestroy();

   RECT rect;   
   GetWindowRect(&rect);
   keyApp.WriteProfileInt("Defaults", "X", rect.left);
   keyApp.WriteProfileInt("Defaults", "Y", rect.top);
}
