/* popen.c: A "safe" pipe open routine.

Portions of this software are Copyright 1995 by Randall Atkinson and Dan
McDonald, All Rights Reserved. All Rights under this copyright are assigned
to the U.S. Naval Research Laboratory (NRL). The NRL Copyright Notice and
License Agreement applies to this software.

	History:

	Modified at NRL for OPIE 2.0.
	Originally from BSD.
*/
/*
 * Copyright (c) 1988 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software written by Ken Arnold and
 * published in UNIX Review, Vol. 6, No. 8.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef lint
static char sccsid[] = "@(#)popen.c	5.9 (Berkeley) 2/25/91";

#endif	/* not lint */

#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "opie_cfg.h"

char **ftpglob __ARGS((register char *));
char **copyblk __ARGS((char **));
void blkfree __ARGS((char **));

/*
 * Special version of popen which avoids call to shell.  This insures noone
 * may create a pipe to a hidden program as a side effect of a list or dir
 * command.
 */
#ifdef IS_A_BSD
static int *pids;
#else
static pid_t *pids;
#endif

static int fds;

FILE *ftpd_popen(program, type)
char *program, *type;
{
  register char *cp;
  FILE *iop;
  int argc, gargc, pdes[2];
  char **pop, *argv[100], *gargv[1000], *vv[2];

#ifdef IS_A_BSD
  int pid;

#else
  pid_t pid;

#endif

  if (*type != 'r' && *type != 'w' || type[1])
    return (NULL);

  if (!pids) {
    if ((fds = sysconf(_SC_OPEN_MAX)) <= 0)
      return (NULL);
#ifdef IS_A_BSD
    if ((pids = (int *) malloc((u_int) (fds * sizeof(int)))) == NULL)
      return (NULL);
    memset((char *) pids, 0, fds * sizeof(int));
#else
    if ((pids = (pid_t *) malloc((u_int) (fds * sizeof(pid_t)))) == NULL)
      return (NULL);
    memset((char *) pids, 0, fds * sizeof(pid_t));
#endif
  }
  if (pipe(pdes) < 0)
    return (NULL);

  /* break up string into pieces */
  for (argc = 0, cp = program;; cp = NULL)
    if (!(argv[argc++] = strtok(cp, " \t\n")))
      break;

  /* glob each piece */
  gargv[0] = argv[0];
  for (gargc = argc = 1; argv[argc]; argc++) {
    if (!(pop = (char **) ftpglob(argv[argc]))) {
      /* globbing failed */
      vv[0] = argv[argc];
      vv[1] = NULL;
      pop = (char **) copyblk(vv);
    }
    argv[argc] = (char *) pop;	/* save to free later */
    while (*pop && gargc < 1000)
      gargv[gargc++] = *pop++;
  }
  gargv[gargc] = NULL;

  iop = NULL;
  switch (pid = vfork()) {
  case -1:	/* error */
    close(pdes[0]);
    close(pdes[1]);
    goto pfree;
    /* NOTREACHED */
  case 0:	/* child */
    if (*type == 'r') {
      if (pdes[1] != 1) {
	dup2(pdes[1], 1);
	dup2(pdes[1], 2);	/* stderr, too! */
	close(pdes[1]);
      }
      close(pdes[0]);
    } else {
      if (pdes[0] != 0) {
	dup2(pdes[0], 0);
	close(pdes[0]);
      }
      close(pdes[1]);
    }
    execv(gargv[0], gargv);
    _exit(1);
  }
  /* parent; assume fdopen can't fail...  */
  if (*type == 'r') {
    iop = fdopen(pdes[0], type);
    close(pdes[1]);
  } else {
    iop = fdopen(pdes[1], type);
    close(pdes[0]);
  }
  pids[fileno(iop)] = pid;

pfree:for (argc = 1; argv[argc] != NULL; argc++) {
    blkfree((char **) argv[argc]);
    free((char *) argv[argc]);
  }
  return (iop);
}

int ftpd_pclose(iop)
FILE *iop;
{
  register int fdes;

#ifdef IS_A_BSD
  union wait *stat_loc;
  int omask;
  int pid;

#else	/* POSIX usage */
  sigset_t *omask;
  pid_t pid;
  pid_t stat_loc;

#endif

  /* pclose returns -1 if stream is not associated with a `popened' command,
     or, if already `pclosed'. */
  if (pids == 0 || pids[fdes = fileno(iop)] == 0)
    return (-1);
  fclose(iop);
  sigprocmask(SIG_BLOCK,
#ifndef IS_A_BSD
	      (sigset_t *)
#endif
	      (sigmask(SIGINT) | sigmask(SIGQUIT) | sigmask(SIGHUP)), omask);

  while ((pid = wait((int *) stat_loc)) != pids[fdes] && (pid != -1));
  sigprocmask(SIG_SETMASK, omask, NULL);
  pids[fdes] = 0;

#ifdef WEXITSTATUS
  /* this is the fully POSIX compliant implementation */
  return (pid == -1 ? -1 : WEXITSTATUS(stat_loc));
#else
  return (pid == -1 ? -1 : stat_loc.w_status);
#endif
}
