/* login.c: The infamous /bin/login

Portions of this software are Copyright 1995 by Randall Atkinson and Dan
McDonald, All Rights Reserved. All Rights under this copyright are assigned
to the U.S. Naval Research Laboratory (NRL). The NRL Copyright Notice and
License Agreement applies to this software.

	History:

        Modified at NRL for OPIE 2.02. Flush stdio after printing login
                prompt. Fixed Solaris shadow password problem introduced
                in OPIE 2.01 (the shadow password structure is spwd, not
                spasswd).
        Modified at NRL for OPIE 2.01. Changed password lookup handling
                to use a static structure to avoid problems with drain-
                bamaged shadow password packages. Make sure to close
                syslog by function to avoid problems with drain bamaged
                syslog implementations. Log a few interesting errors.
	Modified at NRL for OPIE 2.0.
	Modified at Bellcore for the Bellcore S/Key Version 1 software
		distribution.
	Originally from BSD.
*/
/*
 * Portions of this software are
 * Copyright (c) 1980,1987 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */
#ifndef lint
char copyright[] =
"@(#) Copyright (c) 1980 Regents of the University of California,\n\
      All Rights Reserved.\n";

#endif	/* not lint */

#ifndef lint
static char sccsid[] = "@(#)login.c	5.20 (Berkeley) 10/1/87";

#endif	/* not lint */

#include "opie_cfg.h"	/* OPIE: defines symbols for filenames & pathnames */

/*
 * USAGE:
 * login [ name ]
 * login -r hostname    (for rlogind)
 * login -h hostname    (for telnetd, etc.)
 * login -f name        (for pre-authenticated login: datakit, xterm, etc.)
 */


/*---------------------- INCLUDE FILES HERE -----------------------*/
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <sys/file.h>
#include <utmp.h>
#include <signal.h>
#include <pwd.h>	/* POSIX Password routines */
#include <stdio.h>
#include <errno.h>
#include <unistd.h>	/* Basic POSIX macros and functions */
#include <termios.h>	/* POSIX terminal I/O */
#include <string.h>	/* ANSI C string functions */
#include <fcntl.h>	/* File I/O functions */
#include <syslog.h>
#include <grp.h>
#include <netdb.h>
#include <netinet/in.h>	/* contains types needed for next include file */
#include <arpa/inet.h>	/* Inet addr<-->ascii functions */
#include <stdlib.h>

#ifdef	QUOTA
#include <sys/quota.h>
#endif

struct utmp utmp;

#ifdef BSD4_3
#include <sys/ioctl.h>	/* non-portable routines used only a few places */
#include <ttyent.h>
#endif

#include "opie.h"

#ifndef DODISC	/* Is the line discipline code even necessary? */
#define DODISC 0
#endif	/* DODISC */

#if DOUTMPX
/* On many System V systems, we include and use extended utmp -- utmpx
because the ut_host member is present in utmpx but not utmp. The defines in
opie_cfg.h should also help with portability. */

#include <utmpx.h>
struct utmpx utmpx;

#endif	/* DOUTMPX */

void getutmpentry __ARGS((char *, struct UTMPX *));
void pututmpentry __ARGS((char *, struct UTMPX *));

#define TTYGID(gid)	tty_gid(gid)	/* gid that owns all ttys */

#define NMAX	sizeof(UTMPX.ut_name)
#define HMAX	sizeof(UTMPX.ut_host)

#ifdef SUNOS
#include <lastlog.h>
#endif

int rflag = 0;
int usererr = -1;
int stopmotd;
char rusername[NMAX + 1];
char name[NMAX + 1] = "";
char me[MAXHOSTNAMELEN];
char *rhost;
char minusnam[16] = "-";
char *envinit[1];	/* now set by setenv calls */
char term[64] = "\0";	/* important to initialise to a NULL string */
char host[sizeof(UTMPX.ut_host)] = "\0";
struct passwd nouser;
struct passwd thisuser;

#ifdef SOLARIS
#include <shadow.h>
char *ttyprompt;
int first = 1;

#endif	/* SOLARIS */
#ifdef PERMSFILE
extern char *home;

#endif	/* PERMSFILE */
struct termios attr, attrnoecho;

extern int errno;

static int ouroptind;
static char *ouroptarg;

#ifdef SUNOS
/*
 * lastlog is not portable to System V systems, not even to Solaris 2.x
 * It is retained only for historical reasons and only within appropriate
 * defines.  Support for lastlog might be removed in a later release.
 *               rja
 */
#ifndef _PATH_LASTLOG
#define _PATH_LASTLOG "/var/adm/lastlog"
#endif	/* _PATH_LASTLOG */

char lastlog[] = _PATH_LASTLOG;

#endif

/*
 * The "timeout" variable bounds the time given to login.
 * We initialize it here for safety and so that it can be
 * patched on machines where the default value is not appropriate.
 */
int timeout = 300;

void getstr __ARGS((char *, int, char *));
void closelog __ARGS((void));

#ifdef SUNOS
#include <des_crypt.h>
#endif	/* SUNOS */
#ifdef SOLARIS
#include <crypt.h>	/* Solaris 2.x */
#endif	/* SOLARIS */

#ifdef TIOCSWINSZ
/* Windowing variable relating to JWINSIZE/TIOCSWINSZ/TIOCGWINSZ. This is
available on BSDish systems and at least Solaris 2.x, but portability to
other systems is questionable. Use within this source code module is
protected by suitable defines.

I'd be interested in hearing about a more portable approach. rja */

struct winsize win =
{0, 0, 0, 0};

#endif


/*------------------ BEGIN REAL CODE --------------------------------*/

/* We allow the malloc()s to potentially leak data out because we can
only call this routine about four times in the lifetime of this process
and the kernel will free all heap memory when we exit or exec. */
int lookupuser()
{
  struct passwd *pwd;
#ifdef SOLARIS
  struct spwd *spwd;
#endif /* SOLARIS */

  memcpy(&thisuser, &nouser, sizeof(thisuser));

  if (!(pwd = getpwnam(name)))
    return -1;

  thisuser.pw_uid = pwd->pw_uid;
  thisuser.pw_gid = pwd->pw_gid;

  if (!(thisuser.pw_name = malloc(strlen(pwd->pw_name) + 1)))
    goto lookupuserbad;
  strcpy(thisuser.pw_name, pwd->pw_name);

  if (!(thisuser.pw_dir = malloc(strlen(pwd->pw_dir) + 1)))
    goto lookupuserbad;
  strcpy(thisuser.pw_dir, pwd->pw_dir);

  if (!(thisuser.pw_shell = malloc(strlen(pwd->pw_shell) + 1)))
    goto lookupuserbad;
  strcpy(thisuser.pw_shell, pwd->pw_shell);

#ifdef SOLARIS
  if (!(spwd = getspnam(name)))
	goto lookupuserbad;

  pwd->pw_passwd = spwd->sp_pwdp;

  endspent();
#endif	/* SOLARIS */

  if (!(thisuser.pw_passwd = malloc(strlen(pwd->pw_passwd) + 1)))
    goto lookupuserbad;
  strcpy(thisuser.pw_passwd, pwd->pw_passwd);

  endpwent();

  return ((thisuser.pw_passwd[0] == '*') || (thisuser.pw_passwd[0] == '#'));

lookupuserbad:
  memcpy(&thisuser, &nouser, sizeof(thisuser));
  return -1;
}

void getloginname()
{
  register char *namep;
  char c, d;

  memset(name, 0, sizeof(name));

  d = 0;
  while (name[0] == '\0') {
    namep = name;
#ifdef SOLARIS
    if (ttyprompt) {
      if (first)
	first--;
      else
	printf(ttyprompt);
    } else
#endif	/* SOLARIS */
      printf("login: ");
    fflush(stdout);
    if (++d == 3)
      exit(0);
    while (1) {
      c = getchar();
      if (feof(stdin) || (c == EOF)) {
        syslog(LOG_CRIT, "End-of-file on stdin!");
	exit(0);
      }
      c = c & 0x7f;
      if (c == '\n' || c == '\r')
	break;
      if (c < 32)
	continue;
      if (c == ' ')
	c = '_';
      if (namep < (name + sizeof(name)))
	*namep++ = c;
    }
  }
}


void timedout(i)
int i;
{
  /* input variable declared just to keep the compiler quiet */
  printf("Login timed out after %d seconds\n", timeout);
  syslog(LOG_CRIT, "Login timed out after %d seconds!", timeout);
  exit(0);
}

void catch(i)
int i;
{
  /* the input variable is declared to keep the compiler quiet */
  signal(SIGINT, SIG_IGN);
  stopmotd++;
}

void catchexit()
{
  int i;
  tcsetattr(STDIN_FILENO, TCSANOW, &attr);
  putchar('\n');
  closelog();
  for (i = sysconf(_SC_OPEN_MAX); i > 2; i--)
    close(i);
}

int rootterm(ttyn)
char *ttyn;
{
#ifdef BSD4_3
/* The getttynam() call and the ttyent structure first appeared in 4.3 BSD and
are not portable to System V systems such as Solaris 2.x. or modern versions
of IRIX rja */
  register struct ttyent *t;
  char *tty;

  tty = strrchr(ttyn, '/');

  if (tty == NULL)
    tty = ttyn;
  else
    tty++;

  if ((t = getttynam(tty)) != NULL)
    return (t->ty_status & TTY_SECURE);

  return (1);	/* when in doubt, allow root logins */

#elif defined(SVR4)

  FILE *filno;
  char line[128];
  char *next, *next2;

/* SVR4 only permits two security modes for root logins: 1) only from CONSOLE,
if the string "CONSOLE=/dev/console" exists and is not commented out with "#"
characters, or 2) from anywhere.

So we open /etc/default/login file grab the file contents one line at a time
verify that the line being tested isn't commented out check for the substring
"CONSOLE" and decide whether to permit this attempted root login/su. */

  if ((filno = fopen("/etc/default/login", "r")) != NULL) {
    while (fgets(line, 128, filno) != NULL) {
      next = line;

      if ((line[0] != '#') && (next = strstr(line, "CONSOLE"))) {
	next += 7;	/* get past the string "CONSOLE" */

	while (*next && (*next == ' ') || (*next == '\t'))
	  next++;

	if (*(next++) != '=')
	  break;	/* some weird character, get next line */

	next2 = next;
	while (*next2 && (*next2 != '\t') && (*next2 != ' ') &&
	       (*next2 != '\n'))
	  next2++;
	*next2 = 0;

	return !strcmp(ttyn, next);	/* Allow the login if and only if the
					   user's terminal line matches the
					   setting for CONSOLE */
      }
    }	/* end while another line could be obtained */
  }	/* end if could open file */
  return (1);	/* when no CONSOLE line exists, root can login from anywhere */
#elif DOSECURETTY
  {
    FILE *f;
    char buffer[1024], *c;
    int rc = 0;

    if (!(f = fopen(SECURETTY, "r")))
      return 1;

    if (c = strstr(ttyn, "/dev/"))
      ttyn += 5;

    if (c = strrchr(ttyn, '/'))
      ttyn = ++c;

    while (fgets(buffer, sizeof(buffer), f)) {
      if (c = strrchr(buffer, '\n'))
	*c = 0;

      if (!(c = strrchr(buffer, '/')))
	c = buffer;
      else
	c++;

      if (!strcmp(c, ttyn))
	rc = 1;
    };

    fclose(f);
    return rc;
  }
#else
  return (1);	/* when in doubt, allow root logins */
#endif
}

int doremotelogin(host)
char *host;
{
  int rc;

  getstr(rusername, sizeof(rusername), "remuser");
  getstr(name, sizeof(name), "locuser");
  getstr(term, sizeof(term), "Terminal type");
  if (getuid()) {
    memcpy(&thisuser, &nouser, sizeof(thisuser));
    syslog(LOG_ERR, "getuid() failed");
    return (-1);
  }
  if (lookupuser()) {
    syslog(LOG_ERR, "lookup failed for user %s", name);
    return (-1);
  }
  rc = ruserok(host, !thisuser.pw_uid, rusername, name);
  if (rc == -1) {
    syslog(LOG_ERR,
    "ruserok failed, host=%s, uid=%d, remote username=%s, local username=%s",
	   host, thisuser.pw_uid, rusername, name);
  }
  return rc;
}


void getstr(buf, cnt, err)
char *buf;
int cnt;
char *err;
{
  char c;

  do {
    if (read(0, &c, 1) != 1)
      exit(1);
    if (--cnt < 0) {
      printf("%s too long\r\n", err);
      syslog(LOG_CRIT, "%s too long", err);
      exit(1);
    }
    *buf++ = c;
  }
  while ((c != 0) && (c != '~'));
}

struct speed_xlat {
  char *c;
  int i;
}          speeds[] = {

#ifdef B0
  {
    "0", B0
  },
#endif	/* B0 */
#ifdef B50
  {
    "50", B50
  },
#endif	/* B50 */
#ifdef B75
  {
    "75", B75
  },
#endif	/* B75 */
#ifdef B110
  {
    "110", B110
  },
#endif	/* B110 */
#ifdef B134
  {
    "134", B134
  },
#endif	/* B134 */
#ifdef B150
  {
    "150", B150
  },
#endif	/* B150 */
#ifdef B200
  {
    "200", B200
  },
#endif	/* B200 */
#ifdef B300
  {
    "300", B300
  },
#endif	/* B300 */
#ifdef B600
  {
    "600", B600
  },
#endif	/* B600 */
#ifdef B1200
  {
    "1200", B1200
  },
#endif	/* B1200 */
#ifdef B1800
  {
    "1800", B1800
  },
#endif	/* B1800 */
#ifdef B2400
  {
    "2400", B2400
  },
#endif	/* B2400 */
#ifdef B4800
  {
    "4800", B4800
  },
#endif	/* B4800 */
#ifdef B7200
  {
    "7200", B7200
  },
#endif	/* B7200 */
#ifdef B9600
  {
    "9600", B9600
  },
#endif	/* B9600 */
#ifdef B14400
  {
    "14400", B14400
  },
#endif	/* B14400 */
#ifdef B19200
  {
    "19200", B19200
  },
#endif	/* B19200 */
#ifdef B28800
  {
    "28800", B28800
  },
#endif	/* B28800 */
#ifdef B38400
  {
    "38400", B38400
  },
#endif	/* B38400 */
#ifdef B57600
  {
    "57600", B57600
  },
#endif	/* B57600 */
#ifdef B115200
  {
    "115200", B115200
  },
#endif	/* B115200 */
#ifdef B230400
  {
    "230400", B230400
  },
#endif	/* 230400 */
  {
    NULL, 0
  }
};

void doremoteterm(term)
char *term;
{
  register char *cp = strchr(term, '/');
  char *speed;
  struct speed_xlat *x;

  if (cp) {
    *cp++ = '\0';
    speed = cp;
    cp = strchr(speed, '/');
    if (cp)
      *cp++ = '\0';
    for (x = speeds; x->c != NULL; x++)
      if (strcmp(x->c, speed) == 0) {
	cfsetispeed(&attr, x->i);
	cfsetospeed(&attr, x->i);
	break;
      }
  }
}

int tty_gid(default_gid)
int default_gid;
{
  struct group *gr;
  int gid = default_gid;

  gr = getgrnam(TTYGRPNAME);
  if (gr != (struct group *) 0)
    gid = gr->gr_gid;
  endgrent();
  return (gid);
}

int main(argc, argv)
int argc;
char *argv[];
{
  /* variables */
  extern char **environ;
  register char *namep;
  struct opie opie;

  int invalid, quietlog;
  FILE *nlfd;
  char *tty, host[256];
  int pflag = 0, hflag = 0, fflag = 0;
  int t, c;
  int i;
  char *p, *domain;
  char opieprompt[OPIE_CHALLENGE_MAX + 1];
  int pwok, af_pwok;
  char *pp;
  char buf[256];
  int fflags, orig_fflags;
  int uid;
  int opiepassed;

  /* initialisation */
  host[0] = '\0';
  opieprompt[0] = '\0';

  if (p = getenv("TERM")) {
#ifdef DEBUG
    fprintf(stderr, "Environment TERM=%s", p);
#endif /* DEBUG */
    strncpy(term, p, sizeof(term));
  };
  
  memset(&nouser, 0, sizeof(nouser));
  nouser.pw_uid = -1;
  nouser.pw_gid = -1;
  nouser.pw_passwd = "#nope";
  nouser.pw_name = nouser.pw_gecos = nouser.pw_dir = nouser.pw_shell = "";

#ifdef IS_A_BSD
  setpriority(PRIO_PROCESS, 0, 0);
#endif	/* IS_A_BSD */

  signal(SIGALRM, timedout);
  alarm(timeout);
  signal(SIGQUIT, SIG_IGN);
  signal(SIGINT, SIG_IGN);

#ifdef SOLARIS
  ttyprompt = (char *) getenv("TTYPROMPT");
#endif	/* SOLARIS */

#ifdef	QUOTA
  quota(Q_SETUID, 0, 0, 0);
#endif

  gethostname(me, sizeof(me));
  domain = strchr(me, '.');

#ifdef DEBUG
  {
    int foo;

    printf("my args are: (argc=%d)\n", foo = argc);
    while (--foo)
      printf("%d: %s\n", foo, argv[foo]);
  }
#endif	/* DEBUG */

/* Some OSs pass environment variables on the command line. All of them except
   for TERM get eaten. */

  i = argc;
  while (--i)
    if (strchr(argv[i], '=')) {
#ifdef DEBUG
      printf("eating %s\n", argv[i]);
#endif	/* DEBUG */
      argc--;
      if (!strncmp(argv[i], "TERM=", 5)) {
	strncpy(term, &(argv[i][5]), sizeof(term));
	term[sizeof(term) - 1] = 0;
#ifdef DEBUG
	printf("passed TERM=%s, ouroptind = %d\n", term, i);
#endif	/* DEBUG */
      }
    }
/* Implement our own getopt()-like functionality, but do so in a much more
   strict manner to prevent security problems. */
  for (ouroptind = 1; ouroptind < argc; ouroptind++) {
    i = 0;
    if (argv[ouroptind])
      if (argv[ouroptind][0] == '-')
	if (i = argv[ouroptind][1])
	  if (!argv[ouroptind][2])
	    switch (i) {
	    case 'd':
	      if (++ouroptind == argc)
		exit(1);
/*    The '-d' option is apparently a performance hack to get around
   ttyname() being slow. The potential does exist for it to be used
   for malice, and it does not seem to be strictly necessary, so we
   will just eat it. */
	      break;

	    case 'r':
	      if (rflag || hflag || fflag) {
		printf("Other options not allowed with -r\n");
		exit(1);
	      }
	      if (++ouroptind == argc)
		exit(1);

	      ouroptarg = argv[ouroptind];

	      if (!ouroptarg)
		exit(1);

	      rflag = 1;
	      doremotelogin(ouroptarg);

	      strncpy(host, ouroptarg, sizeof(host));
	      if ((domain != NULL) &&
		  (p = strchr(ouroptarg, '.')) &&
		  (strcmp(p, domain) == 0)) {
		*p = 0;
	      }
	      strncpy(host, ouroptarg, sizeof(host));
	      break;

	    case 'h':
	      if (getuid() == 0) {
		if (rflag || hflag || fflag) {
		  printf("Other options not allowed with -h\n");
		  exit(1);
		}
		hflag = 1;

		if (++ouroptind == argc)
		  exit(1);

		ouroptarg = argv[ouroptind];

		if (!ouroptarg)
		  exit(1);

		strncpy(host, ouroptarg, sizeof(host));
		if ((domain != NULL) &&
		    (p = strchr(ouroptarg, '.')) &&
		    (strcmp(p, domain) == 0))
		  *p = 0;
		strncpy(host, ouroptarg, sizeof(host));
	      }
	      break;

	    case 'f':
	      if (rflag) {
		printf("Only one of -r and -f allowed\n");
		exit(1);
	      }
	      fflag = 1;

	      if (++ouroptind == argc)
		exit(1);

	      ouroptarg = argv[ouroptind];

	      if (!ouroptarg)
		exit(1);

	      strncpy(name, ouroptarg, sizeof(name));
	      break;

	    case 'p':
	      pflag = 1;
	      break;
	  } else
	    i = 0;
    if (!i) {
      ouroptarg = argv[ouroptind++];
      strncpy(name, ouroptarg, sizeof(name));
      break;
    }
  }

  for (t = sysconf(_SC_OPEN_MAX); t > 2; t--)
    close(t);

#ifdef TIOCNXCL
  /* BSDism:  not sure how to rewrite for POSIX.  rja */
  ioctl(0, TIOCNXCL, 0);	/* set non-exclusive use of tty */
#endif

  /* get original termio attributes */
  if (tcgetattr(STDIN_FILENO, &attr) != 0)
    return (-1);

/* If talking to an rlogin process, propagate the terminal type and baud rate
   across the network. */
  if (rflag)
    doremoteterm(term);

/* Force termios portable control characters to the system default values as
specified in termios.h. This should help the one-time password login feel the
same as the vendor-supplied login. Common extensions are also set for
completeness, but these are set within appropriate defines for portability. */

#define CONTROL(x) (x - 64);

#ifdef VEOF
#ifdef CEOF
  attr.c_cc[VEOF] = CEOF;
#else	/* CEOF */
  attr.c_cc[VEOF] = CONTROL('D');
#endif	/* CEOF */
#endif	/* VEOF */
#ifdef VEOL
#ifdef CEOL
  attr.c_cc[VEOL] = CEOL;
#else	/* CEOL */
  attr.c_cc[VEOL] = CONTROL('J');
#endif	/* CEOL */
#endif	/* VEOL */
#ifdef VERASE
#ifdef CERASE
  attr.c_cc[VERASE] = CERASE;
#else	/* CERASE */
  attr.c_cc[VERASE] = CONTROL('H');
#endif	/* CERASE */
#endif	/* VERASE */
#ifdef VINTR
#ifdef CINTR
  attr.c_cc[VINTR] = CINTR;
#else	/* CINTR */
  attr.c_cc[VINTR] = CONTROL('C');
#endif	/* CINTR */
#endif	/* VINTR */
#ifdef VKILL
#ifdef CKILL
  attr.c_cc[VKILL] = CKILL;
#else	/* CKILL */
  attr.c_cc[VKILL] = CONTROL('U');
#endif	/* CKILL */
#endif	/* VKILL */
#ifdef VQUIT
#ifdef CQUIT
  attr.c_cc[VQUIT] = CQUIT;
#else	/* CQUIT */
  attr.c_cc[VQUIT] = CONTROL('\\');
#endif	/* CQUIT */
#endif	/* VQUIT */
#ifdef VSUSP
#ifdef CSUSP
  attr.c_cc[VSUSP] = CSUSP;
#else	/* CSUSP */
  attr.c_cc[VSUSP] = CONTROL('Z');
#endif	/* CSUSP */
#endif	/* VSUSP */
#ifdef VSTOP
#ifdef CSTOP
  attr.c_cc[VSTOP] = CSTOP;
#else	/* CSTOP */
  attr.c_cc[VSTOP] = CONTROL('S');
#endif	/* CSTOP */
#endif	/* VSTOP */
#ifdef VSTART
#ifdef CSTART
  attr.c_cc[VSTART] = CSTART;
#else	/* CSTART */
  attr.c_cc[VSTART] = CONTROL('Q');
#endif	/* CSTART */
#endif	/* VSTART */
#ifdef VDSUSP
#ifdef CDSUSP
  attr.c_cc[VDSUSP] = CDSUSP;
#else	/* CDSUSP */
  attr.c_cc[VDSUSP] = 0;
#endif	/* CDSUSP */
#endif	/* VDSUSP */
#ifdef VEOL2
#ifdef CEOL2
  attr.c_cc[VEOL2] = CEOL2;
#else	/* CEOL2 */
  attr.c_cc[VEOL2] = 0;
#endif	/* CEOL2 */
#endif	/* VEOL2 */
#ifdef VREPRINT
#ifdef CRPRNT
  attr.c_cc[VREPRINT] = CRPRNT;
#else	/* CRPRNT */
  attr.c_cc[VREPRINT] = 0;
#endif	/* CRPRNT */
#endif	/* VREPRINT */
#ifdef VWERASE
#ifdef CWERASE
  attr.c_cc[VWERASE] = CWERASE;
#else	/* CWERASE */
  attr.c_cc[VWERASE] = 0;
#endif	/* CWERASE */
#endif	/* VWERASE */
#ifdef VLNEXT
#ifdef CLNEXT
  attr.c_cc[VLNEXT] = CLNEXT;
#else	/* CLNEXT */
  attr.c_cc[VLNEXT] = 0;
#endif	/* CLNEXT */
#endif	/* VLNEXT */

  attr.c_lflag |= ICANON;	/* enable canonical input processing */
  attr.c_lflag &= ~ISIG;	/* disable INTR, QUIT,& SUSP signals */
  attr.c_lflag |= (ECHO | ECHOE);	/* enable echo and erase */
#ifdef ONLCR
  /* POSIX does not specify any output processing flags, but the usage below
     is SVID compliant and is generally portable to modern versions of UNIX. */
  attr.c_oflag |= ONLCR;	/* map CR to CRNL on output */
#endif
#ifdef ICRNL
  attr.c_iflag |= ICRNL;
#endif	/* ICRNL */

  attr.c_oflag |= OPOST;
  attr.c_lflag |= ICANON;	/* enable canonical input */
  attr.c_lflag |= ECHO;
  attr.c_lflag |= ECHOE;	/* enable ERASE character */
  attr.c_lflag |= ECHOK;	/* enable KILL to delete line */
  attr.c_cflag |= HUPCL;	/* hangup on close */

  memcpy(&attrnoecho, &attr, sizeof(struct termios));

  attrnoecho.c_lflag &= ~(ECHO | ECHOK | ECHOE);	/* disable echoing */

  /* Set revised termio attributes */
  if (tcsetattr(STDIN_FILENO, TCSANOW, &attr) != 0)
    return (-1);

  atexit(catchexit);

  tty = ttyname(0);

  if (tty == (char *) 0 || *tty == '\0')
    tty = "UNKNOWN";	/* was: "/dev/tty??" */

#ifdef DEBUG
  fprintf(stderr, "tty = %s\n", tty);
#endif	/* DEBUG */

  openlog("login", LOG_ODELAY, LOG_AUTH);
  atexit(closelog);
  t = 0;
  invalid = TRUE;
  af_pwok = opieaccessfile(host);

  getutmpentry(tty, &UTMPX);

  if (name[0])
    if (name[0] == '-') {
      fprintf(stderr, "User names can't start with '-'.\n");
      syslog(LOG_AUTH, "Attempt to use invalid username: %s.", name);
      exit(1);
    } else
      invalid = lookupuser();

  atexit(catchexit);

  do {
    /* If remote login take given name, otherwise prompt user for something. */
    if (invalid && !name[0]) {
      getloginname();
      invalid = lookupuser();
    }
#ifdef DEBUG
    printf("login name is -%s-, of length %d, [0] = %d\n", name,
	   strlen(name), name[0]);
#endif	/* DEBUG */

    if (fflag) {
      uid = getuid();

      if (uid != 0 && uid != thisuser.pw_uid)
	fflag = 0;
      /* Disallow automatic login for root. */
      if (thisuser.pw_uid == 0)
	fflag = 0;
    }
    if (feof(stdin))
      exit(0);

    /* If no remote login authentication and a password exists for this user,
       prompt for and verify a password. */
    if (!fflag && *thisuser.pw_passwd) {
#ifdef DEBUG
      printf("login name is -%s-, of length %d, [0] = %d\n", name,
	     strlen(name), name[0]);
#endif	/* DEBUG */

      /* Attempt a one-time password challenge */
      i = opiechallenge(&opie, name, opieprompt);

      if (!memcmp(&thisuser, &nouser, sizeof(thisuser)))
	if (host[0])
	  syslog(LOG_WARNING, "Invalid login attempt for %s on %s from %s.",
		 name, tty, host);
	else
	  syslog(LOG_WARNING, "Invalid login attempt for %s on %s.",
		 name, tty);

      pwok = af_pwok && opiealways(thisuser.pw_dir);

      printf("%s\n", opieprompt);
      if (!pwok) 
	printf("(OTP response required)\n");
      printf("Password:");
      fflush(stdout);

      /* Use blocking I/O for now */
      orig_fflags = fcntl(STDIN_FILENO, F_GETFL, 0);
      memcpy(&fflags, &orig_fflags, sizeof(int));

      fflags &= ~O_NONBLOCK;
      fcntl(STDIN_FILENO, F_SETFL, fflags);

      if (tcsetattr(STDIN_FILENO, TCSANOW, &attrnoecho))
	return (-1);

      /* Read password */
      fgets(buf, sizeof(buf), stdin);
      opiestrip_crlf(buf);
      if (feof(stdin))
	invalid = TRUE;

      if (tcsetattr(STDIN_FILENO, TCSANOW, &attr))
	return -1;

      if (strlen(buf) == 0) {
	/* Null line entered, so display appropriate prompt & flush current
	   data. */
	printf(" (echo on)\nPassword:");
	fflush(stdout);
	fgets(buf, sizeof(buf), stdin);
	opiestrip_crlf(buf);
	if (feof(stdin))
	  invalid = TRUE;
      } else
	printf("\n");

      /* Restore previous tty modes */
      fcntl(STDIN_FILENO, F_SETFL, orig_fflags);

/* Once opieverify() is called, this information is gone */
      i = opiegetsequence(&opie);

      opiepassed = !opieverify(&opie, buf);

#ifdef DEBUG
      fprintf(stderr, "opiepassed = %d\n", opiepassed);
#endif /* DEBUG */

      if (!invalid) {
        if (opiepassed) {
	    /* OPIE authentication succeeded */
	    if (i < 5) {
	      printf("Warning: Change your OTP secret pass phrase NOW!\n");
	    } else {
	      if (i < 10) {
		printf("Warning: Change your OTP secret pass phrase.\n");
	      }
            }
        } else {
	    /* Try regular password check, if allowed */
	    if (pwok) {
	      pp = crypt(buf, thisuser.pw_passwd);
	      invalid = strcmp(pp, thisuser.pw_passwd);
	    } else
              invalid = TRUE;
	}
      }
    }

    /* If user not super-user, check for logins disabled. */
    if (thisuser.pw_uid) {
      if (nlfd = fopen(NO_LOGINS_FILE, "r")) {
	while ((c = getc(nlfd)) != EOF)
	  putchar(c);
	fflush(stdout);
	sleep(5);
	exit(0);
      }
    }
    /* If valid so far and root is logging in, see if root logins on this
       terminal are permitted. */
    if (!invalid && !thisuser.pw_uid && !rootterm(tty)) {
      if (host[0])
	syslog(LOG_CRIT, "ROOT LOGIN REFUSED ON %s FROM %.*s",
	       tty, HMAX, host);
      else
	syslog(LOG_CRIT, "ROOT LOGIN REFUSED ON %s", tty);
      invalid = TRUE;
    }
    /* If invalid, then log failure attempt data to appropriate system
       logfiles and close the connection. */
    if (invalid) {
      printf("Login incorrect\n");
      if (host[0])
	  syslog(LOG_ERR, "LOGIN FAILURE ON %s FROM %.*s, %.*s",
		 tty, HMAX, host, sizeof(name), name);
	else
	  syslog(LOG_ERR, "LOGIN FAILURE ON %s, %.*s", 
		 tty, sizeof(name), name);
      if (++t >= 5)
	exit(1);
    }
    if (*thisuser.pw_shell == '\0')
      thisuser.pw_shell = "/bin/sh";
    if ((chdir(thisuser.pw_dir) < 0) && !invalid) {
      if (chdir("/") < 0) {
	printf("No directory!\n");
	invalid = TRUE;
      } else {
	printf("No directory! %s\n", "Logging in with HOME=/");
        strcpy(thisuser.pw_dir, "/");
      }
    }
    /* Remote login invalid must have been because of a restriction of some
       sort, no extra chances. */
    if (invalid) {
      if (!usererr)
	exit(1);
      name[0] = 0;
    }
  }
  while (invalid);
  /* Committed to login -- turn off timeout */
  alarm(0);

#ifdef	QUOTA
  if (quota(Q_SETUID, thisuser.pw_uid, 0, 0) < 0 && errno != EINVAL) {
    if (errno == EUSERS)
      printf("%s.\n%s.\n", "Too many users logged on already",
	     "Try again later");
    else
      if (errno == EPROCLIM)
	printf("You have too many processes running.\n");
      else
	perror("quota (Q_SETUID)");
    sleep(5);
    exit(0);
  }
#endif

  strncpy(UTMPX.ut_host, host, sizeof(UTMPX.ut_host));
  strncpy(UTMPX.ut_name, name, sizeof(UTMPX.ut_name));
#ifdef USER_PROCESS
  UTMPX.ut_type = USER_PROCESS;
#endif	/* USER_PROCESS */

  pututmpentry(tty, &UTMPX);

  quietlog = !access(QUIET_LOGIN_FILE, F_OK);

#ifdef SUNOS
  {
  int f;
  /* Lastlog appears to be unique to SunOS 4.1.x.  It is not portable to
     System V Release 4 systems such as Solaris 2.x. Support for Lastlog is
     currently protected by suitable defines. Support for Lastlog might be
     removed entirely in later releases. rja */
  if ((f = open(lastlog, O_RDWR)) >= 0) {
    struct lastlog ll;

    lseek(f, (long)thisuser.pw_uid * sizeof(struct lastlog), 0);

    if ((sizeof(ll) == read(f, (char *) &ll, sizeof(ll))) &&
	(ll.ll_time != 0) && (!quietlog)) {
      printf("Last login: %.*s ",
	     24 - 5, (char *) ctime(&ll.ll_time));
      if (*ll.ll_host != '\0')
	printf("from %.*s\n", sizeof(ll.ll_host), ll.ll_host);
      else
	printf("on %.*s\n", sizeof(ll.ll_line), ll.ll_line);
    }
    lseek(f, (long)thisuser.pw_uid * sizeof(struct lastlog), 0);

    time(&ll.ll_time);
    strncpy(ll.ll_line, tty, sizeof(ll.ll_line));
    strncpy(ll.ll_host, host, sizeof(ll.ll_line));
    write(f, (char *) &ll, sizeof ll);
    close(f);
  }
  }
#endif

  chown(tty, thisuser.pw_uid, TTYGID(thisuser.pw_gid));

#ifdef TIOCSWINSZ
/* POSIX does not specify any interface to set/get window sizes, so this is
not portable.  It should work on most recent BSDish systems and the defines
should protect it on older System Vish systems.  It does work under Solaris
2.4, though it isn't clear how many other SVR4 systems support it. I'd be
interested in hearing of a more portable approach. rja */
  if (!hflag && !rflag)
    ioctl(0, TIOCSWINSZ, &win);	/* set window size to 0,0,0,0 */
#endif

  chmod(tty, 0622);
  setgid(thisuser.pw_gid);
  initgroups(name, thisuser.pw_gid);

#ifdef	QUOTA
  quota(Q_DOWARN, thisuser.pw_uid, (dev_t) - 1, 0);
#endif

#ifdef PERMSFILE
  home = thisuser.pw_dir;
  permsfile(name, tty, thisuser.pw_uid, thisuser.pw_gid);
  fflush(stderr);
#endif	/* PERMSFILE */

  setuid(thisuser.pw_uid);

  /* destroy environment unless user has asked to preserve it */
  if (!pflag)
    environ = envinit;
  setenv("HOME", thisuser.pw_dir, 1);
  setenv("SHELL", thisuser.pw_shell, 1);
  if (!term[0]) {
#if (defined(BSD4_3) || defined(SUNOS))
/*
 * The getttynam() call and the ttyent structure first appeared in 4.3 BSD.
 * They are not portable to System V systems such as Solaris 2.x.
 *         rja
 */
  register struct ttyent *t;
  register char *c;

  if (c = strrchr(tty, '/'))
    c++;
  else
    c = tty;

  if (t = getttynam(c))
    strncpy(term, t->ty_type, sizeof(term));
  else
#endif	/* BSD4_3 */
    strcpy(term, "unknown");
  }

  setenv("USER", name, 1);
  setenv("LOGNAME", name, 1);
  setenv("PATH", DEFAULT_PATH, 0);
  if (term[0]) {
#ifdef DEBUG
    printf("setting TERM=%s\n", term);
#endif	/* DEBUG */
    setenv("TERM", term, 1);
  }
  if ((namep = strrchr(thisuser.pw_shell, '/')) == NULL)
    namep = thisuser.pw_shell;
  else
    namep++;
  strcat(minusnam, namep);
  if (tty[sizeof("tty") - 1] == 'd')
    syslog(LOG_INFO, "DIALUP %s, %s", tty, name);
  if (!thisuser.pw_uid)
    if (host[0])
      syslog(LOG_NOTICE, "ROOT LOGIN %s FROM %.*s", tty, HMAX, host);
    else
      syslog(LOG_NOTICE, "ROOT LOGIN %s", tty);
#if DOMOTD
  if (!quietlog) {
    struct stat st;
    char buf[128];

    FILE *mf;
    register c;

    signal(SIGINT, catch);
    if ((mf = fopen(MOTD_FILE, "r")) != NULL) {
      while ((c = getc(mf)) != EOF && !stopmotd)
	putchar(c);
      fclose(mf);
    }
    signal(SIGINT, SIG_IGN);

    strcpy(buf, MAIL_DIR);
    strcat(buf, name);
    if (!stat(buf, &st) && st.st_size)
      printf("You have %smail.\n",
	     (st.st_mtime > st.st_atime) ? "new " : "");
  }
#endif	/* DOMOTD */
  signal(SIGALRM, SIG_DFL);
  signal(SIGQUIT, SIG_DFL);
  signal(SIGINT, SIG_DFL);
  signal(SIGTSTP, SIG_IGN);

  attr.c_lflag |= (ISIG | IEXTEN);

  catchexit();
  execlp(thisuser.pw_shell, minusnam, 0);
  perror(thisuser.pw_shell);
  printf("No shell\n");
  exit(0);
}

/* EOF login.c */
