/* opiekey.c: Stand-alone program for computing responses to OTP challenges.

 Takes a sequence number and seed (presumably from an OPIE challenge)
 as command line arguments, prompts for the user's secret pass phrase,
 and outputs a response.

Portions of this software are Copyright 1995 by Randall Atkinson and Dan
McDonald, All Rights Reserved. All Rights under this copyright are assigned
to the U.S. Naval Research Laboratory (NRL). The NRL Copyright Notice and
License Agreement applies to this software.

	History:

        Modified at NRL for OPIE 2.2. Check opiereadpass() return.
                Change opiereadpass() calls to add echo arg. Use FUNCTION
                definition et al. Check seed length here, too. Added back
		hex output. Reworked final output function.
	Modified at NRL for OPIE 2.0.
	Written at Bellcore for the S/Key Version 1 software distribution
		(skey.c).
*/
#include "opie_cfg.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "opie.h"

#ifdef	__MSDOS__
#include <dos.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif /* HAVE_FCNTL_H */

extern char *optarg;
extern int optind, opterr;

/******** Begin real source code ***************/

static VOIDRET usage FUNCTION((s), char *s)
{
  fprintf(stderr, "usage: %s [-v] [-h] [-4 | -5] [-a] [-n count] sequence_number seed\n", s);
  exit(1);
}

int main FUNCTION((argc, argv), int argc AND char *argv[])
{
  /* variable declarations */
  unsigned algorithm = MDX;	/* default algorithm per Makefile's MDX
				   symbol */
  int keynum = 0;
  int i = 0;
  int count = 1;
  char passwd[OPIE_PASS_MAX + 1];
  char key[8];
  char *seed;
  char buf[33];
  char *slash;
  int aflag = 0;
  int hex = 0;

  if (slash = strchr(argv[0], '/'))
    slash++;
  else
    slash = argv[0];

  if (!strcmp(slash, "key") || strstr(slash, "md4"))
    algorithm = 4;

  if (strstr(slash, "md5"))
    algorithm = 5;

  while ((i = getopt(argc, argv, "hvn:x45a")) != EOF) {
    switch (i) {
    case 'v':
      opieversion();

    case 'n':
      count = atoi(optarg);
      break;

    case 'x':
      hex = 1;
      break;

    case '4':
      /* use MD4 algorithm */
      algorithm = 4;
      break;

    case '5':
      /* use MD5 algorithm */
      algorithm = 5;
      break;

    case 'a':
      aflag = 1;
      break;

    default:
      usage(argv[0]);
    }
  }

  if ((argc - optind) < 2)
    usage(argv[0]);

  fprintf(stderr, "Using MD%d algorithm to compute response.\n", algorithm);

  /* get sequence number, which is next-to-last parameter */
  keynum = atoi(argv[optind]);
  if (keynum < 1) {
    fprintf(stderr, "Sequence number %s is not positive.\n", argv[optind]);
    exit(1);
  }
  /* get seed string, which is last parameter */
  seed = argv[optind + 1];
  if (strlen(seed) > OPIE_SEED_MAX) {
    fprintf(stderr, "Seeds must be less than %d characters long.", OPIE_SEED_MAX);
    exit(1);
  }

  fprintf(stderr, "Reminder: Don't use opiekey from telnet or dial-in sessions.\n");

  if (opieinsecure()) {
    fprintf(stderr, "Sorry, but you don't seem to be on the console or a secure terminal.\n");
    exit(1);
  }
  fprintf(stderr, "Enter secret pass phrase: ");
  if (!opiereadpass(passwd, sizeof(passwd), 0)) {
    fprintf(stderr, "Error reading secret pass phrase!\n");
    exit(1);
  }
#if RETYPE
  {
    char verify[OPIE_PASS_MAX + 1];

    fprintf(stderr, "Again secret pass phrase: ");
    if (!opiereadpass(verify, sizeof(verify), 0)) {
      fprintf(stderr, "Error reading secret pass phrase!\n");
      exit(1);
    }
    if (verify[0] && strncmp(verify, passwd, sizeof(passwd))) {
      fprintf(stderr, "They don't match. Try again.\n");
      exit(1);
    }
  }
#endif	/* RETYPE */
  if ((!aflag) && opiepasscheck(passwd)) {
    memset(passwd, 0, sizeof(passwd));
    fprintf(stderr, "Secret pass phrases must be between %d and %d characters long.\n", OPIE_PASS_MIN, OPIE_PASS_MAX);
    exit(1);
  };

  /* Crunch seed and secret password into starting key normally */
  if (opiekeycrunch(algorithm, key, seed, passwd) != 0) {
    memset(passwd, 0, sizeof(passwd));
    fprintf(stderr, "%s: key crunch failed\n", argv[0]);
    return 1;
  }
  memset(passwd, 0, sizeof(passwd));

  for (i = 0; i <= (keynum - count); i++)
    opiehash(key, algorithm);
  
  for (; i <= keynum; i++) {
    if (hex)
      opiebtoa8(buf, key);
    else
      opiebtoe(buf, key);
    if (count != 1)
      printf("%d: ", i);
    puts(buf);
    opiehash(key, algorithm);
  }
  return 0;
}
