/*
 *  SecuDE Release 4.1 (GMD)
 */
/********************************************************************
 * Copyright (C) 1991, GMD. All rights reserved.                    *
 *                                                                  *
 *                                                                  *
 *                         NOTICE                                   *
 *                                                                  *
 *    Acquisition, use, and distribution of this module             *
 *    and related materials are subject to restrictions             *
 *    mentioned in each volume of the documentation.                *
 *                                                                  *
 ********************************************************************/

/*-----------------------aux-certIO.c-------------------------------*/
/*------------------------------------------------------------------*/
/* GMD Darmstadt Institut fuer Telekooperationstechnik (I2)         */
/* Rheinstr. 75 / Dolivostr. 15                                     */
/* 6100 Darmstadt                                                   */
/* Grimm/Nausester/Schneider/Viebeg/Vollmer et alii                 */
/*------------------------------------------------------------------*/
/*                                                                  */
/* PACKAGE   certificate IO  VERSION   3.0                          */
/*                              DATE   19.02.1992                   */
/*                                BY   Grimm/WS                     */
/*                                                                  */
/*                            REVIEW                                */
/*                              DATE                                */
/*                                BY                                */
/* DESCRIPTION                                                      */
/*   This modul presents functions to fill SecuDe                   */
/*   relevant C-structures from stdin (get_*), and to               */
/*   write the C-structures to a given stream (aux_put_*).          */
/*   All functions have int return value (0 or -1) (except          */
/*   get_and_return_cert) and expect a pointer on                   */
/*   the respective structure on input.                             */
/*                                                                  */
/* EXPORT                                                           */
/*                                                                  */
/*                           aux_fprint_PKRoot()                    */
/*                           aux_fprint_PKList()                    */
/*                           aux_fprint_PSEToc()                    */
/*                           aux_fprint_KeyBits()                   */
/*                           aux_fprint_AlgId()                     */
/*                           aux_fprint_KeyInfo()                   */
/*                           aux_fprint_Certificate()               */
/*                           aux_fprint_FCPath()                    */
/*                           aux_fprint_Certificates()              */
/*                           aux_fprint_PemInfo()                   */
/*                           aux_fprint_CertificateSet()            */
/*                           aux_fprint_error()                     */
/*------------------------------------------------------------------*/
#include <stdio.h>
#include "pem.h"
#include "cadb.h"

#define BUFLEN 4096
#define ALGIDLEN 80
#define TIMLEN   40


extern void  aux_xdump();
extern void  aux_xdump2();
extern char	*strcpy();
extern OctetString *aux_dechex();
extern OctetString *aux_enchex();


static char	*VSN           = "    Version:      ";
static char	*SER           = "    SerialNo:     ";
static char	*ISS           = "    IssuerName:   ";
static char	*SPK           = "    SubjectKey:   ";
static char	*SEK           = "    SecretKeyAid: ";
static char	*IAI           = "    IssuerAid:    ";
static char	*SIGN          = "    Signature of issuer: ";
static char	*SAI           = "    SignatureAid: ";
static char	*VALNB         = "    Validity      NotBefore: ";
static char	*VALNA         = "                  NotAfter:  ";
static char	*SUB           = "    SubjectName:  ";
static char	*ROO           = "    RootCaName:   ";
static char	*SNN           = "    SerialNoNew:  ";
static char	*SNO           = "    SerialNoOld:  ";
static char	*PUKOLD        = "    OldPublicKey: ";
static char	*PUKNEW        = "    NewPublicKey: ";
static char	*HSHRESULT     = "    Hash value of DER code of ToBeSigned: ";
static char	*DERCODE       = "    DER code of ";
static char	*EPKROOT       = "PKRoot is NULL pointer";
static char	*EKEYBITS      = "KeyBits is NULL pointer";
static char	*ESERNUMS      = "SerialNumbers is NULL pointer";
static char	*EKEYINFO      = "KeyInfo is NULL pointer";
static char	*EAID          = "AlgId is NULL pointer";
static char	*ECERT         = "Certificate is NULL pointer";
static char	*ECERTS        = "Certificates is NULL pointer";
static char	*ETBS          = "ToBeSigned is NULL pointer";
static char	*EPSETOC       = "PSEToc is NULL pointer";
static char	*LASTUP        = "    Last Update:  ";
static char	*NEXTUP        = "    Next Update:  ";
static char	*AIRCA         = "    Aid of Revoking CA:          ";
static char	*SIGNRCA       = "    Signature of Revoking CA: ";
static char	*SAIRCA        = "    SignatureAid of Revoking CA: ";
static char	*RCA           = "    RevokingCAName:              ";
static char	*RSER          = "    SerialNo:                    ";
static char	*REVDATE       = "    Revocation Date:             ";
static char	*OLDSER	       = "    Table's SerialNo: ";
static char	*ERCLIST       = "Revocation list (X.500) is NULL pointer";
static char	*ERCLISTPEM    = "Revocation list (PEM) is NULL pointer";
static char	*ERCLISTTBS    = "CrlTBS is NULL pointer";
static char	*ERCLISTPEMTBS = "PemCrlTBS is NULL pointer";
static char	*EREVCERT      = "Revoked Certificate is NULL pointer";
static char	*EREVCERTPEM   = "Revoked Certificate is NULL pointer";
static char	*EREVCERTTBS   = "RevCertTBS is NULL pointer";
static char	*EOCLIST       = "List of OldCertificates is NULL pointer";
static char	*ECERTSET      = "SET_OF_Certificate is NULL pointer";
static char	*ECERTPAIR     = "Certificate Pair is NULL pointer";
static char	*EREVLISTSET   = "CrlSet is NULL pointer";
static char	*EREVLISTPSE   = "CrlPSE is NULL pointer";
static char	*EISSCERT      = "IssuedCertificate is NULL pointer";
static char	*EISSCERTSET   = "SET_OF_IssuedCertificate is NULL pointer";
static char     *EPEMCRLWITHCERTSSET = "SET_OF_PemCrlWithCerts is NULL pointer";
static char     *EPEMCRLWITHCERTS = "PemCrlWithCerts is NULL pointer";
static char	*ISSUEDATE     = "    Date of Issue:  ";
static char	*ENAMESET      = "SET_OF_Name is NULL pointer";
static char	*VERRES        = "VerificationResult is NULL pointer";

#ifdef COSINE
int aux_fprint_AuthorisationAttributes();

#define NORMAL "Normal"
#define PRIVILEGED "Privileged"
static char	*AUTHATTR      = "    AuthorisationAttributes: ";
static char	*COID          = "      CountryID:  ";
static char	*GID           = "      GroupID:    ";
static char	*CLID          = "      ClassID:    ";
static char	*ECOID         = "CountryId missing";
static char	*EGID          = "GroupId missing";
static char	*ECLID         = "wrong ClassId";
#endif

RC
aux_fprint_PSESel(ff, pse)
FILE *ff;
PSESel *pse;
{
	fprintf(ff, "app_name    = %s\n", pse->app_name);
	fprintf(ff, "pin         = %s\n", pse->pin);
	fprintf(ff, "object.name = %s\n", pse->object.name);
	fprintf(ff, "object.pin  = %s\n", pse->object.pin);
	fprintf(ff, "app_id      = %d\n", pse->app_id);
	return(0);
}

RC
aux_fprint_Version(ff)
FILE *ff;
{
	char buf[256];
	Boolean flag = FALSE;


	buf[0] = '\0';

	sprintf(&buf[strlen(buf)], "%s       ", secude_version);
	if(af_x500) {

		if(!flag) sprintf(&buf[strlen(buf)], "with ");
	    	sprintf(&buf[strlen(buf)], "X.500");

		if(af_strong) sprintf(&buf[strlen(buf)], ", Strong Authentication");
		flag = TRUE;
	}
	if(sec_sca) {

		if(flag) sprintf(&buf[strlen(buf)], ", ");
		else sprintf(&buf[strlen(buf)], "with ");

		sprintf(&buf[strlen(buf)], "Smartcard Support");
		flag = TRUE;
	}
	if(af_COSINE) {

		if(flag) sprintf(&buf[strlen(buf)], ", ");
		else sprintf(&buf[strlen(buf)], "with ");

		sprintf(&buf[strlen(buf)], "COSINE extensions");
		flag = TRUE;
	}
	fprintf(ff, "%-75s (c)1993 GMD Darmstadt\n", buf);
	return(0);
}

RC
aux_fprint_version(ff)
FILE *ff;
{
	aux_fprint_Version(ff);
	fprintf(stderr, "\n");
	return(0);
}

RC
aux_fprint_PSEToc(ff, toc1, toc2)
FILE *ff;
PSEToc *toc1, *toc2;
{
	struct PSE_Objects *obj;
	PSEToc *toc;
	char *scflag, *update, *create;
	int	i = 1;

	if ( !toc1 ) {
		fprintf(ff, "%s\n", EPSETOC);
		return(-1);
	}

	toc = toc1;

	if(toc2) scflag = " SC";
	else scflag = "\0";

	fprintf(ff, "Owner: %-10s ", toc->owner);
	create = aux_readable_UTCTime(toc->create);
	update = aux_readable_UTCTime(toc->update);
	fprintf(ff, "Created: %s / Changed: %s%s", create + 4, update + 4, scflag);
	if(create) free(create);
	if(update) free(update);
	if(toc->status & ONEKEYPAIRONLY) fprintf(ff, " / ONE keypair \n");
	else fprintf(ff, " / TWO keypairs \n");
	fprintf(ff, "Objects:\n");
	obj = toc->obj;
	while (obj) {
		create = aux_readable_UTCTime(obj->create);
		update = aux_readable_UTCTime(obj->update);
		fprintf(ff, "%2d. %-13s Created: %s / Changed: %s / %6d Octets %s\n", i++, obj->name, create + 4, update + 4, obj->noOctets, scflag);
		if(create) free(create);
		if(update) free(update);
		obj = obj->next;
	}
	if(!toc2 || toc1 == toc2) return(0);
	obj = toc2->obj;
	while (obj) {
		create = aux_readable_UTCTime(obj->create);
		update = aux_readable_UTCTime(obj->update);
		fprintf(ff, "%2d. %-13s Created: %s / Changed: %s / %6d Octets \n", i++, obj->name, create + 4, update + 4, obj->noOctets);
		if(create) free(create);
		if(update) free(update);
		obj = obj->next;
	}
	return(0);
}


RC
aux_fprint_SerialNumbers(ff, serialnums)
FILE * ff;
SerialNumbers * serialnums;
{
	if (! serialnums) {
		fprintf(ff, "%s\n", ESERNUMS);
		return(-1);
	}

	fprintf(ff, "INITIAL value of serialnumber: %6d\n", serialnums->initial);
	fprintf(ff, "ACTUAL value of serialnumber:  %6d\n", serialnums->actual);

	return(0);
}


RC
aux_fprint_KeyInfo(ff, pki)
FILE *ff;
KeyInfo *pki;
{
	BitString  * bis;
        KeyBits *keybits;
	int	n;

	if ( !pki ) {
		fprintf(ff, "%s\n", EKEYINFO);
		return(-1);
	}

	if (print_keyinfo_flag & ALGID) 
		if (aux_fprint_AlgId(ff, pki->subjectAI) < 0) 
			return(-1);
	if (aux_ObjId2AlgEnc(pki->subjectAI->objid) == RSA || aux_ObjId2AlgEnc(pki->subjectAI->objid) == DSA) {
		bis = &(pki->subjectkey);
		n = bis->nbits / 8;
		if ( bis->nbits % 8 ) n++;
		if (print_keyinfo_flag & BSTR) {
			fprintf(ff, "    DER code of Key BitString (no. of octets = %d):\n", n);
			aux_fxdump2(ff, bis->bits, n, 0);
		}
		if (print_keyinfo_flag & KEYBITS) {
			keybits = d_KeyBits(&(pki->subjectkey));
			if(keybits) {
				aux_fprint_KeyBits(ff, keybits);
				aux_free_KeyBits(&keybits);
			}
		}
	} 
	else {
		if (print_keyinfo_flag & (BSTR | KEYBITS)) {
			n = pki->subjectkey.nbits;
			fprintf(ff, "    Key (no. of bits = %d):\n", n);
			aux_fxdump2(ff, pki->subjectkey.bits, n / 8, 0);
		}
	}
	return(0);
}


RC
aux_fprint_KeyBits(ff, keybits)
FILE *ff;
KeyBits *keybits;
{
	int	n;

	if ( !keybits ) {
		fprintf(ff, "%s\n", EKEYBITS);
		return(-1);
	}
	if(keybits->part3.noctets) {
		n = keybits->part1.noctets;
		if (print_keyinfo_flag & PK) {
			fprintf(ff, "    Public y (no. of bits = %d):\n", n * 8);
		} 
		else if (print_keyinfo_flag & SK) {
			fprintf(ff, "    Secret x (no. of bits = %d):\n", n * 8);
		} 
		else {
			fprintf(ff, "    Key part1 (no. of bits = %d):\n", n * 8);
		}
		aux_fxdump2(ff, keybits->part1.octets, n, 0);

		n = keybits->part2.noctets;
		if (print_keyinfo_flag & (PK | SK)) {
			fprintf(ff, "    Public prime p (no. of bits = %d):\n", n * 8);
		} 
		else {
			fprintf(ff, "    Key part2 (no. of bits = %d):\n", n * 8);
		}
		aux_fxdump2(ff, keybits->part2.octets, n, 0);

		n = keybits->part3.noctets;
		if (print_keyinfo_flag & (PK | SK)) {
			fprintf(ff, "    Public prime q (no. of bits = %d):\n", n * 8);
		} 
		else {
			fprintf(ff, "    Key part3 (no. of bits = %d):\n", n * 8);
		}
		aux_fxdump2(ff, keybits->part3.octets, n, 0);

		n = keybits->part4.noctets;
		if (print_keyinfo_flag & (PK | SK)) {
			fprintf(ff, "    Public base g (no. of bits = %d):\n", n * 8);
		} 
		else {
			fprintf(ff, "    Key part4 (no. of bits = %d):\n", n * 8);
		}
		aux_fxdump2(ff, keybits->part4.octets, n, 0);

	} 
	else {
		n = keybits->part1.noctets;
		if (print_keyinfo_flag & PK) {
			fprintf(ff, "    Public modulus (no. of bits = %d):\n", n * 8);
		} 
		else if (print_keyinfo_flag & SK) {
			fprintf(ff, "    Prime p (no. of bits = %d):\n", n * 8);
		} 
		else {
			fprintf(ff, "    Key part1 (no. of bits = %d):\n", n * 8);
		}
		aux_fxdump2(ff, keybits->part1.octets, n, 0);

		n = keybits->part2.noctets;
		if (print_keyinfo_flag & PK) {
			fprintf(ff, "    Public exponent (no. of bits = %d):\n", n * 8);
		} 
		else if (print_keyinfo_flag & SK) {
			fprintf(ff, "    Prime q (no. of bits = %d):\n", n * 8);
		} 
		else {
			fprintf(ff, "    Key part2 (no. of bits = %d):\n", n * 8);
		}
		aux_fxdump2(ff, keybits->part2.octets, n, 0);
	}
	return(0);
}


RC
aux_fprint_ObjId(ff, objid)
FILE *ff;
ObjId *objid;
{
	register int	i;

	fprintf(ff, "{ ");
	for ( i = 0; i < objid->oid_nelem; i++) {
		fprintf(ff, "%d ", objid->oid_elements[i]);
	}
	fprintf(ff, "} ");
	return(0);
}


RC
aux_fprint_AlgId(ff, aid)
FILE *ff;
AlgId *aid;
{
	int	paramchoice;
	rsa_parm_type     * rsa_parm;
	desCBC_parm_type * des_parm;
	char *name;

	if ( !aid || !aid->objid ) {
		fprintf(ff, "%s\n", EAID);
		return(-1);
	}

	fprintf(ff, "Algorithm ");
	aux_fprint_ObjId(ff, aid->objid);

	name = aux_ObjId2Name(aid->objid);
	if(name) {
		fprintf(ff, "(%s)", name);
		free(name);
		if ( aid->parm ) {
			switch ( paramchoice = aux_ObjId2ParmType( aid->objid ) ) {

			case PARM_INTEGER:
				rsa_parm = (rsa_parm_type * )(aid->parm);
				fprintf(ff, ", Keysize = %d\n", *rsa_parm);
				break;

			case PARM_OctetString:
				des_parm = (desCBC_parm_type * )(aid->parm);
				fprintf(ff, ", DES-Initial-Vector(len %d): ", des_parm->noctets);
				if (des_parm->noctets) 
					aux_fxdump2(ff, des_parm->octets, des_parm->noctets, 0);
				fprintf(ff, "\n");
				break;
			default:
				fprintf(ff, "?? Unidentified parameter:\n");
				fprintf(ff, "  Parameter is set to INTEGER 0 (encode) or ");
				fprintf(ff, "ignored (decode).\n");
			}
		} 
	        else switch ( paramchoice = aux_ObjId2ParmType( aid->objid ) ) {
			case PARM_NULL:
				fprintf(ff, ", NULL parameter\n");
				break;
	                case PARM_ABSENT:
	        		fprintf(ff, ", no parameter\n");
	                        break;
	        }
	}
	else fprintf(ff, "(unknown object identifier)\n");
	return(0);
}


RC
aux_fprint_Certificate(ff, cert)
FILE *ff;
Certificate *cert;
{
	BitString   * bis;
	OctetString hash_result, *ostr;
	Name * printrepr;
	char * notbefore, *notafter;
	int	i;

	char	* proc = "aux_fprint_Certificate";

#ifdef TEST
	fprintf(stderr, "%s\n", proc);	
#endif

	if ( !cert ) {
		fprintf(ff, "%s\n", ECERT);
		return(-1);
	}

	if (print_cert_flag & DER) {
		if (print_cert_flag & TBS) {
			fprintf(ff, "%s", DERCODE);
			fprintf(ff, "of ToBeSigned:\n");
			if(cert->tbs_DERcode) aux_fxdump(ff, cert->tbs_DERcode->octets, cert->tbs_DERcode->noctets, 0);
			else fprintf(ff, " NULL\n");
		} else {
			if (print_cert_flag & ISSUER) {
				fprintf(ff, "%s", DERCODE);
				fprintf(ff, "of Issuer DName:\n");
				if(cert->tbs) {
					ostr = e_DName(cert->tbs->issuer);
					if (ostr) {
						aux_fxdump2(ff, ostr->octets, ostr->noctets, 0);
						aux_free_OctetString(&ostr);
					}
				} else fprintf(ff, " NULL\n");
			}
			if (print_cert_flag & KEYINFO) {
				fprintf(ff, "%s", DERCODE);
				fprintf(ff, "of SubjectPK:\n");
				if(cert->tbs) {
					ostr = e_KeyInfo(cert->tbs->subjectPK);
					if (ostr) {
						aux_fxdump2(ff, ostr->octets, ostr->noctets, 0);
						aux_free_OctetString(&ostr);
					}
				} else fprintf(ff, " NULL\n");
			}
		}
		if (print_cert_flag & ALG) {
			fprintf(ff, "%s", DERCODE);
			fprintf(ff, "of AlgorithmIdentifier:\n");
			if(cert->sig) {
				ostr = e_AlgId(cert->sig->signAI);
				if (ostr) {
					aux_fxdump2(ff, ostr->octets, ostr->noctets, 0);
					aux_free_OctetString(&ostr);
				}
			} else fprintf(ff, " NULL\n");
		}
	} else {
		if (print_cert_flag & TBS) aux_fprint_ToBeSigned(ff, cert->tbs);
		else {
			if(cert->tbs) {
				printrepr = aux_DName2Name(cert->tbs->issuer);
				if (print_cert_flag & ISSUER) fprintf(ff, "%s%s\n", ISS, printrepr);
				free(printrepr);
				if (print_cert_flag & VAL) {
					notbefore = aux_readable_UTCTime(cert->tbs->notbefore);
					notafter = aux_readable_UTCTime(cert->tbs->notafter);
					fprintf(ff, "%s%s (%s)\n", VALNB, notbefore, cert->tbs->notbefore);
					fprintf(ff, "%s%s (%s)\n", VALNA, notafter, cert->tbs->notafter);
					if(notbefore) free(notbefore);
					if(notafter) free(notafter);
				}
				if (print_cert_flag & KEYINFO) {
					fprintf(ff, "%s", SPK);
					aux_fprint_KeyInfo(ff, cert->tbs->subjectPK);
				}
			}
		}
		if (print_cert_flag & ALG) {
			fprintf(ff, "%s", IAI);
			if(cert->sig) aux_fprint_AlgId(ff, cert->sig->signAI);
			else fprintf(ff, " NULL\n");
		}
	}
	if ((print_cert_flag & HSH) && cert->sig) {
		sec_hash(cert->tbs_DERcode, &hash_result, END, cert->sig->signAI, (HashInput * )0);
		fprintf(ff, "%s\n", HSHRESULT);
		aux_fxdump2(ff, hash_result.octets, hash_result.noctets, 0);
	}
	if ((print_cert_flag & SIGNAT) && cert->sig) {
		fprintf(ff, "%s\n", SIGN);
		bis = &(cert->sig->signature);
		i = bis->nbits / 8;
		if ( bis->nbits % 8 ) 
			i++;
		aux_fxdump2(ff, bis->bits, i, 0);
	}
	return(0);
}


RC
aux_fprint_ToBeSigned(ff, tbs)
FILE *ff;
ToBeSigned *tbs;
{
	Name    * printrepr;
	char    * notbefore, * notafter;
	char	* proc = "aux_fprint_ToBeSigned";

#ifdef TEST
	fprintf(stderr, "%s\n", proc);	
#endif

	if ( !tbs ) {
		fprintf(ff, "%s\n", ETBS);
		return(-1);
	}

	fprintf(ff, "%s%d\n", VSN, tbs->version);
	fprintf(ff, "%s%d\n", SER, tbs->serialnumber);
	fprintf(ff, "%s", SAI);
	aux_fprint_AlgId(ff, tbs->signatureAI);

	printrepr = aux_DName2Name(tbs->issuer);
	fprintf(ff, "%s%s\n", ISS, printrepr);
	free(printrepr);

	notbefore = aux_readable_UTCTime(tbs->notbefore);
	notafter = aux_readable_UTCTime(tbs->notafter);
	fprintf(ff, "%s%s (%s)\n", VALNB, notbefore, tbs->notbefore);
	fprintf(ff, "%s%s (%s)\n", VALNA, notafter, tbs->notafter);
	if(notbefore) free(notbefore);
	if(notafter) free(notafter);

	printrepr = aux_DName2Name(tbs->subject);
	fprintf(ff, "%s%s\n", SUB, printrepr);
	free(printrepr);

	fprintf(ff, "%s", SPK);
	aux_fprint_KeyInfo(ff, tbs->subjectPK);

#ifdef COSINE
	if (tbs->authatts)
		aux_fprint_AuthorisationAttributes(ff, tbs->authatts);
#endif

	return(0);
}


#ifdef COSINE
RC
aux_fprint_AuthorisationAttributes(ff, authatts )
FILE *ff;
AuthorisationAttributes *authatts;
{

	if ( !authatts )
		return (0);

	fprintf(ff, "%s\n", AUTHATTR);

	if ( !authatts->country ) {
		fprintf(ff, "%s\n", ECOID);
		return(-1);
	}
	fprintf(ff, "%s%s\n", COID, authatts->country);

	if ( !authatts->group ) {
		fprintf(ff, "%s\n", EGID);
		return(-1);
	}
	fprintf(ff, "%s%s\n", GID, authatts->group);

	switch (authatts->class) {
	case Normal:
		fprintf(ff, "%s%s\n", CLID, NORMAL);
		break;
	case Privileged:
		fprintf(ff, "%s%s\n", CLID, PRIVILEGED);
		break;
	default:
		fprintf(ff, "%s\n", ECLID);
		return(-1);
	}
	
	return(0);
}
#endif

RC
aux_fprint_FCPath(ff, fcpath )
FILE *ff;
FCPath *fcpath;
{
	CrossCertificates * ccerts;
	Certificate       * cert;

	while (fcpath) {
		fprintf(ff, "\n ===next certificate in path:===\n");
		ccerts = fcpath->liste;
		while (ccerts) {
			cert = ccerts->element;
			aux_fprint_Certificate(ff, cert); /*print out elements of cert: */
			if (( ccerts = ccerts->next )) {
				fprintf(ff, "\n ===next crosscertificate:===\n");
			}
		}
		fcpath = fcpath->next_forwardpath;
	}
	return(0);
}


RC
aux_fprint_Certificates(ff, certs)
FILE *ff;
Certificates *certs;
{
	if ((!certs)) {
		fprintf(ff, "%s\n", ECERTS);
		return(-1);
	}
	fprintf(ff, "\n ===user certificate:===\n");
	if (aux_fprint_Certificate(ff, certs->usercertificate) == 0) 
		return(aux_fprint_FCPath(ff, certs->forwardpath));
	else 
		return(-1);
}


RC
aux_fprint_PKRoot(ff, pkroot)
FILE *ff;
PKRoot *pkroot;
{
	Name    * printrepr;
	char	* proc = "aux_fprint_PKRoot";

#ifdef TEST
	fprintf(stderr, "%s\n", proc);	
#endif

	if ( !pkroot ) {
		fprintf(ff, "%s\n", EPKROOT);
		return(-1);
	}

	printrepr = aux_DName2Name(pkroot->ca);
	fprintf(ff, "%s%s\n", ROO, printrepr);
	free(printrepr);
	
	if(pkroot->oldkey)
		fprintf(ff, "\n-------------------- Root Key (NEW) --------------------\n\n");

	aux_fprint_RootInfo(ff, pkroot->newkey, TRUE);
	if (! pkroot->oldkey)
		return;

	fprintf(ff, "\n-------------------- Root Key (OLD) --------------------\n\n");
	aux_fprint_RootInfo(ff, pkroot->oldkey, FALSE);

	return(0);
}


RC
aux_fprint_RootInfo(ff, rootinfo, new)
FILE * ff;
Serial * rootinfo;
Boolean new;
{
	char    	* notbefore, * notafter;
	BitString   	* bis;
	int		  i;

	if(new == TRUE) 
		fprintf(ff, "%s%d\n", SNN, rootinfo->serial);
	else 
		fprintf(ff, "%s%d\n", SNO, rootinfo->serial);

	if(rootinfo->version > 0)
		fprintf(ff, "%s%d\n", VSN, rootinfo->version);

	if(rootinfo->notbefore){
		notbefore = aux_readable_UTCTime(rootinfo->notbefore);
		fprintf(ff, "%s%s (%s)\n", VALNB, notbefore, rootinfo->notbefore);
		free(notbefore);
	}
	if(rootinfo->notafter){
		notafter = aux_readable_UTCTime(rootinfo->notafter);
		fprintf(ff, "%s%s (%s)\n", VALNA, notafter, rootinfo->notafter);
		free(notafter);
	}

	if(new == TRUE)
		fprintf(ff, "%s", PUKNEW);
	else
		fprintf(ff, "%s", PUKOLD);
	aux_fprint_KeyInfo(ff, rootinfo->key);

	if(rootinfo->sig){
		fprintf(ff, "%s", SAI);
		aux_fprint_AlgId(ff, rootinfo->sig->signAI);
	}

	if ((print_cert_flag & SIGNAT) && rootinfo->sig) {
		fprintf(ff, "%s\n", SIGN);
		bis = &(rootinfo->sig->signature);
		i = bis->nbits / 8;
		if ( bis->nbits % 8 ) 
			i++;
		aux_fxdump2(ff, bis->bits, i, 0);
	}

	return(0);
}


RC
aux_fprint_PKList(ff, pklist)
FILE *ff;
PKList *pklist;
{
	PKList * actlist;
	int	i = 1;

	actlist = pklist;
	while ( actlist ) {
		fprintf(ff, "%2d. -------------------------------------------------------------\n", i++);
		aux_fprint_ToBeSigned(ff, actlist->element);
		actlist = actlist->next;
	}
	return(0);
}


RC
aux_fprint_PemInfo(ff, peminfo)
FILE *ff;
PemInfo *peminfo;
{
	OctetString * aux_enchex();
	RecpList    * rl;
	OctetString * reckey;
	Name        * printrepr;
	char	    * proc = "aux_fprint_PemInfo";

#ifdef TEST
	fprintf(stderr, "%s\n", proc);	
#endif

	if ((!peminfo )) {
		fprintf(ff, "??No PemInfo specified\n");
		return(-1);
	}
	fprintf(ff, "\n*** ========PemInfo:========\n");
	if (peminfo->confidential ) {
		fprintf(ff, "*** confidential = ENCRYPTED\n");
		fprintf(ff, "*** encryptKEY:");
		if (peminfo->encryptKEY ) {
			fprintf(ff, "\n***    keyinfo: ");
			if (peminfo->encryptKEY->key) {
				aux_fprint_KeyInfo(ff, peminfo->encryptKEY->key);
			} else 
				fprintf(ff, " = (KeyInfo *)0\n");
			fprintf(ff, "***    keyref = %d\n", peminfo->encryptKEY->keyref);
			fprintf(ff, "***    pse_sel : ");
			if (peminfo->encryptKEY->pse_sel) {
				fprintf(ff, "PSE object name = \"%s\"\n",
				    peminfo->encryptKEY->pse_sel->object.name);
			} else 
				fprintf(ff, " = (PSESel *)0\n");
		} else 
			fprintf(ff, " = (Key *)0\n");
	} else {
		fprintf(ff, "*** confidential = MIC-ONLY\n");
		fprintf(ff, "*** encryptKEY: ignored\n");
	}

	fprintf(ff, "\n*** origcert:\n");
	aux_fprint_Certificates(ff, peminfo->origcert);
	fprintf(ff, "\n*** signAI: ");
	aux_fprint_AlgId(ff, peminfo->signAI);
	fprintf(ff, "\n*** recplist:");
	if ((rl = peminfo->recplist)) {
		while (rl) {
			fprintf(ff, "\n*** recipient's certificate:\n");
			if (rl->recpcert) {
				printrepr = aux_DName2Name(rl->recpcert->tbs->subject);
				fprintf(ff, "    owner  = \"%s\"\n", printrepr);
				free(printrepr);

				printrepr = aux_DName2Name(rl->recpcert->tbs->issuer);
				fprintf(ff, "    issuer = \"%s\"\n", printrepr);
				free(printrepr);

				fprintf(ff, "    ser.no = %d\n", rl->recpcert->tbs->serialnumber);
				fprintf(ff, "    subjectPublicKey: ");
				aux_fprint_KeyInfo(ff, rl->recpcert->tbs->subjectPK);
			} else 
				fprintf(ff, "    EMPTY\n");
			if (rl->key) {
				reckey = aux_enchex(rl->key);
				fprintf(ff, "*** recipient's encrypted key: %s\n", reckey->octets);
			} else 
				fprintf(ff, "*** recipient's encrypted key = NULL\n");
			rl = rl->next;
		}
	} else 
		fprintf(ff, " = (RecpList *)0\n");
	return(0);
}


RC
aux_fprint_RevCert(ff, revcert)
FILE *ff;
RevCert *revcert;
{
	BitString * bis;
	int	i;

	if (!revcert) {
		fprintf(ff, "%s\n", EREVCERT);
		return(-1);
	}

	aux_fprint_RevCertTBS(ff, revcert->tbs);

	fprintf(ff, "%s", AIRCA);
	aux_fprint_AlgId(ff, revcert->sig->signAI);

	fprintf(ff, "%s\n", SIGNRCA);
	bis = &(revcert->sig->signature);
	i = bis->nbits / 8;
	if ( bis->nbits % 8 ) 
		i++;
	aux_xdump2(bis->bits, i, 0);

	return(0);
}


RC
aux_fprint_RevCertTBS(ff, tbs)
FILE *ff;
RevCertTBS *tbs;
{
	Name * printrepr;
	char * time;
	char * proc = "aux_fprint_RevCertTBS";

#ifdef TEST
	fprintf(stderr, "%s\n", proc);	
#endif

	if ( !tbs ) {
		fprintf(ff, "%s\n", EREVCERTTBS);
		return(-1);
	}

	fprintf(ff, "%s", SAIRCA);
	aux_fprint_AlgId(ff, tbs->signatureAI);
	printrepr = aux_DName2Name(tbs->issuer);
	fprintf(ff, "%s%s\n", RCA, printrepr);
	free(printrepr);
	fprintf(ff, "%s%d\n", RSER, tbs->subject);
	time = aux_readable_UTCTime(tbs->revocationdate);
	fprintf(ff, "%s%s (%s)\n", REVDATE, time, tbs->revocationdate);
	if(time) free(time);

	return(0);
}


RC
aux_fprint_Crl(ff, rclist)
FILE *ff;
Crl *rclist;
{
	BitString * bis;
	int	i;

	if ( !rclist ) {
		fprintf(ff, "%s\n", ERCLIST);
		return(-1);
	}

	fprintf(ff, " --------------- Revocation List (X.500) ---------------\n");

	aux_fprint_CrlTBS(ff, rclist->tbs);

	fprintf(ff, "%s", IAI);
	aux_fprint_AlgId(ff, rclist->sig->signAI);

	fprintf(ff, "%s\n", SIGN);
	bis = &(rclist->sig->signature);
	i = bis->nbits / 8;
	if ( bis->nbits % 8 ) 
		i++;
	aux_xdump2(bis->bits, i, 0);

	return(0);
}


RC
aux_fprint_CrlTBS(ff, tbs)
FILE *ff;
CrlTBS *tbs;
{
	SEQUENCE_OF_RevCert * revcertseq;
	RevCert 	    * revcert;
	Name		    * printrepr;
	char                * time;
	char		    * proc = "aux_fprint_CrlTBS";

#ifdef TEST
	fprintf(stderr, "%s\n", proc);	
#endif

	if ( !tbs ) {
		fprintf(ff, "%s\n", ERCLISTTBS);
		return(-1);
	}

	fprintf(ff, "%s", SAI);
	aux_fprint_AlgId(ff, tbs->signatureAI);
	printrepr = aux_DName2Name(tbs->issuer);
	fprintf(ff, "%s%s\n", ISS, printrepr);
	free(printrepr);
	time = aux_readable_UTCTime(tbs->lastupdate);
	fprintf(ff, "%s%s (%s)\n", LASTUP, time, tbs->lastupdate);
	if(time) free(time);
	fprintf(ff, "\n --- Sequence of Revoked Certificates: ---\n");
	revcertseq = tbs->revokedcertificates;
	while (revcertseq) {
		revcert = revcertseq->element;
		aux_fprint_RevCert(ff, revcert);  /*print out elements of revcert: */

		if (( revcertseq = revcertseq->next )) {
			fprintf(ff, "\n --- next revoked certificate: ---\n");
		}
	}
	fprintf(ff, "\n\n");

	return(0);
}


RC
aux_fprint_RevCertPem(ff, revcertpem)
FILE *ff;
RevCertPem *revcertpem;
{
	char    * time;

	if ((!revcertpem)) {
		fprintf(ff, "%s\n", EREVCERTPEM);
		return(-1);
	}

	fprintf(ff, "%s%d\n", RSER, revcertpem->serialnumber);
	time = aux_readable_UTCTime(revcertpem->revocationDate);
	fprintf(ff, "%s%s (%s)\n", REVDATE, time, revcertpem->revocationDate);
	if(time) free(time);

	return(0);
}


RC
aux_fprint_PemCrl(ff, pemcrl)
FILE *ff;
PemCrl *pemcrl;
{
	BitString * bis;
	int	i;

	if ( !pemcrl ) {
		fprintf(ff, "%s\n", ERCLISTPEM);
		return(-1);
	}

	fprintf(ff, " --------------- Revocation List (PEM) ---------------\n");

	aux_fprint_PemCrlTBS(ff, pemcrl->tbs);

	fprintf(ff, "%s", IAI);
	aux_fprint_AlgId(ff, pemcrl->sig->signAI);

	fprintf(ff, "%s\n", SIGN);
	bis = &(pemcrl->sig->signature);
	i = bis->nbits / 8;
	if ( bis->nbits % 8 ) 
		i++;
	aux_xdump2(bis->bits, i, 0);

	return(0);
}


RC
aux_fprint_PemCrlTBS(ff, tbs)
FILE *ff;
PemCrlTBS *tbs;
{
	SEQUENCE_OF_RevCertPem   * seq;
	RevCertPem		 * revcertpem;
	Name 			 * printrepr;
	char                     * last, * next;
	char		         * proc = "aux_fprint_PemCrlTBS";

#ifdef TEST
	fprintf(stderr, "%s\n", proc);	
#endif

	if ( !tbs ) {
		fprintf(ff, "%s\n", ERCLISTPEMTBS);
		return(-1);
	}

	fprintf(ff, "%s", SAI);
	aux_fprint_AlgId(ff, tbs->signatureAI);
	printrepr = aux_DName2Name(tbs->issuer);
	fprintf(ff, "%s%s\n", ISS, printrepr);
	free(printrepr);
	last = aux_readable_UTCTime(tbs->lastUpdate);
	next = aux_readable_UTCTime(tbs->nextUpdate);
	fprintf(ff, "%s%s (%s)\n", LASTUP, last, tbs->lastUpdate);
	fprintf(ff, "%s%s (%s)\n", NEXTUP, next, tbs->nextUpdate);
	if(last) free(last);
	if(next) free(next);
	fprintf(ff, "\n --- Sequence of Revoked Certificates: ---\n");
	seq = tbs->revokedCertificates;
	while (seq) {
		revcertpem = seq->element;
		aux_fprint_RevCertPem(ff, revcertpem);  /*print out elements of revcertpem: */

		if (( seq = seq->next )) {
			fprintf(ff, "\n --- next revoked certificate: ---\n");
		}
	}
	fprintf(ff, "\n\n");

	return(0);
}


RC
aux_fprint_OCList(ff, oclist)
FILE *ff;
OCList *oclist;
{

	if ((!oclist )) {
		fprintf(ff, "%s\n", EOCLIST);
		return(-1);
	}

	fprintf(ff, "\n --- Old Certificates: ---\n");
	while (oclist) {
		fprintf(ff, "%s%d\n", OLDSER, oclist->serialnumber);
		fprintf(ff, "\n");
		aux_fprint_Certificate(ff, oclist->ccert);
		if (( oclist = oclist->next )) {
			fprintf(ff, "\n --- next entry in list: ---\n");
		}
	}
	return(0);
}

RC
aux_fprint_CertificateSet(ff, certset)
FILE *ff;
SET_OF_Certificate *certset;
{

	if ((!certset )) {
		fprintf(ff, "%s\n", ECERTSET);
		return(-1);
	}

	fprintf(ff, "\n --------------- Set of Certificates: ---------------\n");

	while (certset) {
		if (!certset->element) 
			break;
		aux_fprint_Certificate(ff, certset->element);
		fprintf(ff, "------------------------------------------------------\n");
		certset = certset->next;
	}
	return(0);
}


RC
aux_fprint_CertificatePairSet(ff, cpair)
FILE *ff;
SET_OF_CertificatePair *cpair;
{

	if ( !cpair ) {
		fprintf(ff, "%s\n", ECERTPAIR);
		return(-1);
	}

	fprintf(ff, "\n --------------- crossCertificatePair ---------------\n\n\n");
	while (cpair) {
		fprintf(ff, "  --- ForwardCertificate ---\n\n");
		aux_fprint_Certificate(ff, cpair->element->forward);
		fprintf(ff, "\n\n  --- ReverseCertificate ---\n\n");
		aux_fprint_Certificate(ff, cpair->element->reverse);
		fprintf(ff, "\n------------------------------------------------------\n\n\n");
		cpair = cpair->next;
	}
	return(0);
}


RC
aux_fprint_CrlPSE(ff, crlpse)
FILE *ff;
CrlPSE *crlpse;
{
	Name       		* printrepr;
	SEQUENCE_OF_RevCertPem  * seq;
	RevCertPem		* revcertpem;
	char       		* update;
	char       		* proc = "aux_fprint_CrlPSE";

#ifdef TEST
	fprintf(stderr, "%s\n", proc);	
#endif

	if ( !crlpse ) {
		fprintf(ff, "%s\n", EREVLISTPSE);
		return(-1);
	}

	fprintf(ff, "\n ------------- Revocation List on PSE: -------------\n");

	printrepr = aux_DName2Name(crlpse->issuer);
	fprintf(ff, "%s%s\n", ISS, printrepr);
	free(printrepr);

	update = aux_readable_UTCTime(crlpse->nextUpdate);
	fprintf(ff, "%s%s (%s)\n", NEXTUP, update, crlpse->nextUpdate);
	if(update) free(update);

	fprintf(ff, "\n --- Sequence of Revoked Certificates: ---\n");
	seq = crlpse->revcerts;
	while (seq) {
		revcertpem = seq->element;
		aux_fprint_RevCertPem(ff, revcertpem);  /*print out elements of revcertpem: */
		if ((seq = seq->next)) fprintf(ff, "\n --- next revoked certificate: ---\n");
	}

	return(0);
}


RC
aux_fprint_CrlSet(ff, crlset)
FILE *ff;
CrlSet *crlset;
{

	if (( !crlset )) {
		fprintf(ff, "%s\n", EREVLISTSET);
		return(-1);
	}

	fprintf(ff, "\n--------------- Set of Revocation Lists: ---------------\n");

	while (crlset) {
		if (!crlset->element) 
			break;
		aux_fprint_CrlPSE(ff, crlset->element);
		fprintf(ff, " ------------------------------------------------------\n");
		crlset = crlset->next;
	}
	return(0);
}


RC
aux_fprint_IssuedCertificate(ff, isscert)
FILE *ff;
IssuedCertificate *isscert;
{
	char *date_of_issue;

	if ( !isscert ) {
		fprintf(ff, "%s\n", EISSCERT);
		return(-1);
	}

	date_of_issue = aux_readable_UTCTime(isscert->date_of_issue);
	fprintf(ff, " Date of Issue: %s,%s%d\n", date_of_issue, SER, isscert->serial);
	if(date_of_issue) free(date_of_issue);
	return(0);
}


RC
aux_fprint_SET_OF_IssuedCertificate(ff, isscertset)
FILE *ff;
SET_OF_IssuedCertificate *isscertset;
{

        int i = 0;
	if ( !isscertset ) {
		fprintf(ff, "%s\n", EISSCERTSET);
		return(-1);
	}

	while (isscertset) {
		if (!isscertset->element) break;
                fprintf(ff, "%d. ", ++i);
		aux_fprint_IssuedCertificate(ff, isscertset->element);
		isscertset = isscertset->next;
	}
	return(0);
}


RC
aux_fprint_SET_OF_Name(ff, nameset)
FILE *ff;
SET_OF_Name *nameset;
{

        int i = 0;
	if ( !nameset ) {
		fprintf(ff, "%s\n", ENAMESET);
		return(-1);
	}

	while (nameset) {
		if (!nameset->element) break;
		fprintf(ff, "%d. %s\n", ++i, nameset->element);
		nameset = nameset->next;
	}
	return(0);
}


RC
aux_fprint_PemCrlWithCerts(ff, arg)
FILE * ff;
PemCrlWithCerts * arg;
{
	if (! arg) {
		fprintf(ff, "%s\n", EPEMCRLWITHCERTS);
		return(- 1);
	}

	aux_fprint_PemCrl(ff, arg->pemcrl);
	if(arg->certificates) aux_fprint_Certificates(ff, arg->certificates);

	return(0);
}


RC
aux_fprint_SET_OF_PemCrlWithCerts(ff, set)
FILE * ff;
SET_OF_PemCrlWithCerts * set;
{
	if (! set) {
		fprintf(ff, "%s\n", EPEMCRLWITHCERTSSET);
		return(- 1);
	}

	while (set) {
		if (! set->element) 
			break;
		aux_fprint_PemCrlWithCerts(ff, set->element);
		fprintf(ff, "------------------------------------------------------\n");
		set = set->next;
	}
	return(0);
}


RC
aux_fprint_VerificationResult(ff, verres)
FILE *ff;
VerificationResult *verres;
{
	int     index = 0;
	Name  * subject, * issuer;
	char  * time = CNULL;

	if (! verres ) {
		fprintf(ff, "%s\n", VERRES);
		return(-1);
	}

	fprintf(ff, "\n");

	if(verres->textverified == TRUE) {
		/* call of af_verify() and af_verify_Certificates() */
		if (verres->success == TRUE) {
			if(! verres->verifstep || ! verres->verifstep[0]){
				fprintf(ff, "Text signature of <%s>\n\n", verres->top_name);
			}
			else {
				subject = aux_DName2Name(verres->verifstep[0]->cert->tbs->subject);
				fprintf(ff, "Text signature of <%s>\n\n", subject);
				if(subject) free(subject);
			}
		}
		else
			fprintf(ff, "Text signature\n\n");
	}
	else {
		/* call of af_verify_Certificates() only */
		if(!verres->verifstep || !verres->verifstep[0] || !verres->verifstep[0]->cert){
			if(verres->trustedKey == - 2){
				fprintf(ff, "Certificate (containing Public Root Key)\n\n");
				fprintf(ff, "    - issued by <%s>\n", verres->top_name);
				fprintf(ff, "    - serial number %d\n\n", verres->top_serial);
				time = aux_readable_UTCTime(verres->date);
				fprintf(ff, "    - E X P I R E D (expiration date: %s (%s))\n\n", time, verres->date);
				if(time) free(time);
				return(0);
			}
			else{
				fprintf(ff, "VerificationResult structure could not be filled due to an unexpected return "); 
				fprintf(ff, "from the af_verify() or af_verify_Certificates() routine.\n\n");
				fprintf(ff, "Print error stack for any further information.\n\n");
				return(0);
			}
		}
		subject = aux_DName2Name(verres->verifstep[0]->cert->tbs->subject);
		fprintf(ff, "Certificate of owner <%s>\n\n", subject);
		if(subject) free(subject);

		if(af_verbose){
			aux_fprint_Certificate(ff, verres->verifstep[0]->cert);
			fprintf(ff, "\n");
		}

		if(verres->verifstep[0]->supplied == 0)
			fprintf(ff, "    - supplied within received 'Certificates'\n");
		else
			fprintf(ff, "    - retrieved from Directory\n");

		switch(verres->verifstep[0]->crlcheck) {
		case REVOKED:
			time = aux_readable_UTCTime(verres->verifstep[0]->date);
			fprintf(ff, "    - R E V O K E D (revocation date: %s (%s))\n\n", time, verres->verifstep[0]->date);
			if (time) free (time);
			if (! verres->verifstep[1])
				return(0);
			break;
		case NOT_REVOKED:
			fprintf(ff, "    - not revoked\n\n");
			break;
		case CRL_NOT_AVAILABLE:
			fprintf(ff, "    - CRL not available\n\n");
			break;
		case CRL_OUT_OF_DATE:
			time = aux_readable_UTCTime(verres->verifstep[0]->date);
			fprintf(ff, "    - CRL is out-of-date (next update: %s (%s))\n\n", time, verres->verifstep[0]->date);
			if (time) free (time);
			break;
		case NOT_REQUESTED:
			fprintf(ff, "    - Checking against CRL was not requested\n\n");
			break;
		case 0:
			fprintf(ff, "\n\n");
			break;
		default:
			fprintf(ff, "undefined value\n");
			break;
		}  /* switch */

		if (verres->success == FALSE && ! verres->verifstep[1] && verres->verifstep[0]->date) {
			/* usercertificate itself has expired */
			if (verres->trustedKey == - 2) {
				time = aux_readable_UTCTime(verres->verifstep[0]->date);
				fprintf(ff, "    - E X P I R E D (expiration date: %s (%s))\n\n", time, verres->verifstep[0]->date);
				if (time) free (time);
				return(0);
			}
		}

		index++;
	}


	while (verres->verifstep &&
	       ((verres->success == TRUE && verres->verifstep[index]) ||
	       (verres->success == FALSE && (verres->trustedKey == - 1 || verres->trustedKey == - 2 || verres->trustedKey == - 4) && verres->verifstep[index + 1]) ||
	       (verres->success == FALSE && verres->trustedKey != - 1 && verres->trustedKey != - 2 && verres->trustedKey != - 4 && verres->verifstep[index]))) {

		fprintf(ff, "... validated with certificate\n\n");

		if(af_verbose){
			aux_fprint_Certificate(ff, verres->verifstep[index]->cert);
			fprintf(ff, "\n");
		}

		issuer = aux_DName2Name(verres->verifstep[index]->cert->tbs->issuer);
		fprintf(ff, "    - issued by <%s>\n", issuer); 
		if (issuer)
			free (issuer);
		fprintf(ff, "    - serial number %d\n", verres->verifstep[index]->cert->tbs->serialnumber);

		if(verres->verifstep[index]->supplied == 0)
			fprintf(ff, "    - supplied within received 'Certificates'\n");
		else
			fprintf(ff, "    - retrieved from Directory\n");

		switch(verres->verifstep[index]->crlcheck) {
		case REVOKED:
			time = aux_readable_UTCTime(verres->verifstep[index]->date);
			fprintf(ff, "    - R E V O K E D (revocation date: %s (%s))\n\n", time, verres->verifstep[index]->date);
			if (time) free (time);
			break;
		case NOT_REVOKED:
			fprintf(ff, "    - not revoked\n\n");
			break;
		case CRL_NOT_AVAILABLE:
			fprintf(ff, "    - CRL not available\n\n");
			break;
		case CRL_OUT_OF_DATE:
			time = aux_readable_UTCTime(verres->verifstep[index]->date);
			fprintf(ff, "    - CRL is out-of-date (next update: %s (%s))\n\n", time, verres->verifstep[index]->date);
			if (time) free (time);
			break;
		case NOT_REQUESTED:
			fprintf(ff, "    - Checking against CRL was not requested\n\n");
			break;
		case 0:
			fprintf(ff, "\n\n");
			break;
		default:
			fprintf(ff, "undefined value\n");
			break;
		}  /* switch */

		index++;

	}   /* while */


	if (verres->success == TRUE) {
		if(verres->trustedKey == 4){
			fprintf(ff, "\n");
			fprintf(ff, "which is your own certificate\n");
			fprintf(ff, "\n");
			return(0);
		}

		fprintf(ff, "... validated with certificate\n\n");

		fprintf(ff, "    - issued by <%s>\n", verres->top_name);
		fprintf(ff, "    - serial number %d\n\n", verres->top_serial);
	
		switch(verres->trustedKey) {
		case 0:
			fprintf(ff, "whose public key was trusted and found in PKRoot(new)\n");
			break;
		case 1:
			fprintf(ff, "whose public key was trusted and found in PKRoot(old)\n");
			break;
		case 2:
			fprintf(ff, "whose public key was trusted and found in PKList\n");
			break;
		case 3:
			fprintf(ff, "whose public key was trusted and found in FCPath\n");
			break;
		}  /* switch */
	}

	if (verres->success == FALSE) {
		if (verres->verifstep && verres->verifstep[index]) {
			fprintf(ff, "... n o t  validated with certificate\n\n");

			if(af_verbose){
				aux_fprint_Certificate(ff, verres->verifstep[index]->cert);
				fprintf(ff, "\n");
			}

			issuer = aux_DName2Name(verres->verifstep[index]->cert->tbs->issuer);
			fprintf(ff, "    - issued by <%s>\n", issuer);
			if (issuer)
				free (issuer);
			fprintf(ff, "    - serial number %d\n", verres->verifstep[index]->cert->tbs->serialnumber);

			if(verres->verifstep[index]->supplied == 0)
				fprintf(ff, "    - supplied within received 'Certificates'\n");
			else
				fprintf(ff, "    - retrieved from Directory\n");
	
			switch(verres->verifstep[index]->crlcheck) {
			case REVOKED:
				time = aux_readable_UTCTime(verres->verifstep[index]->date);
				fprintf(ff, "    - R E V O K E D (revocation date: %s (%s))\n", time, verres->verifstep[index]->date);
				if (time) free (time);
				break;
			case NOT_REVOKED:
				fprintf(ff, "    - not revoked\n");
				break;
			case CRL_NOT_AVAILABLE:
				fprintf(ff, "    - CRL not available\n");
				break;
			case CRL_OUT_OF_DATE:
				time = aux_readable_UTCTime(verres->verifstep[index]->date);
				fprintf(ff, "    - CRL is out-of-date (next update: %s (%s))\n", time, verres->verifstep[index]->date);
				if (time) free (time);
				break;
			case NOT_REQUESTED:
				fprintf(ff, "    - Checking against CRL was not requested\n");
				break;
			case 0:
				if (verres->trustedKey == - 1)
					fprintf(ff, "    - verification  F A I L E D\n\n");
				if (verres->trustedKey == - 2) {
					time = aux_readable_UTCTime(verres->verifstep[index]->date);
					fprintf(ff, "    - E X P I R E D (expiration date: %s (%s))\n\n", time, verres->verifstep[index]->date);
					if (time) free (time);
				}
				if (verres->trustedKey == - 4) {
					subject = aux_DName2Name(verres->verifstep[index - 1]->cert->tbs->subject);
					issuer = aux_DName2Name(verres->verifstep[index - 1]->cert->tbs->issuer);
					fprintf(ff, "    - PEM naming subordination rule  V I O L A T E D, i.e.\n");
					fprintf(ff, "      <%s> is NOT subordinated to <%s>\n\n", subject, issuer);
					if (subject) free(subject);
					if (issuer) free(issuer);
				}
				break;
			default:
				fprintf(ff, "undefined value\n");
				break;
			}  /* switch */
		}

		else if (verres->trustedKey == - 3)
			fprintf(ff, "... n o t  validated (no certificate available)\n\n");

		else {
			fprintf(ff, "... n o t  validated with certificate\n\n");

			fprintf(ff, "    - issued by <%s>\n", verres->top_name);
			fprintf(ff, "    - serial number %d\n", verres->top_serial);

			switch(verres->trustedKey) {
			case 0:
				fprintf(ff, "    - verification  F A I L E D\n\n");
				fprintf(ff, "whose public key was found in PKRoot(new)\n");
				break;
			case 1:
				fprintf(ff, "    - verification  F A I L E D\n\n");
				fprintf(ff, "whose public key was found in PKRoot(old)\n");
				break;
			case 2:
				fprintf(ff, "    - verification  F A I L E D\n\n");
				fprintf(ff, "whose public key was found in PKList\n");
				break;
			case 3:
				fprintf(ff, "    - verification  F A I L E D\n\n");
				fprintf(ff, "whose public key was found in FCPath\n");
				break;
			case - 1:
				break;
			case - 2:
				time = aux_readable_UTCTime(verres->date);
				fprintf(ff, "    - E X P I R E D (expiration date: %s (%s))\n\n", time, verres->date);
				if (time) free (time);
				break;
			default:
				fprintf(ff, "undefined value\n");
				break;
			}  /* switch */
		}
	}


	fprintf(ff, "\n");

	return(0);
}



RC
aux_fprint_error(ff, last_only)
FILE *ff;
int	last_only;
{
	struct ErrStack *err;
	int	n;

	if (!err_stack) return(0);

	for (err = err_stack ; err != 0 ; err = err->next ) {
		for (n = 0; err_list[n].id != 0; n++) if (err_list[n].id == err->e_number) break;
		fprintf(ff, "ERROR in %s: %s (%d)\n", err->e_proc, err->e_text, err->e_number);
		if (err->e_addr) {
			switch (err->e_addrtype) {

			case char_n:
				fprintf(ff, "%s\n", err->e_addr);
				break;
                        case DName_n:
                                fprintf(ff, "%s\n", aux_DName2Name((DName *)err->e_addr));
                                break;
			case Certificate_n:
				aux_fprint_Certificate(ff, (Certificate *)err->e_addr);
				break;
			case CertificatePair_n:
				aux_fprint_Certificate(ff, ((CertificatePair * )err->e_addr)->forward);
				aux_fprint_Certificate(ff, ((CertificatePair * )err->e_addr)->reverse);
				break;
			case Certificates_n:
				aux_fprint_Certificates(ff, (Certificates *)err->e_addr);
				break;
			case PKList_n:
				aux_fprint_PKList(ff, (PKList *)err->e_addr);
				break;
			case OctetString_n:
				aux_fxdump2(ff, ((OctetString * )err->e_addr)->octets, ((OctetString * )err->e_addr)->noctets , 0);
				break;
			case BitString_n:
				aux_fxdump2(ff, ((OctetString * )err->e_addr)->octets, (((OctetString * )err->e_addr)->noctets + 7) / 8 , 0);
				break;
			case SET_OF_Certificate_n:
				aux_fprint_CertificateSet(ff, (SET_OF_Certificate *)err->e_addr);
				break;
			case SET_OF_CertificatePair_n:
				aux_fprint_CertificatePairSet(ff, (SET_OF_CertificatePair *)err->e_addr);
				break;
			case AlgId_n:
				aux_fprint_AlgId(ff, (AlgId *)err->e_addr);
				break;
			case OCList_n:
				aux_fprint_OCList(ff, (OCList *)err->e_addr);
				break;
			case CrlTBS_n:
				aux_fprint_CrlTBS(ff, (CrlTBS *)err->e_addr);
				break;
			case Crl_n:
				aux_fprint_Crl(ff, (Crl *)err->e_addr);
				break;
			case RevCert_n:
				aux_fprint_RevCert(ff, (RevCert *)err->e_addr);
				break;
			case PemCrlTBS_n:
				aux_fprint_PemCrlTBS(ff, (PemCrlTBS *)err->e_addr);
				break;
			case PemCrl_n:
				aux_fprint_PemCrl(ff, (PemCrl *)err->e_addr);
				break;
			case RevCertPem_n:
				aux_fprint_RevCertPem(ff, (RevCertPem *)err->e_addr);
				break;
			case CrlSet_n:
				aux_fprint_CrlSet(ff, (CrlSet *)err->e_addr);
				break;
			case CrlPSE_n:
				aux_fprint_CrlPSE(ff, (CrlPSE *)err->e_addr);
				break;
			case PemInfo_n:
				aux_fprint_PemInfo(ff, (PemInfo *)err->e_addr);
				break;
			case KeyInfo_n:
				aux_fprint_KeyInfo(ff, (KeyInfo *)err->e_addr);
				break;
			case FCPath_n:
				aux_fprint_FCPath(ff, (FCPath *)err->e_addr);
				break;
			case PKRoot_n:
				aux_fprint_PKRoot(ff, (PKRoot *)err->e_addr);
				break;
			case IssuedCertificate_n:
				aux_fprint_IssuedCertificate(ff, (IssuedCertificate *)err->e_addr);
				break;
			case SET_OF_IssuedCertificate_n:
				aux_fprint_SET_OF_IssuedCertificate(ff, (SET_OF_IssuedCertificate *)err->e_addr);
				break;
			case SET_OF_Name_n:
				aux_fprint_SET_OF_Name(ff, (SET_OF_Name *)err->e_addr);
				break;
			case ToBeSigned_n:
				aux_fprint_ToBeSigned(ff, (ToBeSigned *)err->e_addr);
				break;
			case ObjId_n:
				aux_fprint_ObjId(ff, (ObjId *)err->e_addr);
				break;
			case KeyBits_n:
				aux_fprint_KeyBits(ff, (KeyBits *)err->e_addr);
				break;
			case PSEToc_n:
				aux_fprint_PSEToc(ff, (PSEToc *)err->e_addr);
				break;
			case PSESel_n:
				aux_fprint_PSESel(ff, (PSESel *)err->e_addr);
				break;
			case PemCrlWithCerts_n:
				aux_fprint_PemCrlWithCerts(ff, (PemCrlWithCerts *)err->e_addr);
				break;
			case SET_OF_PemCrlWithCerts_n:
				aux_fprint_SET_OF_PemCrlWithCerts(ff, (SET_OF_PemCrlWithCerts *)err->e_addr);
				break;
			}
		}
		if (last_only) return(0);
	}
}


