/*
 * this file is part of "The W Toolkit".
 *
 * (W) 1996, Kay Roemer.
 *
 * pane widget.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <Wlib.h>
#include <Wt.h>
#include "toolkit.h"

typedef struct {
	widget_t w;
	short is_open;
	short is_realized;
	short orient;
	short align;
	short dist;
} pane_widget_t;

static long pane_query_geometry (widget_t *, long *, long *, long *, long *);

static long
pane_init (void)
{
	return 0;
}

static widget_t *
pane_create (widget_class_t *cp)
{
	pane_widget_t *wp = malloc (sizeof (pane_widget_t));
	if (!wp)
		return NULL;
	memset (wp, 0, sizeof (pane_widget_t));
	wp->w.class = wt_pane_class;
	wp->orient = OrientVert;
	wp->align = AlignCenter;
	return (widget_t *)wp;
}

static long
pane_delete (widget_t *_w)
{
	pane_widget_t *w = (pane_widget_t *)_w;
	widget_t *wp, *next;

	for (wp = w->w.childs; wp; wp = next) {
		next = wp->next;
		(*wp->class->delete) (wp);
	}
	if (w->is_realized)
		w_delete (w->w.win);
	free (w);
	return 0;
}

static long
pane_close (widget_t *_w)
{
	pane_widget_t *w = (pane_widget_t *)_w;

	if (w->is_realized && w->is_open) {
		w_close (w->w.win);
		w->is_open = 0;
	}
	return 0;
}

static long
pane_open (widget_t *_w)
{
	pane_widget_t *w = (pane_widget_t *)_w;

	if (w->is_realized && !w->is_open) {
		w_open (w->w.win, w->w.x, w->w.y);
		w->is_open = 1;
	}
	return 0;
}

static long
pane_addchild (widget_t *parent, widget_t *w)
{
	widget_t *wp;

	for (wp = parent->childs; wp && wp->next; wp = wp->next)
		;
	wt_add_after (parent, wp, w);
	return 0;
}

static long
pane_delchild (widget_t *parent, widget_t *w)
{
	wt_remove (w);
	return 0;
}

static long
pane_realize (widget_t *_w, WWIN *parent)
{
	pane_widget_t *w = (pane_widget_t *)_w;
	long x, y, wd, ht, curx, cury;
	widget_t *wp;

	if (w->is_realized)
		return -1;

	pane_query_geometry (_w, &x, &y, &wd, &ht);
	w->w.w = wd;
	w->w.h = ht;

	w->w.win = wt_create_window (parent, wd, ht,
		W_NOBORDER|W_CONTAINER|W_MOVE);
	if (!w->w.win)
		return -1;
	w->w.win->user_val = (long)w;

	if (w->orient == OrientHorz) {
		/*
		 * horizontal
		 */
		curx = 0;
		for (wp = w->w.childs; wp; wp = wp->next) {
			(*wp->class->query_geometry) (wp, &x, &y, &wd, &ht);
			if (w->align < 0) {
				cury = 0;
			} else if (w->align > 0) {
				cury = w->w.h - ht;
			} else {
				cury = (w->w.h - ht)/2;
			}
			(*wp->class->setopt) (wp, WT_XPOS, &curx);
			(*wp->class->setopt) (wp, WT_YPOS, &cury);
			curx += wd + w->dist;
		}
	} else {
		/*
		 * vertical
		 */
		cury = 0;
		for (wp = w->w.childs; wp; wp = wp->next) {
			(*wp->class->query_geometry) (wp, &x, &y, &wd, &ht);
			if (w->align < 0) {
				curx = 0;
			} else if (w->align > 0) {
				curx = w->w.w - wd;
			} else {
				curx = (w->w.w - wd)/2;
			}
			(*wp->class->setopt) (wp, WT_XPOS, &curx);
			(*wp->class->setopt) (wp, WT_YPOS, &cury);
			cury += ht + w->dist;
		}
	}
	for (wp = w->w.childs; wp; wp = wp->next) {
		if ((*wp->class->realize) (wp, w->w.win) < 0) {
			w_delete (w->w.win);
			return -1;
		}
	}
	w->is_realized = 1;
	w->is_open = 1;
	w_open (w->w.win, w->w.x, w->w.y);
	return 0;
}

static long
pane_query_geometry (widget_t *_w, long *xp, long *yp, long *wdp, long *htp)
{
	pane_widget_t *w = (pane_widget_t *)_w;
	long maxw, maxh, totw, toth, cx, cy, cw, ch;
	widget_t *child;

	*xp = w->w.x;
	*yp = w->w.y;
	if (w->w.w > 0 && w->w.h > 0) {
		*wdp = w->w.w;
		*htp = w->w.h;
		return 0;
	}
	maxw = 0;
	maxh = 0;
	totw = 0;
	toth = 0;
	for (child = w->w.childs; child; child = child->next) {
		(*child->class->query_geometry) (child, &cx, &cy, &cw, &ch);
		if (cw > maxw)
			maxw = cw;
		if (ch > maxh)
			maxh = ch;
		totw += cw + w->dist;
		toth += ch + w->dist;
	}
	totw -= w->dist;
	toth -= w->dist;
	if (w->orient == OrientHorz) {
		/*
		 * horizontal
		 */
		*wdp = w->w.w > 0 ? w->w.w : (totw > 0 ? totw : 100);
		*htp = w->w.h > 0 ? w->w.h : (maxh > 0 ? maxh : 100);
	} else {
		/*
		 * vertical
		 */
		*wdp = w->w.w > 0 ? w->w.w : (maxw > 0 ? maxw : 100);
		*htp = w->w.h > 0 ? w->w.h : (toth > 0 ? toth : 100);
	}
	return 0;
}

static long
pane_setopt (widget_t *_w, long key, void *val)
{
	pane_widget_t *w = (pane_widget_t *)_w;
	short mask = 0;

	switch (key) {
	case WT_XPOS:
		w->w.x = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_YPOS:
		w->w.y = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_WIDTH:
		w->w.w = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_HEIGHT:
		w->w.h = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_ORIENTATION:
		w->orient = *(long *)val;
		break;

	case WT_ALIGNMENT:
		w->align = *(long *)val;
		break;

	case WT_HDIST:
	case WT_VDIST:
		w->dist = *(long *)val;
		break;

	default:
		return -1;
	}
	if (mask && w->is_realized)
		wt_change_notify (_w, mask);
	return 0;
}

static long
pane_getopt (widget_t *_w, long key, void *val)
{
	pane_widget_t *w = (pane_widget_t *)_w;

	switch (key) {
	case WT_XPOS:
		*(long *)val = w->w.x;
		break;

	case WT_YPOS:
		*(long *)val = w->w.y;
		break;

	case WT_WIDTH:
		*(long *)val = w->w.w;
		break;

	case WT_HEIGHT:
		*(long *)val = w->w.h;
		break;

	case WT_ORIENTATION:
		*(long *)val = w->orient;
		break;

	case WT_ALIGNMENT:
		*(long *)val = w->align;
		break;

	case WT_HDIST:
	case WT_VDIST:
		*(long *)val = w->dist;
		break;

	default:
		return -1;
	}
	return 0;
}

static long
pane_event (widget_t *_w, WEVENT *ev)
{
	return -1;
}

static long
pane_changes (widget_t *w, widget_t *w2, short changes)
{
	return 0;
}

static widget_class_t _wt_pane_class = {
	"pane", 0,
	pane_init,
	pane_create,
	pane_delete,
	pane_close,
	pane_open,
	pane_addchild,
	pane_delchild,
	pane_realize,
	pane_query_geometry,
	pane_setopt,
	pane_getopt,
	pane_event,
	pane_changes,
	pane_changes
};

widget_class_t *wt_pane_class = &_wt_pane_class;
