/*
 * this file is part of "The W Toolkit".
 *
 * (W) 1996, Kay Roemer.
 *
 * button and label widget.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <Wlib.h>
#include <Wt.h>
#include "toolkit.h"

typedef struct {
	widget_t w;
	char label[64];
	char fontname[32];
	WFONT *font;
	short is_open;
	short is_realized;
	short is_pressed;
	short mode;
	short align;
	enum { PushButton, RadioButton, CheckButton, Label } type;
	void (*press_cb) (widget_t *w, int pressed);
	void (*draw_fn) (widget_t *w, long x, long y, long wd, long ht);
} button_widget_t;


static long button_query_geometry (widget_t *, long *, long *, long *, long *);


static long
button_draw (button_widget_t *w)
{
	int x = 0, y = 0, wd = 0, ht = 0;

	w_setmode (w->w.win, M_CLEAR);
	w_pbox (w->w.win, 0, 0, w->w.w, w->w.h);

	switch (w->type) {
	case Label:
		if (w->mode == LabelModeNoBorder) {
			x = y = 0;
			wd = w->w.w;
			ht = w->w.h;
			break;
		}
		/*fallthrough*/

	case PushButton:
		wt_box3d (w->w.win, 0, 0, w->w.w, w->w.h);
		x = y = 3;
		wd = w->w.w-6;
		ht = w->w.h-6;
		break;

	case RadioButton:
		wt_circle3d (w->w.win, w->w.h/2, w->w.h/2, w->w.h/2 - 3);
		x = w->w.h;
		y = 0;
		wd = MAX (0, w->w.w - w->w.h);
		ht = w->w.h;
		break;

	case CheckButton:
		wt_box3d (w->w.win, 3, 3, w->w.h-6, w->w.h-6);
		x = w->w.h;
		y = 0;
		wd = MAX (0, w->w.w - w->w.h);
		ht = w->w.h;
		break;
	}
	if (w->label[0] && w->font) {
		w_setfont (w->w.win, w->font);
		wt_text (w->w.win, w->font, w->label,
			x+1, y+1, wd-2, ht-2, w->align);
	} else if (w->draw_fn) {
		(*w->draw_fn) ((widget_t *)w, x, y, wd, ht);
	}
	return 0;
}

static long
button_press (button_widget_t *w)
{
	switch (w->type) {
	case RadioButton:
		wt_circle3d_press (w->w.win, w->w.h/2, w->w.h/2, w->w.h/2 - 3);
		wt_circle3d_mark (w->w.win, w->w.h/2, w->w.h/2, w->w.h/2 - 3);
		break;

	case Label:
		if (w->mode == LabelModeNoBorder)
			break;

	case PushButton:
		wt_box3d_press (w->w.win, 0, 0, w->w.w, w->w.h, 1);
		break;

	case CheckButton:
		wt_box3d_press (w->w.win, 3, 3, w->w.h-6, w->w.h-6, 0);
		wt_box3d_mark (w->w.win, 3, 3, w->w.h-6, w->w.h-6);
		break;

	default:
		return 0;
	}
	return 0;
}

static long
button_release (button_widget_t *w)
{
	switch (w->type) {
	case RadioButton:
		wt_circle3d_unmark (w->w.win, w->w.h/2, w->w.h/2,
			w->w.h/2 - 3);
		wt_circle3d_release (w->w.win, w->w.h/2, w->w.h/2,
			w->w.h/2 - 3);
		break;

	case Label:
		if (w->mode == LabelModeNoBorder)
			break;

	case PushButton:
		wt_box3d_release (w->w.win, 0, 0, w->w.w, w->w.h, 1);
		break;

	case CheckButton:
		wt_box3d_unmark (w->w.win, 3, 3, w->w.h-6, w->w.h-6);
		wt_box3d_release (w->w.win, 3, 3, w->w.h-6, w->w.h-6, 0);
		break;

	default:
		return 0;
	}
	return 0;
}

static int
button_loadfont (button_widget_t *w, char *fname)
{
	WFONT *fp = w_loadfont (fname);
	if (!fp)
		return -1;
	w->font = fp;
	strncpy (w->fontname, fname, sizeof (w->fontname));
	w->fontname[sizeof (w->fontname) - 1] = '\0';
	return 0;
}

	
static long
button_init (void)
{
	return 0;
}

static widget_t *
button_create (widget_class_t *cp)
{
	button_widget_t *wp = malloc (sizeof (button_widget_t));
	if (!wp)
		return NULL;
	memset (wp, 0, sizeof (button_widget_t));
	wp->w.class = cp;

	if (cp == wt_pushbutton_class) {
		wp->type = PushButton;
		wp->mode = ButtonModePush;
	} else if (cp == wt_checkbutton_class) {
		wp->type = CheckButton;
		wp->mode = ButtonModeToggle;
	} else if (cp == wt_radiobutton_class) {
		wp->type = RadioButton;
		wp->mode = ButtonModeRadio;
	} else {
		wp->type = Label;
		wp->mode = ButtonModePush;
	}
	if (button_loadfont (wp, "fixed10.wfnt")) {
		free (wp);
		return NULL;
	}
	return (widget_t *)wp;
}

static long
button_delete (widget_t *_w)
{
	button_widget_t *w = (button_widget_t *)_w;

	if (w->is_realized)
		w_delete (w->w.win);
	w_unloadfont (w->font);
	free (w);
	return 0;
}

static long
button_close (widget_t *_w)
{
	button_widget_t *w = (button_widget_t *)_w;

	if (w->is_realized && w->is_open) {
		w_close (w->w.win);
		w->is_open = 0;
	}
	return 0;
}

static long
button_open (widget_t *_w)
{
	button_widget_t *w = (button_widget_t *)_w;

	if (w->is_realized && !w->is_open) {
		w_open (w->w.win, w->w.x, w->w.y);
		w->is_open = 1;
	}
	return 0;
}

static long
button_addchild (widget_t *parent, widget_t *w)
{
	return -1;
}

static long
button_delchild (widget_t *parent, widget_t *w)
{
	return -1;
}

static long
button_realize (widget_t *_w, WWIN *parent)
{
	button_widget_t *w = (button_widget_t *)_w;
	long x, y, wd, ht;

	if (w->is_realized)
		return -1;

	button_query_geometry (_w, &x, &y, &wd, &ht);
	w->w.x = x;
	w->w.y = y;
	w->w.w = wd;
	w->w.h = ht;

	if (w->type == Label) {
		w->w.win = wt_create_window (parent, wd, ht, W_NOBORDER|W_MOVE);
	} else {
		w->w.win = wt_create_window (parent, wd, ht,
			W_MOVE|W_NOBORDER|EV_ACTIVE|EV_MOUSE);
	}
	if (!w->w.win)
		return -1;
	w->w.win->user_val = (long)w;

	button_draw (w);
	if (w->is_pressed)
		button_press (w);

	w->is_realized = 1;
	w->is_open = 1;
	w_open (w->w.win, w->w.x, w->w.y);
	return 0;
}

static long
button_query_geometry (widget_t *_w, long *xp, long *yp, long *wdp, long *htp)
{
	button_widget_t *w = (button_widget_t *)_w;
	int wd, ht;

	*xp = w->w.x;
	*yp = w->w.y;
	*wdp = w->w.w;
	*htp = w->w.h;
	if (*wdp > 0 && *htp > 0)
		return 0;

	wd = w_strlen (w->font, w->label) + w->font->height;
	ht = 2*w->font->height;

	switch (w->type) {
	case Label:
		if (w->mode == LabelModeNoBorder)
			break;

	case PushButton:
		wd += 4;
		ht += 4;
		break;

	case RadioButton:
	case CheckButton:
		wd += ht;
		break;
	}
	if (*wdp <= 0)
		*wdp = wd;
	if (*wdp <= 0)
		*wdp = 100;

	if (*htp <= 0)
		*htp = ht;
	if (*htp <= 0)
		*htp = 25;

	return 0;
}

static long
button_setopt (widget_t *_w, long key, void *val)
{
	button_widget_t *w = (button_widget_t *)_w;
	int needredraw = 0;
	short mask = 0;

	switch (key) {
	case WT_XPOS:
		w->w.x = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_YPOS:
		w->w.y = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_WIDTH:
		w->w.w = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_HEIGHT:
		w->w.h = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_LABEL:
		strncpy (w->label, (char *)val, sizeof (w->label));
		w->label[sizeof (w->label) - 1] = 0;
		needredraw = 1;
		break;

	case WT_FONT:
		if (button_loadfont (w, (char *)val)) {
			return -1;
		}
		needredraw = 1;
		break;

	case WT_DRAW_FN:
		w->draw_fn = val;
		w->label[0] = 0;
		needredraw = 1;
		break;

	case WT_ACTION_CB:
		w->press_cb = val;
		break;

	case WT_MODE:
		if (w->is_realized)
			return -1;
		w->mode = *(long *)val;
		break;

	case WT_STATE:
		if (w->is_pressed && *(long *)val == ButtonStateReleased) {
			w->is_pressed = 0;
			if (w->press_cb)
				(*w->press_cb) (_w, 0);
			if (w->is_realized)
				button_release (w);
		} else if (!w->is_pressed &&
			   *(long *)val == ButtonStatePressed) {
			w->is_pressed = 1;
			if (w->press_cb)
				(*w->press_cb) (_w, 1);
			if (w->is_realized)
				button_press (w);
		}
		break;

	default:
		return -1;
	}
	if (mask && w->is_realized)
		wt_change_notify (_w, mask);
	if (w->is_realized && needredraw) {
		button_draw (w);
		if (w->is_pressed) {
			button_press (w);
		}
	}
	return 0;
}

static long
button_getopt (widget_t *_w, long key, void *val)
{
	button_widget_t *w = (button_widget_t *)_w;

	switch (key) {
	case WT_XPOS:
		*(long *)val = w->w.x;
		break;

	case WT_YPOS:
		*(long *)val = w->w.y;
		break;

	case WT_WIDTH:
		*(long *)val = w->w.w;
		break;

	case WT_HEIGHT:
		*(long *)val = w->w.h;
		break;

	case WT_LABEL:
		strcpy ((char *)val, w->label);
		break;

	case WT_FONT:
		strcpy ((char *)val, w->fontname);
		break;

	case WT_ACTION_CB:
		*(long *)val = (long)w->press_cb;
		break;

	case WT_DRAW_FN:
		*(long *)val = (long)w->draw_fn;
		break;

	case WT_MODE:
		*(long *)val = w->mode;
		break;

	case WT_STATE:
		*(long *)val = w->is_pressed
			? ButtonStatePressed
			: ButtonStateReleased;
		break;

	default:
		return -1;
	}
	return 0;
}

static long
button_event (widget_t *_w, WEVENT *ev)
{
	button_widget_t *w = (button_widget_t *)_w;

	switch (ev->type) {
	case EVENT_MPRESS:
		if (!(ev->key & BUTTON_LEFT))
			break;
		switch (w->mode) {
		case LabelModeNoBorder:
			break;

		case ButtonModePush:
		case ButtonModeRadio:
			if (!w->is_pressed) {
				if (w->press_cb)
					(*w->press_cb) (_w, 1);
				w->is_pressed = 1;
				button_press (w);
			}
			break;

		case ButtonModeToggle:
			if (w->press_cb)
				(*w->press_cb) (_w, !w->is_pressed);
			if (w->is_pressed) {
				w->is_pressed = 0;
				button_release (w);
			} else {
				w->is_pressed = 1;
				button_press (w);
			}
			break;
		}
		break;

	case EVENT_MRELEASE:
		if (!(ev->key & BUTTON_LEFT))
			break;
		if (w->mode == ButtonModePush && w->is_pressed) {
			if (w->press_cb)
				(*w->press_cb) (_w, 0);
			w->is_pressed = 0;
			button_release (w);
		}
		break;

	case EVENT_INACTIVE:
		if (w->mode == ButtonModePush && w->is_pressed) {
			if (w->press_cb)
				(*w->press_cb) (_w, -1);
			w->is_pressed = 0;
			button_release (w);
		}
		break;

	default:
		return -1;
	}
	return 0;
}

static long
button_changes (widget_t *w, widget_t *w2, short changes)
{
	return 0;
}

static widget_class_t _wt_pushbutton_class = {
	"pushbutton", 0,
	button_init,
	button_create,
	button_delete,
	button_close,
	button_open,
	button_addchild,
	button_delchild,
	button_realize,
	button_query_geometry,
	button_setopt,
	button_getopt,
	button_event,
	button_changes,
	button_changes
};

static widget_class_t _wt_radiobutton_class = {
	"radiobutton", 0,
	button_init,
	button_create,
	button_delete,
	button_close,
	button_open,
	button_addchild,
	button_delchild,
	button_realize,
	button_query_geometry,
	button_setopt,
	button_getopt,
	button_event,
	button_changes,
	button_changes
};

static widget_class_t _wt_checkbutton_class = {
	"checkbutton", 0,
	button_init,
	button_create,
	button_delete,
	button_close,
	button_open,
	button_addchild,
	button_delchild,
	button_realize,
	button_query_geometry,
	button_setopt,
	button_getopt,
	button_event,
	button_changes,
	button_changes
};

static widget_class_t _wt_label_class = {
	"label", 0,
	button_init,
	button_create,
	button_delete,
	button_close,
	button_open,
	button_addchild,
	button_delchild,
	button_realize,
	button_query_geometry,
	button_setopt,
	button_getopt,
	button_event,
	button_changes,
	button_changes
};

widget_class_t *wt_pushbutton_class = &_wt_pushbutton_class;
widget_class_t *wt_radiobutton_class = &_wt_radiobutton_class;
widget_class_t *wt_checkbutton_class = &_wt_checkbutton_class;
widget_class_t *wt_label_class  = &_wt_label_class;
