/*
 * this file is part of "The W Toolkit".
 *
 * (W) 1996, Kay Roemer.
 *
 * box widget.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <Wlib.h>
#include <Wt.h>
#include "toolkit.h"

typedef struct {
	widget_t w;
	short is_open;
	short is_realized;
	short orient;
	short xdist, ydist;
} box_widget_t;

static void
vbox_place_childs (box_widget_t *w, long *wdp, long *htp, int movechilds)
{
	widget_t *child;
	long maxy, maxw, curx, cury;
	long cx, cy, ch, cw;

	maxy = w->w.h <= 0 ? LONG_MAX : w->w.h;
	curx = w->xdist;
	cury = w->ydist;
	maxw = 0;
	*wdp = 0;
	*htp = 0;

	for (child = w->w.childs; child; child = child->next) {
		(*child->class->query_geometry) (child, &cx, &cy, &cw, &ch);
		if (maxw > 0 && cury+ch >= maxy) {
			curx += maxw + w->xdist;
			cury = w->ydist;
			maxw = 0;
		}
		if (movechilds) {
			(*child->class->setopt) (child, WT_XPOS, &curx);
			(*child->class->setopt) (child, WT_YPOS, &cury);
		}
		if (curx + cw > *wdp)
			*wdp = curx + cw;
		if (cury + ch > *htp)
			*htp = cury + ch;

		cury += ch + w->ydist;
		if (cw > maxw)
			maxw = cw;
	}
	*wdp = w->w.w > 0 ? w->w.w : (*wdp > w->xdist ? *wdp + w->xdist : 100);
	*htp = w->w.h > 0 ? w->w.h : (*htp > w->ydist ? *htp + w->ydist : 100);	
}

static void
hbox_place_childs (box_widget_t *w, long *wdp, long *htp, int movechilds)
{
	widget_t *child;
	long maxx, maxh, curx, cury;
	long cx, cy, ch, cw;

	maxx = w->w.w <= 0 ? LONG_MAX : w->w.w;
	curx = w->xdist;
	cury = w->ydist;
	maxh = 0;
	*wdp = 0;
	*htp = 0;

	for (child = w->w.childs; child; child = child->next) {
		(*child->class->query_geometry) (child, &cx, &cy, &cw, &ch);
		if (maxh > 0 && curx+cw >= maxx) {
			cury += maxh + w->ydist;
			curx = w->xdist;
			maxh = 0;
		}
		if (movechilds) {
			(*child->class->setopt) (child, WT_XPOS, &curx);
			(*child->class->setopt) (child, WT_YPOS, &cury);
		}
		if (curx + cw > *wdp)
			*wdp = curx + cw;
		if (cury + ch > *htp)
			*htp = cury + ch;

		curx += cw + w->xdist;
		if (ch > maxh)
			maxh = ch;
	}
	*wdp = w->w.w > 0 ? w->w.w : (*wdp > w->xdist ? *wdp + w->xdist : 100);
	*htp = w->w.h > 0 ? w->w.h : (*htp > w->ydist ? *htp + w->ydist : 100);	
}

static long
box_init (void)
{
	return 0;
}

static widget_t *
box_create (widget_class_t *cp)
{
	box_widget_t *wp = malloc (sizeof (box_widget_t));
	if (!wp)
		return NULL;
	memset (wp, 0, sizeof (box_widget_t));
	wp->w.class = wt_box_class;
	wp->xdist = 6;
	wp->ydist = 6;
	wp->orient = OrientHorz;
	return (widget_t *)wp;
}

static long
box_delete (widget_t *_w)
{
	box_widget_t *w = (box_widget_t *)_w;
	widget_t *wp, *next;

	for (wp = w->w.childs; wp; wp = next) {
		next = wp->next;
		(*wp->class->delete) (wp);
	}
	if (w->is_realized)
		w_delete (w->w.win);
	free (w);
	return 0;
}

static long
box_close (widget_t *_w)
{
	box_widget_t *w = (box_widget_t *)_w;

	if (w->is_realized && w->is_open) {
		w_close (w->w.win);
		w->is_open = 0;
	}
	return 0;
}

static long
box_open (widget_t *_w)
{
	box_widget_t *w = (box_widget_t *)_w;

	if (w->is_realized && !w->is_open) {
		w_open (w->w.win, w->w.x, w->w.y);
		w->is_open = 1;
	}
	return 0;
}

static long
box_addchild (widget_t *parent, widget_t *w)
{
	widget_t *wp;

	for (wp = parent->childs; wp && wp->next; wp = wp->next)
		;
	wt_add_after (parent, wp, w);
	return 0;
}

static long
box_delchild (widget_t *parent, widget_t *w)
{
	wt_remove (w);
	return 0;
}

static long
box_realize (widget_t *_w, WWIN *parent)
{
	box_widget_t *w = (box_widget_t *)_w;
	widget_t *wp;
	long wd, ht;

	if (w->is_realized)
		return -1;

	if (w->orient == OrientHorz)
		hbox_place_childs (w, &wd, &ht, 1);
	else 
		vbox_place_childs (w, &wd, &ht, 1);

	w->w.w = wd;
	w->w.h = ht;

	w->w.win = wt_create_window (parent, wd, ht,
		W_NOBORDER|W_CONTAINER|W_MOVE);
	if (!w->w.win)
		return -1;
	w->w.win->user_val = (long)w;
	for (wp = w->w.childs; wp; wp = wp->next) {
		if ((*wp->class->realize) (wp, w->w.win) < 0) {
			w_delete (w->w.win);
			return -1;
		}
	}
	w->is_realized = 1;
	w->is_open = 1;
	w_open (w->w.win, w->w.x, w->w.y);
	return 0;
}

static long
box_query_geometry (widget_t *_w, long *xp, long *yp, long *wdp, long *htp)
{
	box_widget_t *w = (box_widget_t *)_w;
	
	*xp = w->w.x;
	*yp = w->w.y;
	if (w->w.w > 0 && w->w.h > 0) {
		*wdp = w->w.w;
		*htp = w->w.h;
		return 0;
	}
	if (w->orient == OrientHorz)
		hbox_place_childs (w, wdp, htp, 0);
	else
		vbox_place_childs (w, wdp, htp, 0);		
	return 0;
}

static long
box_setopt (widget_t *_w, long key, void *val)
{
	box_widget_t *w = (box_widget_t *)_w;
	short mask = 0;

	switch (key) {
	case WT_XPOS:
		w->w.x = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_YPOS:
		w->w.y = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_WIDTH:
		w->w.w = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_HEIGHT:
		w->w.h = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_ORIENTATION:
		w->orient = *(long *)val;
		break;

	case WT_HDIST:
		w->xdist = *(long *)val;
		break;

	case WT_VDIST:
		w->ydist = *(long *)val;

	default:
		return -1;
	}
	if (mask && w->is_realized)
		wt_change_notify (_w, mask);
	return 0;
}

static long
box_getopt (widget_t *_w, long key, void *val)
{
	box_widget_t *w = (box_widget_t *)_w;

	switch (key) {
	case WT_XPOS:
		*(long *)val = w->w.x;
		break;

	case WT_YPOS:
		*(long *)val = w->w.y;
		break;

	case WT_WIDTH:
		*(long *)val = w->w.w;
		break;

	case WT_HEIGHT:
		*(long *)val = w->w.h;
		break;

	case WT_ORIENTATION:
		*(long *)val = w->orient;
		break;

	case WT_HDIST:
		*(long *)val = w->xdist;
		break;

	case WT_VDIST:
		*(long *)val = w->ydist;
		break;

	default:
		return -1;
	}
	return 0;
}

static long
box_event (widget_t *_w, WEVENT *ev)
{
	return -1;
}

static long
box_changes (widget_t *w, widget_t *w2, short changes)
{
	return 0;
}

static widget_class_t _wt_box_class = {
	"box", 0,
	box_init,
	box_create,
	box_delete,
	box_close,
	box_open,
	box_addchild,
	box_delchild,
	box_realize,
	box_query_geometry,
	box_setopt,
	box_getopt,
	box_event,
	box_changes,
	box_changes
};

widget_class_t *wt_box_class = &_wt_box_class;
