/*
 * server/main.c, part of W
 * (C) 1994,95,96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * main startup / initialization and signal / terminate / exit code
 *
 * changes:
 * - moved terminate code from loop to here.
 * - rewrote 'wconfig' file reading (there are now also some other variables
 *   beside 'menuitem').
 *   ++eero 10/97
 */

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <pwd.h>
#include <termios.h>

#ifdef SVGALIB
#include <vga.h>
#include <vgamouse.h>
#endif

#ifdef linux
#include <linux/kd.h>
#endif

#ifdef __MINT__
# include <support.h>
# include <mintbind.h>
#endif

#if !defined(sun) && !defined(__NetBSD__)
#include <getopt.h>
#endif

#include "config.h"
#include "types.h"
#include "pakets.h"
#include "proto.h"
#include "window.h"


/* 
 * global variables
 */
short glob_uid;
SCREEN *glob_screen = NULL;
int glob_ineth = 0, glob_unixh = 0;
short is_terminating = 0;
short glob_fontsize = 0;
char *glob_fontfamily = NULL;
char *glob_fontpath = NULL;
int glob_debug = 0;

static struct termios glob_termios;
static int forceMono = 0;


/* 
 * read the config file and set global variables and menu accordingly.
 * settings are of a form:
 *	setting = value
 *	setting = title, command
 *
 * White space and (other) control characters are stripped off from
 * both ends.
 */
static void read_config(struct passwd *pw, char *config, char **menufont, char **titlefont)
{
	char *file, *line, *value, *cmd, c;
	int idx, maxlen, eq, have_exit_item = 0;
	FILE *fp = NULL;

	menu.items = 0;
	menu.maxlen = 0;
	*menufont = NULL;
	*titlefont = NULL;

	maxlen = strlen(pw->pw_dir);
	if(maxlen < (idx = strlen(LIBDIR))) {
		maxlen = idx;
	}
	maxlen += strlen(config) + 3;
	if (maxlen < MAX_LINELEN) {
		maxlen = MAX_LINELEN;
	}
	if ((file = malloc(maxlen))) {
		sprintf(file, "%s/.%s", pw->pw_dir, config);
		if(!(fp = fopen(file, "r"))) {
			sprintf(file, "%s/%s", LIBDIR, config);
			if(!(fp = fopen(file, "r"))) {
				/* no configuration file */
				free(file);
				file = 0;
			}
		}
	} else {
		fprintf(stderr, "wserver: out of memory\r\n");
	}

	/* read configuration variables */

	while (file && fgets(file, maxlen, fp)) {

		line = file;
		/* search variable start */
		while(*line && *line <= 32) {
			line++;
		}

		eq = 0;
		for (idx = 0; idx < maxlen && line[idx]; idx++) {
			c = line[idx];
			/* search value end */
			if ((c < 32 && c != '\t') || c == '#') {
				break;
			}
			if (c == '=') {
				/* variable end / value start */
				eq = idx;
			}
		}
		if (!eq) {
			continue;
		}

		/* remove white space at the end and start of the value */
		while(--idx > eq && line[idx] < 32)
			;
		line[++idx] = 0;
		value = line + eq;
		while(*++value && *value <= 32)
			;

		/* remove white space at the end of variable name */
		while(line[--eq] <= 32)
			;
		line[++eq] = 0;

		idx = (line + idx) - value;
		/* 'eq' got now variable and 'idx' value lenght */


		/* add into the root menu */

		if (!strcmp(line, "menuitem")) {
			if ((cmd = strchr(value, ','))) {
				line = cmd;
				while (*line <= 32) {
					*line++ = 0;
				}
				do {
					*cmd++ = 0;
				} while (*cmd && *cmd <= 32);
#ifndef REFRESH
				if (!strcasecmp(cmd, "@refresh")) {
					cmd = NULL;
				}
#endif
			}
			if (cmd && (menu.items < (MAXITEMS-1))) {
				idx = strlen(value);
				if (idx > menu.maxlen) {
					menu.maxlen = idx;
				}
				menu.item[menu.items].title = strdup(value);
				menu.item[menu.items].command = strdup(cmd);
				if (!strcasecmp(cmd, "@exit")) {
					have_exit_item = 1;
				}
				menu.items++;
			}
			continue;
		}

		/* check for global defaults */

		/* server menufont */
		if (!strcmp(line, "menufont")) {
			*menufont = strdup(value);
			continue;
		}
		/* server titlefont */
		if (!strcmp(line, "titlefont")) {
			*titlefont = strdup(value);
			continue;
		}
		/* font loading path */
		if (!strcmp(line, "fontpath")) {
			glob_fontpath = strdup(value);
			continue;
		}
		/* default client font family... */
		if (!strcmp(line, "fontfamily")) {
			glob_fontfamily = strdup(value);
			continue;
		}
		/* ...and size */
		if (!strcmp(line, "fontsize")) {
			glob_fontsize = atoi(value);
			continue;
		}
		/* more to come? */
	}

	if (file) {
		fclose(fp);
		free(file);
	}

	/* no server fonts defined? use defaults... */
	if(!*titlefont) {
		*titlefont = strdup(DEF_WTITLEFONT);
	}
	if(!*menufont) {
		*menufont = strdup(DEF_WMENUFONT);
	}

	/* hmm, no menu defined? use default menu... */
	if (!menu.items) {
		strcpy(menu.item[2].title,   "wterm tiny");
		strcpy(menu.item[2].command, "wterm -s 7");
		strcpy(menu.item[3].title,   "wterm default");
		strcpy(menu.item[3].command, "wterm");
		strcpy(menu.item[4].title,   "wterm large");
		strcpy(menu.item[4].command, "wterm -s 11");
		menu.maxlen = 15;
		menu.items = 3;
	}

	/* if there is no exit item in the menu, add one... */
	if (!have_exit_item) {
		strcpy(menu.item[menu.items].title, "exit");
		strcpy(menu.item[menu.items].command, "@exit");
		if (menu.maxlen < 4) {
			menu.maxlen = 4;
		}
		menu.items++;
	}
}


/*
 * my own way to install signal handlers, always != SA_ONESHOT
 */

void *mysignal(int signum, void *handler)
{
#ifdef __MINT__
  return signal(signum, handler);
#else
  struct sigaction sa, so;

  sa.sa_handler = handler;
  sa.sa_mask = 0;
  sa.sa_flags = 0;
#if 0
  sa.sa_restorer = NULL;
#endif
  sigaction(signum, &sa, &so);

  return so.sa_handler;
#endif
}

static void sigroutine(int sig)
{
  terminate(sig, "signal");
#ifdef linux
  mysignal(sig, sigroutine);
#endif
}

/*
 * a two step terminator...
 */

void terminate(int sig, char *msg)
{
  CLIENT *cptr;
  EVENTP paket;

  mysignal(sig, SIG_IGN);

  /*
   * shall we just prepare exiting?
   */

  if (!is_terminating && !sig) {

    is_terminating = 1;

    /* this will tell all clients to terminate */

    cptr = glob_clients;
    while (cptr) {
      paket.len        = htons(sizeof(EVENTP));
      paket.type       = htons(PAK_EVENT);
      paket.event.type = htons(EVENT_GADGET);
      paket.event.key  = htonl(GADGET_EXIT);
      write(cptr->sh, &paket, sizeof(EVENTP));
      cptr = cptr->next;
    }

    mysignal(SIGALRM, sigroutine);
    alarm(10);

    return;
  }

  /*
   * something more urgent occured (or all clients died)
   */

  wserver_exit(sig, msg);
}

/* this should be called even in case of an error so that text screen
 * will be restored (and in case wserver is suid root, wserver socket
 * file removal would be nice too...
 */
void wserver_exit(int sig, char *msg)
{
#ifdef SVGALIB
    vga_setmode(TEXT);
#else	/* !SVGALIB */
#ifdef linux
    ioctl(0, KDSETMODE, KD_TEXT);
#endif
#if 0	/* I want to see the error messages */
    clear_scr();
#endif
    cursor_on();
#endif	/* !SVGALIB */

  /* who not for MiNT too? ++kay */
  tcsetattr(0, TCSANOW, &glob_termios);

  if (glob_unixh > 0) {
    close(glob_unixh);
    if (unlink(UNIXNAME) < 0) {
      fprintf(stderr, "Unlinking " UNIXNAME " socket failed.\r\n");
    }
  }

  if (sig > 0) {
    fprintf(stderr, "fatal: Wserver received signal #%i\r\n", sig);
  } else if (sig < 0) {
    fprintf(stderr, "fatal: Wserver terminated with message: %s\r\n", msg);
  }
  exit(sig);  
}

/*
 *  a SIGCHLD routine, just to remove the zombies
 */

static void sigchild(int arg)
{
#ifdef SIGTSTP		/* BSD signals need this handler */
  /* uk: ask for information about terminated children, until there is no more,
   * just for the case that several processes died at the same time.
   */
  long res;
#ifdef __MINT__
  union wait status;

  while ((res = waitpid(-1, (__WP)&status, WNOHANG)) > 0) {
#else
  int status;

  while ((res = waitpid(-1, &status, WNOHANG)) > 0) {
#endif
    /* do nothing so far */
  }
#endif
}

/*
 *
 */

static void startup(struct passwd *pw)
{
  char fname[MAX(strlen(pw->pw_dir), strlen(LIBDIR)) + 6];
  short fh;

#ifndef LIBDIR
#error You must define LIBDIR!!!
#endif

  sprintf(fname, "%s/.wrc", pw->pw_dir);
  if (access(fname, X_OK)) {
    sprintf(fname, "%s/wrc", LIBDIR);
    if (access(fname, X_OK)) {
      return;
    }
  }

  switch (fork()) {

    case -1: /* error */
      fprintf(stderr, "Wserver (warning): Can't fork startup sequence\r\n");
      break;

    case 0: /* child */
      if ((fh = open("/dev/null", O_RDWR)) >= 0) {
	dup2(fh, 0);
	dup2(fh, 1);
	dup2(fh, 2);
	close(fh);
      } else {
	fprintf(stderr, "Wserver (warning): Startup sequence can't open /dev/null\r\n");
      }

      /* this shouldn't return */
      execl(pw->pw_shell, pw->pw_shell, fname, NULL);

      /* just to be sure the main program continues as wanted */
      _exit(-99);
      break;

    default: /* parent */
  }

  /* if this failed we can still go on */
}


/*
 *  how to init the sockets?
 */

static int initunixsocket(void)
{
  int sh;
  struct sockaddr unixladdr;

  if ((sh = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
    perror("server: socket(AF_UNIX)");
    return -1;
  }

  unixladdr.sa_family = AF_UNIX;
  strcpy(unixladdr.sa_data, UNIXNAME);
  if (bind(sh, &unixladdr, sizeof(struct sockaddr))) {
    close(sh);
    perror("server: bind(AF_UNIX)");
    return -1;
  }

  if (listen(sh, MAXLISTENQUEUE)) {
    close(sh);
    perror("server: listen(AF_UNIX)");
    return -1;
  }

  return sh;
}


#ifndef AF_UNIX_ONLY

static int initinetsocket(void)
{
  int sh;
  struct sockaddr_in inetladdr;

  if ((sh = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("server: socket(AF_INET)");
    return -1;
  }

  /* bind to anything which is local */

  inetladdr.sin_family = AF_INET;
  inetladdr.sin_addr.s_addr = INADDR_ANY;
  inetladdr.sin_port = htons(SERVERPORT);
  if (bind(sh, (struct sockaddr *)&inetladdr, sizeof(struct sockaddr))) {
    close(sh);
    perror("server: bind(AF_INET)");
    return -1;
  }

  if (listen(sh, MAXLISTENQUEUE)) {
    close(sh);
    perror("server: listen(AF_INET)");
    return -1;
  }

  return sh;
}

#endif


/*
 *  major init code
 *
 * NOTES: SVGALIB programs have to be suid root for screen initialization,
 * after which the library will give up the root priviledges. Therefore
 * it's advisable to do screen intialization as soon as possible (so that
 * e.g. wserver socket on /tmp/wserver won't be owned by root)...
 *
 * SVGALIB also adds it's own signal handlers to most signals so that it
 * can restore text screen in case of an error!
 */

static short initialize (void)
{
  char *ttyn, *ptr = NULL;
  struct passwd *pw;
  struct termios mytermios;
  char *titlefont, *menufont;

  mysignal(SIGHUP, sigroutine);
  mysignal(SIGQUIT, sigroutine);
  mysignal(SIGILL, sigroutine);
  mysignal(SIGTERM, sigroutine);
  mysignal(SIGINT, sigroutine);
  /* can only happen if a client terminated, and we're going to deal with it
   * somewhere else.
   */
  mysignal(SIGPIPE, SIG_IGN);
#ifdef SIGTSTP				/* BSD */
  mysignal(SIGSTOP, SIG_IGN);
  mysignal(SIGTSTP, SIG_IGN);
  mysignal(SIGCHLD, sigchild);
#else
  mysignal(SIGCLD, SIG_IGN);		/* SYSV */
#endif

#ifndef SVGALIB		/* SVGAlib does this itself */
  cursor_off();
#endif

  if (wmouse_init ()) {
    fprintf(stderr, "error: can't find mouse, or it's busy\r\n");
    return -1;
  }

  /* the sequence must be:
   *
   * 1) init screen
   * 2) init colors (because it uses a screen specific function)
   * 3) init windows (because it uses color information)
   */
  if (!(glob_screen = screen_init(forceMono))) {
    fprintf(stderr, "error: this screen is not supported\r\n");
    return -1;
  }

  /*
   *  check for terminal name
   */

  if (!(ttyn = ttyname(0))) {
    fprintf(stderr, "error: can't resolve tty name\r\n");
    return -2;
  }

  while (*ttyn) {
    if (*ttyn++ == '/') {
      ptr = ttyn;
    }
  }

  if (!ptr) {
    fprintf(stderr, "error: illegal tty name\r\n");
    return -2;
  }

#if !defined(linux) && !defined(__NetBSD__)
  /* under linux things aren't that easy to decide...
   *
   * (Phx) nor under NetBSD, which is booted under /dev/ttyeX (where
   * X depends on the graphics device which is used...
   */
  if (strcmp(ptr, "console")) {
    fprintf(stderr, "error: W cannot be started on %s\r\n", ptr);
    return -2;
  }
#endif

  /*
   *  check for sockets
   */

  if ((glob_unixh = initunixsocket()) < 0) {
    fprintf(stderr, "error: can't create local communication socket. if there isn't\r\n");
    fprintf(stderr, "       currently a Wserver running this is an error and you\r\n");
    fprintf(stderr, "       should retry it after removing %s\r\n", UNIXNAME);
    return -2;
  }

  /* from here on we assume that no other Wserver is running */

#ifndef AF_UNIX_ONLY
  if ((glob_ineth = initinetsocket()) < 0) {
    /* this may fail due to missing network */
    fprintf(stderr, "warning: can't create any network communication socket, at least\r\n");
    fprintf(stderr, "         the loopback interface should be enabled\r\n");
  }
#endif

  /*
   *  some more initializations
   */

  /*
   *  check for login name
   */
  if (!(pw = getpwuid(glob_uid = getuid()))) {
    fprintf(stderr, "error: couldn't resolve your login name\r\n");
    return -2;
  }

  if (kbdInit() < 0) {
    return -1;
  }

  /* reads menu definition, font name(s) etc. */
  read_config(pw, "wconfig", &menufont, &titlefont);

  if (font_init(titlefont, menufont)) {
    fprintf(stderr, "error: can't load fonts\r\n");
    return -1;
  }
  if(titlefont)
    free(titlefont);
  if(menufont)
    free(menufont);


  if (colorInit()) {
    wserver_exit(-1, "Color initialization failed!");
  }
  if (window_init() < 0) {
    wserver_exit(-1, "Window initialization failed!");
  }

#ifdef SVGALIB
  /* can not set the mouse to middle of screen */
  glob_mouse.real.x0 = 0;
  glob_mouse.real.y0 = 0;
  mouse_setposition(0,0);
#else
  glob_mouse.real.x0 = glob_screen->bm.width >> 1;
  glob_mouse.real.y0 = glob_screen->bm.height >> 1;
#endif
  glob_mouse.real.x1 = glob_mouse.real.x0 + 15;
  glob_mouse.real.y1 = glob_mouse.real.y0 + 15;

  mouse_show();

  /* set terminal to raw mode */

  tcgetattr(0, &glob_termios);
  mytermios = glob_termios;
  makeraw (&mytermios);
  tcsetattr(0, TCSANOW, &mytermios);

  FD_ZERO(&glob_crfd);

  /*
   *  other stuff...
   */

  switch (fork()) {
    case -1:	/* error */
      fprintf (stderr, "wserver: cannot execute copyright\r\n");
      break;
    case 0:	/* child */
      execlp ("wcpyrgt", "wcpyrgt", NULL);
      fprintf (stderr, "wserver: cannot execute copyright\r\n");
      _exit (-99);
  }

  startup(pw);

  return 0;
}


/*
 * guess what... :)
 */

static void usage (char *prgname)
{
  fprintf(stderr, "usage: %s [--debug] [--forcemono]\r\n", prgname);
}


#if !defined(sun) && !defined(__NetBSD__)
static struct option options[] = {
  {"debug", 0, NULL, 1},
  {"forcemono", 0, NULL, 2},
  {NULL, 0, NULL, 0}
};
#endif


int main (int argc, char *argv[])
{
  int ret;

#ifdef SIGTTOU
  /* (BSD) allow background writes (ignore terminal I/O). */
  mysignal(SIGTTOU, SIG_IGN);
#endif

#if !defined(sun) && !defined(__NetBSD__)
  /* parse arguments
   */
  while ((ret = getopt_long(argc, argv, "", options, NULL)) != -1) {
    switch(ret) {
      case 1:
        glob_debug = 1;
	break;
      case 2:
        forceMono = 1;
	break;
      case ':':
	printf("missing argument for option '%s'\r\n", argv[optind]);
	break;
      default:
	usage(argv[0]);
	return -1;
    }
  }
#endif

#if 0
  /* no additional arguments supported so far
   */
  if (optind < argc) {
    printf("ignoring non option arguments:\r\n");
    while (optind < argc) {
      printf("\t%s\r\n", argv[optind++]);
    }
  }
#endif

  if (initialize()) {
    wserver_exit(-1, "Server initialization failed!");
  }

#ifdef __MINT__
  /* we want the server to be as quick as possible */
  (void)Prenice(Pgetpid(), -50);
#endif

  /* aaaaaand...
   */
  loop();

  /* notreached
   */
  return -99999;
}
