/*
 * server/loop.c, part of W
 * (C) 1994,95,96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * main loop and stuff belonging to it
 *
 * Phx 06/96:
 * - mouse support for NetBSD (Sun Firm_event)
 * - NetBSD /dev/kbd support! ;)
 */

#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <termios.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>

#ifdef __MINT__
# include <support.h>
# include <mintbind.h>
long _stksize = 32000;
#endif

#ifdef __NetBSD__
#include <sys/device.h>
#include <amiga/dev/itevar.h>
#include <amiga/dev/kbdmap.h>
#include <amiga/dev/vuid_event.h>
#endif

#ifdef SVGALIB
#include <vgamouse.h>
#include <vga.h>
#endif

#include "config.h"
#include "types.h"
#include "pakets.h"
#include "proto.h"
#include "window.h"
#include "rect.h"


/*
 *	some constants
 */

#define	EFILNF		-33
#define	FLAG_NONE	0x0
#define	FLAG_EXITED	0x1


/*
 * global variables
 */

WINDOW *glob_leftmousepressed = NULL;
WINDOW *glob_rightmousepressed = NULL;
long glob_savetime, glob_issaving = 0;
fd_set glob_crfd;
CLIENT *glob_clients = NULL, *glob_saveclient = 0;
long glob_bytes = 0, glob_pakets = 0;
short glob_pakettype;


/*
 * place a rectangle somewhere on the screen
 *
 * x0 and y0 should hold the previous position, if there was any, and 'UNDEF'
 * (-32768) if there wasn't any. the mouse will keep its relative position to
 * the rectangle so that if you click on an already existing window and release
 * the button without moving the mouse, the mousepointer won't move to the
 * upper left edge of the window. if you discard the move or the rectangle is
 * bigger than the screen both x0 and y0 will contain 'UNDEF' and something
 * != 0 is returned.
 */

int get_rectangle(short width, short height,
		  short *x0, short *y0,
		  char pressed, char released)
{
  WEVENT *ev = NULL;
  short mdx, mdy;
  short wx, wy, oldwx, oldwy;
  short end;
  static GCONTEXT gc = {M_INVERS, 0, NULL, NULL};

#ifdef CHILDS_INSIDE_PARENT
  if ((width > glob_screen->bm.width) || (height > glob_screen->bm.height)) {
    *x0 = UNDEF;
    *y0 = UNDEF;
    return -1;
  }
#endif

  if ((*x0 == UNDEF) || (*y0 == UNDEF)) {
    wx = glob_mouse.real.x0;
    wy = glob_mouse.real.y0;
  } else {
    wx = *x0;
    wy = *y0;
  }

#ifdef CHILDS_INSIDE_PARENTS
  if (wx + width > glob_screen->bm.width)
    wx = glob_screen->bm.width - width;
  if (wy + height > glob_screen->bm.height)
    wy = glob_screen->bm.height - height;
#endif

  mdx = glob_mouse.real.x0 - wx;
  mdy = glob_mouse.real.y0 - wy;

  mouse_hide();

  gc0 = &gc;
  clip0 = &glob_rootwindow->pos;

  (*glob_screen->box)(&glob_screen->bm, wx, wy, width, height);
  (*glob_screen->box)(&glob_screen->bm, wx+1, wy+1, width-2, height-2);

  end = 0;
  while (!end)
    if (get_eventmask(EV_MOUSE, -1, NULL))
      if ((ev = event_mouse())) {

	oldwx = wx;
	oldwy = wy;

	if (ev->type == EVENT_MMOVE) {
#ifdef CHILDS_INSIDE_PARENTS
	  wx += ev->x;
	  if (wx < 0) {
	    wx = 0;
	  } else if (wx + width > glob_screen->bm.width) {
	    wx = glob_screen->bm.width - width;
	  }
	  wy += ev->y;
	  if (wy < 0) {
	    wy = 0;
	  } else if (wy + height > glob_screen->bm.height) {
	    wy = glob_screen->bm.height - height;
	  }
#else
	  /*
	   * only make sure the mouse stays inside the screen
	   * NOTE: we must ensure (otherwise SIGSEGV):
	   *  0 <= mouse.x < glob_screen->bm.width
	   *  0 <= mouse.y < glob_screen->bm.height
	   */
	  wx += ev->x;
	  if (wx + mdx < 0) {
	    wx = -mdx;
	  } else if (wx + mdx >= glob_screen->bm.width) {
	    wx = glob_screen->bm.width - mdx - 1;
	  }
	  wy += ev->y;
	  if (wy + mdy < 0) {
	    wy = -mdy;
	  } else if (wy + mdy >= glob_screen->bm.height) {
	    wy = glob_screen->bm.height - mdy - 1;
	  }
#endif
	} else {
	  if ((ev->reserved[0] & pressed) || (ev->reserved[1] & released)) {
	    end = 1;
	  }
	}

	if ((wx != oldwx) || (wy != oldwy)) {
	  (*glob_screen->box)(&glob_screen->bm,
			      oldwx, oldwy, width, height);
	  (*glob_screen->box)(&glob_screen->bm,
			      oldwx+1, oldwy+1, width-2, height-2);
	  (*glob_screen->box)(&glob_screen->bm, wx, wy, width, height);
	  (*glob_screen->box)(&glob_screen->bm,
			      wx+1, wy+1, width-2, height-2);
	}
      }

  (*glob_screen->box)(&glob_screen->bm, wx, wy, width, height);
  (*glob_screen->box)(&glob_screen->bm, wx+1, wy+1, width-2, height-2);

  /* right button = discard window? */
  if (ev->reserved[0] & BUTTON_RIGHT) {
    *x0 = UNDEF;
    *y0 = UNDEF;
    return -1;
  }

  *x0 = wx;
  *y0 = wy;

  glob_mouse.real.x1 = (glob_mouse.real.x0 = wx + mdx) + 16;
  glob_mouse.real.y1 = (glob_mouse.real.y0 = wy + mdy) + 16;
  mouse_show();

  glob_leftmousepressed = NULL;
  glob_rightmousepressed = NULL;

  return 0;
}


/*
 * resizeRectangle(): based upon an existing rectangle, allow some of the
 * coordinates to follow the mouse pointer until the(?) mouse button is
 * released. which coordinates can be changed depends on the given mask. of
 * course some simple rules apply in any case.
 */

static void resizeRectangle (short *x0Ptr, short *y0Ptr, short *x1Ptr, short *y1Ptr, int mask)
{
  short mx, my, x0 = *x0Ptr, y0 = *y0Ptr, x1 = *x1Ptr, y1 = *y1Ptr, width, height, end;
  WEVENT *ev = NULL;
  static GCONTEXT gc = {M_INVERS, 0, NULL, NULL};

  mouse_hide ();
  mx = glob_mouse.real.x0;
  my = glob_mouse.real.y0;

  gc0 = &gc;
  clip0 = &glob_rootwindow->pos;

  width = x1 - x0 + 1;
  height= y1 - y0 + 1;
  (*glob_screen->box)(&glob_screen->bm, x0, y0, width, height);
  (*glob_screen->box)(&glob_screen->bm, x0+1, y0+1, width-2, height-2);

  end = 0;
  while (!end) {

    if (get_eventmask (EV_MOUSE, -1, NULL)) if ((ev = event_mouse ())) {

      short oldX0 = x0, oldX1 = x1, oldY0 = y0, oldY1 = y1;

      if (ev->type == EVENT_MMOVE) {

	if ((mx += ev->x) < 0)
	  mx = 0;
	if (mx > glob_screen->bm.width - 1)
	  mx = glob_screen->bm.width - 1;

	if ((my += ev->y) < 0)
	  my = 0;
	if (my > glob_screen->bm.height - 1)
	  my = glob_screen->bm.height - 1;

	/* one more place where the window frame size is hardcoded...
	 */

	if (mask & 1)
	  if ((y1 - (y0 = my)) < 7)
	    y0 = y1 - 7;
	if (mask & 2)
	  if (((y1 = my) - y0) < 7)
	    y1 = y0 + 7;
	if (mask & 4)
	  if ((x1 - (x0 = mx)) < 7)
	    x0 = x1 - 7;
	if (mask & 8)
	  if (((x1 = mx) - x0) < 7)
	    x1 = x0 + 7;

      } else if (ev->reserved[1]) {

	/* terminate this if *any* button is released
	 */

	end = 1;
      }

      if ((x0 != oldX0) || (y0 != oldY0) || (x1 != oldX1) || (y1 != oldY1)) {

	width = oldX1 - oldX0 + 1;
	height= oldY1 - oldY0 + 1;
	(*glob_screen->box)(&glob_screen->bm, oldX0, oldY0, width, height);
	(*glob_screen->box)(&glob_screen->bm, oldX0+1, oldY0+1, width-2, height-2);

	width = x1 - x0 + 1;
	height= y1 - y0 + 1;
	(*glob_screen->box)(&glob_screen->bm, x0, y0, width, height);
	(*glob_screen->box)(&glob_screen->bm, x0+1, y0+1, width-2, height-2);
      }
    }
  }

  width = x1 - x0 + 1;
  height= y1 - y0 + 1;
  (*glob_screen->box)(&glob_screen->bm, x0, y0, width, height);
  (*glob_screen->box)(&glob_screen->bm, x0+1, y0+1, width-2, height-2);

  *x0Ptr = x0;
  *y0Ptr = y0;
  *x1Ptr = x1;
  *y1Ptr = y1;

  glob_mouse.real.x1 = (glob_mouse.real.x0 = mx) + 16;
  glob_mouse.real.y1 = (glob_mouse.real.y0 = my) + 16;
  mouse_show();

  glob_leftmousepressed = NULL;
  glob_rightmousepressed = NULL;
}


/*****************************************************************************/

/*
 * some routines dealing with sockets
 */

CLIENT *createClient(void)
{
  CLIENT *ptr;

  if (!(ptr = (CLIENT *)malloc(sizeof(CLIENT)))) {
    return NULL;
  }

  memset(ptr, 0, sizeof(CLIENT));

  if (!(ptr->buf = malloc(LARGEBUF))) {
    free(ptr);
    return NULL;
  }

  return ptr;
}


void killClient(CLIENT *cptr)
{
  short i;
  CLIENT *tmp = cptr->next;

  /* shut down socket, close windows and remove client from queue */

  shutdown(cptr->sh, 2);
  close(cptr->sh);

  /* kill all windows for a particular client */
  windowKillClient(cptr);

  for (i=0; i<MAXCLIENTFONTS; i++) {
    if (cptr->font[i]) {
      client_unloadfont(cptr, i);
    }
  }

  /* also clear the fd_set bit */
  FD_CLR(cptr->sh, &glob_crfd);

  if (cptr->prev) {
    cptr->prev->next = tmp;
  } else {
    glob_clients = tmp;
  }

  if (tmp) {
    tmp->prev = cptr->prev;
  }

  if (cptr->bbm.data) {
    free (cptr->bbm.data);
    cptr->bbm.data = NULL;
  }

  if (glob_saveclient == cptr) {
    glob_issaving = 0;
    glob_saveclient = 0;
  }

  free(cptr->buf);
  free(cptr);
}


void process_paket(CLIENT *cptr, PAKET *pptr)
{
  PAKET *rpptr = NULL;
  short arg1, arg2;
  static PAKET rawpaket;
  static INITRETP irpaket;
  static SRETP srpaket;
  static S3RETP s3rpaket;
  static RSTATUSP rspaket;
  static LRETP lrpaket;
  static LFONTRETP lfontrpaket;
  char string[2];

  switch (glob_pakettype = pptr->type) {

    case PAK_NULLR:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret = htons(-1);
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_INIT:
      cptr->uid = ntohs(((INITP *)pptr)->uid);
      irpaket.len = sizeof(INITRETP);
      irpaket.type = htons(PAK_INITRET);
      irpaket.vmaj = htons(_WMAJ);
      irpaket.vmin = htons(_WMIN);
      irpaket.pl = htons(_WPL);
      irpaket.width = htons(glob_backgroundwin->bitmap.width);
      irpaket.height = htons(glob_backgroundwin->bitmap.height);
      irpaket.screenType = htons(glob_backgroundwin->bitmap.type);
      irpaket.planes = htons(glob_backgroundwin->bitmap.planes);
#if 0
      irpaket.flags = htons(WSERVER_SHM);
#else
      irpaket.flags = htons(0);
#endif
      if (glob_fontsize) {
        irpaket.fsize = htons(glob_fontsize);
      } else {
        irpaket.fsize = htons(DEF_WFONTSIZE);
      }
      if (glob_fontfamily) {
        strncpy(irpaket.fname, glob_fontfamily, MAXFAMILYNAME);
      } else {
	strncpy(irpaket.fname, DEF_WFONTFAMILY, MAXFAMILYNAME);
      }
      rpptr = (PAKET *)&irpaket;
      break;

    case PAK_CREATE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret =
	htons(client_create(cptr,
			    ntohs(((CREATEP *)pptr)->width),
			    ntohs(((CREATEP *)pptr)->height),
			    ntohs(((CREATEP *)pptr)->flags),
			    ((CREATEP *)pptr)->libPtr));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_CREATE2:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_create2(cptr, ntohs(((CREATEP *)pptr)->width),
			       ntohs(((CREATEP *)pptr)->height),
			       ntohs(((CREATEP *)pptr)->flags),
			       ntohs(((CREATEP *)pptr)->handle),
			       ((CREATEP *)pptr)->libPtr));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_OPEN:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_open(cptr, ntohs(((OPENP *)pptr)->handle),
			    ntohs(((OPENP *)pptr)->x0), ntohs(((OPENP *)pptr)->y0)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_MOVE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_move(cptr, ntohs(((MOVEP *)pptr)->handle),
			    ntohs(((MOVEP *)pptr)->x0), ntohs(((MOVEP *)pptr)->y0)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_RESIZE:
      srpaket.len = sizeof (SRETP);
      srpaket.type = htons (PAK_SRET);
      srpaket.ret
	= htons (client_resize (cptr, ntohs(((RESIZEP *)pptr)->handle),
				ntohs(((RESIZEP *)pptr)->width),
				ntohs(((RESIZEP *)pptr)->height)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_CLOSE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret = htons(client_close(cptr, ntohs(((CLOSEP *)pptr)->handle)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_DELETE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret = htons(client_delete(cptr, ntohs(((DELETEP *)pptr)->handle)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_LOADFONT:
      lfontrpaket.len = sizeof(LFONTRETP);
      lfontrpaket.type = htons(PAK_LFONTRET);
      client_loadfont(cptr, ((LOADFONTP *)pptr)->family,
			     ntohs(((LOADFONTP *)pptr)->size),
			     ntohs(((LOADFONTP *)pptr)->styles),
			     &lfontrpaket);
      rpptr = (PAKET *)&lfontrpaket;
      break;

    case PAK_UNLOADFONT:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_unloadfont(cptr, ntohs(((UNLOADFONTP *)pptr)->fonthandle)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_QWINSZ:
      s3rpaket.len = sizeof(S3RETP);
      s3rpaket.type = htons(PAK_S3RET);
      s3rpaket.ret[2]
	= htons(client_querywinsize(cptr, ntohs(((QWINSZP *)pptr)->handle),
				    ntohs(((QWINSZP *)pptr)->effective),
				    &arg1, &arg2));
      s3rpaket.ret[1] = htons(arg2);
      s3rpaket.ret[0] = htons(arg1);
      rpptr = (PAKET *)&s3rpaket;
      break;

    case PAK_TEST:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret
	= htons(client_test(cptr, htons(((TESTP *)pptr)->handle),
			    ntohs(((TESTP *)pptr)->x0), ntohs(((TESTP *)pptr)->y0)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_QMPOS:
      s3rpaket.len = sizeof(S3RETP);
      s3rpaket.type = htons(PAK_S3RET);
      s3rpaket.ret[2]
	= htons(client_querymousepos(cptr, ntohs(((QMPOSP *)pptr)->handle),
				     &arg1, &arg2));
      s3rpaket.ret[1] = htons(arg2);
      s3rpaket.ret[0] = htons(arg1);
      rpptr = (PAKET *)&s3rpaket;
      break;

    case PAK_QWPOS:
      s3rpaket.len = sizeof(S3RETP);
      s3rpaket.type = htons(PAK_S3RET);
      s3rpaket.ret[2]
	= htons(client_querywindowpos(cptr, ntohs(((QWPOSP *)pptr)->handle),
				      ntohs(((QWPOSP *)pptr)->effective),
				      &arg1, &arg2));
      s3rpaket.ret[1] = htons(arg2);
      s3rpaket.ret[0] = htons(arg1);
      rpptr = (PAKET *)&s3rpaket;
      break;

    case PAK_QSTATUS:
      rspaket.len = sizeof(RSTATUSP);
      rspaket.type = htons(PAK_RSTATUS);
      rspaket.ret
	= htons(client_status(cptr, glob_clients,
			      ntohs(((QSTATUSP *)pptr)->index), &rspaket.status));
      rpptr = (PAKET *)&rspaket;
      break;

    case PAK_PUTBLKREQ:
      lrpaket.len = sizeof(LRETP);
      lrpaket.type = htons(PAK_LRET);
      lrpaket.ret
	= htonl(client_putblockreq (cptr, ntohs(((PUTBLKREQP *)pptr)->width),
				    ntohs(((PUTBLKREQP *)pptr)->height),
				    ntohs(((PUTBLKREQP *)pptr)->handle),
				    ntohs(((PUTBLKREQP *)pptr)->x1),
				    ntohs(((PUTBLKREQP *)pptr)->y1),
				    ((PUTBLKREQP *)pptr)->shmKey));
      rpptr = (PAKET *)&lrpaket;
      break;

    case PAK_SSAVER:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      if ((glob_saveclient && (glob_saveclient != cptr)) ||
	  (ntohs(((SSAVERP *)pptr)->seconds) < 10)) {
	srpaket.ret = htons(-1);
      } else {
	glob_saveclient = cptr;
	glob_savetime = ntohs(((SSAVERP *)pptr)->seconds);
	glob_issaving = 0;
	srpaket.ret = htons(0);   /* boah... */
      }
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_GETBLKREQ:
      lrpaket.len = sizeof(LRETP);
      lrpaket.type = htons(PAK_LRET);
      lrpaket.ret
       = htonl(client_getblockreq(cptr, ntohs(((GETBLKREQP *)pptr)->handle),
				  ntohs(((GETBLKREQP *)pptr)->x0),
				  ntohs(((GETBLKREQP *)pptr)->y0),
				  ntohs(((GETBLKREQP *)pptr)->width),
				  ntohs(((GETBLKREQP *)pptr)->height),
				  ((GETBLKREQP *)pptr)->shmKey));
      rpptr = (PAKET *)&lrpaket;
      break;

    case PAK_GETBLKDATA:
      rawpaket.len
       = client_getblockdata(cptr, rawpaket.data, sizeof(rawpaket.data));
      rawpaket.len += sizeof(rawpaket) - sizeof(rawpaket.data);
      rawpaket.type = htons(PAK_RAWDATA);
      rpptr = (PAKET *)&rawpaket;
      break;

    case PAK_ALLOCCOL:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret =
	htons(clientAllocColor(cptr, ntohs(((ALLOCCOLP *)pptr)->handle),
			       ntohs(((ALLOCCOLP *)pptr)->red),
			       ntohs(((ALLOCCOLP *)pptr)->green),
			       ntohs(((ALLOCCOLP *)pptr)->blue)));
      rpptr = (PAKET *)&srpaket;
      break;

    case PAK_ALLOCCOLRANGE:
      srpaket.len = sizeof(SRETP);
      srpaket.type = htons(PAK_SRET);
      srpaket.ret =
	htons(clientAllocColorRange(cptr,
				    ntohs(((ALLOCCOLRANGEP *)pptr)->handle),
				    ntohs(((ALLOCCOLRANGEP *)pptr)->colors),
				    ntohs(((ALLOCCOLRANGEP *)pptr)->start)));
      rpptr = (PAKET *)&rawpaket;
      break;

/* pakets that don't need a return code */

    case PAK_NULL:
      break;

    case PAK_EXIT:
      cptr->flags |= FLAG_EXITED;
      break;

    case PAK_STEXTSTYLE:
      client_settextstyle(cptr, ntohs(((STEXTSTYLEP *)pptr)->handle),
			  ntohs(((STEXTSTYLEP *)pptr)->flags));
      break;

    case PAK_STITLE:
      client_settitle(cptr, ntohs(((STITLEP *)pptr)->handle),
		      ((STITLEP *)pptr)->title);
      break;

    case PAK_PLOT:
    case PAK_DPLOT:
      client_plot(cptr, ntohs(((PLOTP *)pptr)->handle),
		  ntohs(((PLOTP *)pptr)->x0), ntohs(((PLOTP *)pptr)->y0));
      break;

    case PAK_LINE:
    case PAK_DLINE:
      client_line(cptr, ntohs(((LINEP *)pptr)->handle),
		  ntohs(((LINEP *)pptr)->x0), ntohs(((LINEP *)pptr)->y0),
		  ntohs(((LINEP *)pptr)->xe), ntohs(((LINEP *)pptr)->ye));
      break;

    case PAK_HLINE:
    case PAK_DHLINE:
      client_hline(cptr, ntohs(((HVLINEP *)pptr)->handle),
		   ntohs(((HVLINEP *)pptr)->x0), ntohs(((HVLINEP *)pptr)->y0),
		   ntohs(((HVLINEP *)pptr)->e));
      break;

    case PAK_VLINE:
    case PAK_DVLINE:
      client_vline(cptr, ntohs(((HVLINEP *)pptr)->handle),
		   ntohs(((HVLINEP *)pptr)->x0), ntohs(((HVLINEP *)pptr)->y0),
		   ntohs(((HVLINEP *)pptr)->e));
      break;

    case PAK_BOX:
    case PAK_PBOX:
    case PAK_DBOX:
    case PAK_DPBOX:
      client_box(cptr, ntohs(((BOXP *)pptr)->handle),
		 ntohs(((BOXP *)pptr)->x0), ntohs(((BOXP *)pptr)->y0),
		 ntohs(((BOXP *)pptr)->width), ntohs(((BOXP *)pptr)->height));
      break;

    case PAK_BITBLK:
      client_bitblk(cptr, ntohs(((BITBLKP *)pptr)->handle),
		    ntohs(((BITBLKP *)pptr)->x0), ntohs(((BITBLKP *)pptr)->y0),
		    ntohs(((BITBLKP *)pptr)->width),
		    ntohs(((BITBLKP *)pptr)->height),
		    ntohs(((BITBLKP *)pptr)->x1), ntohs(((BITBLKP *)pptr)->y1));
      break;

    case PAK_BITBLK2:
      client_bitblk2(cptr, ntohs(((BITBLKP *)pptr)->handle),
		     ntohs(((BITBLKP *)pptr)->x0),
		     ntohs(((BITBLKP *)pptr)->y0),
		     ntohs(((BITBLKP *)pptr)->width),
		     ntohs(((BITBLKP *)pptr)->height),
		     ntohs(((BITBLKP *)pptr)->dhandle),
		     ntohs(((BITBLKP *)pptr)->x1),
		     ntohs(((BITBLKP *)pptr)->y1));
      break;

    case PAK_VSCROLL:
      client_vscroll(cptr, ntohs(((VSCROLLP *)pptr)->handle),
		     ntohs(((VSCROLLP *)pptr)->x0),
		     ntohs(((VSCROLLP *)pptr)->y0),
		     ntohs(((VSCROLLP *)pptr)->width),
		     ntohs(((VSCROLLP *)pptr)->height),
		     ntohs(((VSCROLLP *)pptr)->y1));
      break;

    case PAK_PRINTC:
      string[0] = ntohs(((PRINTCP *)pptr)->c) & 0xff;
      string[1] = '\0';
      client_prints(cptr, ntohs(((PRINTCP *)pptr)->handle),
		    ntohs(((PRINTCP *)pptr)->x0),
		    ntohs(((PRINTCP *)pptr)->y0),
		    string);
      break;

    case PAK_PRINTS:
      client_prints(cptr, ntohs(((PRINTSP *)pptr)->handle),
		    ntohs(((PRINTSP *)pptr)->x0),
		    ntohs(((PRINTSP *)pptr)->y0),
		    ((PRINTSP *)pptr)->s);
      break;

    case PAK_CIRCLE:
    case PAK_PCIRCLE:
    case PAK_DCIRCLE:
    case PAK_DPCIRCLE:
      client_circle(cptr, ntohs(((CIRCLEP *)pptr)->handle),
		    ntohs(((CIRCLEP *)pptr)->x0),
		    ntohs(((CIRCLEP *)pptr)->y0),
		    ntohs(((CIRCLEP *)pptr)->r));
      break;

    case PAK_ELLIPSE:
    case PAK_PELLIPSE:
    case PAK_DELLIPSE:
    case PAK_DPELLIPSE:
      client_ellipse(cptr, ntohs(((ELLIPSEP *)pptr)->handle),
		    ntohs(((ELLIPSEP *)pptr)->x0),
		    ntohs(((ELLIPSEP *)pptr)->y0),
		    ntohs(((ELLIPSEP *)pptr)->rx),
		    ntohs(((ELLIPSEP *)pptr)->ry));
      break;

    case PAK_POLY:
    case PAK_PPOLY:
    case PAK_DPOLY:
    case PAK_DPPOLY:
      client_poly(cptr, ntohs(((POLYP *)pptr)->handle),
		  ntohs(((POLYP *)pptr)->numpoints), ((POLYP *)pptr)->points);
      break;

    case PAK_BEZIER:
    case PAK_DBEZIER:
      client_bezier(cptr, ntohs(((BEZIERP *)pptr)->handle),
		  ((BEZIERP *)pptr)->points);
      break;

    case PAK_RAWDATA:
      arg1 = pptr->len - (sizeof(*pptr) - sizeof(pptr->data));
      client_putblockdata(cptr, pptr->data, arg1);
      break;

    case PAK_BEEP:
      client_beep(cptr);
      break;

    case PAK_SMODE:
      client_setmode(cptr, ntohs(((SMODEP *)pptr)->handle),
		     ntohs(((SMODEP *)pptr)->mode));
      break;

    case PAK_SFONT:
      client_setfont(cptr, ntohs(((SFONTP *)pptr)->handle),
		     ntohs(((SFONTP *)pptr)->fonthandle));
      break;

    case PAK_SPATTERN:
      client_setpattern(cptr, ntohs(((SPATTERNP *)pptr)->handle),
			ntohs(((SPATTERNP *)pptr)->pattern));
      break;

    case PAK_FREECOL:
      clientFreeColor(cptr, ntohs(((FREECOLP *)pptr)->handle),
		      ntohs(((FREECOLP *)pptr)->color));
      break;

    case PAK_CHANGECOL:
      clientChangeColor(cptr, ntohs(((CHANGECOLP *)pptr)->handle),
			ntohs(((CHANGECOLP *)pptr)->color),
			ntohs(((CHANGECOLP *)pptr)->red),
			ntohs(((CHANGECOLP *)pptr)->green),
			ntohs(((CHANGECOLP *)pptr)->blue));
      break;

    case PAK_SETFGCOL:
    case PAK_SETBGCOL:
      clientSetColor(cptr, ntohs(((SETFGCOLP *)pptr)->handle),
		     ntohs(((SETFGCOLP *)pptr)->color));
      break;

    default:
      fprintf (stderr,
	       "wserver: got paket 0x%04x from client 0x%p, killing client\r\n",
	       pptr->type, cptr);
      cptr->flags |= FLAG_EXITED;
      cptr->inbuf = 0;
  }

  if (rpptr) {
    int len = rpptr->len;
    /* we don't care if the connection is broken */
    rpptr->len = htons(rpptr->len);
    write(cptr->sh, rpptr, len);
  }
}


/*****************************************************************************/

/*
 * some routines dealing with events, the first one just reports a bitmask of
 * what events are ready to be processed and the others actually process them.
 */

#if defined(__MINT__)
#define	MBUFSIZE 381
#elif defined(__NetBSD__)
#define MBUFSIZE 3*sizeof(Firm_event)
#else
#define	MBUFSIZE 3
#endif

short mevents = 0;
char mbuf[MBUFSIZE];
#ifdef __NetBSD__
Firm_event *mptr;
#else
char *mptr;
#endif

long get_eventmask(long wanted, long timeout, fd_set *retrfd)
{
  fd_set rfd;
  struct timeval tv, *tvp;
#ifndef SVGALIB
  int rfds; 
#else
  int was_mouseevent;
#endif
  long events = 0;

  do {

    /* set up file descriptors */
    if (wanted & EV_CLIENT) {
      rfd = glob_crfd;
    } else {
      FD_ZERO(&rfd);
    }
    if (wanted & EV_UCONN) {
      FD_SET(glob_unixh, &rfd);
    }
#ifndef AF_UNIX_ONLY
    if ((wanted & EV_ICONN) && (glob_ineth >= 0)) {
      FD_SET(glob_ineth, &rfd);
    }
#endif
    if (wanted & EV_KEYS) {
      FD_SET(glob_kbd, &rfd);
    }

#ifndef SVGALIB
    if (wanted & EV_MOUSE) {
      if (mevents) {
	events |= EV_MOUSE;
	timeout = 0;
      } else {
	FD_SET(glob_mouse.fh, &rfd);
      }
    }
#endif

    /* set up timeout */
    if (timeout < 0) {
      tvp = NULL;
    } else {
      tv.tv_sec = timeout / 1000;
      tv.tv_usec = timeout - 1000 * tv.tv_sec;
      tvp = &tv;
    }

    /* select the descriptors */
#ifndef SVGALIB
    if ((rfds = select(FD_SETSIZE, &rfd, NULL, NULL, tvp)) < 0) {
      /* experience has shown that it is not save
       * to test the fd_sets after an error
       */
      return 0;
    }
#else
    if (wanted & EV_MOUSE) {
      if ((was_mouseevent = 
	   vga_waitevent(VGA_MOUSEEVENT, &rfd, NULL, NULL, tvp)) < 0) {
	return 0;
      }
    }
    else {
      if ((was_mouseevent = 
	   vga_waitevent(0, &rfd, NULL, NULL, tvp)) < 0) {
	return 0;
      }
    }
#endif

    /* what have we got? */
#ifndef SVGALIB
    if (FD_ISSET(glob_mouse.fh, &rfd)) {
      long ret;
      if ((ret = read(glob_mouse.fh, mbuf, MBUFSIZE)) > 0) {
	events |= EV_MOUSE;
	rfds--;
#ifdef __NetBSD__
	mptr = (Firm_event *)mbuf;  /* Phx */
	mevents = ret / sizeof(Firm_event);
#else
	mptr = mbuf;
	mevents = ret / 3;
#endif
      }
    }
#else
    if (was_mouseevent & VGA_MOUSEEVENT) {
      events |= EV_MOUSE;
    }
#endif

    if (FD_ISSET(glob_kbd, &rfd)) {
      events |= EV_KEYS;
#ifndef SVGALIB
      rfds--;
#endif
    }

    if (FD_ISSET(glob_unixh, &rfd)) {
      events |= EV_UCONN;
#ifndef SVGALIB
      rfds--;
#endif
    }

#ifndef AF_UNIX_ONLY
    if (glob_ineth >= 0) if (FD_ISSET(glob_ineth, &rfd)) {
      events |= EV_ICONN;
#ifndef SVGALIB
      rfds--;
#endif
    }
#endif

#ifndef SVGALIB
    if (rfds)
#endif
      {
	/* what remains? */
	events |= EV_CLIENT;
	if (retrfd) {
	  *retrfd = rfd;
	}
      }
    
  } while (!timeout && !events);

  return events;
}


void event_connect(int sokh)
{
  int rlen, shtmp;
  struct sockaddr_in raddr;
  CLIENT *cptr;

  /* it seems to be save to expect these to stay empty for AF_UNIX accepts()
   */
  raddr.sin_addr.s_addr = 0;

  rlen = sizeof(struct sockaddr);
  if ((shtmp = accept(sokh, (struct sockaddr *)&raddr, &rlen)) < 0) {
    /* oops, I ran out of file handles? */
    return;
  }

  if ((cptr = createClient())) {
    /* ok, we can do it */
    cptr->sh = shtmp;
    cptr->raddr = raddr.sin_addr.s_addr;
    FD_SET(shtmp, &glob_crfd);
    cptr->prev = NULL;
    cptr->next = glob_clients;
    if (glob_clients) {
      glob_clients->prev = cptr;
    }
    glob_clients = cptr;
  } else {
    /* shut it down immediately */
    shutdown(shtmp, 2);
    close(shtmp);
  }
}


void event_client(fd_set *rfd)
{
  CLIENT *next, *cptr = glob_clients;
  int len;
  PAKET *pptr;
  long ret, offset;

  while (cptr) {
    /* keep a local copy of next client, killclient might trash cptr */
    next = cptr->next;

    if (FD_ISSET(cptr->sh, rfd)) {

      if ((ret = read(cptr->sh, cptr->buf + cptr->inbuf,
		      LARGEBUF - cptr->inbuf)) < 1) {
	cptr->flags |= FLAG_EXITED;
      } else {
	offset = 0;

	/* try to extract pakets */
	glob_bytes += ret;
	cptr->inbuf += ret;
	pptr = (PAKET *)cptr->buf;

	while ((cptr->inbuf >= 4) && (cptr->inbuf >= (len = ntohs(pptr->len)))) {
	  glob_pakets++;
	  cptr->pakets++;
	  cptr->bytes += len;
	  cptr->inbuf -= len;
	  pptr->len = len;
	  pptr->type = ntohs(pptr->type);
	  process_paket(cptr, pptr);
	  offset += len;
	  pptr = (PAKET *)(((char *)pptr) + len);
	}

	/* move remaining data down to beginning of buffer */
	if (offset && cptr->inbuf) {
	  bcopy(cptr->buf + offset, cptr->buf, cptr->inbuf);
	}
      }
    }

    if (cptr->flags & FLAG_EXITED)
      killClient (cptr);

    cptr = next;
  }
}


/*
 * normally this routine assumes that a previous check has shown that mouse
 * data is available, but perhaps it's better to be a bit carefull.
 *
 * other than that, we still lack a way to simultainously report mouse
 * movement *and* button presses which is why the stuff here tries to treat
 * them seperately. perhaps I should really implement a higher level buffer
 * of mouse events...
 */

#ifdef __NetBSD__
/* Phx 06/96
 * This is the mouse event handler for NetBSD, which uses an emulation
 * of Sun's Firm_events.
 */

WEVENT *event_mouse(void)
{
  short val, dx=0, dy=0;
  char button;
  static char lastbutton = BUTTON_ALL;
  static WEVENT someevent;

  button = lastbutton;
  while (mevents) {
    val = (short)mptr->value;
    switch (mptr->id) {
      case LOC_X_DELTA: 
        dx += val;
	break;
      case LOC_Y_DELTA:
	dy += val;
	break;
      case MS_LEFT:
	if (val)
	  button &= ~BUTTON_LEFT;
	else
	  button |= BUTTON_LEFT;
	break;
      case MS_MIDDLE:
	if (val)
	  button &= ~BUTTON_MID;
	else
	  button |= BUTTON_MID;
	break;
      case MS_RIGHT:
	if (val)
	  button &= ~BUTTON_RIGHT;
	else
	  button |= BUTTON_RIGHT;
	break;
    }
    --mevents;
    ++mptr;
  }

  /* unfortunately the only solution in the current environment: */
  /* if a button event was received, the mouse movements are discarded, */
  /* so it is best to allow only a single event in the buffer... */
  if (button != lastbutton) {
    someevent.type = EVENT_MPRESS; /* or EVENT_MRELEASE, we don't care here. */
    someevent.x = glob_mouse.rx;
    someevent.y = glob_mouse.ry;
    someevent.reserved[0] = lastbutton & ~button;
    someevent.reserved[1] = ~lastbutton & button;
    lastbutton = button;

    return &someevent;
  }

  if (dx || dy) {
    someevent.type = EVENT_MMOVE;
    someevent.x = dx;
    someevent.y = dy;

    return &someevent;
  }

  return NULL;
}

#else /* !NetBSD */

WEVENT *event_mouse(void)
{
#ifdef SVGALIB
  static WEVENT someevent;
  static int last_x     = 0; 
  static int last_y     = 0;
  static int lastbutton = 7;
  int  x, y, button, dx, dy;

  x  = mouse_getx();
  y  = mouse_gety();
  dx = x - last_x;
  dy = y - last_y;
  
  if(dx || dy) {
    /* mouse position changed */
    someevent.type = EVENT_MMOVE;
    someevent.x = dx;
    someevent.y = dy;
    last_x = x;
    last_y = y;
    return &someevent;
  }

  button = 0;
  x = mouse_getbutton();
  if (x & MOUSE_LEFTBUTTON)
    button |= BUTTON_LEFT;
  if (x & MOUSE_RIGHTBUTTON)
    button |= BUTTON_RIGHT;
  if (x & MOUSE_MIDDLEBUTTON)
    button |= BUTTON_MID;
 
  if(lastbutton != button) {
    /* button status changed */
    
    someevent.type = EVENT_MPRESS; /* or EVENT_MRELEASE, we don't care here. */
    someevent.x = glob_mouse.real.x0;
    someevent.y = glob_mouse.real.y0;
    someevent.reserved[0] = ~lastbutton & button;
    someevent.reserved[1] = lastbutton & ~button;
    lastbutton = button;

    return &someevent;      
  }
  
  /* nothing changed??? */

  return NULL;

#else /* !SVGALIB */

  short dx, dy, button;
  static char lastbutton = 7;
  static WEVENT someevent;

  dx = 0;
  dy = 0;

  /* first read movement only data */

  while (mevents && ((*mptr++ & 7) == lastbutton)) {
    dx += *mptr++;
    dy -= *mptr++;
    mevents--;
  }

  if (mevents)
    mptr--;   /* adjust pointer */

  if (dx || dy) {
    /* the mouse moved */
    someevent.type = EVENT_MMOVE;
    someevent.x = dx;
    someevent.y = dy;

    return &someevent;
  }

  if (mevents && (*mptr != lastbutton)) {
    /* some button event occured */
    button = *mptr++ & 7;
    mptr++; mptr++;		/* ignore movement here */
    mevents--;

    someevent.type = EVENT_MPRESS; /* or EVENT_MRELEASE, we don't care here. */
    someevent.x = glob_mouse.real.x0;
    someevent.y = glob_mouse.real.y0;
    someevent.reserved[0] = lastbutton & ~button;
    someevent.reserved[1] = ~lastbutton & button;
    lastbutton = button;

    return &someevent;
  }

  /* obviously nothing happened */

  return NULL;

#endif /* !SVGALIB */
}

#endif /* !NetBSD */


void event_mousemove(WEVENT *ev)
{
  short	oldx, oldy;

  oldx = glob_mouse.real.x0;
  oldy = glob_mouse.real.y0;

  glob_mouse.real.x0 += ev->x;
  glob_mouse.real.y0 += ev->y;

  if (glob_mouse.real.x0 < 0)
    glob_mouse.real.x0 = 0;
  if (glob_mouse.real.x0 >= glob_screen->bm.width)
    glob_mouse.real.x0 = glob_screen->bm.width - 1;
  if (glob_mouse.real.y0 < 0)
    glob_mouse.real.y0 = 0;
  if (glob_mouse.real.y0 >= glob_screen->bm.height)
    glob_mouse.real.y0 = glob_screen->bm.height - 1;

#if 0
  /* nobody actually uses the w,h,x1,y1 fields of the 'real' values...
   */
  glob_mouse.real.x1 += glob_mouse.real.x0 + 15;
  glob_mouse.real.y1 += glob_mouse.real.y0 + 15;
#endif

  /* watch out for active windows. this may also hide the mouse.
   */
  w_changeActiveWindow ();

  /* if we don't redraw the mouse immediately it really looks stupid
   */
  if ((glob_mouse.real.x0 != oldx) || (glob_mouse.real.y0 != oldy)) {
    mouse_hide ();
    mouse_show ();
  }
}


void event_mousebutton (WEVENT *ev)
{
  static char *perrorMsg = "wserver: event_mousebutton(): write()";
  short	x, y, wx, wy;
  EVENTP paket;
  WINDOW *win;

  paket.len = htons (sizeof(EVENTP));
  paket.type = htons (PAK_EVENT);

  /* first deal with release events seperately, because they might go to
   * different windows...
   */

  if (glob_leftmousepressed)
    if ((ev->reserved[1] & BUTTON_LEFT) &&
	(glob_leftmousepressed != glob_backgroundwin)) {

      paket.event.type = htons(EVENT_MRELEASE);
      paket.event.win = glob_leftmousepressed->libPtr;
      paket.event.key = htonl(BUTTON_LEFT);
      x = ev->x - glob_leftmousepressed->work.x0;
      y = ev->y - glob_leftmousepressed->work.y0;
      if ((x < 0) || (x >= glob_leftmousepressed->work.w) ||
	  (y < 0) || (y >= glob_leftmousepressed->work.h)) {
	x = -1;
	y = -1;
      }
      paket.event.x = htons(x);
      paket.event.y = htons(y);
      if (write(glob_leftmousepressed->cptr->sh, &paket,
		sizeof(EVENTP)) != sizeof(EVENTP))
	perror (perrorMsg);
      glob_leftmousepressed = NULL;
    }

  if (glob_rightmousepressed)
    if ((ev->reserved[1] & BUTTON_RIGHT) &&
	(glob_rightmousepressed != glob_backgroundwin)) {

      paket.event.type = htons(EVENT_MRELEASE);
      paket.event.win = glob_rightmousepressed->libPtr;
      paket.event.key = htonl(BUTTON_RIGHT);
      x = ev->x - glob_rightmousepressed->work.x0;
      y = ev->y - glob_rightmousepressed->work.y0;
      if ((x < 0) || (x >= glob_rightmousepressed->work.w) ||
	  (y < 0) || (y >= glob_rightmousepressed->work.h)) {
	x = -1;
	y = -1;
      }
      paket.event.x = htons(x);
      paket.event.y = htons(y);
      if (write(glob_rightmousepressed->cptr->sh, &paket,
		sizeof(EVENTP)) != sizeof(EVENTP))
	perror (perrorMsg);
      glob_rightmousepressed = NULL;
    }

  /* can we already break here?
   */
  if (!ev->reserved[0])
    return;

  /* there was a `press' event, deal with it...
   */
  x = ev->x;
  y = ev->y;
  if ((win = window_find (x, y, 1)) == glob_backgroundwin) {
    /* action on background
     */
    if (ev->reserved[0] & BUTTON_LEFT)
      menu_domenu();
    return;
  }

  /* action on window
   */
  ev->win = win->libPtr;

  if (rect_cont_point (&win->work, x, y)) {

    /* action in work area of window
     */
    if (ev->reserved[0] & BUTTON_LEFT)
      glob_leftmousepressed = win;
    if (ev->reserved[0] & BUTTON_RIGHT)
      glob_rightmousepressed = win;

    if (win->flags & EV_MOUSE) {

      /* we won't deal with the `middle' button emulation so far...
       */
      ev->key = htonl (ev->reserved[0] & (BUTTON_LEFT | BUTTON_RIGHT));
      ev->x = htons (x - win->work.x0);
      ev->y = htons (y - win->work.y0);
      paket.event = *ev;
      paket.event.type = htons (EVENT_MPRESS);

      if (write(win->cptr->sh, &paket, sizeof(EVENTP)) != sizeof(EVENTP))
	perror (perrorMsg);
      
    } else {

      /* just cancel the old state
       */
      if (ev->reserved[0] & BUTTON_LEFT)
	glob_leftmousepressed = NULL;
      if (ev->reserved[0] & BUTTON_RIGHT)
	glob_rightmousepressed = NULL;
    }

    return;
  }

  wx = x - win->pos.x0;
  wy = y - win->pos.y0;

  if (rect_cont_point (&win->area[AREA_CLOSE], wx, wy)) {

    /* close gadget clicked
     */
    ev->type = htons(EVENT_GADGET);
    ev->key = htonl(GADGET_CLOSE);
    paket.event = *ev;

    if (write(win->cptr->sh, &paket, sizeof(EVENTP)) != sizeof(EVENTP))
      perror (perrorMsg);

    if (ev->reserved[0] & BUTTON_LEFT)
      glob_leftmousepressed = NULL;
    if (ev->reserved[0] & BUTTON_RIGHT)
      glob_rightmousepressed = NULL;

    return;
  }

  if (rect_cont_point (&win->area[AREA_ICON], wx, wy)) {

    /* icon gadget clicked
     */
    ev->type = htons(EVENT_GADGET);
    ev->key = htonl(GADGET_ICON);
    paket.event = *ev;

    if (write(win->cptr->sh, &paket, sizeof(EVENTP)) != sizeof(EVENTP))
      perror (perrorMsg);

    if (ev->reserved[0] & BUTTON_LEFT)
      glob_leftmousepressed = NULL;
    if (ev->reserved[0] & BUTTON_RIGHT)
      glob_rightmousepressed = NULL;

    return;
  }

/*****************************************************************************/

  /* action on frame of window
   *
   * Left button resizes and moves windows.  If window isn't/can't be
   * resized, it will be moved.  Right button tops/bottoms window.
   */

  /* for moving */
  wx = BUTTON_RIGHT;		/* check pressed */
  wy = BUTTON_LEFT;		/* check release */

  if ((win->flags & W_RESIZE) &&
      (ev->reserved[0] & BUTTON_LEFT)) {

    int idx = 0;

    if (y - win->pos.y0 < 4)
      idx |= 1;   /* upper edge */

    if (win->pos.y1 - y < 4)
      idx |= 2;   /* lower edge */

    if (x - win->pos.x0 < 4)
      idx |= 4;   /* left edge */

    if (win->pos.x1 - x < 4)
      idx |= 8;   /* right edge */

    if (idx) {

      short x0 = win->pos.x0, y0 = win->pos.y0, xe = win->pos.x1, ye = win->pos.y1;
      short oldX0 = x0, oldY0 = y0, oldXe = xe, oldYe = ye;

      resizeRectangle (&x0, &y0, &xe, &ye, idx);

      if ((oldX0 != x0) || (oldY0 != y0) || (oldXe != xe) || (oldYe != ye)) {

	paket.len = htons (sizeof (EVENTP));
	paket.type = htons (PAK_EVENT);
	paket.event.type = htons (EVENT_RESIZE);
	paket.event.win = win->libPtr;
	paket.event.x = htons (x0);
	paket.event.y = htons (y0);
	/* size for work area.
	 * beware: y border depends on whether window has titlebar or not.
	 */
	paket.event.w = htons (xe-x0-(win->pos.w-win->work.w)+1);
 	paket.event.h = htons (ye-y0-(win->pos.h-win->work.h)+1);

	if (write (win->cptr->sh, &paket,
		   sizeof (EVENTP)) != sizeof (EVENTP))
	  perror (perrorMsg);

	/* resize made */
	return;
      }
      /* else try to move window until left button is pressed again */
      wx = BUTTON_LEFT;
      wy = BUTTON_RIGHT;
    }
  }

  if ((win->flags & W_MOVE) && !(ev->reserved[0] & wx)) {
    x = win->pos.x0;
    y = win->pos.y0;
    if (!get_rectangle(win->pos.w, win->pos.h, &x, &y, wx, wy))
      client_move(NULL, win->id, x, y);
  }

  if (!(win->flags & W_TOP) && (ev->reserved[0] & BUTTON_RIGHT)) {
    /* top or down the window */
    w_topDown(win);
  }
}


/*
 * event_key(): send up to MAXKEYS of keyboard input to the active window
 *
 * Phx 06/96 - support for NetBSD/Amiga. /dev/kbd supplies us with
 * Firm_events containing (Amiga specific) raw key codes.
 */

#define MAXKEYS 8

#ifdef __NetBSD__
extern struct kbdmap netbsd_kmap;  /* the active keymap (us, german, etc.) */

void event_key(void)
{
  static unsigned char key_mod=0;
  EVENTP paket[MAXKEYS];
  uchar	c[MAXKEYS];
  Firm_event fe;
  int nchr=0, i;
  struct key key;
  unsigned char mask, code;
  char *str;

  if (read(glob_kbd,&fe,sizeof(Firm_event)) > 0) {
    code = (unsigned char)fe.id;

    /* check for qualifier */
    if (code >= KBD_LEFT_SHIFT) {
      mask = 1 << (code - KBD_LEFT_SHIFT);
      if (!fe.value)
	key_mod &= ~mask;  /* qualifier released */
      else
	key_mod |= mask;  /* qualifier pressed */
    }

    else if (fe.value) {
      /* a new key was pressed - translate it */
      if (key_mod & KBD_MOD_SHIFT) {
	if (key_mod & KBD_MOD_ALT)
	  key = netbsd_kmap.alt_shift_keys[code];
	else 
	  key = netbsd_kmap.shift_keys[code];
      } else if (key_mod & KBD_MOD_ALT)
	key = netbsd_kmap.alt_keys[code];
      else {
	key = netbsd_kmap.keys[code];
	if ((key_mod & KBD_MOD_CAPS) && (key.mode & KBD_MODE_CAPS))
	  key = netbsd_kmap.shift_keys[code];
      }
      code = key.code;

      /* 
       * For simplicity KBD_MODE_KPAD is ignored, so the keypad-keys are
       * treated as normals keys, instead of generating strings.
       */

      if (key.mode & KBD_MODE_STRING) {
	/* key generates a string */
	str = netbsd_kmap.strings + code;
	for (i=(int)*str++; i>0; i--)
	  c[nchr++] = *str++;
      }
      /* Dead keys and accents are (currently) ignored */
    
      else {
	if (key_mod & KBD_MOD_CTRL)
	  code &= 0x1f;
	if (key_mod & KBD_MOD_META)
	  code |= 0x80;
	
	/* return single ASCII character */
	c[nchr++] = code;
      }
    }
  }

  if (nchr) {
    if ((glob_activewindow == glob_rootwindow) ||
	!(glob_activewindow->flags & EV_KEYS)) {
      /* want no keys */
      return;
    }

    for (i=0; i<nchr; i++) {
      paket[i].len = sizeof(EVENTP);
      paket[i].type = PAK_EVENT;
      paket[i].event.type = EVENT_KEY;
      paket[i].event.key = c[i];
      paket[i].event.win = glob_activewindow->libPtr;
    }
    
    write(glob_activewindow->cptr->sh, paket, nchr * sizeof(EVENTP));
  }
}

#else /* !NetBSD */

void event_key(void)
{
  static EVENTP paket[MAXKEYS];
  uchar	c[MAXKEYS];
  int max, i;

  if (ioctl(glob_kbd, FIONREAD, &max))
    /* how could this happen?
     */
    return;

  if (max > MAXKEYS)
    max = MAXKEYS;

  if ((max = read(glob_kbd, c, max)) < 1)
    /* how could this happen?
     */
    return;

  if ((glob_activewindow == glob_backgroundwin) ||
      !(glob_activewindow->flags & EV_KEYS))
    /* want no keys
     */
    return;

  for (i=0; i<max; i++) {
    paket[i].len = htons(sizeof(EVENTP));
    paket[i].type = htons(PAK_EVENT);
    paket[i].event.type = htons(EVENT_KEY);
    paket[i].event.key = htonl(c[i]);
    paket[i].event.win = glob_activewindow->libPtr;   /* no change */
  }

#if 0
  for (i=0; i<max; i++) {
    int ret, size = sizeof(EVENTP);
    ret = write(glob_activewindow->cptr->sh, &paket[i], size);
    if (size != ret)
      perror ("event_key(): write()");
  }
#else
  write(glob_activewindow->cptr->sh, paket, max * sizeof(EVENTP));
#endif
}

#endif /* !NetBSD */


/*****************************************************************************/

/*
 * guess what... :-)
 */

void loop(void)
{
  long events;
  fd_set rfd;
  WEVENT *ev;
  time_t now, last = time(0);

  while (42) {

    events = get_eventmask(EV_KEYS | EV_MOUSE | EV_UCONN | EV_ICONN | EV_CLIENT, 1000, &rfd);
    now = time(0);

/* check for new connection requests */
    if (events & EV_UCONN) {
      event_connect(glob_unixh);
    }
#ifndef AF_UNIX_ONLY
    if (events & EV_ICONN) {
      event_connect(glob_ineth);
    }
#endif

/* check for client pakets */
    if (events & EV_CLIENT) {
      event_client(&rfd);
    }

/* check for mouse events */
    if (events & EV_MOUSE) {
      last = now;
      if ((ev = event_mouse())) switch (ev->type) {
        case EVENT_MMOVE:
	  event_mousemove(ev);
	  break;
	case EVENT_MPRESS:
	  event_mousebutton(ev);
	  break;
      }
    }

/* update everything that may have been changed by the client or the mouse */
    windowRedrawAllIfDirty();

/* check for keys */
    if (events & EV_KEYS) {
      last = now;
      event_key();
    }

/* how about the saver? */
    if (glob_saveclient && !glob_issaving && (now - last > glob_savetime)) {
      EVENTP	paket;
      glob_issaving = 1;
      paket.len = htons(sizeof(EVENTP));
      paket.type = htons(PAK_EVENT);
      paket.event.type = EVENT_SAVEON;
      if (write(glob_saveclient->sh, &paket, sizeof(EVENTP))
	  != sizeof(EVENTP)) {
	perror("server: enable_ssaver: write()");
      }
    }

    if (glob_issaving && (now - last < glob_savetime)) {
      EVENTP	paket;
      glob_issaving = 0;
      paket.len = htons(sizeof(EVENTP));
      paket.type = htons(PAK_EVENT);
      paket.event.type = EVENT_SAVEOFF;
      if (write(glob_saveclient->sh, &paket, sizeof(EVENTP))
	  != sizeof(EVENTP)) {
	perror("server: disable_ssaver: write()");
      }
    }

/* re-enable the mouse cursor if necessary and possible */
#ifdef LAZYMOUSE
    if ((glob_mouse.rx != glob_mouse.dx) || (glob_mouse.ry != glob_mouse.dy))
#endif
      mouse_show();

/* complete exit wanted? */
    if (is_terminating && !glob_clients) {
      /* this won't return */
      terminate(0, "");
    }

/* deal with VT switches if necessary (Linux68k only) */
    if (glob_screen->vtswitch) {
      (glob_screen->vtswitch)();
    }
  }
}
