/*
 * server/graph/svgalib.c, part of W
 * (C) 94-07/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * SVGA section (C) 06/96 by Torsten Will
 * itwill@techfak.uni-bielefeld.de
 *
 * a graphic driver for the Linux-System on an Intel-based PC
 *
 * CHANGES:
 * - Rewrote the whole thing to use both DIRECT8 and BMONO drivers.
 * 11/97, ++eero
 *
 * with BMONO this sets a couple of vga registers here, so beware!
 */

#if !defined(i386) || !defined(linux) || !defined(SVGALIB)
#error "this is a *x86-linux* *SVGAlib* graphics driver"
#endif

#if !defined(BMONO) && !defined(DIRECT8)
#error "you have to define at least one of BMONO or DIRECT8 graphics drivers!"
#endif

#include <stdio.h>
#include "../config.h"
#include "../types.h"
#include "gproto.h"
#include "generic/generic.h"
#ifdef BMONO
#include "monochrome/bmono.h"
#endif
#ifdef DIRECT8
#include "direct8/direct8.h"
#endif

#include <vga.h>

/*
 *
 */

#define GRA_I   0x3CE		/* Graphics Controller Index */
#define GRA_D   0x3CF		/* Graphics Controller Data Register */

/* set some vga registers and hope svgalib restores them at exit... */
static __inline__ void port_out(int value, int port)
{
	__asm__ volatile ("outb %0,%1"
	      ::"a" ((unsigned char) value), "d"((unsigned short) port));
}

/*
 * the big init functions
 */

static int svgalib_screen_init(BITMAP *bm)
{
	int svgamode, i;
	vga_modeinfo *modeinfo;
	int switch_to_linear = 0;	   /* 1 if neccessary */

	vga_init();

	/* get mode to switch to */

	/* from $GSVGAMODE environment variable.  another alternative would
	 * be to present user with a menu like many other SVGAlib programs
	 * do...
	 */
	svgamode = vga_getdefaultmode();
	modeinfo = vga_getmodeinfo(svgamode);

	/* mode is linear one with 1 byte per pixe?l */
	if ((modeinfo->flags & CAPABLE_LINEAR) && modeinfo->bytesperpixel == 1) {
		/* force switch to linear */
		switch_to_linear = 1;
	} else {
		printf ("wserver: Mode not capable for 1bpp linear adressing. Using 640x480x2!\r\n");
		svgamode = G640x480x2;
	}

	if (svgamode < 0) {
		/* standard VGA resolution */
		svgamode = G640x480x2;
	}

	/* switch to mode */
	vga_setmode(svgamode);
	if (switch_to_linear) {
		if (vga_setlinearaddressing() < 0) {
			printf ("wserver: Could not set linear adressing. Using 640x480x2!\r\n");
			svgamode = G640x480x2;
			vga_setmode(svgamode);
		}
	}

	modeinfo = vga_getmodeinfo(svgamode);

	bm->data	= vga_getgraphmem();
	bm->width	= modeinfo->width;
	bm->height	= modeinfo->height;

	if (modeinfo->bytesperpixel != 1) {
		if (bm->width & 31) {
			fprintf(stderr, "warning: monochrome screen width not a multiple of 32 pixels,\r\n");
			fprintf(stderr, "         can't use this graphic mode!\r\n");
			return -1;
		}
		bm->type	= BM_PACKEDMONO;
		bm->upl		= bm->width >> 5;
		bm->planes	= 1;
		bm->unitsize	= 4;

		/* set palette registers correctly */
		vga_setpalette(0, 63, 63, 63);
		for (i = 255; i > 0; i--) {
			vga_setpalette(i, 0, 0, 0);
		}
		/* disable Set/Reset Register */
		port_out(0x01, GRA_I);
		port_out(0x00, GRA_D);
	} else {
		/* 8-bit linear */
		bm->type	= BM_DIRECT8;
		bm->upl		= bm->width; 
		bm->planes	= 8;
		bm->unitsize	= 1;
	}

	printf ("wserver: %ix%i screen with %i colors.\r\n",
		bm->width, bm->height, 1 << bm->planes);

	return 0;
}


/*
 * set color palette
 */

static void dummyPutCmap(COLORTABLE *colTab) {}

static void svgaPutCmap(COLORTABLE *colTab)
{
	int i, max;

	max = 1 << theScreen->bm.planes;
	max = max < colTab->colors ? max : colTab->colors;

	for (i = 0; i < max; i++) {
		/* vga palette registers use 6 bits */
		vga_setpalette(i, colTab->red[i] >> 2,
		                  colTab->green[i] >> 2, colTab->blue[i] >> 2);
	}
}

#ifdef BMONO
static SCREEN mono_screen = {
	{}, dummyPutCmap, NULL, bmono_mouseShow, bmono_mouseHide,
	bmono_plot, bmono_test, bmono_line, bmono_hline,
	bmono_vline, generic_box, generic_pbox, bmono_dvline,
	bmono_dhline, generic_dbox, generic_dpbox, generic_circ,
	generic_pcirc, generic_ellipse, generic_pellipse, bmono_bitblk,
	bmono_scroll, bmono_normalc, bmono_stylec, generic_prints,
	bmono_dplot, bmono_dline, generic_dcirc, generic_dpcirc,
	generic_dellipse, generic_dpellipse, generic_poly, generic_dpoly,
	generic_ppoly, generic_dppoly, generic_bezier, generic_dbezier,
	bmono_createbm
};
#endif

#ifdef DIRECT8
static SCREEN svga_screen = {
	{}, svgaPutCmap, NULL, direct8_mouseShow, direct8_mouseHide,
	direct8_plot, direct8_test, direct8_line, direct8_hline,
	direct8_vline, generic_box, generic_pbox, direct8_dvline,
	direct8_dhline, generic_dbox, generic_dpbox, generic_circ,
	generic_pcirc, generic_ellipse, generic_pellipse, direct8_bitblk,
	direct8_scroll, direct8_normalc, direct8_stylec, generic_prints,
	direct8_dplot, direct8_dline, generic_dcirc, generic_dpcirc,
	generic_dellipse, generic_dpellipse, generic_poly, generic_dpoly,
	generic_ppoly, generic_dppoly, generic_bezier, generic_dbezier,
	direct8_createbm
};
#endif

#if 0	/* use stuff in svga/ */
static SCREEN gl_screen = {
  	{}, svgalibPutCmap, NULL, svga_mouseShow, svga_mouseHide,
	svga_plot, svga_test, svga_line, svga_hline,
	svga_vline, generic_box, generic_pbox, svga_dvline,
	svga_dhline, generic_dbox, generic_dpbox, generic_circ,
	generic_pcirc, generic_ellipse, generic_pellipse, svga_bitblk,
	svga_scroll, svga_normalc, svga_stylec, generic_prints,
	svga_dplot, svga_dline, generic_dcirc, generic_dpcirc,
	generic_dellipse, generic_dpellipse, generic_poly, generic_dpoly,
	generic_ppoly, generic_dppoly, generic_bezier, generic_dbezier,
	svga_createbm
};
#endif


/*
 * the real init function
 */

SCREEN *svgalib_init()
{
	BITMAP bm;

	if (svgalib_screen_init(&bm)) {
		fprintf(stderr, "fatal: unknown or unsupported graphics mode requested!\r\n");
		return NULL;
	}
#ifdef BMONO
	if (bm.type == BM_PACKEDMONO) {
		mono_screen.bm = bm;
		return &mono_screen;
	}
#endif
#ifdef DIRECT8
	if (bm.type == BM_DIRECT8) {
		svga_screen.bm = bm;
		return &svga_screen;
	}
#endif
	fprintf(stderr, "fatal: SCREEN struct for mode missing!\n");
	return NULL;
}
