/*
 * server/graph/packed/text.c, part of W
 * (C) 94-04/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * character output routines for the Atari graphics driver
 *
 * CHANGES:
 *
 * - Do clipping and most of the effects in prints instead of printc.
 * - Two separate functions for character output.  One that implements text
 *   style effects (bold, light, reverse, italic) and another that does just
 *   normal text and is therefore much faster.
 * - No inlines.  Now character functions are driver specific and prints a
 *   generic function.  Prints implements underline effect.
 * - Made source ready for font references.
 *   ++eero 11/96
 *
 * TODO:
 * - Optimize the color and 68020 versions like TeSche's original printc()
 *   functions were.
 */

#include <stdio.h>
#include "../../config.h"
#include "../../types.h"
#include "../gproto.h"
#include "packed.h"


#ifdef COLOR

/* Do all the text styles, no clipping. */
void FUNCTION(stylec)(bm, x0, y0, c)
     BITMAP *bm;
     long x0;
     long y0;
     ulong c;
{
  register ulong *cptr, cbit, cdata, fbit, prevbit;
  register ushort *dptr, *dlptr, dlbit;
  register ushort *mask, lighten = 0xffff;
  register short lcwidth, idx;
  FONT *ref = gc0->font, *font = gc0->font;	/* gc0->fref, fref->font */
  short cheight, cwidth, planes = bm->planes;
  ushort dbit, *bgmask, *fgmask;
  ulong bold = 0, skew = 0;
  
  c &= 0xff;
  cheight = font->hdr.height;
  cwidth = font->widths[c];

  /* set styles */
  if (ref->effects & F_BOLD) {
    bold = 0xffffffff;
   }

  if (ref->effects & F_LIGHT) {
    lighten = font->hdr.lighten;
  }

  if (ref->effects & F_ITALIC) {
    x0 += font->hdr.slant_size;
    skew = font->hdr.skew;
  }

  if (ref->effects & F_REVERSE) {
    fgmask = gc0->bgColMask;
    bgmask = gc0->fgColMask;
  } else {
    fgmask = gc0->fgColMask;
    bgmask = gc0->bgColMask;
  }

  dbit = 0x8000 >> (x0 & 15);
  dptr = (ushort *)bm->data + y0 * bm->upl + (x0 >> 4) * planes;

  cptr = font->data + font->offsets[c];
  cbit = 0x80000000;
  cdata = *cptr++;

  while (--cheight >= 0) {

    if (skew & 1) {
      skew |= 0x10000;	/* could also be rolled like `lighten' */
      /* shift lower part of character (co-ordinate) to left */
      if (dbit >= 0x8000) {
	dptr -= planes;
	dbit = 1;
      } else {
	dbit <<= 1;
      }
    }
    skew >>= 1;

    lighten = ((lighten & 1) << 15) | (lighten >> 1);	/* in asm: rol.w */
    prevbit = 0;
    dlbit = dbit;
    dlptr = dptr;
    lcwidth = cwidth;

    while (--lcwidth >= 0) {

      fbit = cdata & cbit;
      fbit |= bold & prevbit;
      prevbit = fbit & cbit;
      if (fbit) {
	mask = fgmask;
      } else {
	mask = bgmask;
      }
      idx = planes;
      while (--idx >= 0) {
	dlptr[idx] = (dlptr[idx] & ~dlbit) | (mask[idx] & (dlbit & lighten));
      }

      if (!(cbit >>= 1)) {
	cbit = 0x80000000;
	cdata = *cptr++;
      }
      if (!(dlbit >>= 1)) {
	dlbit = 0x8000;
	dlptr += planes;
      }
    }
    dptr += bm->upl;
  }
}


/* do normal text, no clipping */
void FUNCTION(normalc)(bm, x0, y0, c)
     BITMAP *bm;
     long x0;
     long y0;
     ulong c;
{
  register ulong *cptr, cbit, cdata;
  register ushort *dptr, *dlptr, dbit, dlbit;
  register short cheight, lcwidth;
  register FONT *font = gc0->font;		/* gc0->fref->font */
  register short idx, planes = bm->planes;
  register ushort *mask;
  short cwidth;
  
  c &= 0xff;
  cheight = font->hdr.height;
  cwidth = font->widths[c];

  dbit = 0x8000 >> (x0 & 15);
  dptr = (ushort *)bm->data + y0 * bm->upl + (x0 >> 4) * planes;

  cptr = font->data + font->offsets[c];
  cbit = 0x80000000;
  cdata = *cptr++;

  while (--cheight >= 0) {			/* TODO: graphics modes */
    dlbit = dbit;
    dlptr = dptr;
    lcwidth = cwidth;
    while (lcwidth-- > 0) {
      if (cdata & cbit) {
	mask = gc0->fgColMask;
      } else {
	mask = gc0->bgColMask;
      }
      idx = planes;
      while (--idx >= 0) {
	dlptr[idx] = (dlptr[idx] & ~dlbit) | (mask[idx] & dlbit);
      }
      if (!(cbit >>= 1)) {
	cbit = 0x80000000;
	cdata = *cptr++;
      }
      if (!(dlbit >>= 1)) {
	dlbit = 0x8000;
	dlptr += planes;
      }
    }
    dptr += bm->upl;
  }
}

#else


/* Do all the text styles, no clipping. */
void FUNCTION(stylec)(bm, x0, y0, c)
     BITMAP *bm;
     long x0;
     long y0;
     ulong c;
{
  register ulong *cptr, cbit, cdata, fbit, prevbit;
#ifdef MONO
  register ulong *dptr, *dlptr, dlbit, dldata;
  ulong dbit, lighten = 0xffffffff;
#else
  register ushort *dptr, *dlptr, dlbit, dldata;
  ushort dbit, lighten = 0xffff;
#endif
  register short lcwidth;
  short cheight, cwidth, reverse = 0;
  FONT *ref = gc0->font, *font = gc0->font;	/* gc0->fref, fref->font */
  ulong bold = 0, skew = 0;
  
  c &= 0xff;
  cheight = font->hdr.height;
  cwidth = font->widths[c];

  /* set styles */
  reverse = ref->effects & F_REVERSE;

  if (ref->effects & F_BOLD) {
    bold = 0xffffffff;
   }

  if (ref->effects & F_LIGHT) {
    lighten = font->hdr.lighten;
#ifdef MONO
    lighten |= (lighten << 16);
#endif
  }

  if (ref->effects & F_ITALIC) {
    x0 += font->hdr.slant_size;
    skew = font->hdr.skew;
  }

#ifdef MONO
  dbit = 0x80000000 >> (x0 & 31);
  dptr = (ulong *)bm->data + y0 * bm->upl + (x0 >> 5);
#else
  dbit = 0x8000 >> (x0 & 15);
  dptr = (ushort *)bm->data + y0 * bm->upl + (x0 >> 4) * bm->planes;
#endif

  cptr = font->data + font->offsets[c];
  cbit = 0x80000000;
  cdata = *cptr++;


  while (--cheight >= 0) {

    if (skew & 1) {
      skew |= 0x10000;	/* could also be rolled like `lighten' */
      /* shift lower part of character (co-ordinate) to left */
#ifdef MONO
      if (dbit >= 0x80000000) {
	dptr--;
#else
      if (dbit >= 0x8000) {
	dptr -= bm->planes;
#endif
	dbit = 1;
      } else {
	dbit <<= 1;
      }
    }
    skew >>= 1;
#ifdef MONO
    lighten = ((lighten & 1) << 31) | (lighten >> 1);	/* in asm: rol.l */
#else
    lighten = ((lighten & 1) << 15) | (lighten >> 1);	/* in asm: rol.w */
#endif
    prevbit = 0;
    dlbit = dbit;
    dlptr = dptr;
    dldata = *dlptr;
    lcwidth = cwidth;

    while (--lcwidth >= 0) {

      fbit = cdata & cbit;		/* font bit */
      fbit |= bold & prevbit;		/* F_BOLD */
      prevbit = fbit & cbit;
      if (reverse) {			/* F_REVERSE */
        dldata |= dlbit;
        if (fbit) {
	  dldata &= ~(dlbit & lighten);	/* F_LIGHT */
        }
      } else {
        dldata &= ~dlbit;
        if (fbit) {
	  dldata |= (dlbit & lighten);	/* F_LIGHT */
        }
      }
      if (!(cbit >>= 1)) {		/* next font bit */
	cbit = 0x80000000;
	cdata = *cptr++;
      }
      if (!(dlbit >>= 1)) {
#ifdef MONO
	dlbit = 0x80000000;
	*dlptr++ = dldata;
#else
	dlbit = 0x8000;
	*dlptr = dldata;
	dlptr += bm->planes;
#endif
	dldata = *dlptr;
      }
    }
    *dlptr = dldata;
    dptr += bm->upl;
  }
}


/* do normal text, no clipping. */
void FUNCTION(normalc)(bm, x0, y0, c)
     BITMAP *bm;
     long x0;
     long y0;
     register ulong c;
{
  register ulong *cptr, cbit, cdata;
#ifdef MONO
  register ulong *dptr, *dlptr, dbit, dlbit, dldata;
#else
  register ushort *dptr, *dlptr, dbit, dlbit, dldata;
#endif
  register short cheight, lcwidth;
  register FONT *font = gc0->font;		/* gc0->fref->font */
  short cwidth;

  c &= 0xff;
  cheight = font->hdr.height;
  cwidth = font->widths[c];

#ifdef MONO
  dbit = 0x80000000 >> (x0 & 31);
  dptr = (ulong *)bm->data + y0 * bm->upl + (x0 >> 5);
#else
  dbit = 0x8000 >> (x0 & 15);
  dptr = (ushort *)bm->data + y0 * bm->upl + (x0 >> 4) * bm->planes;
#endif

  cptr = font->data + font->offsets[c];
  cbit = 0x80000000;
  cdata = *cptr++;

  while (--cheight >= 0) {		/* TODO: graphics modes */
    dlbit = dbit;
    dlptr = dptr;
    dldata = *dlptr;
    lcwidth = cwidth;
    while (lcwidth-- > 0) {
      dldata &= ~dlbit;
      if (cdata & cbit) {
	dldata |= dlbit;
      }
      if (!(cbit >>= 1)) {
	cbit = 0x80000000;
	cdata = *cptr++;
      }
      if (!(dlbit >>= 1)) {
#ifdef MONO
	dlbit = 0x80000000;
	*dlptr++ = dldata;
#else
	dlbit = 0x8000;
	*dlptr = dldata;
	dlptr += bm->planes;
#endif
	dldata = *dlptr;
      }
    }
    *dlptr = dldata;
    dptr += bm->upl;
  }
}

#endif

