/*
 * server/graph/monochrome/text.c, part of W
 * (C) 94-04/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * character output routines for the monochrome byte graphics driver
 *
 * CHANGES:
 *
 * - Do clipping and most of the effects in prints instead of printc.
 * - Two separate functions for character output.  One that implements text
 *   style effects (bold, light, reverse, italic) and another that does just
 *   normal text and is therefore much faster.
 * - No inlines.  Now character functions are driver specific and prints a
 *   generic function.  Prints implements underline effect.
 * - Made source ready for font references.
 *   ++eero 11/96
 */

#include <stdio.h>
#include <netinet/in.h>
#include "../../config.h"
#include "../../types.h"
#include "../gproto.h"
#include "bmono.h"


/* Do all the text styles, no clipping. */
void FUNCTION(stylec)(bm, x0, y0, c)
     BITMAP *bm;
     long x0;
     long y0;
     ulong c;
{
  ulong *cptr, cbit, cdata, fbit, prevbit;
  ulong *dptr, *dlptr, dlbit, dldata;
  ulong dbit, lighten = 0xffffffff;
  short lcwidth;
  short cheight, cwidth, reverse = 0;
  FONT *ref = gc0->font, *font = gc0->font;	/* gc0->fref, fref->font */
  ulong bold = 0, skew = 0;
  
  c &= 0xff;
  cheight = font->hdr.height;
  cwidth = font->widths[c];

  /* set styles */
  reverse = ref->effects & F_REVERSE;

  if (ref->effects & F_BOLD) {
    bold = 0xffffffff;
   }

  if (ref->effects & F_LIGHT) {
    lighten = font->hdr.lighten;
    lighten |= (lighten << 16);
  }

  if (ref->effects & F_ITALIC) {
    x0 += font->hdr.slant_size;
    skew = font->hdr.skew;
  }

  dbit = 0x80000000 >> (x0 & 31);
  dptr = (ulong *)bm->data + y0 * bm->upl + (x0 >> 5);

  cptr = font->data + font->offsets[c];
  cbit = 0x80000000;
  cdata = *cptr++;


  while (--cheight >= 0) {

    if (skew & 1) {
      skew |= 0x10000;	/* could also be rolled like `lighten' */
      /* shift lower part of character (co-ordinate) to left */
      if (dbit >= 0x80000000) {
	dptr--;
	dbit = 1;
      } else {
	dbit <<= 1;
      }
    }
    skew >>= 1;
    lighten = ((lighten & 1) << 31) | (lighten >> 1);	/* in asm: rol.l */
    prevbit = 0;
    dlbit = dbit;
    dlptr = dptr;
    dldata = htonl(*dlptr);
    lcwidth = cwidth;

    while (--lcwidth >= 0) {

      fbit = cdata & cbit;		/* font bit */
      fbit |= bold & prevbit;		/* F_BOLD */
      prevbit = fbit & cbit;
      if (reverse) {			/* F_REVERSE */
        dldata |= dlbit;
        if (fbit) {
	  dldata &= ~(dlbit & lighten);	/* F_LIGHT */
        }
      } else {
        dldata &= ~dlbit;
        if (fbit) {
	  dldata |= (dlbit & lighten);	/* F_LIGHT */
        }
      }
      if (!(cbit >>= 1)) {		/* next font bit */
	cbit = 0x80000000;
	cdata = *cptr++;
      }
      if (!(dlbit >>= 1)) {
	dlbit = 0x80000000;
	*dlptr++ = ntohl(dldata);
	dldata = htonl(*dlptr);
      }
    }
    *dlptr = ntohl(dldata);
    dptr += bm->upl;
  }
}


/* do normal text, no clipping. */
void FUNCTION(normalc)(bm, x0, y0, c)
     BITMAP *bm;
     long x0;
     long y0;
     ulong c;
{
  ulong *cptr, cbit, cdata;
  ulong *dptr, *dlptr, dbit, dlbit, dldata;
  short cheight, lcwidth;
  FONT *font = gc0->font;		/* gc0->fref->font */
  short cwidth;

  c &= 0xff;
  cheight = font->hdr.height;
  cwidth = font->widths[c];

  dbit = 0x80000000 >> (x0 & 31);
  dptr = (ulong *)bm->data + y0 * bm->upl + (x0 >> 5);

  cptr = font->data + font->offsets[c];
  cbit = 0x80000000;
  cdata = *cptr++;

  while (--cheight >= 0) {		/* TODO: graphics modes */
    dlbit = dbit;
    dlptr = dptr;
    dldata = htonl(*dlptr);
    lcwidth = cwidth;
    while (lcwidth-- > 0) {
      dldata &= ~dlbit;
      if (cdata & cbit) {
	dldata |= dlbit;
      }
      if (!(cbit >>= 1)) {
	cbit = 0x80000000;
	cdata = *cptr++;
      }
      if (!(dlbit >>= 1)) {
	dlbit = 0x80000000;
	*dlptr++ = ntohl(dldata);
	dldata = htonl(*dlptr);
      }
    }
    *dlptr = ntohl(dldata);
    dptr += bm->upl;
  }
}
