/*
 * server/graph/circle.c, part of W
 * (C) 94-02/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * routines dealing with circles
 *
 * CHANGES:
 *
 * - added patterned functions (untested). ++kay, 1/96
 * - added comments on how to speed things up. ++kay, 1/96
 * - changed pattern functions to use solid ones.  One more function
 *   call shouldn't slow down that much? ++eero 8/96
 * - added ellipse functions. ++eero 8/96
 */

#include <stdio.h>
#include "../../config.h"
#include "../../types.h"
#include "../gproto.h"
#include "../clip.h"
#include "generic.h"

static void (*PlotFn)(BITMAP *, long, long);
static void (*HlineFn)(BITMAP *, long, long, long);
static void (*VlineFn)(BITMAP *, long, long, long);

/*
 *
 */

static void circle (bm, x0, y0, r)
     BITMAP *bm;
     register long x0;
     register long y0;
     long r;
{
  register void (*plot_fn) (BITMAP *, long, long);
  register long x, y, da;
  REC *oclip = clip0;

  plot_fn = PlotFn;

  if (!BOX_NEEDS_CLIPPING (x0-r-1, y0-r-1, r+r+2, r+r+2, clip0))
    clip0 = NULL;

  if (!r) {
    (*plot_fn)(bm, x0, y0);
    clip0 = oclip;
    return;
  }

  if (gc0->drawmode == M_INVERS) {
    (*plot_fn)(bm, x0-r, y0);
    (*plot_fn)(bm, x0+r, y0);
    (*plot_fn)(bm, x0, y0-r);
    (*plot_fn)(bm, x0, y0+r);
  }

  x = 0;
  y = r;
  da = r - 1;

  do {
    if (da < 0) {
      y--;
      da += y + y;
    }

    (*plot_fn)(bm, x0+x, y0+y);
    (*plot_fn)(bm, x0-x, y0+y);
    (*plot_fn)(bm, x0+x, y0-y);
    (*plot_fn)(bm, x0-x, y0-y);
    (*plot_fn)(bm, x0+y, y0+x);
    (*plot_fn)(bm, x0-y, y0+x);
    (*plot_fn)(bm, x0+y, y0-x);
    (*plot_fn)(bm, x0-y, y0-x);

    da = da - x - x - 1;
    x++;

  } while (x < y);

  if ((gc0->drawmode == M_INVERS) && (x != y)) {
    (*plot_fn)(bm, x0-y, y0-y);
    (*plot_fn)(bm, x0+y, y0-y);
    (*plot_fn)(bm, x0+y, y0+y);
    (*plot_fn)(bm, x0-y, y0+y);
  }
  clip0 = oclip;
}

static void pcircle (bm, x0, y0, r)
     BITMAP *bm;
     register long x0;
     register long y0;
     long r;
{
  register void (*hline_fn) (BITMAP *, long, long, long);
  register long dx, dy, rq = r * r;
  REC *oclip = clip0;

  hline_fn = HlineFn;

  if (!BOX_NEEDS_CLIPPING (x0-r-1, y0-r-1, r+r+2, r+r+2, clip0))
    clip0 = NULL;

  if (!r) {
    (*PlotFn)(bm, x0, y0);
    clip0 = oclip;
    return;
  }

  (*hline_fn)(bm, x0-r, y0, x0+r);

  dy = 0;
  dx = r;
  while (dx) {

    dy++;
    while (dx*dx + dy*dy > rq)
      dx--;

    (*hline_fn)(bm, x0-dx, y0-dy, x0+dx);
    (*hline_fn)(bm, x0-dx, y0+dy, x0+dx);
  }
  clip0 = oclip;
}

void generic_circ (bm, x0, y0, r)
     BITMAP *bm;
     register long x0;
     register long y0;
     register long r;
{
  PlotFn = theScreen->plot;
  circle (bm, x0, y0, r);
}


void generic_pcirc (bm, x0, y0, r)
     BITMAP *bm;
     register long x0;
     register long y0;
     long r;
{
  PlotFn = theScreen->plot;
  HlineFn = theScreen->hline;
  pcircle (bm, x0, y0, r);
}

void generic_dcirc (bm, x0, y0, r)
     BITMAP *bm;
     register long x0;
     register long y0;
     register long r;
{
  PlotFn = theScreen->dplot;
  circle (bm, x0, y0, r);
}


void generic_dpcirc (bm, x0, y0, r)
     BITMAP *bm;
     register long x0;
     register long y0;
     long r;
{
  PlotFn = theScreen->dplot;
  HlineFn = theScreen->dhline;
  pcircle (bm, x0, y0, r);
}

/* 
 * ellipses
 */

static void ellipse (bm, x0, y0, rx, ry)
     BITMAP *bm;
     register long x0;
     register long y0;
     long rx;
     long ry;
{
  register void (*plot_fn) (BITMAP *, long, long);
  register long dx, dy, dd, a = rx*rx, b = ry*ry;
  REC *oclip = clip0;

  plot_fn = PlotFn;

  if (!BOX_NEEDS_CLIPPING (x0-rx-1, y0-ry-1, rx+rx+2, ry+ry+2, clip0))
    clip0 = NULL;

  if (!rx) {
    (*VlineFn) (bm, x0, y0-ry, y0+ry);
    clip0 = oclip;
    return;
  }

  if (!ry) {
    (*HlineFn) (bm, x0-rx, y0, x0+ry);
    clip0 = oclip;
    return;
  }

  /* this is fastest when ry > rx */
  (*plot_fn) (bm, x0-rx, y0);
  (*plot_fn) (bm, x0+rx, y0);

  dy = 0;
  dx = rx;
  dd = a*b;	/* seems to be quite accurate even with overflow? */
  for (;;) {

    dy++;
    while (dx*dx*b + dy*dy*a > dd)
    {
      (*plot_fn) (bm, x0-dx, y0-dy);
      (*plot_fn) (bm, x0+dx, y0-dy);
      (*plot_fn) (bm, x0-dx, y0+dy);
      (*plot_fn) (bm, x0+dx, y0+dy);
      dx--;
    }
    if(!dx)
    {
      (*VlineFn) (bm, x0, y0-dy, y0-ry);
      (*VlineFn) (bm, x0, y0+dy, y0+ry);
      clip0 = oclip;
      return;
    }
    (*plot_fn) (bm, x0-dx, y0-dy);
    (*plot_fn) (bm, x0+dx, y0-dy);
    (*plot_fn) (bm, x0-dx, y0+dy);
    (*plot_fn) (bm, x0+dx, y0+dy);
  }
}

static void pellipse (bm, x0, y0, rx, ry)
     BITMAP *bm;
     register long x0;
     register long y0;
     long rx;
     long ry;
{
  register void (*hline_fn) (BITMAP *, long, long, long);
  register long dx, dy, dd, a = rx*rx, b = ry*ry;
  REC *oclip = clip0;

  hline_fn = HlineFn;

  if (!BOX_NEEDS_CLIPPING (x0-rx-1, y0-ry-1, rx+rx+2, ry+ry+2, clip0))
    clip0 = NULL;

  if (!rx) {
    (*VlineFn) (bm, x0, y0-ry, y0+ry);
    clip0 = oclip;
    return;
  }

  if (!ry) {
    (*hline_fn) (bm, x0-rx, y0, x0+rx);
    clip0 = oclip;
    return;
  }

  (*hline_fn) (bm, x0-rx, y0, x0+rx);

  dy = 0;
  dx = rx;
  dd = a*b;
  while (dx) {

    dy++;
    while (dx*dx*b + dy*dy*a > dd)
      dx--;

    (*hline_fn) (bm, x0-dx, y0-dy, x0+dx);
    (*hline_fn) (bm, x0-dx, y0+dy, x0+dx);
  }
  if (dy < ry) {
    dy++;
    /* ellipses that are only a couple of pixels wide... */
    (*VlineFn) (bm, x0, y0-dy, y0-ry);
    (*VlineFn) (bm, x0, y0+dy, y0+ry);
  }
  clip0 = oclip;
}

void generic_ellipse (bm, x0, y0, rx, ry)
     BITMAP *bm;
     register long x0;
     register long y0;
     long rx;
     long ry;
{
  PlotFn = theScreen->plot;
  HlineFn = theScreen->hline;
  VlineFn = theScreen->vline;
  ellipse (bm, x0, y0, rx, ry);
}

void generic_pellipse (bm, x0, y0, rx, ry)
     BITMAP *bm;
     register long x0;
     register long y0;
     long rx;
     long ry;
{
  HlineFn = theScreen->hline;
  VlineFn = theScreen->vline;
  pellipse (bm, x0, y0, rx, ry);
}

void generic_dellipse (bm, x0, y0, rx, ry)
     BITMAP *bm;
     register long x0;
     register long y0;
     long rx;
     long ry;
{
  PlotFn = theScreen->dplot;
  HlineFn = theScreen->dhline;
  VlineFn = theScreen->dvline;
  ellipse (bm, x0, y0, rx, ry);
}

void generic_dpellipse (bm, x0, y0, rx, ry)
     BITMAP *bm;
     register long x0;
     register long y0;
     long rx;
     long ry;
{
  HlineFn = theScreen->dhline;
  VlineFn = theScreen->dvline;
  pellipse (bm, x0, y0, rx, ry);
}
