/*
 * server/graph/atari.c, part of W
 * (C) 94-07/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * a graphic driver for the Atari ST/TT, either MiNT or Linux68k
 */

#include <stdio.h>

#if (defined(__MINT__) || defined(linux)) && !defined(i386)

#include "../config.h"
#include "../types.h"
#include "gproto.h"
#include "packed/packed.h"
#include "generic/generic.h"

#ifdef PACKEDMONO
#error is mono
#endif

#ifdef PACKEDCOLOR
#error is color
#endif


/*
 * the big init functions
 */

#ifdef linux

#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <linux/fb.h>
#include <linux/vt.h>
#include <linux/kd.h>

static struct fb_fix_screeninfo finfo;
static struct fb_var_screeninfo vinfo;
static int vsize, fh;
static ushort *origScreen = NULL;
static volatile int sigVt;


/*
 *
 */

static void sigvtswitch(int sig);


static void linuxVtSwitch(void)
{
  struct sigaction sa;

  /* this routine will be called at vt-switch by the main loop when all
   * graphics has been done, therefore we needn't any kind of semaphore
   * for the screen, just copy it...
   */

  switch(sigVt) {

    case SIGUSR1:
      if (!origScreen) {
	origScreen = theScreen->bm.data;
	if ((theScreen->bm.data = malloc(vsize))) {
	  memcpy(theScreen->bm.data, origScreen, vsize);
	  ioctl(0, VT_RELDISP, 1);
	}
      }
      break;

    case SIGUSR2:
      if (origScreen) {
	memcpy(origScreen, theScreen->bm.data, vsize);
	free(theScreen->bm.data);
	theScreen->bm.data = origScreen;
	origScreen = NULL;
	ioctl(0, VT_ACKACQ, 1);
      }
      break;

    default:
      fprintf(stderr, "unknown signal received\r\n");
      exit(-1);
  }

  /* ready for further signals */

  theScreen->vtswitch = NULL;
  sa.sa_handler = sigvtswitch;
  sa.sa_flags = 0;
  sa.sa_mask = 0;

  sigaction(SIGUSR1, &sa, NULL);
  sigaction(SIGUSR2, &sa, NULL);
}


static void sigvtswitch(int sig)
{
  struct sigaction sa;

  if ((sig == SIGUSR1) || (sig == SIGUSR2)) {

    /* ignore further signals until this one is served */

    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    sigaction(SIGUSR1, &sa, NULL);
    sigaction(SIGUSR2, &sa, NULL);

    sigVt = sig;
    theScreen->vtswitch = linuxVtSwitch;
  }
}


static int linux_screen_init(BITMAP *bm)
{
  caddr_t addr;
  struct vt_mode vt;

  if ((fh = open("/dev/fb0current", O_RDWR)) < 0) {
    perror("open()");
    return -1;
  }

  if (ioctl(fh, FBIOGET_FSCREENINFO, &finfo)) {
    perror("ioctl()");
    return -1;
  }
  finfo.id[15] = 0;
  switch(finfo.type) {
    case FB_TYPE_PACKED_PIXELS:
      printf("%s: packed pixels\r\n", finfo.id);
      break;
    case FB_TYPE_PLANES:
      printf("%s: planes\r\n", finfo.id);
      break;
    case FB_TYPE_INTERLEAVED_PLANES:
      printf("%s: interleaved planes\r\n", finfo.id);
      break;
    default:
      fprintf(stderr, "%s: unknown type\r\n", finfo.id);
      return -1;
  }

  ioctl(fh, FBIOGET_VSCREENINFO, &vinfo);
  printf("using %ix%i of %ix%i pixels, %i bits per pixel\r\n",
	 vinfo.xres, vinfo.yres,
	 vinfo.xres_virtual, vinfo.yres_virtual,
	 vinfo.bits_per_pixel);
  vsize = (vinfo.xres * vinfo.bits_per_pixel >> 3) * vinfo.yres;
  if ((addr = mmap(0, vsize, PROT_READ | PROT_WRITE, MAP_SHARED, fh, 0)) < 0) {
    perror("mmap()");
    return -1;
  }
  printf("%ik videoram mapped to 0x%08x\r\n", vsize >> 10,(int)addr);

  if (!glob_debug) {
    /* sigh, I often have got to disable this code because if wserver crashes
     * under Linux68k then the screen is locked in graphic mode and I've got to
     * reboot my machine... :(
     *
     * in fact I had to change this so often that it is now a runtime-option.
     */

    /* set up signal handlers for vtswitch
     */
    signal(SIGUSR1, sigvtswitch);
    signal(SIGUSR2, sigvtswitch);

    /* disable vty switches
     */
    vt.mode = VT_PROCESS;
    vt.waitv = 0;
    vt.relsig = SIGUSR1;
    vt.acqsig = SIGUSR2;
    if (ioctl(0, VT_SETMODE, &vt)) {
      perror("ioctl(VT_SETMODE)");
      return -1;
    }

    /* disable screensaver */
    ioctl(0, KDSETMODE, KD_GRAPHICS);
  }

  sleep(2);

  vinfo.xoffset = 0;
  vinfo.yoffset = 0;
  ioctl(fh, FBIOPUT_VSCREENINFO, &vinfo);

  bm->width = vinfo.xres;
  bm->height = vinfo.yres;
  bm->planes = vinfo.bits_per_pixel;
  bm->data = addr;

  return 0;
}


static void linuxPutCmap(COLORTABLE *colTab)
{
  static struct fb_cmap cmap = {0, 0, NULL, NULL, NULL, NULL};
  int i;

  /* sadly enough, linux uses 'ushort' as RGB values - allthough I heavily
   * doubt there's any need for this, but it forces us to copy the color table
   * to a different format. we use a static struct here because we assume that
   * all color tables will have the same number of colors - this may be legal
   * for a beginning, but may change.
   */
  if (!cmap.len) {
    cmap.start = 0;
    cmap.len = colTab->colors;
    /* let's hope this never fails!!!
     */
    cmap.red = malloc (cmap.len * sizeof (ushort));
    cmap.green = malloc (cmap.len * sizeof (ushort));
    cmap.blue = malloc (cmap.len * sizeof (ushort));
  }

  for (i=0; i<colTab->colors; i++) {
    cmap.red[i] = colTab->red[i] << 8;
    cmap.green[i] = colTab->green[i] << 8;
    cmap.blue[i] = colTab->blue[i] << 8;
  }

  ioctl(fh, FBIOPUTCMAP, &cmap);
}

#endif


#ifdef __MINT__

#include <osbind.h>
#include <linea.h>

int mint_screen_init(BITMAP *bm)
{
  linea0();

  bm->width = V_X_MAX;
  bm->height = V_Y_MAX;
  bm->planes = VPLANES;

  bm->data = Logbase();

  if (bm->planes == 1) {
    printf("Mint monochrome, %i * %i pixel\r\n", bm->width, bm->height);
  } else {
    printf("Mint color, %i * %i pixel\r\n", bm->width, bm->height);
  }

  return 0;
}

#endif


/*
 * the real init function
 */

/*
 * a dummy routine monochrome (or yet unimplemented color) drivers may use
 */

static void dummyPutCmap(COLORTABLE *colTab)
{
  /* dummy, dummy, dummy... */
}


#ifdef PCOLOR
static SCREEN packed_color_screen = {
  {}, dummyPutCmap, NULL, packed_color_mouseShow,
  packed_color_mouseHide, packed_color_plot, packed_color_test,
  packed_color_line, packed_color_hline, packed_color_vline,
  generic_box, generic_pbox, packed_color_dvline, packed_color_dhline,
  generic_dbox, generic_dpbox, generic_circ, generic_pcirc,
  generic_ellipse, generic_pellipse, packed_color_bitblk,
  packed_color_scroll, packed_color_normalc, packed_color_stylec,
  generic_prints, packed_color_dplot, packed_color_dline, generic_dcirc,
  generic_dpcirc, generic_dellipse, generic_dpellipse, generic_poly,
  generic_dpoly, generic_ppoly, generic_dppoly, generic_bezier,
  generic_dbezier, packed_color_createbm };
#endif

#ifdef PCOLORMONO
static SCREEN packed_colormono_screen = {
  {}, dummyPutCmap, NULL, packed_colormono_mouseShow,
  packed_colormono_mouseHide, packed_colormono_plot,
  packed_colormono_test, packed_colormono_line, packed_colormono_hline,
  packed_colormono_vline, generic_box, generic_pbox,
  packed_colormono_dvline, packed_colormono_dhline, generic_dbox,
  generic_dpbox, generic_circ, generic_pcirc, generic_ellipse,
  generic_pellipse, packed_colormono_bitblk, packed_colormono_scroll,
  packed_colormono_normalc, packed_colormono_stylec, generic_prints,
  packed_colormono_dplot, packed_colormono_dline, generic_dcirc,
  generic_dpcirc, generic_dellipse, generic_dpellipse, generic_poly,
  generic_dpoly, generic_ppoly, generic_dppoly, generic_bezier,
  generic_dbezier, packed_colormono_createbm };
#endif

#ifdef PMONO
static SCREEN packed_mono_screen = {
  {}, dummyPutCmap, NULL, packed_mono_mouseShow, packed_mono_mouseHide,
  packed_mono_plot, packed_mono_test, packed_mono_line,
  packed_mono_hline, packed_mono_vline, generic_box, generic_pbox,
  packed_mono_dvline, packed_mono_dhline, generic_dbox, generic_dpbox,
  generic_circ, generic_pcirc, generic_ellipse, generic_pellipse,
  packed_mono_bitblk, packed_mono_scroll, packed_mono_normalc,
  packed_mono_stylec, generic_prints, packed_mono_dplot,
  packed_mono_dline, generic_dcirc, generic_dpcirc, generic_dellipse,
  generic_dpellipse, generic_poly, generic_dpoly, generic_ppoly,
  generic_dppoly, generic_bezier, generic_dbezier, packed_mono_createbm };
#endif

SCREEN *atari_init (int forceMono)
{
  int ret;
  BITMAP bm;

#if defined(linux)
  ret = linux_screen_init(&bm);
#ifdef PCOLOR
  packed_color_screen.putCmap = linuxPutCmap;
#endif
#ifdef PCOLORMONO
  packed_colormono_screen.putCmap = linuxPutCmap;
#endif
#elif defined(__MINT__)
  ret = mint_screen_init(&bm);
#else
  ret = -1;
#endif

  if (ret) {
    fprintf(stderr, "fatal: unknown graphics or system is not supported!\r\n");
    return NULL;
  }

#ifdef PMONO
  if (bm.planes == 1) {
    if (bm.width & 31) {
      fprintf(stderr, "warning: monochrome screen width not a multiple of 32 pixels,\r\n");
      fprintf(stderr, "         can't use this graphic mode!\r\n");
    } else {
      bm.type = BM_PACKEDMONO;
      bm.unitsize = 4;
      bm.upl = bm.width >> 5;
      packed_mono_screen.bm = bm;
      return &packed_mono_screen;
    }
  }
#endif

  bm.unitsize = 2;
  bm.upl = (bm.width * bm.planes) >> 4;

#ifdef PCOLORMONO
  if (forceMono) {
    bm.type = BM_PACKEDCOLORMONO;
    packed_colormono_screen.bm = bm;
    return &packed_colormono_screen;
  }
#endif

#ifdef PCOLOR
  bm.type = BM_PACKEDCOLOR;
  packed_color_screen.bm = bm;
  return &packed_color_screen;
#endif

  fprintf(stderr, "fatal: you didn't configure any graphic driver?!\r\n");

  return NULL;
}

#endif
