/*
 * server/font.c, part of W
 * (C) 1994,95,96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * font loading and handling stuff
 *
 * CHANGES
 * - modified to work with the new font format. ++eero 11/96
 * - besides FONTPATH, fonts can be loaded from 'glob_fontpath'.
 * - unloadfont frees font->name too. ++eero 10/97
 */

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#ifdef __MINT__
# include <mintbind.h>
#endif

#include "config.h"
#include "types.h"
#include "pakets.h"
#include "proto.h"


/*
 * global variables
 */

FONT glob_font[MAXFONTS];


/*
 * some private functions
 */

int load_font(int font_h, char *fname)
{
  FONT *font = &glob_font[font_h];
  FONTHDR *hdr = &font->hdr;
  long fh, i, longs, needs;
  int isLittleEndian = 0;
  char *buf, *path;

#ifndef FONTDIR
#error you must define FONTDIR!!!
#endif

  if (glob_fontpath) {
    path = glob_fontpath;
  } else {
    path = FONTDIR;
  }
  needs = strlen(path);
  buf = malloc(needs + strlen(fname) + 2);
  strcpy(buf, path);
  buf[needs++] = '/';
  strcpy(buf + needs, fname);

  if ((fh = open(buf, O_RDONLY)) < 0) {
    free(buf);
    return -1;
  }
  free(buf);

  if (read(fh, hdr, sizeof(FONTHDR)) != sizeof(FONTHDR)) {
    close(fh);
    return -2;
  }

  /* hey, we may be running on an Intel box!
   */
  switch (hdr->magic) {
    case 0x314e4657:
      isLittleEndian = 1;
      /* fall through */
    case 0x57464e31:
      /* font header / data size */
      hdr->header       = ntohs(hdr->header);
      hdr->lenght       = ntohl(hdr->lenght);
      /* font type information */
      hdr->height       = ntohs(hdr->height);
      hdr->flags        = ntohs(hdr->flags);
      hdr->styles       = ntohs(hdr->styles);
      /* font effect generation vars */
      hdr->thicken      = ntohs(hdr->thicken);
      hdr->slant_size   = ntohs(hdr->slant_size);
      hdr->slant_offset = ntohs(hdr->slant_offset);
      hdr->skew         = ntohs(hdr->skew);
      hdr->lighten      = ntohs(hdr->lighten);
      hdr->effect_mask  = ntohs(hdr->effect_mask);
      /* font character cell size */
      hdr->baseline     = ntohs(hdr->baseline);
      hdr->maxwidth     = ntohs(hdr->maxwidth);
      hdr->top          = ntohs(hdr->top);
      hdr->bottom       = ntohs(hdr->bottom);
      hdr->left         = ntohs(hdr->left);
      hdr->right        = ntohs(hdr->right);
      break;
    case 0x544e4657:
    case 0x57464e54:
      fprintf(stderr, "%s is a W1R3 format font\r\n", fname);
    default:
      close(fh);
      return -3;
  }

  i = hdr->header - sizeof(FONTHDR);
  if (i) {
    /* disregard font information */
    lseek(fh, i, SEEK_CUR);
  }

  if (read(fh, font->widths, 256) != 256) {
    close(fh);
    return -4;
  }

  longs = 0;
  for (i = 0; i < 256; i++) {
    font->offsets[i] = longs;
    longs += ((font->widths[i] * hdr->height + 31) & ~31) >> 5;
  }

  needs = longs << 2;
  if (needs != hdr->lenght || !(font->data = malloc(needs))) {
    close(fh);
    return -5;
  }

  if (read(fh, font->data, needs) != needs) {
    close(fh);
    free(font->data);
    return -6;
  }

  if (isLittleEndian) {
    for (i = 0; i < longs; i++) {
      font->data[i] = ntohl(font->data[i]);
    }
  }

  font->name = strdup(fname);
  font->numused = 1;
  close(fh);

  return font_h;
}


/*
 * the exported functions
 */

int font_init(char *titlefont, char *menufont)
{
  int i, err1, err2;

  if(!(titlefont && menufont)) {
    fprintf(stderr, "fatal: font initialisation error (%s/%s)\r\n", titlefont, menufont);
    return -1;
  }

  for (i=0; i<MAXFONTS; i++) {
    glob_font[i].numused = 0;
  }

  err1 = load_font(TITLEFONT, titlefont);
  err2 = load_font(MENUFONT, menufont);

  if ((err1 < 0) || (err2 < 0)) {
    fprintf(stderr, "fatal: font initialisation error (%d/%d)\r\n", err1, err2);
    return -1;
  }

  return 0;
}


short font_loadfont(char *fontname)
{
  short i, fglobh;

  fglobh = -1;
  for (i = 0; i < MAXFONTS; i++) {
    if (!glob_font[i].numused) {
      fglobh = i;
      break;
    }
  }

  if (fglobh == -1) {
    return -1;
  }

  return load_font(fglobh, fontname);
}


short font_unloadfont(FONT *fp)
{
  if (fp->numused > 0) {
    return -1;
  }

  fp->numused = 0;
  free(fp->name);
  free(fp->data);

  return 0;
}


/*
 * calculate the length of a string in pixel
 */

int fontStrLen(FONT *f, char *s)
{
  int len = 0;

  while (*s) {
    /*
     * kay: I have those sign extension bugs!
     */
    len += f->widths[*s++ & 0xff];
  }

  return len;
}
