/*
 * server/color.c, part of W
 * (C) 1994,95,96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * 07/96 TeSche:
 * functions for maintaining color tables
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#ifdef __MINT__
# include <mintbind.h>
#endif

#include "config.h"
#include "types.h"
#include "pakets.h"
#include "proto.h"


/*
 *
 */

#define SETBIT(ptr,bit) ptr[bit>>3] |= 0x80>>(bit&7)
#define CLRBIT(ptr,bit) ptr[bit>>3] &= ~(0x80>>(bit&7))
#define TSTBIT(ptr,bit) (ptr[bit>>3] & 0x80>>(bit&7))


/*
 *
 */

COLORTABLE *glob_colortable;
static ulong *colorCount;
static short colors, planes;


/*
 * private helpers
 */

static void makePrivateTable (WINDOW *win)
{
  COLORTABLE *new;

  if (!(new = malloc (sizeof (COLORTABLE)))) {
    wserver_exit(-1, "fatal: out of memory in makePrivateTable()!");
  }

  new->colors = colors;
  new->red = malloc (colors);
  new->green = malloc (colors);
  new->blue = malloc (colors);
  new->used = malloc (colors >> 3);

  if (!new->red || !new->green || !new->blue || !new->used) {
    wserver_exit(-1, "fatal: out of memory in makePrivateTable()!");
  }

  memcpy(new->red, win->colTab->red, colors);
  memcpy(new->green, win->colTab->green, colors);
  memcpy(new->blue, win->colTab->blue, colors);
  memcpy(new->used, win->colTab->used, colors >> 3);

  new->win = win;
  win->colTab = new;
}


/*
 * needed for PACKED driver
 */

inline void colorGetMask (short color, ushort *mask)
{
  register short i = planes;
  register ushort bit = 0x0001;

  while (--i >= 0) {
    if (color & bit) {
      *mask++ = 0xffff;
    } else {
      *mask++ = 0x0000;
    }
    bit <<= 1;
  }
}


/*
 * initialize a the basic color table.  the colors used are 0=white
 * (background) and 1=black (foreground).  others are set some exemplary
 * colors, don't expect them to be anything definitive until you have set
 * them yourself.
 */

short colorInit (void)
{
  short size, idx, div;

  if (!(glob_colortable = malloc (sizeof (COLORTABLE)))) {
    fprintf(stderr, "error: not enough memory for system color table\r\n");
    return -1;
  }

  planes = glob_screen->bm.planes;
  colors = glob_colortable->colors = 1 << planes;
  if ((colors < 2) || (colors > 256)) {
    fprintf(stderr, "error: can't support %i-color screens\r\n", colors);
    return -1;
  }
  glob_colortable->win = NULL;

  glob_colortable->red = malloc (colors);
  glob_colortable->green = malloc (colors);
  glob_colortable->blue = malloc (colors);
  if (!glob_colortable->red ||
      !glob_colortable->green ||
      !glob_colortable->blue) {
    fprintf(stderr, "error: not enough memory for system color table\r\n");
    return -1;
  }

  /* first color is white, next black, others 'something' */
  glob_colortable->red[0] = 255;
  glob_colortable->green[0] = 255;
  glob_colortable->blue[0] = 255;
  glob_colortable->red[1] = 0;
  glob_colortable->green[1] = 0;
  glob_colortable->blue[1] = 0;

  div = (colors - 2) / 3;
  for (idx = 2; idx < colors; idx++) {
    if (idx < div) {
      glob_colortable->red[1] = 0;
      glob_colortable->green[1] = 0;
      glob_colortable->blue[1] = ((idx % div) * 256) / div;
    } else if (idx*2 < div) {
      glob_colortable->red[1] = 0;
      glob_colortable->green[1] = ((idx % div) * 256) / div;
      glob_colortable->blue[1] = 0;
    } else {
      glob_colortable->red[1] = ((idx % div) * 256) / div;
      glob_colortable->green[1] = 0;
      glob_colortable->blue[1] = 0;
    }
  }

  size = colors >> 3;
  if (!(glob_colortable->used = malloc (size))) {
    fprintf(stderr, "error: not enough memory for system color table\r\n");
    return -1;
  }
  memset(glob_colortable->used, 0, size);
  SETBIT(glob_colortable->used, 0);
#if 1
  colors--;
  SETBIT(glob_colortable->used, colors);
  colors++;
#else
  SETBIT(glob_colortable->used, colors-1);
#endif

  size = colors * sizeof (ulong);
  if (!(colorCount = malloc (size))) {
    fprintf(stderr, "error: not enough memory for system color table\r\n");
    return -1;
  }
  memset(colorCount, 0, size);
  /* this will help us to only use these colors if absolutely necessary
   */
  colorCount[0] = 0xf0000000;
  colorCount[colors-1] = 0xf0000000;

  (*glob_screen->putCmap)(glob_colortable);

  return 0;
}


void colorSetColorTable(COLORTABLE *colTab, short force)
{
  static COLORTABLE *last = NULL;
  short i;

  if ((colTab != last) || force) {
    for (i=0; i<colors; i++) {
      if (TSTBIT(colTab->used, i)) {
	glob_colortable->red[i] = colTab->red[i];
	glob_colortable->green[i] = colTab->green[i];
	glob_colortable->blue[i] = colTab->blue[i];
      }
    }
    (*glob_screen->putCmap)(glob_colortable);
    last = colTab;
  }
}


/*
 * some window orientated functions
 */

/* colorCopyVirtualTable(): called when creating a new window. increments
 * global counters for color usage for those colors currently in use by
 * parent, but doesn't give the child a private color table yet. that's done
 * later when it turns out to be really necessary.
 */
short colorCopyVirtualTable (WINDOW *win, WINDOW *parent)
{
  short i;
  ulong *cnt;
  uchar *ptr;

  win->colTab = parent->colTab;

  cnt = colorCount;
  ptr = win->colTab->used;
  for (i=0; i<colors; i++) {
    if (TSTBIT(ptr, i)) {
      (*cnt)++;
    }
    cnt++;
  }

  return 0;
}


short colorFreeTable (WINDOW *win)
{
  short i;
  ulong *cnt;
  uchar *ptr;
  COLORTABLE *colTab = win->colTab;

  /* always decrement global counters
   */
  cnt = colorCount;
  ptr = colTab->used;
  for (i=0; i<colors; i++) {
    if (TSTBIT(ptr, i)) {
      (*cnt)--;
    }
    cnt++;
  }

  /* if this was really our table, physically remove it
   */
  if (colTab->win == win) {
    free (colTab->used);
    free (colTab->blue);
    free (colTab->green);
    free (colTab->red);
    free (colTab);
  }

  return 0;
}


short colorAllocColor (WINDOW *win, uchar newRed, uchar newGreen, uchar newBlue)
{
  COLORTABLE *colTab = win->colTab;
  uchar *red, *green, *blue;
  short i, cnt;
  ulong min;

  /* first see if user was stupid and this color already exists in this
   * window.
   */
  cnt = 0;
  red = colTab->red;
  green = colTab->green;
  blue = colTab->blue;
  for (i=0; i<colors; i++) {
    if (TSTBIT(colTab->used, i)) {
      cnt++;
      if ((*red++ == newRed) &&
	  (*green++ == newGreen) &&
	  (*blue++ == newBlue)) {
	/* it does exist
	 */
	return i;
      }
    }
  }

  /* it doesn't exist, but are there any free colors at all?
   */
  if (cnt == colors)
    return -1;

  /* do we've got to install a private colortable?
   */
  if (colTab->win != win)
    makePrivateTable(win);

  /* now we must search the least used color in the global table, but which is
   * still unused in our table.
   */
  cnt = -1;
  red = win->colTab->used;   /* steal this variable */
  min = 0xffffffff;
  for (i=0; i<colors; i++) {
    if (!TSTBIT(red, i) && (colorCount[i] < min)) {
      min = colorCount[i];
      cnt = i;
    }
  }

  /* at this point we can be sure we've found one, therefore no check
   */
  SETBIT(red, cnt);
  colorCount[cnt]++;
  win->colTab->red[cnt] = newRed;
  win->colTab->green[cnt] = newGreen;
  win->colTab->blue[cnt] = newBlue;

  return cnt;
}


short colorFreeColor (WINDOW *win, short color)
{
  if ((color < 0) || (color > colors))
    return -1;

  if (!TSTBIT(win->colTab->used, color))
    return -1;

  /* it may still be that this is only a virtual table, if this was called to
   * free the system default colors directly after creating a new window.
   */
  if (win->colTab->win != win)
    makePrivateTable(win);

  colorCount[color]--;
  CLRBIT(win->colTab->used, color);

  return 0;
}


short colorChangeColor (WINDOW *win, short color,
			uchar red, uchar green, uchar blue)
{
  if ((color < 0) || (color > colors)) {
    return -1;
  }

  if (!TSTBIT(win->colTab->used, color)) {
    return -1;
  }

  /* this can also be the first table modifying call
   */
  if (win->colTab->win != win)
    makePrivateTable (win);

  win->colTab->red[color] = red;
  win->colTab->green[color] = green;
  win->colTab->blue[color] = blue;

  return 0;
}


short colorSetFGColor (WINDOW *win, short color)
{
  if ((color < 0) || (color > colors))
    return -1;

  if (!TSTBIT(win->colTab->used, color))
    return -1;

  win->gc.fgCol = color;
  colorGetMask(color, win->gc.fgColMask);

  return 0;
}


short colorSetBGColor (WINDOW *win, short color)
{
  if ((color < 0) || (color > win->colTab->colors))
    return -1;

  if (!TSTBIT(win->colTab->used, color))
    return -1;

  win->gc.bgCol = color;
  colorGetMask(color, win->gc.bgColMask);

  return 0;
}
