/*
 * server/client_noret.c, part of W
 * (C) 1994,95,96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * these are functions (called by clients) that don't need to return anything
 */

#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#ifdef __MINT__
# include <mintbind.h>
#endif

#include "config.h"
#include "types.h"
#include "pakets.h"
#include "proto.h"
#include "window.h"
#include "rect.h"


/*
 * and here are the items...
 *
 * the first one, client_plot() will serve as a demo of how this works and
 * therefore contains extensive comments, the others just follow this scheme.
 */

void client_plot(CLIENT *cptr,
		 ushort handle,
		 short x0, short y0)
{
  WINDOW *win;
  void (*plot)(BITMAP *, long, long);

  /* first thing is to get the window pointer from the handle
   */
  if (!(win = windowLookup(handle))) {
    return;
  }

  /* you cannot draw to W_CONTAINER windows
   */
  if (win->flags & W_CONTAINER) {
    return;
  }

  plot = (glob_pakettype == PAK_PLOT) ? glob_screen->plot : glob_screen->dplot;

  /* then we must get the effective coordinates to draw to as relative to
   * the window. for now this is merely just adding the size of the border.
   * this does not yet specify if and/or where on the screen the plot will
   * occur.
   */
  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;

  /* set the graphic context for this window. `gc0' is a global variable
   * used by all graphics calls.
   */
  gc0 = &win->gc;

  /* if the window is (at least partially) hidden or not open at all we'll draw
   * into the window bitmap, mark the window as dirty and let a later routine
   * do the job of updating the screen contents. since this is always done via
   * bitblk operations we can save quite a lot of time by doing so only when
   * it's really necessary, say, when all requests from this client currently
   * in the receive buffer are processed and we don't know if anything more
   * will come in time.
   *
   * if the server was compiled with -DREFRESH all drawing stuff will always go
   * to the window bitmap. if the bitmap is always guaranteed to be up to date
   * we'll be able to do an easy refresh of the whole screen.
   */
#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {

    /* window is open and completely visible, so there's no point in using
     * the bitmap. we better draw directly on the screen because that saves
     * double work. therefore we actually MUST disable the mouse here if
     * the pointer interferes with the region in question.
     *
     * If the mouse isn't in the *visible* area of the window, this wouldn't
     * be necessary. However, checking that would take more time...
     */
    if (mouse_rcintersect(win->pos.x0 + x0,
			  win->pos.y0 + y0, 1, 1)) {

      /* this will only hide it if it's not already hidden
       */
      mouse_hide();
    }

    /* set clipping to the work area of the window on screen
     */
    clip0 = &win->work;

    /* plot into screen
     */
    (*plot)(&glob_screen->bm, win->pos.x0 + x0,
			 win->pos.y0 + y0);

    /* mark that the screen is more up-to-date than the bitmap
     */
    win->is_dirty = 1;
  } else
#endif
  {
    /* set clipping to the work area of the bitmap
     */
    clip0 = &win->area[AREA_WORK];

    /* plot into the window bitmap
     */
    (*plot)(&win->bitmap, x0, y0);

    /* if the window is in fact open but hidden we must mark it as dirty, else
     * we don't care about that: since the window isn't on the screen there's
     * no point in updating it's contents.
     */
    if (win->is_open) {

      /* if the section we've just drawn is under the mouse pointer we must
       * disable that. note that strictly speaking this isn't necessary here
       * because we actually didn't yet write anything on the screen, but if we
       * do it here the later routine to do the update mustn't deal with it.
       */
      if (mouse_rcintersect(win->pos.x0 + x0,
			    win->pos.y0 + y0, 1, 1)) {

	/* this will only hide it if it's not already hidden
	 */
	mouse_hide();
      }

      /* mark this part of the window as dirty
       */
      rectUpdateDirty(&win->dirty, x0, y0, 1, 1);
    }
  }
}


void client_settitle(CLIENT *cptr,
		     ushort handle,
		     char *s)
{
  WINDOW *win;
  static GCONTEXT gc = {M_DRAW, F_NORMAL,
			  &glob_font[TITLEFONT], DefaultPattern,
			  1, 0,
			  {0xffff,0x0,0x0,0x0,0x0,0x0,0x0,0x0},
			  {0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}};

  if (!(win = windowLookup (handle)))
    return;

  if (!(win->flags & W_TITLE))
    return;

  strncpy (win->title, s, MAXTITLE);

  if (win->flags & W_CONTAINER)
    return;

  gc0 = &gc;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if (mouse_rcintersect(win->pos.x0 + win->area[AREA_TITLE].x0,
			  win->pos.y0 + win->area[AREA_TITLE].y0,
			  win->area[AREA_TITLE].w,
			  win->area[AREA_TITLE].h)) {
      mouse_hide();
    }
    clip0 = &win->pos;
    (*glob_screen->dpbox)(&glob_screen->bm,
			  win->pos.x0 + win->area[AREA_TITLE].x0,
			  win->pos.y0 + win->area[AREA_TITLE].y0,
			  win->area[AREA_TITLE].w,
			  win->area[AREA_TITLE].h);
    (*glob_screen->prints)(&glob_screen->bm,
			   win->pos.x0 + win->area[AREA_TITLE].x0,
			   win->pos.y0 + win->area[AREA_TITLE].y0,
			   s);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_TITLE];
    (*glob_screen->dpbox)(&win->bitmap,
			  win->area[AREA_TITLE].x0,
			  win->area[AREA_TITLE].y0,
			  win->area[AREA_TITLE].w,
			  win->area[AREA_TITLE].h);
    (*glob_screen->prints)(&win->bitmap,
			   win->area[AREA_TITLE].x0,
			   win->area[AREA_TITLE].y0,
			   s);
    if (win->is_open) {
      if (mouse_rcintersect(win->pos.x0 + win->area[AREA_TITLE].x0,
			    win->pos.y0 + win->area[AREA_TITLE].y0,
			    win->area[AREA_TITLE].w,
			    win->area[AREA_TITLE].h)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty,
		      win->area[AREA_TITLE].x0,
		      win->area[AREA_TITLE].y0,
		      win->area[AREA_TITLE].w,
		      win->area[AREA_TITLE].h);
    }
  }

  win->has_title = 1;
}


void client_line(CLIENT *cptr,
		 ushort handle,
		 short x0, short y0,
		 short xe, short ye)
{
  WINDOW *win;
  void (*line)(BITMAP *, long, long, long, long);
  short ex0, ey0, ewidth, eheight;

  if (!(win = windowLookup(handle))) {
    return;
  }

  if (win->flags & W_CONTAINER) {
    return;
  }

  line = (glob_pakettype == PAK_LINE) ? glob_screen->line : glob_screen->dline;

  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;
  xe += win->area[AREA_WORK].x0;
  ye += win->area[AREA_WORK].y0;

  if ((ex0 = x0) > xe)
    ex0 = xe;
  ewidth = abs(xe - x0) + 1;
  if ((ey0 = y0) > ye)
    ey0 = ye;
  eheight = abs(ye - y0) + 1;

  gc0 = &win->gc;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if (mouse_rcintersect(win->pos.x0+ex0,
			  win->pos.y0+ey0, ewidth, eheight)) {
      mouse_hide();
    }
    clip0 = &win->work;
    (*line)(&glob_screen->bm,
	    win->pos.x0+x0, win->pos.y0+y0, win->pos.x0+xe, win->pos.y0+ye);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_WORK];
    (*line)(&win->bitmap, x0, y0, xe, ye);
    if (win->is_open) {
      if (mouse_rcintersect(win->pos.x0+ex0,
			    win->pos.y0+ey0, ewidth, eheight)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, ex0, ey0, ewidth, eheight);
    }
  }
}


void client_hline(CLIENT *cptr, ushort handle, short x0, short y0, short xe)
{
  WINDOW *win;
  void (*hline)(BITMAP *, long, long, long);

  if (!(win = windowLookup(handle))) {
    return;
  }

  if (win->flags & W_CONTAINER) {
    return;
  }

  hline = (glob_pakettype == PAK_HLINE) ? glob_screen->hline : glob_screen->dhline;

  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;
  xe += win->area[AREA_WORK].x0;

  gc0 = &win->gc;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if (mouse_rcintersect(win->pos.x0 + x0, win->pos.y0 + y0,
			  xe - x0 + 1, 1)) {
      mouse_hide();
    }
    clip0 = &win->work;
    (*hline)(&glob_screen->bm,
	     win->pos.x0 + x0, win->pos.y0 + y0,
	     win->pos.x0 + xe);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_WORK];
    (*hline)(&win->bitmap, x0, y0, xe);
    if (win->is_open) {
      if (mouse_rcintersect(win->pos.x0 + x0,
			    win->pos.y0 + y0, xe - x0 + 1, 1)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, x0, y0, xe - x0 + 1, 1);
    }
  }
}


void client_vline(CLIENT *cptr, ushort handle, short x0, short y0, short ye)
{
  WINDOW *win;
  void (*vline)(BITMAP *, long, long, long);

  if (!(win = windowLookup(handle))) {
    return;
  }

  if (win->flags & W_CONTAINER) {
    return;
  }

  vline = (glob_pakettype == PAK_VLINE) ? glob_screen->vline : glob_screen->dvline;

  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;
  ye += win->area[AREA_WORK].y0;

  gc0 = &win->gc;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if (mouse_rcintersect(win->pos.x0+x0, win->pos.y0+y0,
			  1, abs(ye-y0)+1)) {
      mouse_hide();
    }
    clip0 = &win->work;
    (*vline)(&glob_screen->bm,
	     win->pos.x0 + x0, win->pos.y0 + y0, win->pos.y0 + ye);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_WORK];
    (*vline)(&win->bitmap, x0, y0, ye);
    if (win->is_open) {
      if (mouse_rcintersect(win->pos.x0+x0,
			    win->pos.y0+y0, 1, abs(ye-y0)+1)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, x0, y0, 1, ye - y0 + 1);
    }
  }
}


void client_box(CLIENT *cptr, ushort handle, short x0, short y0, short width,
		short height)
{
  WINDOW *win;
  void (*box)(BITMAP *, long, long, long, long);

  if (!(win = windowLookup(handle))) {
    return;
  }

  if (win->flags & W_CONTAINER) {
    return;
  }

  switch (glob_pakettype) {
    case PAK_BOX:
      box = glob_screen->box;
      break;
    case PAK_PBOX:
      box = glob_screen->pbox;
      break;
    case PAK_DBOX:
      box = glob_screen->dbox;
      break;
    default:   /* PAK_DPBOX */
      box = glob_screen->dpbox;
  }

  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;

  if (width < 0) {
    x0 += width;
    width = -width;
  }
  if (height < 0) {
    y0 += height;
    height = -height;
  }

  gc0 = &win->gc;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if (mouse_rcintersect(win->pos.x0+x0, win->pos.y0+y0,
			  width, height)) {
      mouse_hide();
    }
    clip0 = &win->work;
    (*box)(&glob_screen->bm,
	   win->pos.x0+x0, win->pos.y0+y0, width, height);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_WORK];
    (*box)(&win->bitmap, x0, y0, width, height);
    if (win->is_open) {
      if (mouse_rcintersect(win->pos.x0+x0,
			    win->pos.y0+y0, width, height)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, x0, y0, width, height);
    }
  }
}


void client_bitblk (CLIENT *cptr, ushort handle, short x0, short y0,
		    short width, short height, short x1, short y1)
{
  WINDOW *win;

  if (!(win = windowLookup(handle)))
    return;

  if (win->flags & W_CONTAINER)
    return;

  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;
  x1 += win->area[AREA_WORK].x0;
  y1 += win->area[AREA_WORK].y0;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if ((mouse_rcintersect(win->pos.x0+x0,
			   win->pos.y0+y0, width, height)) ||
	(mouse_rcintersect(win->pos.x0+x1,
			   win->pos.y0+y1, width, height))) {
      mouse_hide();
    }
    clip0 = &win->work;
    clip1 = &win->work;
    (*glob_screen->bitblk)(&glob_screen->bm, win->pos.x0+x0,
			   win->pos.y0+y0, width, height,
			   &glob_screen->bm, win->pos.x0+x1,
			   win->pos.y0+y1);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_WORK];
    clip1 = &win->area[AREA_WORK];
    (*glob_screen->bitblk)(&win->bitmap, x0, y0, width, height,
			   &win->bitmap, x1, y1);
    if (win->is_open) {
      if ((mouse_rcintersect(win->pos.x0+x0,
			     win->pos.y0+y0, width, height)) ||
	  (mouse_rcintersect(win->pos.x0+x1,
			     win->pos.y0+y1, width, height))) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, x1, y1, width, height);
    }
  }
}


void client_bitblk2(CLIENT *cptr, ushort shandle, short x0, short y0,
		    short width, short height, ushort dhandle, short x1,
		    short y1)
{
  WINDOW *swin, *dwin;

  if (!(swin = windowLookup(shandle)) || !(dwin = windowLookup(dhandle))) {
    return;
  }

  if ((swin->flags & W_CONTAINER) || (dwin->flags & W_CONTAINER)) {
    return;
  }

  x0 += swin->area[AREA_WORK].x0;
  y0 += swin->area[AREA_WORK].y0;
  x1 += dwin->area[AREA_WORK].x0;
  y1 += dwin->area[AREA_WORK].y0;

#ifndef REFRESH
  if (dwin->is_open && !dwin->is_hidden) {
    /* destination: screen */
    if (!swin->is_open || swin->is_hidden) {
      /* source: bitmap */
      if (mouse_rcintersect(dwin->pos.x0+x1,
			    dwin->pos.y0+y1, width, height)) {
	mouse_hide();
      }
      clip0 = &swin->area[AREA_WORK];
      clip1 = &dwin->work;
      (*glob_screen->bitblk)(&swin->bitmap, x0, y0, width,
			     height, &glob_screen->bm,
			     dwin->pos.x0+x1,
			     dwin->pos.y0+y1);
    } else {
      /* source: screen */
      if ((mouse_rcintersect(swin->pos.x0+x0,
			     swin->pos.y0+y0, width, height)) ||
	  (mouse_rcintersect(dwin->pos.x0+x1,
			     dwin->pos.y0+y1, width, height))) {
	mouse_hide();
      }
      clip0 = &swin->work;
      clip1 = &dwin->work;
      (*glob_screen->bitblk)(&glob_screen->bm, swin->pos.x0+x0,
			     swin->pos.y0+y0, width, height,
			     &glob_screen->bm, dwin->pos.x0+x1,
			     dwin->pos.y0+y1);
    }
    dwin->is_dirty = 1;
  } else
#endif
  {
    /* destination: bitmap */
#ifndef REFRESH
    if (swin->is_open && !swin->is_hidden) {
      /* source: screen */
      if (mouse_rcintersect(swin->pos.x0 + x0,
			    swin->pos.y0 + y0, width, height)) {
	mouse_hide();
      }
      clip0 = &swin->work;
      clip1 = &dwin->area[AREA_WORK];
      (*glob_screen->bitblk)(&glob_screen->bm, swin->pos.x0+x0,
			     swin->pos.y0+y0, width, height,
			     &dwin->bitmap, x1, y1);
    } else
#endif
    {
      /* source: bitmap */
      clip0 = &swin->area[AREA_WORK];
      clip1 = &dwin->area[AREA_WORK];
      (*glob_screen->bitblk)(&swin->bitmap, x0, y0, width,
			     height, &dwin->bitmap, x1, y1);
    }
    if (dwin->is_open) {
      if (mouse_rcintersect(dwin->pos.x0+x0,
			    dwin->pos.y0+y0, width, height)) {
	mouse_hide();
      }
      rectUpdateDirty(&dwin->dirty, x1, y1, width, height);
    }
  }
}


void client_vscroll(CLIENT *cptr, ushort handle, short x0, short y0,
		    short width, short height, short y1)
{
  WINDOW *win;

  if (!(win = windowLookup(handle))) {
    return;
  }

  if (win->flags & W_CONTAINER) {
    return;
  }

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if (mouse_rcintersect(win->work.x0+x0, win->work.y0+y0,
			  width, height) ||
	mouse_rcintersect(win->work.x0+x0, win->work.y0+y1,
			  width, height)) {
      mouse_hide();
    }
    clip0 = &win->work;
    (*glob_screen->scroll)(&glob_screen->bm, win->work.x0+x0,
			   win->work.y0+y0, width, height,
			   win->work.y0+y1);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_WORK];
    (*glob_screen->scroll)(&win->bitmap, win->area[AREA_WORK].x0 + x0,
			   win->area[AREA_WORK].y0 + y0, width,
			   height, win->area[AREA_WORK].y0 + y1);
    if (win->is_open) {
      if (mouse_rcintersect(win->work.x0+x0,
			    win->work.y0+y0, width, height) ||
	  mouse_rcintersect(win->work.x0+x0,
			    win->work.y0+y1, width, height)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, x0, y1, width, height);
    }
  }
}


void client_prints(CLIENT *cptr, ushort handle, short x0, short y0, uchar *s)
{
  WINDOW *win;
  FONT *f;
  int len;

  if (!(win = windowLookup(handle))) {
    return;
  }

  if (win->flags & W_CONTAINER) {
    return;
  }

  if (!(f = win->gc.font)) {
    return;
  }

  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;

  len = fontStrLen(f, s);
  gc0 = &win->gc;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if (mouse_rcintersect(win->pos.x0+x0, win->pos.y0+y0,
			  len, f->hdr.height)) {
      mouse_hide();
    }
    clip0 = &win->work;
    (*glob_screen->prints)(&glob_screen->bm, win->pos.x0+x0,
			   win->pos.y0+y0, s);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_WORK];
    (*glob_screen->prints)(&win->bitmap, x0, y0, s);
    if (win->is_open) {
      if (mouse_rcintersect(win->pos.x0+x0,
			    win->pos.y0+y0, len, f->hdr.height)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, x0, y0,
		     len, f->hdr.height);
    }
  }
}


void client_circle(CLIENT *cptr, ushort handle, short x0, short y0, short r)
{
  WINDOW *win;
  void (*circ)(BITMAP *, long, long, long);
  long size = (r << 1) + 1;

  if (r < 0) {
    return;
  }

  if (!(win = windowLookup(handle))) {
    return;
  }

  if (win->flags & W_CONTAINER) {
    return;
  }

  switch (glob_pakettype) {
    case PAK_CIRCLE:
      circ = glob_screen->circ;
      break;
    case PAK_PCIRCLE:
      circ = glob_screen->pcirc;
      break;
    case PAK_DCIRCLE:
      circ = glob_screen->dcirc;
      break;
    default:   /* PAK_DPCIRCLE */
      circ = glob_screen->dpcirc;
  }

  x0 += win->area[AREA_WORK].x0;
  y0 += win->area[AREA_WORK].y0;
  gc0 = &win->gc;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    if (mouse_rcintersect(win->pos.x0+x0-r,
			  win->pos.y0+y0-r, size, size)) {
      mouse_hide();
    }
    clip0 = &win->work;
    (*circ)(&glob_screen->bm, win->pos.x0 + x0,
	    win->pos.y0 + y0, r);
    win->is_dirty = 1;
  } else
#endif
  {
    clip0 = &win->area[AREA_WORK];
    (*circ)(&win->bitmap, x0, y0, r);
    if (win->is_open) {
      if (mouse_rcintersect(win->pos.x0+x0-r,
			    win->pos.y0+y0-r, size, size)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, x0-r, y0-r, size, size);
    }
  }
}


void client_poly(CLIENT *cptr, ushort handle, short numpoints, short *points)
{
  static long lpoints[MAXPOLYPOINTS<<1];
  long *lptr = lpoints;
  short x, y, xmin, ymin, xmax, ymax, wx0, wy0, count = numpoints;
  WINDOW *win;
  void (*poly)(BITMAP *, long, long *);

  if ((numpoints < 3) || (numpoints > MAXPOLYPOINTS)) {
    return;
  }

  if (!(win = windowLookup(handle))) {
    return;
  }

  if (win->flags & W_CONTAINER) {
    return;
  }

  switch (glob_pakettype) {
    case PAK_POLY:
      poly = glob_screen->poly;
      break;
    case PAK_PPOLY:
      poly = glob_screen->ppoly;
      break;
    case PAK_DPOLY:
      poly = glob_screen->dpoly;
      break;
    default:   /* PAK_DPPOLY */
      poly = glob_screen->dppoly;
  }

  wx0 = win->area[AREA_WORK].x0;
  wy0 = win->area[AREA_WORK].y0;
  xmin = 32767;
  ymin = 32767;
  xmax = -32768;
  ymax = -32768;

  gc0 = &win->gc;

#ifndef REFRESH
  if (win->is_open && !win->is_hidden) {
    wx0 += win->pos.x0;
    wy0 += win->pos.y0;
    while (count--) {
      /*
       * ntohs evaluates it's argument more than once
       */
      x = *points++;
      y = *points++;
      if ((x = ntohs (x)) < xmin) xmin = x;
      if (x > xmax) xmax = x;
      if ((y = ntohs (y)) < ymin) ymin = y;
      if (y > ymax) ymax = y;
      *lptr++ = x + wx0;
      *lptr++ = y + wy0;
    }
    xmax = xmax - xmin + 1;   /* this really is 'width' */
    ymax = ymax - ymin + 1;
    if (mouse_rcintersect(wx0 + xmin, wy0 + ymin, xmax, ymax)) {
      mouse_hide();
    }
    clip0 = &win->work;
    (*poly)(&glob_screen->bm, numpoints, lpoints);
    win->is_dirty = 1;
  } else
#endif
  {
    while (count--) {
      x = *points++;
      y = *points++;
      if ((x = ntohs(x)) < xmin) xmin = x;
      if (x > xmax) xmax = x;
      if ((y = ntohs(y)) < ymin) ymin = y;
      if (y > ymax) ymax = y;
      *lptr++ = x + wx0;
      *lptr++ = y + wy0;
    }
    xmax = xmax - xmin + 1;   /* this really is 'width' */
    ymax = ymax - ymin + 1;
    clip0 = &win->area[AREA_WORK];
    (*poly)(&win->bitmap, numpoints, lpoints);
    if (win->is_open) {
      if (mouse_rcintersect(win->work.x0 + xmin,
			    win->work.y0 + ymin, xmax, ymax)) {
	mouse_hide();
      }
      rectUpdateDirty(&win->dirty, wx0 + xmin, wy0 + ymin, xmax, ymax);
    }
  }
}


void client_beep(CLIENT *cptr)
{
#ifdef __MINT__
  Bconout(2, 7);
#else
  fprintf(stdout, "\007");
  fflush(stdout);
#endif
}


void client_setmode(CLIENT *cptr, ushort handle, short mode)
{
  WINDOW *win;

  if ((win = windowLookup(handle))) {
    win->gc.drawmode = mode;
  }
}


void client_setfont(CLIENT *cptr, ushort handle, short fonthandle)
{
  WINDOW *win;

  if ((fonthandle < 0) || (fonthandle >=MAXCLIENTFONTS)) {
    return;
  }

  if ((win = windowLookup(handle))) {
    if (cptr->font[fonthandle]) {
      win->gc.font = cptr->font[fonthandle];
    }
  }
}


void client_settextstyle(CLIENT *cptr, ushort handle, short flags)
{
  WINDOW *win;

  if ((win = windowLookup(handle)))
    win->gc.textstyle = flags & F_STYLEMASK;
}


/* Manufacture 16x16 pattern according to the pattern id. */

void client_setpattern(CLIENT *cptr, ushort handle, ushort id)
{
  static ushort DM64[8][8] = {			/* courtesy of Kay... */
    {  0, 32,  8, 40,  2, 34, 10, 42 },
    { 48, 16, 56, 24, 50, 18, 58, 26 },
    { 12, 44,  4, 36, 14, 46,  6, 38 },
    { 60, 28, 52, 20, 62, 30, 54, 22 },
    {  3, 35, 11, 43,  1, 33,  9, 41 },
    { 51, 19, 59, 27, 49, 17, 57, 25 },
    { 15, 47,  7, 39, 13, 45,  5, 37 },
    { 63, 31, 55, 23, 61, 29, 53, 21 }
  };
  WINDOW *win;
  ushort patt, *patbuf;
  ushort *intensity;
  int i;

  if (!(win = windowLookup(handle)))
    return;

  patbuf = win->patbuf;

  /* grayscale patterns */
  if(id <= MAX_GRAYSCALES)
  {
    for (i=0; i < 8; i++)
    {
      patt = 0;
      intensity = DM64[i];
      /* matrix values [0-63] -> id 0=black, 64=white */
      if(id <= *intensity++) patt  = 0x0101;
      if(id <= *intensity++) patt |= 0x0202;
      if(id <= *intensity++) patt |= 0x0404;
      if(id <= *intensity++) patt |= 0x0808;
      if(id <= *intensity++) patt |= 0x1010;
      if(id <= *intensity++) patt |= 0x2020;
      if(id <= *intensity++) patt |= 0x4040;
      if(id <= *intensity++) patt |= 0x8080;
      patbuf[8] = patt;
      *patbuf++ = patt;
    }
    return;
  }

  /* line patterns (rolled) */
  if(id >= 0xff)
  {
    for(i = 0; i < 16; i++)
    {
      *patbuf++ = id;
      id = ((id & 1) << 15) | (id >> 1);
    }
    return;
  }

  /* misc patterns (eg. images) */
  switch(id)
  {
    case W_PATTERN:
      memcpy(win->patbuf, BackGround, sizeof(win->patbuf));
      break;
    default:
      memcpy(win->patbuf, DefaultPattern, sizeof(win->patbuf));
  }
}


/*
 * some color functions
 */

short clientFreeColor(CLIENT *cptr, ushort handle, short color)
{
  WINDOW *win;

  if (!(win = windowLookup (handle)))
    return -1;

  colorFreeColor (win, color);

  return 0;
}


void clientChangeColor(CLIENT *cptr, ushort handle, short color,
		       ushort red, ushort green, ushort blue)
{
  WINDOW *win;

  if (!(win = windowLookup (handle)))
    return;

  colorChangeColor (win, color, red & 255, green & 255, blue & 255);

  if (win == glob_activewindow)
    colorSetColorTable (win->colTab, 1);
}


void clientSetColor (CLIENT *cptr, ushort handle, short color)
{
  WINDOW *win;

  if (!(win = windowLookup (handle)))
    return;

  switch (glob_pakettype) {
    case PAK_SETFGCOL:
      colorSetFGColor (win, color);
      break;
    case PAK_SETBGCOL:
      colorSetBGColor (win, color);
  }
}
