/*
 * lib/socket.c, part of W
 * (C) 1994,95,96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * all low-level functions and global variables
 *
 * Changes:
 * - WEVENT buffer will now discard oldest events (instead of newest ones)
 *   in case of an overflow.
 * - _remove_events() will now move events in buffer only when needed.
 *   ++eero, 10/96
 * - Moved _spaces() into init.c.
 *
 * FIXME: needs a more dynamic event buffering scheme
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <stdlib.h>
#include <netdb.h>
#include <string.h>
#include "Wlib.h"
#include "proto.h"

#define	BUFSIZE		1024
#define	MAXWEVENTS	32


/*
 * some public global variables
 */

WWIN *WROOT = NULL;


/*
 * some private variables...
 */

static int _sh = -1;
static long _optr = 0;
static char _obuf[BUFSIZE];
static short _wevents, _wev_write, _wev_read;
static WEVENT _wevent[MAXWEVENTS];
int _trace = 0, _traceIndent = 0;
WSERVER _wserver;


/*
 *	some private functions...
 */

char *_check_window(WWIN *ptr)
{
  if (!ptr)
    return "no window";

  if (ptr->magic != MAGIC_W)
    return "not W window (anymore?)";

  return NULL;
}


void _wexit(void)
{
  if (_sh >= 0) {
    close(_sh);
  }
}

long _initialize()
{
  char	*wdisplay;

  if (_sh >= 0) {
    fprintf(stderr, "Wlib: server is already connected.\r\n");
    return -1;
  }

  if ((wdisplay = getenv("WDISPLAY"))) {

    struct hostent *he;
    struct sockaddr_in raddr;

    if (!(he = gethostbyname(wdisplay))) {
      fprintf(stderr,
	      "Wlib: can't resolve adress for server `%s'.\r\n", wdisplay);
      return -1;
    }

    if ((_sh = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
      fprintf(stderr, "Wlib: can't create AF_INET socket.\r\n");
      return -1;
    }

    raddr.sin_family = AF_INET;
    raddr.sin_addr = *(struct in_addr *)he->h_addr_list[0];
    raddr.sin_port = htons(SERVERPORT);
    if (connect(_sh, (struct sockaddr *)&raddr, sizeof(struct sockaddr))) {
      fprintf(stderr, "Wlib: can't connect to server `%s' (%s).\r\n",
	      wdisplay, inet_ntoa(raddr.sin_addr));
      close(_sh);
      return -1;
    }

  } else {

    struct sockaddr raddr;

    if ((_sh = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
      fprintf(stderr, "Wlib: can't create AF_UNIX socket.\r\n");
      return -1;
    }

    raddr.sa_family = AF_UNIX;
    strcpy(raddr.sa_data, "/tmp/wserver");
    if (connect(_sh, &raddr, sizeof(struct sockaddr))) {
      fprintf(stderr, "Wlib: can't connect to local server.\r\n");
      close(_sh);
      return -1;
    }
  }

  _wevents = _wev_write = _wev_read = 0;

  return 0;
}


void w_flush(void)
{
  long	ret;
  char	*ptr = _obuf;

  /*
   * shit, on the one hand side this is TCP and therefore
   * guaranteed to only return if either all data is written
   * or the connection is broken, but on the other hand side
   * this involves an OS call which may be interrupted by a
   * signal, so what can we do here?
   */

  while (_optr > 0) {

    if ((ret = write(_sh, ptr, _optr)) < 0) {
      perror("Wlib: _send_paket() returned");
      exit(-99);
    }

    ptr += ret;
    _optr -= ret;
  }

  /* _optr should now be zero */
}


/*
 * _send_paket(): this version will send only complete pakets. this
 * requires that each paket is smaller than our buffer, but that
 * assumption was already made earlier at other places, so what the heck.
 */

void _send_paket (PAKET *pptr)
{
  short	size = pptr->len;

  pptr->len = htons(pptr->len);
  pptr->type = htons(pptr->type);

  if (_optr + size >= BUFSIZE)
    w_flush();

  memcpy(_obuf + _optr, pptr, size);

  _optr += size;
}


/*
 * some stuff dealing with buffered WEVENTS, the push routine
 * also deals with net2host byte order problems and converting
 * between the internal and external event types
 *
 * Buffer is circular and will overwrite older events in case of
 * an overflow.
 */

static WEVENT *_wevent_pop(void)
{
  WEVENT *ptr = NULL;

  if (_wevents) {
    ptr = &_wevent[_wev_read];
    if (++_wev_read == MAXWEVENTS) {
      _wev_read = 0;
    }
    _wevents--;
  }

  return ptr;
}


static void _event_push(WEVENT *ptr)
{
  _wevent[_wev_write].win = ptr->win;
  _wevent[_wev_write].type = ntohs(ptr->type);
  _wevent[_wev_write].x = ntohs(ptr->x);
  _wevent[_wev_write].y = ntohs(ptr->y);
  _wevent[_wev_write].w = ntohs(ptr->w);
  _wevent[_wev_write].h = ntohs(ptr->h);
  _wevent[_wev_write].key = ntohl(ptr->key);
  if (++_wev_write == MAXWEVENTS) {
    _wev_write = 0;
  }

  if (_wevents >= MAXWEVENTS) {
#ifdef SIGTTOU
    void (*old)();
    /* Better to allow background write as overflow isn't that uncommon (eg.
     * if program has been suspended while user gave input to it's window),
     * and stopping the program might confuse the user...
     */
    old = signal(SIGTTOU, SIG_IGN);
    fprintf(stderr, "Wlib: event buffer overflow.\r\n");
    signal(SIGTTOU, old);
#else
    fprintf(stderr, "Wlib: event buffer overflow.\r\n");
#endif
    _wevent_pop();
  }
  _wevents++;
}


/*
 * routine for reading the socket and buffering event pakets. if called with
 * wantEvent==0 it means caller is looking for a real paket. in this case it
 * returns when the first complete paket is received. the fd_sets should be
 * NULL to not disturb selecting the socket. if called with wantEvent==1 it
 * means caller wants events. in this case it returns when either an event
 * is/becomes available or the timeout or the supplied fd_sets suggest to
 * return.
 */

static PAKET *scanSocket (fd_set *rfd, fd_set *wfd, fd_set *xfd, long timeout,
			  int *numReady, int wantEvent)
{
  static long inbuf = 0;
  static char ibuf[BUFSIZE];
  static PAKET paket;
  long this;
  PAKET *pptr, *ret = NULL;
  short len, type, idx;
  int forceBreak = 0;
  fd_set myRfd;
  struct timeval tv, *tvp;
  int fds = 0;

  while (42) {

    /*
     * analyze input buffer
     */

    idx = 0;
    while (inbuf >= 4) {

      /* check if there's a complete paket
       */
      pptr = (PAKET *)&ibuf[idx];
      len = ntohs(pptr->len);
      type = ntohs(pptr->type);
      if (inbuf < len)
	break;

      inbuf -= len;
      idx += len;

      if (type == PAK_EVENT) {
	_event_push ((WEVENT *)&((EVENTP *)pptr)->event);
	continue;
      }

      /* we've got a non-event paket
       */
      memcpy (&paket, pptr, len);
      paket.len = len;
      paket.type = type;
      ret = &paket;
      break;
    }

    if (idx) {
#ifdef sun
      bcopy (&ibuf[idx], ibuf, inbuf);
#else
      memmove (ibuf, &ibuf[idx], inbuf);
#endif
    }

    if (ret)
      /* this is either exactly what we've been waiting for, or a severe error
       * in communication, so always return and let the caller process errors
       */
      break;

    if (forceBreak)
      /* we're in the second run through this loop after the first one looked
       * for an event but didn't find one without reading the socket. either
       * there was one in the newly received data or not - we return in any case
       */
      break;

    /*
     * prepare for select()
     */

    if (!wantEvent)
      timeout = -1;   /* use blocking select when waiting for a paket */
    else if (_wevents)
      timeout = 0;   /* can return immediately */

    if (!wantEvent || !_wevents) {
      if (!rfd) {
	FD_ZERO (&myRfd);
	rfd = &myRfd;
      }
      FD_SET (_sh, rfd);
    }

    tvp = &tv;
    if (timeout < 0) {
#ifdef __MINT__
      tv.tv_sec = -1;
      tv.tv_usec = 0;
#else
      tvp = NULL;
#endif
    } else {
      tv.tv_sec = timeout / 1000;
      tv.tv_usec = 1000 * (timeout - 1000 * tv.tv_sec);
    }

    if ((fds = select (FD_SETSIZE, rfd, wfd, xfd, tvp)) < 0) {
      fprintf (stderr, "Wlib: select() failed in w_queryevent().\r\n");
      fds = 0;
    }

    /*
     * refill input buffer from socket, if possible
     */

    if (rfd && FD_ISSET (_sh, rfd)) {
      fds--;
      if ((this = read (_sh, ibuf + inbuf, BUFSIZE - inbuf)) < 1) {
	fprintf (stderr, "Wlib: connection broken, sending SIGPIPE to my own.\r\n");
	kill (getpid(), SIGPIPE);
	/* not reached?
	 */
	fprintf (stderr, "Wlib: connection broken, how did we get here? DIE!!\r\n");
	_exit (-999);
      }
      inbuf += this;
    }

    if (wantEvent && _wevents)
      /* we wanted an event and got one without reading the socket
       */
      break;

    if (!wantEvent)
      /* we wanted a paket, so restart the loop to analyze the newly received
       * data. this will continue until a complete paket is found.
       */
      continue;

    /* we wanted an event but there wasn't one available without reading the
     * socket. rerun the loop to scan for newly received events, but ensure
     * that we don't actually select again. if there still is no event then
     * we can return at least the selected filehandles.
     */
    forceBreak = 1;
  }

  if (numReady)
    *numReady = fds;

  if (!fds) {
    /* we might not have used select() so have to clear these */
    if(rfd)
      FD_ZERO(rfd);
    if(wfd)
      FD_ZERO(wfd);
    if(xfd)
      FD_ZERO(xfd);
  }

  return ret;
}


/* Called by w_delete() to remove buffered events for window in question
 */
void _remove_events (WWIN *win)
{
  short nr, nw;

  if(!_wevents)
    return;

  while (_wevent[_wev_read].win == win) {
    if (++_wev_read == MAXWEVENTS) {	/* skip events */
      _wev_read = 0;
    }
    _wevents--;
    if (!_wevents) {			/* no events? */
      return;
    }
  }

  /* first non-removable event */
  nr = _wev_read;			/* at least one event to preserve */
  do {
    if (++nr == MAXWEVENTS) {
      nr = 0;
    }
    if (nr == _wev_write) {		/* all preserved? */
      return;
    }
  } while (_wevent[nr].win != win);

  /* first removable event after it */
  nw = nr;				/* first empty place holder */
  for(;;) {
    if (++nr == MAXWEVENTS) {
      nr = 0;
    }
    if (nr == _wev_write) {
      break;
    }
    if (_wevent[nr].win != win) {	/* move valid event? */
      _wevent[nw] = _wevent[nr];
      if (++nw == MAXWEVENTS) {
	nw = 0;
      }
      _wevents--;
    }
  }
  /* first empty event place */
  _wev_write = nw;
}


/*
 * next are two routines for listening on the socket: one for waiting for
 * pakets (return codes) and one for waiting for events
 */

PAKET *_wait4paket (short type)
{
  PAKET *pptr;

  w_flush ();

  if (!(pptr = scanSocket (NULL, NULL, NULL, -1, NULL, 0))) {
    fprintf (stderr, "Wlib: socket communication garbled, aborting\r\n");
    fprintf (stderr, "      expected 0x%04x paket, received NULL\r\n", type);
    w_exit ();
    _exit (-1);
  }

  if (type != pptr->type) {
    fprintf (stderr, "Wlib: socket communication garbled, aborting\r\n");
    fprintf (stderr, "      expected 0x%04x paket, received 0x%04x\r\n",
	     type, pptr->type);
    w_exit ();
    _exit (-1);
  }

  return pptr;
}

WEVENT *w_queryevent (fd_set *rfd,
		      fd_set *wfd,
		      fd_set *xfd,
		      long timeout)
{
  int *numReady = NULL;		/* ATM not used */
  PAKET *pptr;

  w_flush ();

  if ((pptr = scanSocket (rfd, wfd, xfd, timeout, numReady, 1))) {
    fprintf (stderr, "Wlib: socket communication garbled, aborting\r\n");
    fprintf (stderr, "      expected event paket, received 0x%04x\r\n",
	     pptr->type);
    w_exit ();
    _exit (-1);
  }

  return _wevent_pop ();
}
