/*
 * lib/font.c, part of Wlib, part of W
 * (C) 94-07/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * CHANGES
 * 11/97 ++eero:
 * - new fontscheme where we'll request font by attributes instead of it's
 *   filename.
 * - font structure will be freed and unlinked regardless of what server
 *   returns in w_unloadfont().
 * - w_fonttype() which will convert fontname to corresponding attributes.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "Wlib.h"
#include "proto.h"


/*
 * the root for the fonts list
 */

static WFONT *WFonts = NULL;


/*
 *
 */

WFONT *w_loadfont(char *family, short size, ushort stylemask)
{
  WFONT *fptr;
  LOADFONTP paket;
  LFONTRETP *lfontretp;
  char *checkname;
  short checksize;

  TRACESTART();


  if(family && *family) {
    checkname = family;
  } else {
    checkname = _wserver.fname;
  }
  if (size) {
    checksize = size;
  } else {
    checksize = _wserver.fsize;
  }

  TRACEPRINT(("w_loadfont(%s,%d,%d)... ", checkname, checksize, stylemask));

  /* first try to see if the font is already loaded */

  fptr = WFonts;
  while (fptr) {
    if (!strcmp(fptr->family, checkname) &&
        fptr->height == checksize && fptr->styles == stylemask) {

      /* it is, so return this one */
      fptr->used++;
      TRACEPRINT(("0x%p (cached)\n", fptr));
      TRACEEND();
      return fptr;
    }
    fptr = fptr->next;
  }

  /* it isn't, so load it
   */
  if (!(fptr = calloc(1, sizeof(WFONT)))) {
    TRACEPRINT(("NULL (out of memory)\n"));
    TRACEEND();
    return NULL;
  }

  paket.len = sizeof(LOADFONTP);
  paket.type = PAK_LOADFONT;
  paket.size = htons(size);
  paket.styles = htons(stylemask);
  if (family && *family) {
    strncpy(paket.family, family, sizeof(paket.family));
    paket.family[sizeof(paket.family)-1] = 0;
  } else {
    paket.family[0] = 0;
  }
  _send_paket((PAKET *)&paket);

  lfontretp = (LFONTRETP *)_wait4paket(PAK_LFONTRET);

  if ((fptr->handle = ntohs(lfontretp->handle)) < 0) {
    TRACEPRINT(("NULL\n"));
    TRACEEND();
    free(fptr);
    return NULL;
  }

  fptr->magic = MAGIC_F;

  /* checksize would be better as font existence is checked against
   * that above...
   */
  fptr->height = ntohs(lfontretp->height);
  fptr->flags  = ntohs(lfontretp->flags);
  fptr->styles = ntohs(lfontretp->styles);
  fptr->top    = ntohs(lfontretp->top);
  fptr->bottom = ntohs(lfontretp->bottom);
  fptr->left   = ntohs(lfontretp->left);
  fptr->right  = ntohs(lfontretp->right);
  fptr->maxwidth = ntohs(lfontretp->maxwidth);
  fptr->baseline = ntohs(lfontretp->baseline);
  fptr->family = strdup(lfontretp->family);
  memcpy(fptr->widths, lfontretp->widths, 256);

  fptr->used = 1;
  fptr->prev = NULL;
  if ((fptr->next = WFonts))
    WFonts->prev = fptr;
  WFonts = fptr;

  TRACEPRINT(("w_loadfont(%s,%d,%d) -> %p\n", fptr->family, \
	     fptr->height, fptr->styles, fptr));
  TRACEEND();

  return fptr;
}


short w_unloadfont(WFONT *font)
{
  UNLOADFONTP paket;
  short ret;

  TRACESTART();
  TRACEPRINT(("w_unloadfont(%p)... ", font));

  if (!font) {
    TRACEPRINT(("zero pointer\n"));
    TRACEEND();
    return -1;
  }

  if (font->magic != MAGIC_F) {
    TRACEPRINT(("not a WFONT pointer\n"));
    TRACEEND();
    return -1;
  }

  /* first see if we really need to unload it
   */
  if (--font->used > 0) {
    /* no, it's still needed somewhere
     */
    TRACEPRINT(("0 (ok, still used)\n"));
    TRACEEND();

    return 0;
  }

  paket.len = sizeof(UNLOADFONTP);
  paket.type = PAK_UNLOADFONT;
  paket.fonthandle = htons(font->handle);
  _send_paket((PAKET *)&paket);
  ret = ntohs(((SRETP *)_wait4paket(PAK_SRET))->ret);

  /* we should unlink this in any case ++eero */
  font->magic = 0;
  if (font->prev)
    font->prev->next = font->next;
  else
    WFonts = font->next;
  if (font->next)
    font->next->prev = font->prev;
  free(font->family);
  free(font);

  if (ret) {
    TRACEPRINT(("server unloading failed\n"));
  } else {
    TRACEPRINT(("%i\n", ret));
  }
  TRACEEND();

  return ret;
}


WFONT *w_setfont(WWIN *win, WFONT *font)
{
  char *cptr;
  SFONTP paket;
  WFONT *oldfont;

  TRACESTART();

  if ((cptr = _check_window(win))) {
    TRACEPRINT(("w_setfont(%p,%p) -> %s\n", win, font, cptr));
    TRACEEND();
    return NULL;
  }

  if (!font || font->magic != MAGIC_F) {
    TRACEPRINT(("w_setfont(%p,%p) -> font error\n", win, font));
    TRACEEND();
    return NULL;
  }

  paket.len = sizeof(SFONTP);
  paket.type = PAK_SFONT;
  oldfont = win->font;
  paket.handle = htons(win->handle);
  if (font) {
    paket.fonthandle = htons((win->font = font) -> handle);
  } else {
    paket.fonthandle = htons(-1);
  }
  _send_paket((PAKET *)&paket);
  TRACEPRINT(("w_setfont(%p,%p) -> %p\n", win, font, oldfont));

  TRACEEND();
  return oldfont;
}


/* convert font filename into family name, size and styles */
char *w_fonttype(char *filename, short *size, short *styles)
{
  static char name[MAXFAMILYNAME + 4];
  int idx;

  TRACESTART();

  if(!(filename && size && styles && *filename)) {
    TRACEPRINT(("w_loadfont(%s,%p,%p) -> NULL\n", filename, size, styles));
    TRACEEND();
    return NULL;
  }

  /* remove path */
  idx = strlen(filename);
  while (--idx >= 0 && filename[idx] != '/');
  filename += idx + 1;

  /* remove extension */
  idx = strlen(filename);
  while(--idx > 0 && filename[idx] != '.');
  if (!idx) {
    idx = strlen(filename);
  }

  /* get styles */
  *styles = 0;
  for(;--idx > 0;) {
    switch(filename[idx]) {
       case 'b':
	 *styles |= F_BOLD;
	 break;
       case 'i':
	 *styles |= F_ITALIC;
	 break;
       case 'l':
	 *styles |= F_LIGHT;
	 break;
       case 'r':
	 *styles |= F_REVERSE;
	 break;
       case 'u':
	 *styles |= F_UNDERLINE;
	 break;
       default:
         goto out_of_loop;
    }
  }
out_of_loop:
  strncpy(name, filename, ++idx);
  name[idx] = 0;

  /* get size and the rest is family name */
  while(--idx > 0 && (name[idx] >= '0' && name[idx] <= '9'));
  *size = atoi(&name[++idx]);
  name[idx] = 0;

  TRACEPRINT(("w_loadfont(%s,%p,%p) -> %s (%d,%d)\n", filename, \
	     size, styles, name, *size, *styles));
  TRACEEND();

  return name;
}
