/* 
 * lib/convert.c, part of W
 * (C) 94-04/96 by Torsten Scherer (TeSche)
 * itschere@techfak.uni-bielefeld.de
 *
 * functions to convert, allocate, copy and free BITMAPs.
 * 
 * Direct<->packed routines are moved here from TeSche's earlier block.c,
 * other stuff is mine. ++eero 5-10/97.
 *
 * NOTES:
 *
 * Unitsizes currently used by conversion routines:
 * - packed mono:  4	(M_UNIT define, 32 pixels)
 * - packed color: 2	(16 pixels interleaved on number planes: ATARI_SERVER)
 * - direct-8:     1	(one, but images are still long aligned!!!)
 * - direct-24:    3	(one pixel)
 *
 * mono->color conversion will use color index zero for a cleared bit and
 * index one for a set bit in the monochrome bitmap if the destination
 * bitmap format uses a palette. W server should always assign white and
 * black colors to these indeces.
 *
 * Missing conversions:
 * - PACKEDCOLOR -> PACKEDCOLOR. Only server should use that type so
 *   conversion where only either end is PACKEDCOLOR should be enough.
 * - Conversion from DIRECT24 to another color type.  I assume that
 *   applications want to do that themselves, because it needs quite
 *   complicated (=slow) image / palette handling and then there's the
 *   question of dithering etc...
 *
 * If you'll want PACKEDCOLOR->PACKEDMONO conversion to use dithering,
 * you'll have to convert image first to DIRECT8 format.
 *
 * DIRECT24 images are always FS-dithered to PACKEDMONO before other
 * conversions are done. This is done whole bitmap at the time.
 *
 * If you want to use w_bitmapFunction() yourself, you might do the
 * conversions 'in place' when destination bitmap has less 'planes'
 * than source.
 *
 * CHANGES:
 *
 * - w_convertBitmap() function is now w_convertFunction which returns
 *   pointer to the line-conversion function between the formats of
 *   the two argument bitmaps.
 * - In order to make things byte order neutral, some conversion routines
 *   do things byte at the time instead of longs longs and shorts as
 *   earlier and some use netinet/in.h byte order conversion routines.
 *
 * TODO:
 * - w_convertBitmap() and FS-dithering routines could be on another
 *   object file so that they won't be linked to all w_get/putblock()
 *   using programs. Unfortunately I would then need to do Expand and
 *   Graymap global variables :-(.
 * - Put 'in_place' option back to w_convertBitmap()?
 * - Test with server and client(s) residing on different byte order using
 *   machines!
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include "Wlib.h"
#include "proto.h"
#include "dmatrix.h"

/* set what routines are compiled in */

/* conversion routines understand packed (interleaved) bitmaps */
#define ATARI_SERVER


/* monochrome unit size is long = 4 bytes, 32 pixels */
#define M_UNIT	4


static int Expand = 1;		/* fs-dither contrast expansion flag */
static int Matrixrow = 0;	/* ordered dithering row */
static uchar Grayscale[256];	/* own mapping table */
static uchar *Graymap = Grayscale;


/* Using an own graymap is faster if there are lot of images with the same
 * palette. Graymap is of course needed only for mapped (2-8bit) bitmaps.
 *
 * Animations which update only parts of the output view should at least
 * disable contrast expansion.
 */
void
w_ditherOptions(uchar *graymap, int expand)
{
	if (graymap) {
		/* use this mapping instead of mapping palette every time */
		Graymap = graymap;
	} else {
		Graymap = Grayscale;
	}

	/* do/not use contrast expansion */
	Expand = expand;
}


static uchar *
init_dither(BITMAP *bm)
{
	rgb_t *color;
	uchar *gray;
	ulong value;
	int idx;

	Matrixrow = 0;
	if (Graymap != Grayscale) {
		return Graymap;
	}

	gray = Grayscale;
	if (bm->palette) {
		/* color mapping */
		color = bm->palette;
		for (idx = bm->colors; idx > 0; idx--, color++) {
			value =  (*color).red   * 307UL;
			value += (*color).green * 599UL;
			value += (*color).blue  * 118UL;
			*gray++ = value >> 10;
		}
	} else {
		/* no palette (DIRECT24) -> linear grayscale */
		for (idx = 0; idx < 256; idx++) {
			*gray++ = idx;
		}
	}
	return Grayscale;
}


/* ------------------------------------------------------------
 * FS-dither whole image.
 */

/*
 * FS-dither 8-bit BM_DIRECT8 bitmap `bm' (with any unitsize) into long
 * aligned monochrome BM_PACKEDMONO bitmap `mono_bm' of same size using
 * color index -> grayscale value mapping array.  `mono_bm' can be same as
 * `bm'.
 */
static void
FSditherDirect(BITMAP *bm, BITMAP *mono_bm,
             register uchar *mapping, short *FSerrors, int expand)
{
	short threshold;
	register uchar *color;
	register short mingray, maxgray;
	ulong *mono, result, last, bit;
	int x, y, longs, bits;

	register short cur;		/* current error or pixel value */
	register short delta;
	register short *errorptr;	/* FSerrors at column before current */
	register short belowerr;	/* error for pixel below cur */
	register short bpreverr;	/* error for below/prev col */
	register short bnexterr;	/* error for below/next col */

	if (expand) {
		maxgray = 0;
		mingray = 255;
		for (cur = bm->colors - 1; cur >= 0; cur--) {
			if (mapping[cur] < mingray) {
				mingray = mapping[cur];
			}
			if (mapping[cur] > maxgray) {
				maxgray = mapping[cur];
			}
		}
		maxgray -= mingray;
		if (maxgray < 2) {
			threshold = 1;
		} else {
			threshold = maxgray / 2;
		}
	} else {
		mingray = 0;
		threshold = 127;
		maxgray = 255;
	}

	/* longs in the source - 1 */
	longs = (bm->width + (M_UNIT*8-1)) / (M_UNIT*8) - 1;

	/* bit/pixel counter for the last long */
	last = bm->upl * bm->unitsize - longs * (M_UNIT*8);

	color = bm->data;
	mono = mono_bm->data;
	for (y = bm->height; y > 0; y--) {
		errorptr = FSerrors;
		cur = belowerr = bpreverr = 0;

		for(x = longs; x >= 0; x--) {
			result = 0;
			if (!x) {
				bits = last;
			} else {
				bits = M_UNIT*8;
			}
			/* go through one long of pixels */
			for(bit = 1UL << (M_UNIT*8-1); --bits >= 0; bit >>= 1) {

				/* color + error to range */
				cur = (cur + errorptr[1] + 8) >> 4;
				cur += (mapping[*color++] - mingray);
				if(cur < 0) {
					cur = 0;
				} else {
					if(cur > maxgray) {
						cur = maxgray;
					}
				}
				/* resulting color & error */
				if(cur < threshold) {
					result |= bit;
				} else {
					cur -= maxgray;
				}
				/* propagate error fractions */
				bnexterr = cur;
				delta = cur * 2;
				cur += delta;		/* form error * 3 */
				errorptr[0] = bpreverr + cur;
				cur += delta;		/* form error * 5 */
				bpreverr = belowerr + cur;
				belowerr = bnexterr;
				cur += delta;		/* form error * 7 */
				errorptr++;
			}
			*mono++ = ntohl(result);
		}
		errorptr[0] = bpreverr;		/* unload prev err into array */
	}
}


/* FS-dither BM_DIRECT24 bitmap `bm' to a long aligned monochrome `mono_bm'.
 * `mono_bm' may be same as `bm'.  If DIRECT24 unitsize != 3, you'll have to
 * skip alignment bytes after every row when searching through bitmap for
 * min/maxgray values! If expand is specified, this will go through the
 * bitmap twice, first to determine min and max grayscale and second time
 * to dither.
 */
static void
FSditherTruecolor(BITMAP *bm, BITMAP *mono_bm, short *FSerrors, int expand)
{
	register ulong value;
	register uchar *color;
	register short mingray, maxgray;
	ulong *mono, result, last, bit;
	int x, y, longs, bits;
	short threshold;

	register short cur;		/* current error or pixel value */
	register short delta;
	register short *errorptr;	/* FSerrors at column before current */
	short belowerr;			/* error for pixel below cur */
	short bpreverr;			/* error for below/prev col */
	short bnexterr;			/* error for below/next col */

	if (expand) {
		maxgray = 0;
		mingray = 255;
		color = bm->data;
		for (y = bm->height; y > 0; y--) {
			for (cur = bm->width; cur > 0; cur--) {
				value =  (*color++ * 307UL);
				value += (*color++ * 599UL);
				value += (*color++ * 118UL);
				value >>= 10;
				if (value < mingray) {
					mingray = value;
				}
				if (value > maxgray) {
					maxgray = value;
				}
			}
		}
		maxgray -= mingray;
		if (maxgray < 2) {
			threshold = 1;
		} else {
			threshold = maxgray / 2;
		}
	} else {
		maxgray = 255;
		threshold = 127;
		mingray = 0;
	}

	/* longs in the destination - 1 */
	longs = (bm->width + (M_UNIT*8-1)) / (M_UNIT*8) - 1;

	/* bit/pixel counter for the last long, expects bm->upl to be pixels
	 * (ie. unitsize = pixel `size', ATM three bytes)!
	 */
	last = bm->upl - longs * (M_UNIT*8);

	color = bm->data;
	mono = mono_bm->data;
	for (y = bm->height; y > 0; y--) {
		errorptr = FSerrors;
		cur = belowerr = bpreverr = 0;

		for(x = longs; x >= 0; x--) {
			result = 0;
			if (!x) {
				bits = last;
			} else {
				bits = M_UNIT*8;
			}
			/* go through one long of pixels */
			for(bit = 1UL << (M_UNIT*8-1); --bits >= 0; bit >>= 1) {

				/* color + error to range */
				cur = (cur + errorptr[1] + 8) >> 4;
				value =  (*color++ * 307UL);
				value += (*color++ * 599UL);
				value += (*color++ * 118UL);
				value >>= 10;
				cur += value - mingray;
				if(cur < 0) {
					cur = 0;
				} else {
					if(cur > maxgray) {
						cur = maxgray;
					}
				}
				/* resulting color & error */
				if(cur < threshold) {
					result |= bit;
				} else {
					cur -= maxgray;
				}
				/* propagate error fractions */
				bnexterr = cur;
				delta = cur * 2;
				cur += delta;		/* form error * 3 */
				errorptr[0] = bpreverr + cur;
				cur += delta;		/* form error * 5 */
				bpreverr = belowerr + cur;
				belowerr = bnexterr;
				cur += delta;		/* form error * 7 */
				errorptr++;
			}
			*mono++ = ntohl(result);
		}
		errorptr[0] = bpreverr;	/* unload prev err into array */
	}
}


/* convert DIRECT color bitmap to monochrome.  If `in_place' is set,
 * conversion is done to the source bitmap, otherwise a new one is
 * allocated.  This is intended to be called only from w_convertBitmap().
 */
static BITMAP *
fs_direct2mono(BITMAP *bm, int in_place, char **error)
{
	BITMAP *dest;
	short *FS;

	/* first checks */
	if (bm->type != BM_DIRECT8 && bm->type != BM_DIRECT24) {
		*error = "source bitmap type error";
		return NULL;
	}
	if (bm->type == BM_DIRECT8 && !bm->palette) {
		*error = "bitmap palette missing";
		return NULL;
	}
	if (bm->upl * bm->unitsize < M_UNIT) {
		/* mono takes more space than color... */
		in_place = 0;
	}

	/* then allocations */
	FS = calloc(1, (bm->upl * bm->unitsize + 1) * sizeof(*FS));
	if (!FS) {
		*error = "FS-array alloc failed";
		return NULL;
	}
	if (in_place) {
		dest = bm;
	} else {
		dest = w_allocbm(bm->width, bm->height, BM_PACKEDMONO, 2);
		if (!dest) {
			free(FS);
			*error = "destination bitmap alloc failed";
			return NULL;
		}
	}

	/* do the conversion */
	if (bm->type == BM_DIRECT24) {
		FSditherTruecolor(bm, dest, FS, Expand);
	} else {
		FSditherDirect(bm, dest, init_dither(bm), FS, Expand);
	}
	free(FS);

	/* in-place conversion? */
	if (dest == bm) {
		bm->type = BM_PACKEDMONO;
		bm->upl = (bm->width + (M_UNIT*8-1)) / (M_UNIT*8);
		bm->unitsize = M_UNIT;
		bm->colors = 2;
		bm->planes = 1;
	}
	return dest;
}

/* ------------------------------------------------------------
 * line-by-line conversion functions
 */

/* ordered dither DIRECT8 bitmap to monochrome, short at the time */
static uchar *
odither_direct(BITMAP *src, BITMAP *dst, uchar *line)
{
	int x, shorts, last, bits;
	register const uchar *mat;
	register uchar *gray;
	register ushort bit;
	ushort result, *mono;		/* pattern lenght = bits in short */

	/* shorts in the source - 1 */
	shorts = (src->width + 15) / 16 - 1;

	/* bit/pixel counter for the last short */
	last = src->upl * src->unitsize - shorts * 16;

	gray = line;
	mono = dst->data;
	for(x = shorts; x >= 0; x--) {
		mat = DMatrix[Matrixrow];
		result = 0;
		if (x) {
			bits = 16;
		} else {
			bits = last;
		}
		/* go through one pattern line (short) of pixels */
		for(bit = 1U << 15; --bits >= 0; bit >>= 1) {

			/* color/matrix->black? */
			if(Graymap[*gray++] < *mat++) {
				result |= bit;
			}
		}
		*mono++ = ntohs(result);
	}

	Matrixrow = (Matrixrow + 1) & 15;
	return line + src->unitsize * src->upl;
}


/* DIRECT24 should be RGB-triple-byte aligned, with *no* padding */
static uchar *
odither_truecolor(BITMAP *src, BITMAP *dst, uchar *line)
{
	int x, shorts, last, bits;
	register const uchar *mat;
	register ulong value;
	register uchar *gray;
	register ushort bit;
	ushort result, *mono;		/* pattern lenght = bits in short */

	/* shorts in the source - 1 */
	shorts = (src->width + 15) / 16 - 1;

	/* bit/pixel counter for the last short, expects bm->upl to be
	 * pixels (ie. unitsize = pixel `size', ATM three bytes)!
	 */
	last = src->upl - shorts * 16;

	gray = line;
	mono = dst->data;
	for(x = shorts; x >= 0; x--) {
		mat = DMatrix[Matrixrow];
		result = 0;
		if (x) {
			bits = 16;
		} else {
			bits = last;
		}
		/* go through one pattern line (short) of pixels */
		for(bit = 1U << 15; --bits >= 0; bit >>= 1) {

			value =  (*gray++ * 307UL);
			value += (*gray++ * 599UL);
			value += (*gray++ * 118UL);
			if(Graymap[value >> 10] < *mat++) {
				result |= bit;
			}
		}
		*mono++ = ntohs(result);
	}

	Matrixrow = (Matrixrow + 1) & 15;
	return line + src->unitsize * src->upl;
}


#ifdef ATARI_SERVER

/* ------------------------------------------------------------
 * Atari specific packed<->direct8 conversions
 */

/* convert a packed bitmap into a monochrome one, all colors besides
 * one with index zero, will be treated as `black' ie. there's no
 * dithering.
 */
static uchar *
packed2mono (BITMAP *src, BITMAP *dst, uchar *line)
{
	short height, count, planes, plane, shorts, skip;
	register ushort *dptr, *sptr;

	planes = src->planes;
	shorts = src->upl / planes;
	skip = (src->width + (M_UNIT*8-1)) / 16 - shorts;
	height = src->height;

	sptr = (ushort *)line;
	dptr = dst->data;

	planes--;
	while (--height >= 0) {
		count = shorts;
		while (--count >= 0) {
			*dptr = *sptr++;
			plane = planes;
			while (--plane >= 0) {
				*dptr |= *sptr++;
			}
			dptr++;
		}
		dptr += skip;
	}
	return line + src->unitsize * src->upl;
}


/*
 * these functions convert single lines of DIRECT8 data to PACKED data and
 * vice versa (either 8, 4, 2 or 1 target plane).
 *
 * 'dptr' is long so that we can convert 16 pixels on two planes
 * at the time.
 */

static uchar *
direct8packed (BITMAP *src, BITMAP *dst, uchar *line)
{
	uchar *sptr = (uchar *)line;
	ulong *dptr = (ulong *)dst->data;
	short count = dst->width;
	ulong d0 = 0, d1 = 0, d2 = 0, d3 = 0;
	register ulong dbitlo = 0x80000000, dbithi = 0x00008000;
	register uchar sval;

	while (--count >= 0) {

		if ((sval = *sptr++) & 0x80)
			d0 |= dbitlo;
		if (sval & 0x40)
			d0 |= dbithi;
		if (sval & 0x20)
			d1 |= dbitlo;
		if (sval & 0x10)
			d1 |= dbithi;
		if (sval & 0x08)
			d2 |= dbitlo;
		if (sval & 0x04)
			d2 |= dbithi;
		if (sval & 0x02)
			d3 |= dbitlo;
		if (sval & 0x01)
			d3 |= dbithi;

		dbitlo >>= 1;
		if (!(dbithi >>= 1)) {
			dbitlo = 0x80000000;
			dbithi = 0x00008000;

			*dptr++ = htonl(d0);
			*dptr++ = htonl(d1);
			*dptr++ = htonl(d2);
			*dptr++ = htonl(d3);
			d0 = d1 = d2 = d3 = 0;
		}
	}

	if (dbithi != 0x00008000) {
		*dptr++ = htonl(d0);
		*dptr++ = htonl(d1);
		*dptr++ = htonl(d2);
		*dptr = htonl(d3);
	}
	return line + src->unitsize * src->upl;
}


static uchar *
direct4packed (BITMAP *src, BITMAP *dst, uchar *line)
{
	uchar *sptr = (uchar *)line;
	ulong *dptr = (ulong *)dst->data;
	short count = dst->width;
	ulong d0 = 0, d1 = 0;
	register ulong dbitlo = 0x80000000, dbithi = 0x00008000;
	register uchar sval;

	while (--count >= 0) {

		if ((sval = *sptr++) & 0x08)
			d0 |= dbitlo;
		if (sval & 0x04)
			d0 |= dbithi;
		if (sval & 0x02)
			d1 |= dbitlo;
		if (sval & 0x01)
			d1 |= dbithi;

		dbitlo >>= 1;
		if (!(dbithi >>= 1)) {
			dbitlo = 0x80000000;
			dbithi = 0x00008000;
			*dptr++ = htonl(d0);
			*dptr++ = htonl(d1);
			d0 = d1 = 0;
		}
	}

	if (dbithi != 0x00008000) {
		*dptr++ = htonl(d0);
		*dptr = htonl(d1);
	}
	return line + src->unitsize * src->upl;
}


static uchar *
direct2packed (BITMAP *src, BITMAP *dst, uchar *line)
{
	uchar *sptr = (uchar *)line;
	ulong *dptr = (ulong *)dst->data;
	short count = dst->width;
	ulong d0 = 0;
	register ulong dbitlo = 0x80000000, dbithi = 0x00008000;
	register uchar sval;

	while (--count >= 0) {

		if ((sval = *sptr++) & 0x02)
			d0 |= dbitlo;
		if (sval & 0x01)
			d0 |= dbithi;

		dbitlo >>= 1;
		if (!(dbithi >>= 1)) {
			dbitlo = 0x80000000;
			dbithi = 0x00008000;
			*dptr++ = htonl(d0);
			d0 = 0;
		}
	}

	if (dbithi != 0x00008000) {
		*dptr = htonl(d0);
	}
	return line + src->unitsize * src->upl;
}

/* 'sptr' is long so that we can convert 16 pixels on two planes
 * at the time.
 */
static uchar *
packed8direct (BITMAP *src, BITMAP *dst, uchar *line)
{
	ulong *sptr = (ulong *)line;
	uchar *dptr = (uchar *)dst->data;
	short count = dst->width;
	ulong d0 = 0, d1 = 0, d2 = 0, d3 = 0;
	register ulong himask = 0, lomask = 0;
	register uchar d;

	while (--count >= 0) {
		if (!lomask) {
			d0 = ntohl(*sptr++);
			d1 = ntohl(*sptr++);
			d2 = ntohl(*sptr++);
			d3 = ntohl(*sptr++);
			himask = 0x80000000;
			lomask = 0x00008000;
		}
		d = 0;
		if (d0 & himask)
			d |= 0x80;
		if (d0 & lomask)
			d |= 0x40;
		if (d1 & himask)
			d |= 0x20;
		if (d1 & lomask)
			d |= 0x10;
		if (d2 & himask)
			d |= 0x08;
		if (d2 & lomask)
			d |= 0x04;
		if (d3 & himask)
			d |= 0x02;
		if (d3 & lomask)
			d |= 0x01;
		*dptr++ = d;
		himask >>= 1;
		lomask >>= 1;
	}
	return line + src->unitsize * src->upl;
}


static uchar *
packed4direct (BITMAP *src, BITMAP *dst, uchar *line)
{
	ulong *sptr = (ulong *)line;
	uchar *dptr = (uchar *)dst->data;
	short count = dst->width;
	ulong d0 = 0, d1 = 0;
	register ulong himask = 0, lomask = 0;
	register uchar d;

	while (--count >= 0) {
		if (!lomask) {
			d0 = ntohl(*sptr++);
			d1 = ntohl(*sptr++);
			himask = 0x80000000;
			lomask = 0x00008000;
		}
		d = 0;
		if (d0 & himask)
			d |= 0x08;
		if (d0 & lomask)
			d |= 0x04;
		if (d1 & himask)
			d |= 0x02;
		if (d1 & lomask)
			d |= 0x01;
		*dptr++ = d;
		himask >>= 1;
		lomask >>= 1;
	}
	return line + src->unitsize * src->upl;
}


static uchar *
packed2direct (BITMAP *src, BITMAP *dst, uchar *line)
{
	ulong *sptr = (ulong *)line;
	uchar *dptr = (uchar *)dst->data;
	short count = dst->width;
	register ulong d0 = 0;
	register ulong himask = 0, lomask = 0;
	register uchar d;

	while (--count >= 0) {
		if (!lomask) {
			d0 = ntohl(*sptr++);
			himask = 0x80000000;
			lomask = 0x00008000;
		}
		d = 0;
		if (d0 & himask) {
			d |= 0x02;
		}
		if (d0 & lomask) {
			d |= 0x01;
		}
		*dptr++ = d;
		himask >>= 1;
		lomask >>= 1;
	}
	return line + src->unitsize * src->upl;
}


/* ------------------------------------------------------------
 * mono->color conversions
 */

/* on types with palette, bits in mono bitmaps are set to color index 1.
 * this expects both bitmaps to be at least short aligned.
 */
static uchar *
mono2packed (BITMAP *src, BITMAP *dst, uchar *line)
{
	ushort *sptr = (ushort *)line;
	register ushort *dptr = dst->data;
	register short plane;
	short planes = dst->planes;
	short count = (dst->width + 15) >> 4;

	while (--count >= 0) {
		*dptr++ = *sptr++;
		plane = planes;
		while (--plane > 0) {
			*dptr++ = 0;
		}
	}
	return line + src->unitsize * src->upl;
}

#endif /* ATARI_SERVER */

static uchar *
mono2direct (BITMAP *src, BITMAP *dst, uchar *line)
{
	uchar *sptr = line;
	uchar *dptr = dst->data;
	short count = dst->width;
	uchar d0 = *sptr++;
	register uchar mask = 0x80;

	while (--count >= 0) {
		if (!mask) {
			d0 = *sptr++;
			mask = 0x80;
		}
		if (d0 & mask) {
			*dptr++ = 0x01;
		} else {
			*dptr++ = 0x00;
		}
		mask >>= 1;
	}
	return line + src->unitsize * src->upl;
}


static uchar *
mono2truecolor (BITMAP *src, BITMAP *dst, uchar *line)
{
	uchar *sptr = line;
	register uchar *dptr = dst->data;
	register uchar mask = 0x80;
	short count = dst->width;
	uchar d0 = *sptr++;

	while (--count >= 0) {
		if (!mask) {
			d0 = *sptr++;
			mask = 0x80;
		}
		if (d0 & mask) {
			*dptr++ = 0xff;
			*dptr++ = 0xff;
			*dptr++ = 0xff;
		} else {
			*dptr++ = 0x00;
			*dptr++ = 0x00;
			*dptr++ = 0x00;
		}
		mask >>= 1;
	}
	return line + src->unitsize * src->upl;
}

/* ------------------------------------------------------------
 * other 'conversions'
 */

static uchar *
direct2truecolor (BITMAP *src, BITMAP *dst, uchar *line)
{
	uchar *sptr = line;
	register uchar *dptr = dst->data;
	register uchar *rgb;
	short count = dst->width;
	rgb_t *color = src->palette;

	while (--count >= 0) {
		rgb = &color[*sptr++].red;
		*dptr++ = *rgb++;
		*dptr++ = *rgb++;
		*dptr++ = *rgb;
	}
	return line + src->unitsize * src->upl;
}


static uchar *
copy_bm (BITMAP *src, BITMAP *dst, uchar *line)
{
	size_t count = src->unitsize * src->upl;
	memcpy(dst->data, line, count);
	return line + count;
}


/* change destination palette to monochrome */
static void
set_monopal (BITMAP *dst)
{
	rgb_t *rgb;

	if (dst->palette) {
		/* color zero is white */
		rgb = dst->palette;
		rgb->red   = 255;
		rgb->green = 255;
		rgb->blue  = 255;

		/* color one is black */
		rgb++;
		rgb->red   = 0;
		rgb->green = 0;
		rgb->blue  = 0;

		dst->colors = 2;
	}
}

/* ------------------------------------------------------------
 * conversion initialization
 */

/* check that argument BITMAP unitsizes equal ones used here */
static int
wrong_type(BITMAP *bm)
{
	if ((bm->type == BM_PACKEDMONO  && bm->unitsize != M_UNIT) ||
	    (bm->type == BM_PACKEDCOLOR && bm->unitsize != 2)      ||
	    (bm->type == BM_DIRECT8     && bm->unitsize != 1)      ||
	    (bm->type == BM_DIRECT24    && bm->unitsize != 3)) {
	    	return -1;
	}
	return 0;	/* OK */
}


/* generic conversion routine */
uchar *
(*w_convertFunction (BITMAP *src, BITMAP *dst)) (BITMAP *, BITMAP *, uchar *)
{
	char *error = "request for unknown BITMAP type";

	/* first checks */
	if (!(src && dst && src->data && dst->data)) {
		TRACEPRINT(("w_convertFunction(%p,%p) -> no bitmap\n",src,dst));
		return NULL;
	}
	if(wrong_type(src) || wrong_type(dst)) {
		/* this should be known regardless of tracing */
		fprintf(stderr, "w_convertFunction(%p,%p) -> wrong unitsize",
			src, dst);
		return NULL;
	}

	switch (dst->type) {

	    case BM_PACKEDMONO:
		switch (src->type) {
		    case BM_PACKEDMONO:
			return copy_bm;

		    case BM_DIRECT8:
			init_dither(src);
			return odither_direct;

		    case BM_DIRECT24:
			init_dither(src);
			return odither_truecolor;

#ifdef ATARI_SERVER
		    case BM_PACKEDCOLOR:
			/* non-background color -> black */
			return packed2mono;
#endif /* ATARI_SERVER */
		}
		break;

#ifdef ATARI_SERVER
	    case BM_PACKEDCOLOR:
		switch (src->type) {
		    case BM_PACKEDMONO:
			set_monopal(dst);
			return mono2packed;

		    case BM_PACKEDCOLOR:
			if (src->planes == dst->planes) {
				return copy_bm;
			}
			/* Only server can be packed color so packed bitmaps
			 * should always have the same number of planes!
			 */
			 error = "illegal conversion: PACKEDCOLOR -> PACKEDCOLOR";
			break;

		    case BM_DIRECT8:
			switch(dst->planes) {
			    case 2:
				return direct2packed;
			    case 4:
				return direct4packed;
			    case 8:
				return direct8packed;
			}
			error = "unsupported number of DIRECT8 -> PACKEDCOLOR planes";
			break;

		    case BM_DIRECT24:
			/* where the image data (=colors) -> palette
			 * conversion would be done?
			 */
			error = "DIRECT24 -> PACKEDCOLOR not implemented";
			break;
		}
		break;
#endif /* ATARI_SERVER */

	    case BM_DIRECT8:
		switch (src->type) {
		    case BM_PACKEDMONO:
			set_monopal(dst);
			return mono2direct;

#ifdef ATARI_SERVER
		    case BM_PACKEDCOLOR:
			switch(src->planes) {
			    case 2:
				return packed2direct;
			    case 4:
				return packed4direct;
			    case 8:
				return packed8direct;
			}
			error = "unsupported number of PACKEDCOLOR -> DIRECT8 planes";
			break;
#endif /* ATARI_SERVER */

		    case BM_DIRECT8:
			return copy_bm;

		    case BM_DIRECT24:
			/* where the image data (=colors) -> palette
			 * conversion would be done? Graycscale conversion
			 * would be easy...
			 */
			error = "DIRECT24 -> DIRECT8 not yet implemented";
			break;
		}
		break;

	    case BM_DIRECT24:
		/* these would be fairly easy to implement, but I don't see
		 * much point in it.  We haven't got a TC server and apps
		 * manipulating TC images surely do graphics operations
		 * themselves, instead of getting bitmaps from the server...
		 */
		switch (src->type) {
		    case BM_PACKEDMONO:
			return mono2truecolor;

		    case BM_DIRECT8:
			return direct2truecolor;

		    case BM_DIRECT24:
			return copy_bm;

		    default:
			error = "DIRECT24 destination type not yet implemented";
			break;
		}
		break;
	}

	/* this should be known regardless of tracing */
	fprintf(stderr, "w_convertFunction(%p,%p): %s", src, dst, error);
	return NULL;
}


/* ------------------------------------------------------------ */

/* this one allocates a bitmap which is perfectly aligned for directly
 * sending it to the server.  For example DIRECT8 has line lengths padded to
 * a multiple of 4 so that they end on a long boundary and PACKED ones have
 * line lengths padded to a multiple of 16 so that they end on a short
 * boundary.  applications filling in data must know and obey this (means
 * just using `unitsize' and `upl' members).
 *
 * NOTE: PACKEDCOLOR will always have as many planes as W server
 * and hence palette size is same too (fixed if differs).
 */
BITMAP *
w_allocbm (short width, short height, short type, short colors)
{
	BITMAP *bm;

	TRACESTART();
	if (type != BM_PACKEDMONO &&
	    type != BM_PACKEDCOLOR &&
	    type != BM_DIRECT8 &&
	    type != BM_DIRECT24) {
		TRACEPRINT(("w_allocbm(%d,%d,%d,%d) -> illegal bm type\n",\
		           width, height, type, colors));
		TRACEEND();
		return NULL;
	}
	if (width < 1 || height < 1) {
		TRACEPRINT(("w_allocbm(%d,%d,%d,%d) -> illegal bm size\n",\
		           width, height, type, colors));
		TRACEEND();
		return NULL;
	}
	/* bogus color count (too large is fixed to suitable value) */
	if ((type == BM_PACKEDCOLOR || type ==BM_DIRECT8) && colors < 2) {
		TRACEPRINT(("w_allocbm(%d,%d,%d,%d) -> illegal colors\n",\
		           width, height, type, colors));
		TRACEEND();
		return NULL;
	}

	/* alloc and _zero_ BITMAP variables */
	if (!(bm = calloc(1, sizeof(BITMAP)))) {
		TRACEPRINT(("w_allocbm(%d,%d,%d,%d) -> bm alloc failed\n",\
		           width, height, type, colors));
		TRACEEND();
		return NULL;
	}
	bm->type = type;
	bm->width = width;
	bm->height = height;

	/* set type specific variables */
	switch (type) {

	case BM_PACKEDMONO:
		bm->upl = (width + 31) / 32;
		bm->unitsize = 4;		/* align width to long */
		bm->colors = 2;
		bm->planes = 1;
		break;

	case BM_PACKEDCOLOR:
		bm->colors = colors;
		bm->planes = _wserver.planes;
		colors = 1 << _wserver.planes;
		if (bm->colors > colors) {
			bm->colors = colors;
		}
		bm->upl = (((width + 15) & ~15) * bm->planes) / 16;
		bm->unitsize = 2;		/* align width to short */
		break;

	case BM_DIRECT8:
		bm->upl = (width + 3) & ~3;
		bm->unitsize = 1;
		bm->planes = 8;
		if (colors > 256) {
			colors = 256;
		}
		bm->colors = colors;
		break;

	case BM_DIRECT24:
		bm->upl = width;
		bm->unitsize = 3;		/* align width to rgb_t */
		bm->planes = 24;
		bm->colors = colors;
		break;
	}

	if (!(bm->data = malloc(bm->unitsize * bm->upl * height))) {
		free(bm);
		TRACEPRINT(("w_allocbm(%d,%d,%d,%d) -> data alloc failed\n",\
		           width, height, type, colors));
		TRACEEND();
		return NULL;
	}

	if (type == BM_DIRECT8 || type == BM_PACKEDCOLOR) {
		if (!(bm->palette = malloc(colors * sizeof(rgb_t)))) {
			free(bm->data);
			free(bm);
			TRACEPRINT(("w_allocbm(%d,%d,%d,%d) -> palette alloc failed\n",\
			           width, height, type, colors));
			TRACEEND();
			return NULL;
		}
	}
	TRACEPRINT(("w_allocbm(%d,%d,%d,%d) -> %p\n",\
	           width, height, type, colors, bm));
	TRACEEND();
	return bm;
}


BITMAP *
w_copybm(BITMAP *bm)
{
	BITMAP *new;

	TRACESTART();

	if (!bm) {
		TRACEPRINT(("w_dublicatebm(NULL) -> no BITMAP\n"));
		TRACEEND();
		return NULL;
	}

	if (!(new = w_allocbm(bm->width, bm->height, bm->type, bm->colors))) {
		TRACEPRINT(("w_dublicatebm(%p) -> bitmap allocation failed\n",\
		           bm));
		TRACEEND();
		return NULL;
	}
	memcpy(new->data, bm->data, bm->unitsize * bm->upl * bm->height);
	if (new->palette) {
		memcpy(new->palette, bm->palette, bm->colors * sizeof(rgb_t));
	}
	return new;
}

/* DIRECT24 BITMAP conversions goes through PACKEDMONO and takes therefore
 * a bit more memory...
 */
BITMAP *
w_convertBitmap (BITMAP *bm, short type, short colors)
{
	char *cptr;
	BITMAP *use = NULL, *old = bm;
	uchar *data, *line, *(*conv)(BITMAP *, BITMAP *, uchar *);
	int width, height;

	TRACESTART();

	if (!bm) {
		cptr = "no BITMAP";
		goto error;
	}

	if (bm->type == type && bm->colors == colors) {
		return bm;
	}

	if (bm->type == BM_DIRECT24 ||
	    (bm->type == BM_DIRECT8 && type == BM_PACKEDMONO)) {

		/* line-by-line conversion uses inferior ordered
		 * dithering as that's independent of other pixels,
		 * but here we can use fs-dither.
		 */
		if (!(bm = fs_direct2mono(bm, 0, &cptr))) {
			goto error;
		}
		if (type == BM_PACKEDMONO) {
			TRACEPRINT(("w_convertBitmap(%p,%i,%i) -> %p\n",\
				bm, type, colors, bm));
			TRACEEND();
			return bm;
		}
	}

	width = bm->width;
	height = bm->height;

	if (!colors) {
		/* set explicitly */
		if (bm->type == BM_PACKEDMONO) {
			colors = 2;
		} else {
			if (type == BM_DIRECT8) {
				colors = 256;
			} else if (type == BM_PACKEDCOLOR) {
				colors = 1 << _wserver.planes;
			}
		}
	}

	if (!(use = w_allocbm(width, height, type, colors))) {
		cptr = "work bitmap allocation failed";
		goto error;
	}

	if (!(conv = w_convertFunction(bm, use))) {
		cptr = "no bitmap conversion function";
		goto error;
	}

	/* transfer as much of the palette as possible, a real conversion
	 * would really not belong here...
	 */
	if (bm->palette && use->palette) {
		colors = (bm->colors < use->colors ? bm->colors : use->colors);
		memcpy(use->palette, bm->palette, colors * sizeof(rgb_t));
		use->colors = colors;
	}

	data = use->data;
	line = bm->data;

	while (--height >= 0) {

		line = (*conv)(bm, use, line);
		(uchar *)use->data += use->unitsize * use->upl;
	}

	if (bm && bm != old) {
		/* free temporary monochrome bitmap */
		w_freebm(bm);
	}
	use->data = data;

	TRACEPRINT(("w_convertBitmap(%p,%i,%i) -> %p\n",
		bm, type, colors, use));
	TRACEEND();
	return use;

error:
	if (use) {
		w_freebm(use);
	}
	if (bm && bm != old) {
		w_freebm(bm);
	}
	TRACEPRINT(("w_convertBitmap(%p,%i,%i) -> %s\n",\
		bm, type, colors, cptr));
	TRACEEND();
	return NULL;
}


void
w_freebm(BITMAP *bm)
{
	TRACESTART();
	if (bm) {
		if (bm->palette) {
			free(bm->palette);
			bm->palette = NULL;	/* just to be sure... */
		}
		if (bm->data) {
			free(bm->data);
			bm->data = NULL;
		}
		free(bm);
		TRACEPRINT(("w_freebm(%p)\n", bm));
	} else {
		TRACEPRINT(("w_freebm(%p) -> no BITMAP\n", bm));
	}
	TRACEEND();
}

